
#ifndef IIRSCRAM_STATEMENT_HH
#define IIRSCRAM_STATEMENT_HH

// Copyright (c) 1996-2001 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO
// THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
// PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES
// SUFFERED BY LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING
// OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found
// in the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer
//          Malolan Chetlur    
//          Umesh Kumar V. Rajasekaran

//---------------------------------------------------------------------------

#include "IIRBase_Statement.hh"

template <class type> class dl_list;
class IIRScram_WaitStatement;
class IIR_StatementList;
class IIR_Statement;
class IIR_TypeDefinition;
class IIR_SubprogramDeclaration;
class IIR_Name;
class IIR_List;
class IIR_CaseStatementAlternativeList;
class IIR_WaveformList;
class IIR_SimpleName;

class IIRScram_Statement : public IIRBase_Statement {

public:
  virtual set<IIR_Declaration> *_find_declarations( IIR_Name * );
  virtual IIR_Boolean _has_statement_list();

  IIR_Boolean _is_iir_statement(){ return TRUE; }
  /** Returns true if this is an IIR_AssertionStatement or an
     IIR_ConcurrentAssertionStatement.
  */
  virtual IIR_Boolean _is_assert_statement(){ return false; }
  
  virtual void _type_check();
					
  // This method does the type checking and resolution for a boolean
  // condition.  Pass in a conditional, and it will return it
  // resolved.  It will report any errors it finds (like that value
  // not being boolean).  This method is called by IIR_IfStatement,
  // IIR_ElsifStatement, and IIR_WhileLoopStatement...  (For it to be called 
  // from IIR_Elsif, this must be public!
  static IIR *_type_check_and_resolve_boolean_condition( IIR *condition );

  // These virtual methods are needed to support the type checking functions in the
  // protected section
  virtual IIR *_get_target();
  virtual void _set_target( IIR * );

  virtual IIR_DelayMechanism _get_delay_mechanism();
  virtual void _set_delay_mechanism( IIR_DelayMechanism );

  virtual IIR *_get_reject_time_expression();
  virtual void _set_reject_time_expression( IIR * );
	
  // These methods are needed for the type checking of case statements that occurs in
  // IIR_CaseStatement, and IIR_ConcurrentSelectedSignalAssignment.
  virtual IIR *_get_case_statement_expression();
  virtual void _set_case_statement_expression( IIR * );

  /** This method overloads the generic access method
      IIRScram::_get_declarator() to return an declarator associated with the
      IIR hierarchy derived from "this" class.  The declarator associated with a
      Statement is its lablel.  Hence, this method merely returns the mangled
      label associated with this class [obtained by a call to
      IIRScram_Statement::_get_label()->_get_declarator].

      @return A IIR_TextLiteral* that points to the mangled declarator
      associated with this class.  The caller must not modify or delete the
      pointer returned by a call to this method virtual
  */
  IIR_TextLiteral* _get_declarator();

  /** This method is defined only in IIRScram_Statement and is valid for all IIR
      hierarchies derived from IIR_Statement.  This method returns a pointer to
      the mangled label associated with this IIR tree.  Mangling of labels is
      "lazy" .ie. a mangled label is obtained (by a call to
      IIR_Statement::_mangle_label()) during the first call to this method (for
      each IIR tree) and is saved [in the _mangled_label class member object].
      The saved mangled-label is used in subsequent calls to this method.

      @return A IIR_Label* that points to a mangled label associated with the
      IIR hierarchy derived from "this" node.  The caller must not modify or
      delete the pointer returned by a call to this method.
  */
  virtual IIR_Label* _get_label() const;

  /** For assertion statements this method is overloaded to return the
      assertion condtion.  For anything else it returns 0. */
  virtual IIR *_get_assertion_condition(){ return 0; }
  
  /** For report and assertion statements this method is overloaded to
      return the report expression.  For anything else it returns 0. */
  virtual IIR *_get_report_expression(){ return 0; }
  /** For report and assertion statements this method is overloaded to
      return the report expression. For anything else it returns 0. */
  virtual IIR *_get_severity_expression(){ return 0; }

  /**  This is overloaded for assertion and report statements.  Generate
      runtime error for anything else.  */
  virtual void _set_assertion_condition( IIR * );

  /**  This is overloaded for assertion and report statements.  Generate
      runtime error for anything else.  */
  virtual void _set_report_expression( IIR * );

  /**  This is overloaded for assertion and report statements.  Generate
      runtime error for anything else.  */
  virtual void _set_severity_expression( IIR * );

  /**
     Entry into publishing routines.
  */
  virtual void _publish_cc( published_file &_cc_out ) = 0;

  /** All of the assertion-like statements use this method for
      publishing.  This includes assertion statements, report
      statements, and concurrent assertion statements. */
  void _publish_cc_assertion_or_report_statement( published_file &_cc_out );

  /** By default this method publishes code for the expression of an
      assertion statement.  For report statements, it's overridden to print
      "true".  */
  virtual void _publish_cc_assertion_condition( published_file &_cc_out );

  /** If this node returns something non-NULL from
      _get_assertion_condition(), this method will end the conditional that 
      _publish_cc_assertion_condition started. */
  void _publish_cc_assertion_condition_end( published_file &_cc_out );


  ostream &_print( ostream & );

  virtual IIR_Label *_find_instantiate_label( IIR_SimpleName * );

  virtual void _make_interface_visible( symbol_table * );

protected:

  IIRScram_Statement() { _mangled_label = NULL; }
  virtual ~IIRScram_Statement() = 0;

  // These methods are used in the type checking of signal assignment statements,
  // and concurrent selected signal assignment statements.
  void _type_check_target_and_waveform( IIR_WaveformList * );
  void _type_check_mechanism_and_time( IIR_WaveformList * );

  // This method handles the type checking of case statement expressions and their 
  // alternatives.  The return value is the resolved expression (if there is no error),
  // and NULL if there is an error.  The list is type checked in place.
  void _type_check_case_statement_expression( IIR_CaseStatementAlternativeList * );

  // The following methods apply to assertion statements, concurrent assertion
  // statements, and report statements.
  void _type_check_assertion_condition();
  void _type_check_report_expression();
  void _type_check_severity_expression();

  // This method prints the statement label for all statements.
  void _publish_vhdl_stmt_label(ostream &);

  // This method prints the delay mechanism
  // (transport/reject...inertial) for all statements. 
  void _publish_vhdl_delay_mechanism(ostream &, IIR_DelayMechanism, IIR *);

  virtual void _mangle_label() const;
  virtual void _set_label(IIR_Label*);
  
private:
  IIR_Label *_mangled_label;
};
#endif
