/* ---------------------------------------------------------------------
*
*  -- ScaLAPACK routine (version 1.0) --
*     University of Tennessee, Knoxville, Oak Ridge National Laboratory,
*     and University of California, Berkeley.
*     November 17, 1996
*
*  ---------------------------------------------------------------------
*/
/*
*  Include files
*/
#include "pblas.h"

void pdsyr_( uplo, n, alpha, X, ix, jx, desc_X, incx, A, ia, ja,
             desc_A )
/*
*  .. Scalar Arguments ..
*/
   F_CHAR      uplo;
   int         * ia, * incx, * ix, * ja, * jx, * n;
   double      * alpha;
/*  ..
*   .. Array Arguments ..
*/
   int          desc_A[], desc_X[];
   double      A[], X[];
{
/*
*  Purpose
*  =======
*
*  PDSYR   performs the symmetric rank 1 operation
*
*     sub( A ) := alpha*sub( X )*sub( X )' + sub( A ),
*
*  where sub( A ) denotes A(IA:IA+N-1,JA:JA+N-1),
*
*        sub( X ) denotes X(IX,JX:JX+N-1) if INCX = M_X,
*                         X(IX:IX+N-1,JX) if INCX = 1 and INCX <> M_X,
*
*  alpha is a real scalar, sub( X ) is an N element distributed vector
*  and sub( A ) is an N-by-N distributed symmetric matrix.
*
*  Notes
*  =====
*
*  Each global data object is described by an associated description
*  vector.  This vector stores the information required to establish
*  the mapping between an object element and its corresponding process
*  and memory location.
*
*  Let A be a generic term for any 2D block cyclicly distributed array.
*  Such a global array has an associated description vector descA.
*  In the following comments, the character _ should be read as
*  "of the global array".
*
*  NOTATION        STORED IN      EXPLANATION
*  --------------- -------------- --------------------------------------
*  DT_A   (global) descA[ DT_ ]   The descriptor type.  In this case,
*                                 DT_A = 1.
*  CTXT_A (global) descA[ CTXT_ ] The BLACS context handle, indicating
*                                 the BLACS process grid A is distribu-
*                                 ted over. The context itself is glo-
*                                 bal, but the handle (the integer
*                                 value) may vary.
*  M_A    (global) descA[ M_ ]    The number of rows in the global
*                                 array A.
*  N_A    (global) descA[ N_ ]    The number of columns in the global
*                                 array A.
*  MB_A   (global) descA[ MB_ ]   The blocking factor used to distribu-
*                                 te the rows of the array.
*  NB_A   (global) descA[ NB_ ]   The blocking factor used to distribu-
*                                 te the columns of the array.
*  RSRC_A (global) descA[ RSRC_ ] The process row over which the first
*                                 row of the array A is distributed.
*  CSRC_A (global) descA[ CSRC_ ] The process column over which the
*                                 first column of the array A is
*                                 distributed.
*  LLD_A  (local)  descA[ LLD_ ]  The leading dimension of the local
*                                 array.  LLD_A >= MAX(1,LOCr(M_A)).
*
*  Let K be the number of rows or columns of a distributed matrix,
*  and assume that its process grid has dimension p x q.
*  LOCr( K ) denotes the number of elements of K that a process
*  would receive if K were distributed over the p processes of its
*  process column.
*  Similarly, LOCc( K ) denotes the number of elements of K that a
*  process would receive if K were distributed over the q processes of
*  its process row.
*  The values of LOCr() and LOCc() may be determined via a call to the
*  ScaLAPACK tool function, NUMROC:
*          LOCr( M ) = NUMROC( M, MB_A, MYROW, RSRC_A, NPROW ),
*          LOCc( N ) = NUMROC( N, NB_A, MYCOL, CSRC_A, NPCOL ).
*  An upper bound for these quantities may be computed by:
*          LOCr( M ) <= ceil( ceil(M/MB_A)/NPROW )*MB_A
*          LOCc( N ) <= ceil( ceil(N/NB_A)/NPCOL )*NB_A
*
*  Because vectors may be seen as particular matrices, a distributed
*  vector is considered to be a distributed matrix.
*
*  If INCX = M_X, the process column having the first entries of sub( X )
*  must also contain the first block of sub( A ), similarly, if INCX = 1
*  and INCX <> M_X, the process row having the first entries of sub( X )
*  must also contain the first block of sub( A ).  It is also required
*  that MB_A = NB_A and MOD(IA-1,MB_A) = MOD(JA-1,NB_A).
*
*  If INCX = M_X (resp. (INCX = 1 and INCX <> M_X), the column (resp.
*  row) offset of sub( X ) must be equal to the row offset of sub( A ),
*  i.e MOD(JX-1,NB_X) (resp. MOD(IX-1,MB_X)) = MOD(IA-1,MB_A). Moreover,
*  the column (resp. row) blocksize of X must be equal to the column
*  blocksize of A, i.e NB_X = NB_A (resp. MB_X = NB_A).
*
*  Parameters
*  ==========
*
*  UPLO    (global input) pointer to CHARACTER
*          On entry, UPLO specifies whether the upper or lower
*          triangular part of the distributed matrix sub( A ) is to be
*          referenced as follows:
*
*             UPLO = 'U' or 'u'   Only the upper triangular part of
*                                 sub( A ) is to be referenced.
*
*             UPLO = 'L' or 'l'   Only the lower triangular part of
*                                 sub( A ) is to be referenced.
*
*  N       (global input) pointer to INTEGER
*          The order of the distributed matrix sub( A ). N >= 0.
*
*  ALPHA   (global input) pointer to DOUBLE PRECISION
*          On entry, ALPHA specifies the scalar alpha.
*
*  X       (local input) DOUBLE PRECISION array containing the local
*          pieces of a distributed matrix of dimension of at least
*                  ( (JX-1)*M_X + IX + ( N - 1 )*abs( INCX ) )
*          This array contains the entries of the distributed vector
*          sub( X ).
*
*  IX      (global input) pointer to INTEGER
*          The global row index of the submatrix of the distributed
*          matrix X to operate on.
*
*  JX      (global input) pointer to INTEGER
*          The global column index of the submatrix of the distributed
*          matrix X to operate on.
*
*  DESCX   (global and local input) INTEGER array of dimension 8.
*          The array descriptor of the distributed matrix X.
*
*  INCX    (global input) pointer to INTEGER
*          The global increment for the elements of X. Only two values
*          of INCX are supported in this version, namely 1 and M_X.
*
*  A       (local input/local output) DOUBLE PRECISION pointer into the
*          local memory to an array of dimension (LLD_A,LOCc(JA+N-1).
*          Before entry with  UPLO = 'U' or 'u', the leading N-by-N
*          upper triangular part of the distributed matrix sub( A )
*          must contain the upper triangular part of the symmetric
*          distributed matrix and the strictly lower triangular part
*          of sub( A ) is not referenced. On exit, the upper triangular
*          part of sub( A ) is overwritten by the upper triangular
*          part of the updated distributed matrix.
*          Before entry with UPLO = 'L' or 'l', the leading N-by-N
*          lower triangular part of the distributed matrix sub( A )
*          must contain the lower triangular part of the symmetric
*          distributed matrix and the strictly upper triangular part
*          of sub( A ) is not referenced. On exit, the lower triangular
*          part of sub( A ) is overwritten by the lower triangular
*          part of the updated distributed matrix.
*
*  IA      (global input) pointer to INTEGER
*          The global row index of the submatrix of the distributed
*          matrix A to operate on.
*
*  JA      (global input) pointer to INTEGER
*          The global column index of the submatrix of the distributed
*          matrix A to operate on.
*
*  DESCA   (global and local input) INTEGER array of dimension 8.
*          The array descriptor of the distributed matrix A.
*
*  =====================================================================
*
*  .. Local Scalars ..
*/
   char        * top, UploA, xdist;
   int         iacol, iarow, icoffa, ictxt, iia, iix, info, iroffa,
               ixcol, ixpos, ixrow, iszcmp, jja, jjx, lcm, lcmp, lcmq,
               locincx, mullen, mycol, myrow, nca, nn, np0, nprow,
               npcol, nra, nq0, szcmp, tmp1, tmp2, wksz;
/* ..
*  .. PBLAS Buffer ..
*/
   double      * buff;
/* ..
*  .. External Functions ..
*/
   void        blacs_gridinfo_();
   void        pbchkmat();
   void        pbchkvect();
   void        pberror_();
   char        * getpbbuf();
   char        * ptop();
   F_VOID_FCT  pbdsyr_();
   F_INTG_FCT  ilcm_();
/* ..
*  .. Executable Statements ..
*
*  Get grid parameters
*/
   ictxt = desc_X[CTXT_];
   blacs_gridinfo_( &ictxt, &nprow, &npcol, &myrow, &mycol );
/*
*  Test the input parameters
*/
   info = 0;
   if( nprow == -1 )
      info = -(700+CTXT_+1);
   else
   {
      UploA = Mupcase( F2C_CHAR( uplo )[0] );
      iroffa = (*ia-1) % desc_A[MB_];
      icoffa = (*ja-1) % desc_A[NB_];
      pbchkvect( *n, 2, *ix, *jx, desc_X, *incx, 7, &iix, &jjx,
                 &ixrow, &ixcol, nprow, npcol, myrow, mycol, &info );
      pbchkmat( *n, 2, *n, 2, *ia, *ja, desc_A, 12, &iia, &jja,
                &iarow, &iacol, nprow, npcol, myrow, mycol,
                &nra, &nca, &info );
      if( info == 0 )
      {
         if( ( UploA != 'U' ) && ( UploA != 'L' ) )
            info = -1;
         else if( iroffa != icoffa )
            info = -11;
         if( *incx == desc_X[M_] )
         {
            if( ( (*jx-1) % desc_X[NB_] ) != iroffa )
               info = -10;
            else if( desc_A[MB_] != desc_X[NB_] )
               info = -(1200+MB_+1);
            else if( desc_A[MB_] != desc_A[NB_] )
               info = -(1200+NB_+1);
         }
         else if( ( *incx == 1 ) && ( *incx != desc_X[M_] ) )
         {
            if( ( ( (*ix-1) % desc_X[MB_] ) != iroffa ) ||
                ( ixrow != iarow ) )
               info = -10;
            else if( desc_A[MB_] != desc_X[MB_] )
               info = -(1200+MB_+1);
            else if( desc_A[MB_] != desc_A[NB_] )
               info = -(1200+NB_+1);
         }
         else
         {
            info = -13;
         }
         if( ictxt != desc_A[CTXT_] )
            info = -(1200+CTXT_+1);
      }
   }
   if( info )
   {
      pberror_( &ictxt, "PDSYR", &info );
      return;
   }
/*
*  Quick return if possible.
*/
   if( ( *n == 0 ) || ( *alpha == ZERO ) )
      return;
/*
*  Figure out the arguments to be passed to pbdsyr
*/
   nn = *n + iroffa;
   mullen = MULLENFAC * desc_A[MB_];
   lcm = ilcm_( &nprow, &npcol );

   if( *incx == desc_X[M_] )
   {
      lcmp = lcm / nprow;
      lcmq = lcm / npcol;
      top = ptop( BROADCAST, COLUMN, TOPGET );
      ixpos = ixrow;
      xdist = 'R';
      locincx = desc_X[LLD_];
      tmp1 = nn / desc_A[MB_];
      np0 = MYROC0( tmp1, nn, desc_A[MB_], nprow );
      tmp2 = nn / desc_A[NB_];
      nq0 = MYROC0( tmp2, nn, desc_A[NB_], npcol );
      tmp2 = lcmq * desc_A[MB_];
      iszcmp = CEIL( mullen, tmp2 );
      szcmp = iszcmp * iszcmp * tmp2 * lcmp * desc_A[MB_];
      tmp1 = np0 / desc_A[MB_];
      tmp2 = CEIL( nn, desc_A[MB_] );
      tmp1 = MYROC0( tmp1, np0, desc_A[MB_], lcmp ) * MIN( lcmp, tmp2 );
      wksz = np0 + nq0 + MAX( szcmp, tmp1 );
   }
   else
   {
      lcmp = lcm / nprow;
      lcmq = lcm / npcol;
      top = ptop( BROADCAST, ROW, TOPGET );
      ixpos = ixcol;
      xdist = 'C';
      locincx = 1;
      tmp1 = nn / desc_A[MB_];
      np0 = MYROC0( tmp1, nn, desc_A[MB_], nprow );
      tmp2 = nn / desc_A[NB_];
      nq0 = MYROC0( tmp2, nn, desc_A[NB_], npcol );
      tmp2 = lcmq * desc_A[MB_];
      iszcmp = CEIL( mullen, tmp2 );
      szcmp = iszcmp * iszcmp * tmp2 * lcmp * desc_A[MB_];
      tmp1 = nq0 / desc_A[NB_];
      tmp2 = CEIL( nn, desc_A[NB_] );
      tmp1 = MYROC0( tmp1, nq0, desc_A[NB_], lcmq ) * MIN( lcmq, tmp2 );
      wksz = np0 + nq0 + MAX( szcmp, tmp1 );
   }
   buff = (double *)getpbbuf( "PDSYR", wksz*sizeof(double) );
/*
*  Call PB-BLAS routine
*/
   pbdsyr_( &ictxt, uplo, C2F_CHAR( &xdist ), n, &desc_A[MB_], &iroffa,
            alpha, &X[iix-1+(jjx-1)*desc_X[LLD_]], &locincx,
            &A[iia-1+(jja-1)*desc_A[LLD_]], &desc_A[LLD_],
            &ixpos, &iarow, &iacol, C2F_CHAR( top ), C2F_CHAR( NO ),
            C2F_CHAR( NO ), &mullen, buff );
}
