module Schleuder
  def log
    SchleuderLogger.new unless Log4r::Logger['log4r']
    # If there's a ListLogger we use that, else the global SchleuderLogger.
    # The ListLogger is being set up in List::initialize().
    Log4r::Logger['list'] || Log4r::Logger['log4r']
  end
  module_function :log

  def config(config_file=nil)
    # read the base config (class overloads itself with conf/schleuder.conf)
    @config ||= SchleuderConfig.new(config_file)
  end
  module_function :config

  def list
    @list || nil
  end
  module_function :list

  def list=(list)
    @list = list
  end
  module_function :list=

  def origmsg=(msg)
    @origmsg = msg
  end
  module_function :origmsg=

  def origmsg
    @origmsg || nil
  end
  module_function :origmsg

  def self.listname(listname=nil)
    @listname ||= listname
  end

  class Processor
    def self.run(listname, message)
      Schleuder.listname listname
      Schleuder.origmsg = message
      Schleuder.log.debug "Testing if list-dir exists"
      unless File.directory?(List.listdir(listname))
        msg = "No such list: #{listname}"
        $stderr.puts msg
        Schleuder.log.warn msg
        exit 1
      end

      Schleuder.log.debug "Creating list"
      Schleuder.list = List.new(listname)

      Schleuder.log.debug "Parsing incoming message"
      mail = Mail.parse(message)
      Schleuder.log.info "New mail incoming from #{mail.from}"

      msize = message.size / 1024
      lsize = Schleuder.list.config.max_message_size
      if msize > lsize
        self.bounce_or_drop "too big (#{msize}kB > #{lsize}kB)", "This address accepts only messages up to #{Schleuder.list.config.max_message_size} kilobyte.", mail
      end

      #  if mail is a bounce we forward it directly to the admins, as
      #  dealing with those probably fails (encrypted attachments etc.)
      if mail.bounce?
        Schleuder.log.info "This is a bounce, forwarding to admins and exiting"
        Schleuder.log.notify_admin "Problem", [ "Hello,\n\nI caught a bounce. Please take care of it.", message ]
        Schleuder.log.info "Exiting cleanly"
        exit(0)
      end

      if Schleuder.list.config.dump_incoming_mail
        dump_dir = File.join(Schleuder.list.listdir,'dumps')
        Dir.mkdir dump_dir unless File.directory? dump_dir
        require 'digest/sha1'
        msg_file = File.join(dump_dir,"#{Digest::SHA1.hexdigest(message[0..10000])}.msg")
        Schleuder.log.info "Dumping message to #{msg_file}"
        File.open(msg_file,"w") { |f| f << message }
        Schleuder.log.notify_admin "Dump notification", "Hello,\n\nI dumped the current incoming message to #{msg_file} \n\nYou should erase that file after inspection!"
      end

      # if mail is a send-key-request we answer directly
      if mail.to.to_a.include?(Schleuder.list.sendkey_addr) || (mail.subject.strip.downcase == 'send key!' && mail.body.strip.empty?)
        Schleuder.log.info "Found send-key-request"
        # TODO: refactor with Plugin::reply
        out = Mail.new
        out.subject = "Re: #{mail.subject}"
        Schleuder.log.info "Building reply"
        iout = out.individualize(Member.new({'email' => mail.from.first}))
        iout.in_reply_to = mail.message_id
        Schleuder.log.debug "Filling body with key-id and key"
        iout.body = "#{Schleuder.list.key.to_s}\n#{mail.crypt.export(Schleuder.list.key_fingerprint).to_s}"
        Schleuder.log.debug "Signing outgoing email"
        rawmail = iout.sign
        Schleuder.log.info "Handing over to Mailer"
        Mailer.send(rawmail, iout.to)
        Schleuder.log.info "Exiting"
        exit 0
      end

      # Analyse data (hopefully an incoming mail).
      # Is it multi-part? encrypted? pgp/mime? signed?
      Schleuder.log.debug "Analysing incoming mail"
      begin
        mail.decrypt!
      rescue GPGME::Error::DecryptFailed => e
        Schleuder.log.error "#{e}\n\nOriginal message:\n#{message}"
        $stdout.puts "Schleuder speaking. Cannot decrypt. Please check your setup.\nMessages to this list need to be encrypted with this key:\n#{Schleuder.list.key}"
        exit 100
      end

      if !mail.to.nil? && mail.to.include?(Schleuder.list.owner_addr)
        Schleuder.log.info "This is a message to the owner(s), forwarding to admins and exiting"
        Schleuder.log.notify_admin "Message", [ "Hello,\n\nthe attached message is directed to you as list-owner(s). Please take care of it!\n\n", mail ]
        Schleuder.log.info "Exiting cleanly"
        exit(0)
      end

      if Schleuder.list.config.receive_signed_only && !mail.in_signed
        self.bounce_or_drop "not validly signed", "This address accepts only messages validly signed in an OpenPGP-compatible way", mail
      end

      if Schleuder.list.config.receive_encrypted_only && !mail.in_encrypted
        self.bounce_or_drop 'not encrypted', "This address accepts only messages encrypted with this key:\n#{Schleuder.list.key}", mail, message
      end

      if Schleuder.list.config.receive_authenticated_only && !(mail.from_member || mail.from_admin)
        self.bounce_or_drop "not authenticated", "This address accepts only messages validly signed by a list-member's key.", mail
      end

      if Schleuder.list.config.receive_from_member_emailaddresses_only && !(mail.from_member_address?||mail.from_admin_address?)
        self.bounce_or_drop "not from a member e-mail address", "This address accepts only messages from member addresses.", mail
      end

      if Schleuder.list.config.receive_admin_only && !mail.from_admin
        self.bounce_or_drop "not authenticated as admin", "This address accepts only messages validly signed by a list-admin's key.", mail
      end

      if mail.to.to_a.include?(Schleuder.list.request_addr)
        if (mail.from_member || mail.from_admin) && mail.in_encrypted
          process_plugins(mail)
          # If we reach this point no plugin took over
          self.bounce_or_drop "not a valid request", "No valid command found in message. This address (-request) accepts only messages containing valid schleuder-keyword-commands.".fmt, mail
        end
        self.bounce_or_drop "not authenticated", "This address (-request) accepts only encrypted messages by list-members.".fmt, mail
      end

      unless mail.from_member || mail.from_admin
        Schleuder.log.debug "Mail is not from a list-member, adding prefix_in"
        mail.add_prefix_in!
      end

      # Checking for keywords in mail-body (e.g. X-RESEND)
      if mail.from_member || mail.from_admin
        if mail.in_encrypted
          Schleuder.log.info "Incoming mail is encrypted and authorized, processing plugins"
          process_plugins(mail)
        else
          Schleuder.log.info "Incoming mail is not encrypted or authorized, skipping plugins"
        end
      end

      # first encrypt and send message to external recipients and collect
      # information about the process
      mail.resend_to.each do |receiver|
        if receiver.email.to_s.empty?
          errmsg = "Invalid value for receiver.email: %s. Skipping."
          Schleuder.log.warn { errmsg % receiver.email.inspect }
          next
        end

        imail = mail.individualize(receiver)
        imail.add_public_footer!

        if Schleuder.list.config.send_encrypted_only
          enc_only = true
        else
          enc_only = receiver.encrypted_only
        end

        begin
          ret = self.send(imail, receiver, enc_only)
          if ret.first
            # store meta-data about the sent message
            crypt = " (#{ret[1]})"
            mail.metadata[:resent_to] << receiver.email + crypt
          else
            mail.metadata[:error] << "Not resent to #{receiver.email}: #{ret[1]}"
          end
        rescue => e
          Schleuder.log.error e
        end
      end

      mail.add_prefix_out! unless mail.metadata[:resent_to].empty?
      mail.add_prefix!
      mail.add_metadata!

      # archive message if necessary
      Schleuder.list.archive(mail) if Schleuder.list.config.archive

      # encrypt message and send mail for each list-member
      Schleuder.log.info { 'Looping over all list-members to send out' }
      Schleuder.list.members.each do |receiver|
        Schleuder.log.debug { "Looping for #{receiver.inspect}" }

        if receiver.email.to_s.empty?
          errmsg = "Invalid value for receiver.email: %s. Skipping."
          Schleuder.log.warn { errmsg % receiver.email.inspect }
          next
        end

        imail = mail.individualize_member(receiver)

        if Schleuder.list.config.send_encrypted_only
          enc_only = true
        else
          enc_only = receiver.encrypted_only
        end

        # encrypt for each receiver
        begin
          sent = self.send(imail, receiver, enc_only)
          unless sent.first
            Schleuder.log.error sent[1]
          end
        rescue => e
          Schleuder.log.error e
        end
      end
      Schleuder.log.info { 'Processing done, this is the end.' }
    end

    def self.send(mail, receiver, encrypted_only=true, sender=Schleuder.list.bounce_addr)
      begin
        encrypted, errmsg = mail.encrypt!(receiver)
      rescue GPGME::Error::UnusablePublicKey => e
        # This exception is thrown, if the public key of a certain list
        # member is not usable (because it is revoked, expired, disabled or
        # invalid).
        k = e.keys.first
        key = mail.crypt.get_key(k.fpr).first
        errmsg = "#{e.message}: (#{k.class})\n#{key.to_s}"
        Schleuder.log.error "Encryption failed (#{errmsg})"
        encrypted = false
      rescue GPGME::Error::General => e
        errmsg = e.message
        Schleuder.log.error "Encryption failed (#{errmsg})"
        encrypted = false
      end
      if encrypted
        Mailer.send(mail, nil, sender)
        return [true, 'encrypted']
      else
        if encrypted_only
          Schleuder.log.debug "Sending plaintext not allowed, message not sent to #{receiver.inspect}!"
          return [false, "Encrypting to #{receiver.email.inspect} failed: #{errmsg} (sending plaintext disallowed)"]
        else
          Schleuder.log.info "Sending plaintext allowed, doing so"
          # sign msg
          if rawmail = mail.sign
            Mailer.send(rawmail, mail.to, sender)
            return [true, "unencrypted (#{errmsg})"]
          else
            return [false, "signing failed"]
          end
        end
      end
    end

    def self.test(listname=nil)
      # Doing things that trigger exceptions if they fail, which we rescue and
      # put to stderr
      # TODO: test #{smtp_host}:25
      begin
        # test listdir
        Dir.entries Schleuder.config.lists_dir

        # test superadminaddr
        Utils::verify_addr('superadminaddr', Schleuder.config.superadminaddr)

        if listname
          # testwise create a list-object (reads list-config etc.)
          list = List.new listname
          # test for listdir
          Dir.entries list.listdir
          # test admins
          list.config.admins.each do |a|
            Utils::verify_addr('adminaddr', a.email)
            key, msg = a.key
            if !key
              raise "Admin: #{a}'s key lookup fails! Problem: #{msg}"
            end
          end

          crypt = Crypt.new(list.config.gpg_password)

          list.members.each do |m|
            Utils::verify_addr('member address', m.email)
            key, msg = m.key
            if !key
              raise "#{m}'s key lookup fails! Problem: #{msg}"
            end
          end
        end
      rescue => e
        $stderr.puts e.message
        $stderr.puts e.backtrace
        exit 1
      end
    end

    def self.newlist(listname, interactive='true', args=nil)
      created_list = Schleuder::ListCreator.create(listname,interactive,args)
    end

    def self.bounce_or_drop(status, bounce_msg, mail)
      Schleuder.log.warn "Mail is #{status}, not passing it along"

      if Schleuder.list.config.bounces_drop_all
        Schleuder.log.warn "Found bounces_drop_all being true: dropping!"
        self.bounce_notify_admin status, "bounces_drop_all is true"
        exit 0
      end

      Schleuder.log.debug "Testing bounces_drop_on_headers"
      Schleuder.list.config.bounces_drop_on_headers.each do |header,value|
        Schleuder.log.debug "Testing #{header} => #{value}"
        if mail.header[header].to_s.downcase == value.to_s.downcase
          Schleuder.log.warn "Found matching drop-header: #{header} => #{value} -- dropping!"
          self.bounce_notify_admin status, "Forbidden header found: '#{header.capitalize}: #{value.capitalize}'"
          exit 0
        end
      end

      # if we're still alive: bounce message
      Schleuder.log.info "bouncing mail to sender"
      self.bounce_notify_admin status
      $stdout.puts bounce_msg
      exit 100
    end

    def self.bounce_notify_admin(reason, drop_reason='')
      msg = "The attached incoming email has not been passed to the list.\nReason: #{reason}.\n"
      if drop_reason.empty?
        msg += "\nIt has been bounced to the sender.\n"
      else
        msg += "\nIt has *not* been bounced but dropped.\nReason: #{drop_reason}.\n"
      end

      Schleuder.log.notify_admin("Notice", [ msg, Mail.parse(Schleuder.origmsg) ]) if Schleuder.list.config.bounces_notify_admin
    end

    def self.process_plugins(mail)
      Schleuder.log.info 'Email is encrypted and authorized, processing plugins'
      # process plugins
      Schleuder.log.debug 'Processing plugins'
      mail.process_plugins!
    end
  end
end
