/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_clipboard.h"
#include "sg_worksheet.h"
#include "sg_plot.h"

static GList *selection;
static GdkPixmap *pixmap_selection;

void
sg_clipboard_copy(SGworksheet *sg_sheet, gboolean clear)
{
  gint row, col;
  gchar *text, *ftext, *stext;
  GtkSheetCell *cell=NULL;
  SGhiddencell *link=NULL;
  GtkSheet *sheet=GTK_SHEET(sg_sheet->sheet);
  gint srow,scol,num_selected=0;

  sg_clipboard_clear();

  gtk_sheet_freeze(sheet);
  gtk_sheet_get_active_cell(sheet,&srow,&scol);  

  num_selected=((sheet->range.rowi-sheet->range.row0)+1)*
               ((sheet->range.coli-sheet->range.col0)+1);
  for(row = sheet->range.row0; row <= sheet->range.rowi; row++){
    for(col = sheet->range.col0; col <= sheet->range.coli; col++){
       ftext=NULL;
       stext=NULL;
       text = sg_worksheet_cell_get_text(sg_sheet, row, col);
       link = (SGhiddencell *)gtk_sheet_get_link(sheet, row, col);

       if (link) ftext = link->formula;

       if (row==srow && col==scol && num_selected==1)
        stext=gtk_entry_get_text(GTK_ENTRY(sheet->sheet_entry));

       if (stext && strlen(stext)>0) text=stext;
/*
       else if (ftext && strlen(ftext)>0) text=ftext;
*/
              
       if(text && strlen(text)>0){
           cell = g_new0(GtkSheetCell, 1);
           cell->text = g_strdup(text);
           cell->row = row - sheet->range.row0;
           cell->col = col - sheet->range.col0;
          /* This way we tell the paste function we are sending a formula: */
           cell->link = link;
           selection = g_list_append(selection, cell);
       }
    }
  }

  if (clear)
   for (row = sheet->range.row0; row <= sheet->range.rowi; row++)
    for (col = sheet->range.col0; col <= sheet->range.coli; col++)
      { 
        gtk_sheet_remove_link(sheet, row, col);
        gtk_sheet_cell_delete(sheet, row, col);
        if (row==srow && col==scol)
            gtk_entry_set_text(GTK_ENTRY(sheet->sheet_entry),g_strdup(""));
      }      

  gtk_sheet_thaw(sheet);
}

void
sg_clipboard_paste(SGworksheet *sg_sheet)
{
  GtkSheetCell *cell;
  GList *list;
  GtkSheet *sheet=GTK_SHEET(sg_sheet->sheet);
  gboolean selected;
  gint row,col,srow,scol;
  gint nrows, ncols;

  ncols = gtk_sheet_get_columns_count(sheet);
  nrows = gtk_sheet_get_rows_count(sheet);

  gtk_sheet_freeze(sheet);

  list = selection;
  gtk_sheet_get_active_cell(sheet,&srow,&scol);
  while(list){
   cell = (GtkSheetCell *)list->data;
   row=sheet->range.row0 + cell->row;
   col=sheet->range.col0 + cell->col;

   if(row > nrows-1) {
        sg_worksheet_add_rows(sg_sheet, row-nrows+1);
        nrows = gtk_sheet_get_rows_count(sheet);
   }
   if(col > ncols-1) {
        sg_worksheet_add_columns(sg_sheet, row-nrows+1);
        ncols = gtk_sheet_get_columns_count(sheet);
   }

   if (cell->link){ /* We were sent a hidden cell, we clone it */
    SGhiddencell *hidden;
    gboolean formula = FALSE;
    sg_worksheet_hidden_cell_clear(sg_sheet, row, col);
   
    hidden = g_new0(SGhiddencell, 1); 
    *hidden = *((SGhiddencell *)cell->link);
    if(hidden->formula) hidden->formula = g_strdup(hidden->formula);
    if(hidden->type == SG_TYPE_TEXT && hidden->value.val_char)
          hidden->value.val_char = g_strdup(hidden->value.val_char);
    gtk_sheet_link_cell(sheet, row, col, hidden);
 
    sg_worksheet_cell_set_text(sg_sheet, row, col, cell->text);
   }else{ /* we were sent plain text */
    sg_worksheet_hidden_cell_clear(sg_sheet, row, col);
    sg_worksheet_cell_set(sg_sheet, row, col, 
                          cell->text, FALSE, FALSE);
   }

   /* We are pasting into a selected cell, put text in the entry widget */
/*
   if (row==srow && col==scol)
    { 
      if (cell->link)
         gtk_entry_set_text(GTK_ENTRY(sheet->sheet_entry),g_strdup(cell->text));
      else
         gtk_entry_set_text(GTK_ENTRY(sheet->sheet_entry),g_strdup(""));
    }
*/
    list = list->next;
  }
  gtk_sheet_thaw(sheet);
}

void
sg_clipboard_clear()
{
  GtkSheetCell *cell;
  GList *list;

  list = selection;
  while(list){
    cell = (GtkSheetCell *)list->data;
    g_free(cell->text);
    g_free(cell);

    selection = g_list_remove_link(selection, list);
    g_list_free_1(list);
    list = selection;
  }

  if(pixmap_selection)
      gdk_pixmap_unref(pixmap_selection);

  sg_clipboard_init();
}

void
sg_clipboard_init()
{
  selection = NULL;
  pixmap_selection = NULL;
}

void
sg_clipboard_copy_pixmap(SGplot *plot)
{
  GdkPixmap *pixmap_orig;
  GtkPlotCanvas *canvas;
  GList *plots;
  gint width, height;
  gdouble m;

  if(pixmap_selection) g_free(pixmap_selection);
  pixmap_selection = NULL;

  canvas = GTK_PLOT_CANVAS(plot->real_canvas);

  m = canvas->magnification;
  pixmap_orig = canvas->pixmap;
  width = canvas->pixmap_width;
  height = canvas->pixmap_height;

  plots = canvas->plots;
  while(plots){
    GtkPlot *child;
    child = GTK_PLOT(plots->data);
    child->magnification = 1.0;

    GTK_WIDGET(child)->allocation.width = canvas->width;
    GTK_WIDGET(child)->allocation.height = canvas->height;

    plots = plots->next;
  }

  canvas->pixmap_width = canvas->width;
  canvas->pixmap_height = canvas->height;

  

  pixmap_selection = gdk_pixmap_new(plot->real_canvas->window, 
                                    canvas->width, canvas->height, -1); 
  canvas->pixmap = pixmap_selection;
  GTK_PLOT_GDK(canvas->pc)->drawable = pixmap_selection;

  gtk_plot_canvas_paint(canvas);

  canvas->magnification = m;
  canvas->pixmap = pixmap_orig;
  canvas->pixmap_width = width;
  canvas->pixmap_height = height;
  GTK_PLOT_GDK(canvas->pc)->drawable = pixmap_orig;

  plots = canvas->plots;
  while(plots){
    GtkPlot *child;
    child = GTK_PLOT(plots->data);
    child->magnification = m;

    GTK_WIDGET(child)->allocation.width = width;
    GTK_WIDGET(child)->allocation.height = height;

    plots = plots->next;
  }

}

void
sg_clipboard_paste_pixmap(SGplot *plot,
                          gdouble x, gdouble y)
{
  GtkPlotCanvas *canvas;
  GtkPlotCanvasChild *child;
  GdkPixmap *pixmap;
  gint width, height;
  gdouble x1, x2, y1, y2;

  if(!pixmap_selection) return;

  canvas = GTK_PLOT_CANVAS(plot->real_canvas);

  gdk_window_get_size(pixmap_selection, &width, &height);

  pixmap = gdk_pixmap_new(plot->real_canvas->window, 
                          width, height, -1); 

  gdk_draw_pixmap(pixmap, 
                  GTK_WIDGET(canvas)->style->fg_gc[0], 
                  pixmap_selection, 0, 0, 0, 0, -1, -1);

  x1 = x;
  y1 = y;

  x2 = x1 + (gdouble)width / (gdouble)canvas->width * .5;
  y2 = y1 + (gdouble)height / (gdouble)canvas->height * .5;

  child = gtk_plot_canvas_put_pixmap(canvas, 
                                     pixmap,
                                     x, y);
  gtk_plot_canvas_child_move_resize(canvas, child, x1, y1, x2, y2);

  gtk_plot_canvas_refresh(canvas);
}
