/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <string.h>
#include "sg_text_dialog.h"
#include "sg_dialogs.h"
#include "sg_stock.h"
#include "sg_misc.h"
#include "sg_entry.h"

typedef struct _SGtextdialogClass  SGtextdialogClass;
struct _SGtextdialogClass
{
  GtkWindowClass parent_class;
};

static void sg_text_dialog_class_init		(SGtextdialogClass *klass);
static void sg_text_dialog_init                 (SGtextdialog *dialog);
static void sg_text_dialog_destroy              (GtkObject *dialog);
static void update_text				(GtkWidget *widget, 
						 gpointer data);
static void new_fg				(GtkColorCombo *combo, 
                                                 gint selection, 
                                                 gchar *name, gpointer data);
static void new_bg				(GtkColorCombo *combo, 
                                                 gint selection, 
                                                 gchar *name, gpointer data);
static void init_dialog				(SGtextdialog *dialog);

static GtkWindowClass *parent_class = NULL;


GtkType
sg_text_dialog_get_type (void)
{
  static GtkType dialog_type = 0;
  
  if (!dialog_type)
    {
      GtkTypeInfo dialog_info =
      {
	"SGtextdialog",
	sizeof (SGtextdialog),
	sizeof (SGtextdialogClass),
	(GtkClassInitFunc) sg_text_dialog_class_init,
	(GtkObjectInitFunc) sg_text_dialog_init,
	/* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };
      
      dialog_type = gtk_type_unique (gtk_window_get_type(), &dialog_info);
    }
  
  return dialog_type;
}

GtkWidget*
sg_text_dialog_new (GtkPlotText *text)
{
  GtkWidget *widget;
  SGtextdialog *dialog;

  widget = gtk_widget_new (sg_text_dialog_get_type(), NULL);

  dialog = SG_TEXT_DIALOG(widget);
  dialog->text = text;

  gtk_entry_set_text(GTK_ENTRY(dialog->text_entry), text->text);
  gtk_entry_select_region(GTK_ENTRY(dialog->text_entry), 0, strlen(text->text));

  init_dialog(dialog);
  gtk_widget_show_all(dialog->main_box);

  gtk_widget_grab_focus(dialog->text_entry);

  return widget;
}

static void
sg_text_dialog_class_init (SGtextdialogClass *klass)
{
  GtkWidgetClass *widget_class;
  GtkObjectClass *object_class;
  
  widget_class = (GtkWidgetClass*) klass;
  parent_class = (GtkWindowClass*) gtk_type_class (gtk_window_get_type ());
  object_class = (GtkObjectClass*) klass;

  object_class->destroy = sg_text_dialog_destroy;
 
}


static void
sg_text_dialog_init (SGtextdialog *dialog)
{
  GtkWidget *main_box;
  GtkWidget *main_table;
  GtkWidget *frame;
  GtkWidget *box;
  GtkWidget *table;
  GtkRequisition req;
  GtkAdjustment *adj;

  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
  gtk_window_set_title(GTK_WINDOW(dialog), "Edit Text");

  /* Create widgets */
  dialog->main_box = main_box = gtk_hbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 5);
  gtk_container_add(GTK_CONTAINER(dialog), main_box);

  /* Create widgets */
  main_table = gtk_table_new(2,3,FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 2);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 2);
  gtk_box_pack_start (GTK_BOX (main_box), main_table, FALSE, FALSE, 0);
  frame = gtk_frame_new("Position");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach_defaults(GTK_TABLE(main_table), frame, 0, 1, 0, 1);

  table=gtk_table_new(2, 3, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  gtk_table_attach_defaults(GTK_TABLE(table),
                            gtk_label_new("Rotate (deg): "), 0, 1, 0, 1);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 270., 90., 90., 0.);
  dialog->angle_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->angle_spin), TRUE);
  gtk_entry_set_editable(GTK_ENTRY(dialog->angle_spin), FALSE);
  gtk_widget_size_request(dialog->angle_spin, &req);
  req.width /= 2;
  gtk_widget_set_usize(dialog->angle_spin, req.width, req.height);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->angle_spin, 1, 2, 0, 1);

  dialog->center_item = gtk_check_item_new_with_label("Centered");
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->center_item, 0, 2, 1, 2);

 /*--------------------------------------*/

  frame = gtk_frame_new("Properties");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach_defaults(GTK_TABLE(main_table), frame, 1, 2, 0, 1);

  table=gtk_table_new(3, 3, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  gtk_table_attach_defaults(GTK_TABLE(table),
                            gtk_label_new("Foreground"), 0, 1, 0, 1);
  dialog->fg_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->fg_combo, 1, 2, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table),
                            gtk_label_new("Background"), 0, 1, 1, 2);
  dialog->bg_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->bg_combo, 1, 2, 1, 2);

  gtk_table_attach_defaults(GTK_TABLE(table),
                            gtk_label_new("Border"), 0, 1, 2, 3);
  dialog->border_combo = gtk_combo_new();
  gtk_widget_set_usize(dialog->border_combo, req.width, req.height);

  gtk_table_attach_defaults(GTK_TABLE(table), dialog->border_combo, 1, 2, 2, 3);

  dialog->transparent_item = gtk_check_item_new_with_label("Transparent");
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->transparent_item, 0, 2, 3, 4);
 /*----------------------------------------------*/

  frame = sg_entry_new();
  gtk_frame_set_label(GTK_FRAME(frame), "Text");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach_defaults(GTK_TABLE(main_table), frame, 0, 2, 1, 2);

  dialog->text_frame = frame;
  dialog->font_combo = SG_ENTRY(frame)->font_combo;
  dialog->text_entry = SG_ENTRY(frame)->text_entry;


  /*--------------------------------------*/

  /* connect signals */

  /* Action area */

  box = gtk_vbutton_box_new();
  gtk_container_set_border_width (GTK_CONTAINER (box), 5);
  gtk_table_attach_defaults(GTK_TABLE(main_table), box, 2, 3, 0, 1);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(box), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(box), 5);
  gtk_widget_show(box);

  dialog->ok_button = sg_stock_button(GNOME_STOCK_BUTTON_OK);
  gtk_box_pack_start (GTK_BOX (box), dialog->ok_button, TRUE, TRUE, 0);

  dialog->apply_button = sg_stock_button(GNOME_STOCK_BUTTON_APPLY);
  gtk_box_pack_start (GTK_BOX (box), dialog->apply_button, TRUE, TRUE, 0);

  dialog->cancel_button = sg_stock_button(GNOME_STOCK_BUTTON_CLOSE);
  gtk_box_pack_start (GTK_BOX (box), dialog->cancel_button, TRUE, TRUE, 0);

  gtk_signal_connect (GTK_OBJECT(SG_TEXT_DIALOG(dialog)->ok_button),
                      "clicked",
                      GTK_SIGNAL_FUNC (update_text), dialog);

  gtk_signal_connect (GTK_OBJECT(SG_TEXT_DIALOG(dialog)->apply_button),
                      "clicked",
                      GTK_SIGNAL_FUNC (update_text), dialog);
}

static void
update_text(GtkWidget *widget, gpointer data)
{
  SGtextdialog *dialog;
  GtkWidget *child;
  gchar *font;
  gint height;
  gint transparent;
  gint angle;
  GtkJustification justification;
  GtkPlotBorderStyle border;
  GtkColorCombo *combo;
  GdkColor color, bg, fg;
  gchar *color_name;
  gchar *text;
  gboolean state;

  dialog = (SGtextdialog *)data;

  font = GTK_FONT_COMBO(dialog->font_combo)->psfont->psname;
  height = GTK_FONT_COMBO(dialog->font_combo)->height;
  transparent = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->transparent_item));

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->border_combo)->list)->selection->data;
  border = (GtkPlotBorderStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->border_combo)->list), child);

  combo = GTK_COLOR_COMBO(dialog->fg_combo);
  color_name = gtk_color_combo_get_color_at(combo, combo->row, combo->column);
  gdk_color_parse(color_name, &color);
  gdk_color_alloc(dialog->fg_combo->style->colormap, &color);
  fg = color;

  combo = GTK_COLOR_COMBO(dialog->bg_combo);
  color_name = gtk_color_combo_get_color_at(combo, combo->row, combo->column);
  gdk_color_parse(color_name, &color);
  gdk_color_alloc(dialog->bg_combo->style->colormap, &color);
  bg = color;

  angle = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->angle_spin));
  state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->center_item));
  justification = state ? GTK_JUSTIFY_CENTER : GTK_JUSTIFY_LEFT;


  text = g_strdup(gtk_entry_get_text(GTK_ENTRY(dialog->text_entry)));

  if(widget == dialog->ok_button)
          gtk_widget_destroy(GTK_WIDGET(dialog));

  g_free(dialog->text->font);
  g_free(dialog->text->text);

  dialog->text->angle = angle;
  dialog->text->border = border;
  dialog->text->font = g_strdup(font);
  dialog->text->height = height;
  dialog->text->fg = fg;
  dialog->text->bg = bg;
  dialog->text->transparent = transparent;
  dialog->text->justification = justification;
  dialog->text->text = text;
}

static void
new_fg(GtkColorCombo *combo, gint selection, gchar *name, gpointer data)
{
  GtkWidget *text_entry;
  GdkColormap *colormap;
  GdkGC *gc;
  GdkColor color;
  GtkStyle *style;

  text_entry = GTK_WIDGET(data);

  colormap =gdk_colormap_get_system();
  gc = gdk_gc_new(GTK_PIXMAP(GTK_BIN(GTK_COMBO_BOX(combo)->button)->child)->pixmap);

  gdk_color_parse(name, &color);
  gdk_color_alloc(colormap, &color);
  gdk_gc_set_foreground(gc, &color);

  gdk_draw_rectangle(GTK_PIXMAP(GTK_BIN(GTK_COMBO_BOX(combo)->button)->child)->pixmap,
                     gc,
                     TRUE,
                     4,4,33,13);

  gdk_gc_unref(gc);
  gtk_widget_draw(GTK_BIN(GTK_COMBO_BOX(combo)->button)->child, NULL);

  style = gtk_style_copy(text_entry->style);
  style->fg[0] = color;
  gtk_widget_set_style (text_entry, style);
  gtk_style_unref(style);
}


static void
new_bg(GtkColorCombo *combo, gint selection, gchar *name, gpointer data)
{
  GtkWidget *text_entry;
  GdkColormap *colormap;
  GdkGC *gc;
  GdkColor color;
  GtkStyle *style;

  text_entry = GTK_WIDGET(data);

  colormap =gdk_colormap_get_system();
  gc = gdk_gc_new(GTK_PIXMAP(GTK_BIN(GTK_COMBO_BOX(combo)->button)->child)->pixmap);

  gdk_color_parse(name, &color);
  gdk_color_alloc(colormap, &color);
  gdk_gc_set_foreground(gc, &color);

  gdk_draw_rectangle(GTK_PIXMAP(GTK_BIN(GTK_COMBO_BOX(combo)->button)->child)->pixmap,
                     gc,
                     TRUE,
                     4,4,33,13);

  gdk_gc_unref(gc);
  gtk_widget_draw(GTK_BIN(GTK_COMBO_BOX(combo)->button)->child, NULL);

  style = gtk_style_copy(text_entry->style);
  style->base[0] = color;
  gtk_widget_set_style (text_entry, style);
  gtk_style_unref(style);

}

static void
init_dialog(SGtextdialog *dialog)
{
  GtkPSFont *font;
  GtkPlotText *text;
  gint row, column;
  gchar *color_name;
  gint state;

  text = dialog->text;

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->fg_combo), text->fg);
  sg_color_combo_init(GTK_COLOR_COMBO(dialog->bg_combo), text->bg);

  row = GTK_COLOR_COMBO(dialog->fg_combo)->row;
  column = GTK_COLOR_COMBO(dialog->fg_combo)->column;

  if(row != -1 && column != -1){
   color_name = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(dialog->fg_combo),
                                             row, column);
   new_fg(GTK_COLOR_COMBO(dialog->fg_combo), 0, color_name, dialog->text_entry);
  }

  row = GTK_COLOR_COMBO(dialog->bg_combo)->row;
  column = GTK_COLOR_COMBO(dialog->bg_combo)->column;
  if(row != -1 && column != -1){
   color_name = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(dialog->bg_combo),
                                             row, column);
   new_bg(GTK_COLOR_COMBO(dialog->bg_combo), 0, color_name, dialog->text_entry);
  }

  state = (text->justification == GTK_JUSTIFY_CENTER);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->center_item), state);

  state = text->transparent;
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->transparent_item), state);

  sg_combo_set_items(GTK_COMBO(dialog->border_combo), border_styles);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->border_combo)->entry), FALSE);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->border_combo)->list), 
                       text->border);


  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->angle_spin), text->angle);

  font = gtk_psfont_get_font(text->font);
  gtk_font_combo_select(GTK_FONT_COMBO(dialog->font_combo),
                        font->family,
                        FALSE,
                        FALSE,
                        text->height);

}

static void 
sg_text_dialog_destroy                 (GtkObject *text_dialog)
{
  SGtextdialog *dialog;

  dialog = SG_TEXT_DIALOG(text_dialog);

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (text_dialog);
}
