//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [M] = CL_fr_topoNMat(orig)
// Terrestrial frame to topocentric North frame transformation matrix
//
// Calling Sequence
// [M] = CL_fr_topoNMat(lon,lat)
//
// Description
// <itemizedlist><listitem>
// Computes the frame transformation matrix <emphasis role="bold">M</emphasis> from the terrestrial reference frame to the local topocentric North frame.
// <para> By convention, multiplying <emphasis role="bold">M</emphasis> by coordinates relative to the terrestrial frame yields coordinates relative to the topocentric North frame. </para>
// </listitem>
// <listitem>
// Notes: 
// <para> - The origin of the topocentric North frame is not the center of the planet. It is the location defined by <emphasis role="bold">orig</emphasis>. </para>
// <para> - This function can be used for another planet than the Earth. </para>
// </listitem>
// <listitem>
// See <link linkend="LocalFrames">Local Frames</link> for more details on the definition of local frames. 
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// orig: [lon;lat;alt] topocentric frame origin in elliptical (geodetic) coordinates [rad;rad;m] (3xN)
// M : terrestrial to topocentric North frame transformation matrix (3x3xN)
//
// Bibliography
// CNES - MSLIB FORTRAN 90, Volume T (mt_def_topo_N)
//
// See also
// CL_fr_topoN2ter
// CL_fr_ter2topoN
//
// Authors
// CNES - DCT/SB
//
// Examples
// // Conversion from pos_ter to pos_topoN : 
// orig = [CL_deg2rad(40) ; CL_deg2rad(10) ; 0];
// pos_orig = CL_co_ell2car(orig);
// pos_ter = [ 1000.e3 ; 6578.e3 ; 2000.e3 ];
// pos_ter_orig = pos_ter - pos_orig;
// M = CL_fr_topoNMat(orig);
// pos_topoN = M * pos_ter_orig;
// pos_topoN_2 = CL_fr_ter2topoN(orig,pos_ter);
//
// // Conversion from pos_topoN to pos_ter :
// orig = [CL_deg2rad(40) ; CL_deg2rad(10) ; 0];
// pos_orig = CL_co_ell2car(orig);
// // 1000 km above 40deg / 10deg :
// pos_topoN = [ 0.e3 ; 0.e3 ; 1000.e3 ];  
// M = CL_fr_topoNMat(orig);
// pos_ter = M' * pos_topoN + pos_orig;
// pos_ter_2 = CL_fr_topoN2ter(orig,pos_topoN);

// Declarations:


// Code:

lon = orig(1,:);
lat = orig(2,:);

N = size(lat,2);

//vect_i computation: horizontal towards norh
//vect_i = (-sin(lat)cos(long),-sin(lat)sin(long),cos(lat))
vect_i(1,:) = -sin(lat).*cos(lon);
vect_i(2,:) = -sin(lat).*sin(lon);
vect_i(3,:) = cos(lat);

//vect_j computation: local paralel tangent towards west
//vect_j = (sin(long),-cos(long),0)
vect_j(1,:) = sin(lon);
vect_j(2,:) = -cos(lon);
vect_j(3,:) = zeros(1,N);

//vect_k computation: local vertical
//vect_k = (cos(lat)cos(long),cos(lat)sin(long),sin(lat))
vect_k(1,:) = cos(lat).*cos(lon);
vect_k(2,:) = cos(lat).*sin(lon);
vect_k(3,:) = sin(lat);

// Matrix M :
M = hypermat([3 3 N] , [vect_i(1,:); vect_j(1,:); vect_k(1,:); ...
            vect_i(2,:); vect_j(2,:); vect_k(2,:); ...
            vect_i(3,:); vect_j(3,:); vect_k(3,:)]);
if(N == 1)
    M = M(:,:,1);
end

endfunction
