//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [pos_car,vel_car,jacob]=CL_oe_cir2car(cir,mu)
// Circular adapted to cartesian orbital elements
//
// Calling Sequence
// [pos_car,vel_car[,jacob]] = CL_oe_cir2car(cir[,mu])
//
// Description
// <itemizedlist><listitem>
// Converts orbital elements adapted to near-circular orbits to position and velocity 
// for an elliptic orbit.
// <para>The transformation jacobian is optionally computed.</para>
// <para>See <link linkend="OrbitalElements">Orbital elements</link> for more details</para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// cir: orbital elements adapted to near-circular orbits [sma;ex;ey;inc;raan;pom+anm] [m,rad] (6xN)
// mu : (optionnal) gravitational constant. [m^3/s^2] (default value is %CL_mu)
// pos_car: position [X;Y;Z] [m] (3xN)
// vel_car: velocity [Vx;Vy;Vz] [m/s] (3xN)
// jacob: (optional) transformation jacobian d(X,Y,Z,Vx,Vy,Vz)/d(sma,ex,ey,inc,raan,pom+anm) (6x6xN)
//
// Bibliography
// 1 CNES - MSLIB FORTRAN 90, Volume V (mv_cir_car)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_oe_car2cir
//
// Examples
// // Example 1
// cir = [7204649,-2.9d-4,1.34d-3,1.7233,1.5745,0.5726]';
// [pos_car,vel_car,jacob1] = CL_oe_cir2car(cir);
// [cir2,jacob2] = CL_oe_car2cir(pos_car,vel_car);

//!************************************************************************
//! But:  Passage des parametres orbitaux adaptes a l'orbite CIRculaire, aux parametres CARtesiens
//! ===
//!
//! Note d'utilisation:  1) Applicable aux orbites circulaires ou elliptiques, equatoriales ou non
//! ==================   3) Non applicable aux orbites hyperboliques et paraboliques
//!                      4) Pour les orbites hyperboliques ou paraboliques, voir mv_kep_car.
//!                      5) L'element (i,j) de la jacobienne correspond a la derivee partielle du parametre cartesien numero i
//!                         par rapport a la derivee partielle du parametre dit adapte a l'orbite circulaire non equatoriale numero j.
//!                      6) Les unites en entree doivent etre coherentes entre elles. Ex.:  mu en m**3/s**2,cir en metres et
//!                         radians,ainsi pos_car sera en metres et vel_car en m/s.
//!                      7) La transformation inverse (sauf en cas d'orbite equatoriale) peut s'effectuer
//!                         par mv_car_cir.
//!************************************************************************


// Declarations:
if(~exists('%CL_epsilon')) then global %CL_epsilon; end;
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

[lhs,rhs]=argn();
if ~exists('mu','local') then mu=%CL_mu; end
if lhs==3
  compute_jacob=%t
else
  compute_jacob=%f
end

a  = cir(1,:)
ex = cir(2,:)
ey = cir(3,:)
ai = cir(4,:)
agom = cir(5,:)
um = cir(6,:)

pos_car = [];
vel_car = [];

eps100 = 100 * %eps

//Controle des parametres d'entree
e2 = ex.^2 + ey.^2
ii = find(sqrt(e2)>1-%CL_epsilon.parab)
if ii~=[] then CL__error('parabolic or hyperbolic orbit'); end

//calcul ue (= petit omega + E) par equation de kepler generalisee
ue = CL_kp_M2Ecir(ex,ey,um)


//*************************************************************
// 1- calcul vecteur position r = a (ld1*o + mu1*h)
//        et vecteur vitesse rv = sqrt(mu/a) (nu1*o + ki1*h)
//*************************************************************

cue = cos(ue) // cosinus de pom + E
sue = sin(ue) // sinus de pom + E

cai = cos(ai)  //     cosinus de i
sai = sin(ai)  //     sinus de i
cgo = cos(agom)//     cosinus de gomega
sgo = sin(agom)//     sinus degomega

sq = sqrt(1-e2)//     calcul de sqrt(1-ex**2-ey**2) remarque: 1-e2 > 0 car on a teste si e etait bien dans [0., 1.[
exey = ex.*ey
ex2 = ex.^2
ey2 = ey.^2

ecceys = (ex .* cue) + (ey .* sue)
eycexs = (ey .* cue) - (ex .* sue)

o(1,:) = cgo
o(2,:) = sgo          // calcul des composantes du vecteur p en fonction des angles i,gomega
o(3,:) = 0

h(1,:) = - cai .* sgo
h(2,:) = cai .* cgo    // calcul des composantes du vecteur q en fonction des angles i,gomega
h(3,:) = sai

betaa = 1.0./(1 + sq) //     calcul variable beta = 1 / 1+sqrt(1-ex2-ey2)

rld1 = -ex + ((1 - (betaa.*ey2)) .* cue) + (betaa .* exey .* sue) // calcul de ld1 et de
rmu1 = -ey + ((1 - (betaa.*ex2)) .* sue) + (betaa .* exey .* cue) // mu1

sqmusa = sqrt(mu./a) // avec mu et a > 0 a ce stade des calculs

rnu1 = (-sue + (betaa.*ey.*ecceys)) ./ (1 - ecceys) //  remarque: (1-ecceys) different de 0 car e < 1 a ce stade des calculs
rki1 = (cue - (betaa.*ex.*ecceys)) ./ (1 - ecceys)

//calcul des composantes des vecteurs position et vitesse
pos_car = (a.*.ones(3,1)) .* ( ((rld1.*.ones(3,1)) .* o) + ((rmu1.*.ones(3,1)) .* h) )
vel_car = (sqmusa.*.ones(3,1)) .* (((rnu1.*.ones(3,1)) .* o) + ((rki1.*.ones(3,1)) .* h))

//********************************************
// 2- calcul du jacobien de la transformation
//********************************************
if compute_jacob
  xdoai = zeros(o) //derivees des composantes de o par rapport a i

  xdhai(1,:) = sai .* sgo
  xdhai(2,:) = -sai .* cgo //derivees des composantes de h par rapport a i
  xdhai(3,:) = cai

  xdogo(1,:) = - sgo
  xdogo(2,:) = cgo        //derivees des composantes de o par rapport a gomega
  xdogo(3,:) = zeros(xdogo(1,:))

  xdhgo(1,:) = - cai .* cgo
  xdhgo(2,:) = - cai .* sgo //derivees des composantes de h par rapport a gomega
  xdhgo(3,:) = zeros(xdhgo(1,:))

  xdbex = (ex ./ sq) .* (1.0 ./ (1+sq).^2) //calcul derivee de beta par rapport a ex
  xdbey = (ey ./ sq) .* (1.0 ./ (1+sq).^2) //et ey

  xdldex = -1 - (ey2.*cue.*xdbex) + (betaa.*ey.*sue) + (xdbex.*exey.*sue) //calcul des derivees de ld1 et mu1 par rapport a ex
  xdmuex = (betaa .* (eycexs - (ex.*sue))) + (xdbex.*ex.*eycexs)

  xdldey = - (betaa .* ((ey.*cue) + eycexs)) - (xdbey.*ey.*eycexs) //par rapport a ey
  xdmuey = -1 - (ex2.*sue.*xdbey) + (betaa.*ex.*cue) + (xdbey.*exey.*cue)

  xdldue = ((1 - (betaa.*ey2)) .* (-sue)) + (betaa.*exey.*cue)          //et par rapport a ue
  xdmuue = ((1 - (betaa.*ex2)) .* cue) - (betaa.*exey.*sue)

  umey = 1 - ecceys
  betey = betaa.*ey.*ecceys
  betex = betaa.*ex.*ecceys

  //calcul des derivees de nu1 et ki1 par rapport a ex,ey et ue
  xdnuex = ((-cue .* (sue - betey)) ./ umey.^2)+ (((xdbex.*ey.*ecceys) + (betaa.*ey.*cue)) ./ umey)
  xdkiex = ((cue .* (cue - betex)) / umey.^2)- (((xdbex.*ex.*ecceys) + (betaa .* (ecceys+(ex.*cue)))) ./umey)
  xdnuey = ((-sue .* (sue - betey)) / umey.^2)+ (((xdbey.*ey.*ecceys) + (betaa .* (ecceys+(ey.*sue)))) ./umey)
  xdkiey = ((sue .* (cue - betex)) / umey.^2)- (((xdbey.*ex.*ecceys) + (betaa.*ex.*sue)) ./ umey)
  xdnuue = ((-cue + (betaa.*ey.*eycexs)) ./ umey)+ (((-sue + betey) .* eycexs) / umey.^2)
  xdkiue = ((-sue - (betaa.*ex.*eycexs)) ./ umey)+ (((cue - betex) .* eycexs) / umey.^2)

  xdueum = 1.0 ./ umey     // calcul derivees de ue par rapport a um , ex et ey
  xdueex = sue .* xdueum   //due/dum = 1 / 1-ecceys, due/dex = sinue * due/dum
  xdueey = -cue .* xdueum  //due/dey = -cosue * due/dum


  //===========================================================
  // calcul des derivees partielles des vecteurs position et
  // vitesse par rapport a a,ex,ey,i,gomega,um
  //===========================================================

  //par rappport a a
  j131 = ((rld1.*.ones(3,1)) .* o) + ((rmu1.*.ones(3,1)) .* h)
  j461 = ((-0.5.*sqmusa./a).*.ones(3,1)) .* (((rnu1.*.ones(3,1)) .* o) + ((rki1.*.ones(3,1)) .* h))

  //par rappport a ex ( = derivee/ex + derivee/ue * derivee ue/ex)
  j132 = (a.*.ones(3,1)) .* ((((xdldex.*.ones(3,1)).*o) + ((xdmuex.*.ones(3,1)).*h)) +(((xdldue.*.ones(3,1)).*o) + ((xdmuue.*.ones(3,1)).*h)) .* (xdueex.*.ones(3,1)))
  j462 = (sqmusa.*.ones(3,1)) .* ((((xdnuex.*.ones(3,1)).*o) + ((xdkiex.*.ones(3,1)).*h)) +(((xdnuue.*.ones(3,1)).*o) + ((xdkiue.*.ones(3,1)).*h)) .* (xdueex.*.ones(3,1)))

  //par rappport a ey ( = derivee/ey + derivee/ue * derivee ue/ey)
  j133 = (a.*.ones(3,1)) .* ((((xdldey.*.ones(3,1)).*o) + ((xdmuey.*.ones(3,1)).*h)) +(((xdldue.*.ones(3,1)).*o) + ((xdmuue.*.ones(3,1)).*h)) .* (xdueey.*.ones(3,1)))
  j463 = (sqmusa.*.ones(3,1)) .* ((((xdnuey.*.ones(3,1)).*o) + ((xdkiey.*.ones(3,1)).*h)) +(((xdnuue.*.ones(3,1)).*o) + ((xdkiue.*.ones(3,1)).*h)) .* (xdueey.*.ones(3,1)))

  //par rappport a i
  j134 = (a.*.ones(3,1)) .* (((rld1.*.ones(3,1)) .* xdoai) + ((rmu1.*.ones(3,1)) .* xdhai))
  j464 = (sqmusa.*.ones(3,1)) .* (((rnu1.*.ones(3,1)) .* xdoai) + ((rki1.*.ones(3,1)) .* xdhai))

  //par rappport a gomega
  j135 = (a.*.ones(3,1)) .* (((rld1.*.ones(3,1)) .* xdogo) + ((rmu1.*.ones(3,1)) .* xdhgo))
  j465 = (sqmusa.*.ones(3,1)) .* (((rnu1.*.ones(3,1)) .* xdogo) + ((rki1.*.ones(3,1)) .* xdhgo))

  //par rappport a um ( = derivee/ue * derivee de ue/um)
  j136 = (a.*.ones(3,1)) .* (((xdldue.*.ones(3,1)) .* o) + ((xdmuue.*.ones(3,1)) .* h)) .* (xdueum.*.ones(3,1))
  j466 = (sqmusa.*.ones(3,1)) .* (((xdnuue.*.ones(3,1)) .* o) + ((xdkiue.*.ones(3,1)) .* h)) .*(xdueum.*.ones(3,1))

  //jacobien
  jacob=hypermat([6 6 size(cir,2)])

  for i=1:size(cir,2) //this for should be replaced by a vectorized operation
    jacob(:,:,i)=[j131(:,i) j132(:,i) j133(:,i) j134(:,i) j135(:,i) j136(:,i);...
                  j461(:,i) j462(:,i) j463(:,i) j464(:,i) j465(:,i) j466(:,i)];
  end

  if size(cir,2)==1 then jacob=jacob(:,:,1); end

end

endfunction
