//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [varargout]=CL_gm_visiParams(sat_radius,target_radius,type_par1,par1,type_par2)
// Visibility parameters (angles, distance, ...) for a spherical planet
//
// Calling Sequence
// [par2]=CL_gm_visiParams(sat_radius,target_radius,type_par1,par1,type_par2)
// [res1,..resN]=CL_gm_visiParams(sat_radius,target_radius,type_par1,par1,[type1,..,typeN])
// [result]=CL_gm_visiParams(sat_radius,target_radius,type_par1,par1,'all')
//
// Description
// <itemizedlist><listitem>
// This function computes various visibility parameters: 
// <para>- satellite's semi view-angle (<emphasis role="bold">sat</emphasis>): 
// angle between the descending vertical and the direction of the target location </para>
// <para>- satellite's elevation (<emphasis role="bold">elev</emphasis>): 
// elevation of the satellite as seen from the target location </para>
// <para>- satellite's incidence (<emphasis role="bold">incid</emphasis> = pi/2 - elev): 
// incidence of the target-satellite direction from the target location </para>
// <para>- distance (<emphasis role="bold">dist</emphasis>): 
// distance between the satellite and the target location </para>
// <para>- centre angle (<emphasis role="bold">cen</emphasis>): 
// angle between the (planet centre -> satellite) direction and the (planet centre -> target location) direction. </para>
// <para>Given the distance from the planet centre to the satellite (<emphasis role="bold">sat_radius</emphasis>),
// the distance from the planet centre to the target (<emphasis role="bold">target_radius</emphasis>) 
// and one of the following parameters (<emphasis role="bold">type_par1</emphasis>):</para>
// <para>type_par1 = 'sat' , par1 = satellite's semi view angle in radians.</para>
// <para>type_par1 = 'elev' , par1 = elevation from the target location in radians.</para>
// <para>type_par1 = 'incid' , par1 = incidence (=pi/2-elev) from the target location in radians.</para>
// <para>type_par1 = 'dist' , par1 = distance between the target location and the satellite in meters.</para>
// <para>type_par1 = 'cen' , par1 = centre angle: angle from the planet centre between the target location and the satellite in radians.</para>
// <para>The function computes <emphasis role="bold">par2</emphasis> whose type is defined by <emphasis role="bold">type_par2</emphasis></para> 
// <para><emphasis role="bold">type_par2</emphasis> can also be an array of strings (any of the 5 parameters above), 
// or 'all' and in that case the result is a structure whose fields are the 5 parameters above</para>
// <para><inlinemediaobject><imageobject><imagedata fileref="visiParams.gif"/></imageobject></inlinemediaobject></para>
// </listitem>
// <listitem>
// <para>Notes: </para>
// <para> - A spherical planet is assumed.</para>
// <para> - If the input parameter is 'sat' there usually exist 2 possible solutions.
// But only the one for which the elevation is positive is computed. </para>
// </listitem></itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// sat_radius: Distance from the satellite to the planet centre (satellite altitude + planet radius) [m] (1xN)
// target_radius: Distance from the target to the planet centre (target altitude + planet radius) [m] (1xN)
// type_par1: (string) Type of input parameter par1. It can be 'sat', 'elev', 'incid', 'dist', 'cen'
// par1: Satellite's semi view angle, elevation, indicence, distance or centre angle [rad,m] (1xN)
// type_par2: (string) Type of ouput parameter(s) to be computed. It can be 'sat', 'elev', 'incid', 'dist', 'cen' or a vector contaning any of them, or 'all'. If type_par2 == 'all', a structure is returned. 
// par2: Computed value(s): [rad,m] (1xN)
//
// Authors
// CNES - DCT/SB
//
// Examples
// sat_r = %CL_eqRad + 700.e3; // 700 km altitude
// target_r = %CL_eqRad + 200; // 200 m altitude
// // incidence for a 800km distance : 
// [incid]=CL_gm_visiParams(sat_r,target_r,'dist',800.e3,'incid')
//
// // elevation for a 10deg roll (swath) :
// [elev]=CL_gm_visiParams(sat_r,target_r,'sat',CL_deg2rad(10),'elev') 
//
// // distance for a 7deg centre angle :
// [dist]=CL_gm_visiParams(sat_r,target_r,'cen',CL_deg2rad(7),'dist') 
//
// // satellites's semi view angle and distance for a 15 deg incidence :
// [sat,dist]=CL_gm_visiParams(sat_r,target_r,'incid',CL_deg2rad(15),['sat','dist'])
//
// // all parameters for a 37deg swath :
// [result]=CL_gm_visiParams(sat_r,target_r,'sat',CL_deg2rad(37),'all');
// 

//


// Declarations:


// Code:

[lhs,rhs]=argn();

if (rhs ~= 5) 
   CL__error('This function requires 5 input arguments'); 
end

if (typeof(type_par2) <> 'string') 
   CL__error('Wrong type for argument type_par2'); 
end

if ~(size(type_par2,2) == lhs | (type_par2 == 'all' & lhs == 1) ) 
   CL__error('Wrong number of output parameters'); 
end

// calculation of angle at centre depending on input type_par1
if type_par1=='elev' | type_par1=='incid' // par1 = elevation or incidence

  if type_par1=='incid'
    angsit = %pi/2 - par1
  else
    angsit = par1
  end
  i1 = find( abs(angsit)>%pi/2 )
  if i1~=[] then CL__error('Elevation greater than pi/2'); end
  angcen = acos( (target_radius./sat_radius).*cos(angsit) ) - angsit

elseif type_par1=='sat'  //par1 is satellite angle

  angsat = par1
  i2 = find( abs(angsat)>asin(target_radius./sat_radius) )
  if i2~=[] then CL__error('Satellite angle greater than asin(target_radius/sat_radius) : no intersection with the Earth'); end
  angsit = acos( (sat_radius./target_radius) .* sin(angsat) )
  //2 solutions
  angcen = %pi/2 - angsat - angsit
  //warning('only solution for elev>=0 is computed');
  //angcen = %pi/2 - angsat + angsit

elseif type_par1=='dist' //par1 is distance

  dist = par1
  i3 = find( dist<sat_radius-target_radius | dist>sat_radius+target_radius )
  if i3~=[] then CL__error('Distance too low (<sat_radius-target_radius) or too high (>sat_radius+target_radius)'); end
  angcen = acos( (sat_radius.^2 + target_radius.^2 - dist.^2) ./ (2.*sat_radius.*target_radius))

elseif type_par1=='cen'  //par1 is angle at centre

  angcen = par1

else
  CL__error('Unknown type_par1 value');
end

// calculation of output parameters as function of centre angle
res = struct('sat', 0 , 'elev', 0 , 'incid', 0 , 'dist', 0, 'cen', 0);

res.elev = atan(cos(angcen) - target_radius./sat_radius, sin(angcen)); 
res.incid = %pi/2 - res.elev; 
res.sat = %pi/2 - angcen - res.elev;
res.dist = sqrt(sat_radius.^2 + target_radius.^2 - 2.*sat_radius.*target_radius.*cos(angcen));
res.cen = angcen;

// output
if (type_par2 == 'all') 
  varargout(1) = res;
else
  for k = 1:size(type_par2,2)
    varargout(k) = res(type_par2(k));
  end
end

endfunction
