//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [cir,jacob]=CL_oe_car2cir(pos_car,vel_car,mu)
// Cartesian to circular adapted orbital elements
//
// Calling Sequence
// [cir[,jacob]]=CL_oe_car2cir(pos_car,vel_car[,mu])
//
// Description
// <itemizedlist><listitem>
// <p>Converts position and velocity to orbital elements adapted to near-circular orbits (elliptic orbit only).</p>
// <p>The transformation jacobian is optionally computed.</p>
// <p>See <link linkend="Orbital Elements">Orbital elements</link> for more details</p>
// </listitem>
// </itemizedlist>
//
// Parameters
// pos_car: position [X;Y;Z] [m] (3xN)
// vel_car: velocity [Vx;Vy;Vz] [m/s] (3xN)
// mu : (optional) gravitational constant [m^3/s^2] (default value is %CL_mu)
// cir: orbital elements adapted to near-circular orbits [sma;ex;ey;inc;raan;pom+anm] [m,rad] (6xN)
// jacob: (optional) transformation jacobian (See <link linkend="Orbital Elements">Orbital elements</link> for more details) (6x6xN)
//
// Bibliography
// 1) CNES - MSLIB FORTRAN 90, Volume V (mv_car_cir)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_oe_cir2car
// CL_oe_car2cirEqua
// CL_co_car2sph
// CL_co_car2ell
// CL_oe_car2kep
//
// Examples
// // Example 1
// pos_car=[-5910180;4077714;-620640];
// vel_car=[129;-1286;-7325];
// [cir,jacob1] = CL_oe_car2cir(pos_car,vel_car);
// [pos_car2,vel_car2,jacob2] = CL_oe_cir2car(cir);

//!************************************************************************
//! But:  Passage des parametres CARtesiens aux parametres orbitaux dits adaptes aux orbites circulaires
//! ===   non equatoriales. (x,y,z,x',y',z') --> (a, e cos (pom), e sin(pom), i, gom, pom+M)
//!
//! Note d'utilisation: 0) Applicable aux orbites circulaires et elliptiques, NON equatoriales.
//! ==================  1) Pour les orbites hyperboliques ou paraboliques, voir la routine mv_car_kep.
//!                     2) Pour les orbites circulaires et/ou elliptiques ET equatoriales, voir mv_car_cir_equa.
//!                     3) Les unites en entree doivent etre coherentes entre elles. Ex.: pos_car en metres, vel_car en m/s,
//!                        mu en m**3/s**2, et les parametres orbitaux dits adaptes aux orbites circulaires seront en metres
//!                        et radians.
//!                     4) L'element (i,j) de la jacobienne correspond a la derivee partielle du parametre adapte aux orbites
//!                        circulaires numero i par rapport a la derivee partielle du parametre cartesien numero j.
//!                     5) La transformation inverse se fait par mv_cir_car.
//!************************************************************************


// Declarations:
if(~exists('%CL_epsilon')) then global %CL_epsilon; end;
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

[lhs,rhs]=argn();
if ~exists('mu','local') then mu=%CL_mu; end
if lhs==2 then compute_jacob=%t; else compute_jacob=%f; end

N = size(pos_car,2);

eps100 = 100*%eps;


r = CL_norm(pos_car);
ii = find(r < eps100);
if (ii~=[]); CL__error("Null position vector"); end

r2 = r.^2; //carre du rayon vecteur r2

v = CL_norm(vel_car);
jj = find(v < eps100);
if (jj~=[]); CL__error("Null velocity vector "); end

v2 = v.^2; //carre de la norme de la vitesse v2

unsm = 1.0./mu
t1 = r.*v2.*unsm - 1  //calcul de t1=r*v2/mu - 1  avec t1 < 1 pour une orbite circulaire ou elliptique
                      //t1 = eccentricite* cos(anom. eccentrique)
jj = find(t1>1-%CL_epsilon.parab)
if jj~=[] then CL__error('semi major axis not > 0'); end
//condition non suffisante pour verifier la non ellipticite
//mais assure un demi-grand axe > 0 et la definition de unmt1

unmt1 = 1.0./(1-t1)
a = r.*unmt1 //calcul du demi-grand axe

cir(1,:) = a

rrp = CL_cross(pos_car,vel_car)
anorm = CL_norm(rrp) //call mu_prod_vect(pos_car,vel_car,rrp, code_retour)   ! rrp(3) = produit vectoriel position * vitesse
w = rrp./(anorm.*.ones(3,1))     //call mu_norme(rrp, anorm, code_retour,vect_norme = w) ! calcul de w (produit vectoriel norme position-vitesse)

aa = find(anorm < eps100)
if aa~=[] then CL__error('position and velocity vectors are colinear'); end

unsan = 1.0./anorm

//calcul de l'inclinaison
si2 = w(1,:).^2 + w(2,:).^2 //>0 par calcul
si = sqrt(si2)
ci = w(3,:)
xi = CL__sc2angle(ci,si) //call mu_angle2(ci,si,xi,code_retour)!code retour = 0 car (w(1)**2+w(2)**2) et w(3) differents de 0
                     //en meme temps car anorm different de 0
pp = find(sin(xi)<%CL_epsilon.equa)
if pp~=[] then
  CL__error('equatorial orbit not handled');
end

cir(4,:) = xi

omega = CL__sc2angle(-w(2,:),w(1,:)) //call mu_angle2(-w(2),w(1),omega,code_retour)   ! calcul de l'ascension droite du noeud ascendant
                                 //code_retour%valeur = 0 sinon w(1)=w(2)=0 => sin(i)=0 => orb equatoriale interdite
cir(5,:) = omega

//calcul du vecteur eccentricite
usi = 1.0./si
crh = -pos_car(1,:).*w(2,:) + pos_car(2,:).*w(1,:)
crl = -pos_car(1,:).*w(1,:) - pos_car(2,:).*w(2,:)
rh = usi.*crh
rl = w(3,:).*usi.*crl + pos_car(3,:).*si

srrp = CL_dot(pos_car, vel_car) //produit scalaire position.vitesse

unsra = 1.0./sqrt(mu.*a)
t2 = unsra.*srrp  //calcul de t2 = produit scalaire (position * vitesse)/sqrt(rmu*a) = eccentricite * sin(anom. eccentrique)
                  //on ne teste pas t2: voir test sur tau2 ci-dessous

tau2 = t1.^2 + t2.^2 //eccentricite**2 = t1**2 + t2**2
mm = find(sqrt(tau2)>1-%CL_epsilon.parab)
if mm~=[] then CL__error('parabolic or hyperbolic orbit'); end

c1 = t1 - tau2
c2 = sqrt(1.0 - tau2) //c2 non nul car tau2 < 1
asr2 = a./r2
ac2sr2 = asr2.*c2
cof1 = c1.*asr2
cof2 = ac2sr2.*t2

ex = cof1.*rh + cof2.*rl
ey = cof1.*rl - cof2.*rh

//vecteur eccentricite
cir(2,:) = ex
cir(3,:) = ey

//calcul de la longitude moyenne
betaa = 1.0./(1+c2)
t2beta = t2.*betaa
cle = rh./a + ex - t2beta.*ey
sle = rl./a + ey + t2beta.*ex

xle = CL__sc2angle(cle,sle) //call mu_angle2(cle,sle,xle,code_retour)   !calcul de (petit omega + e)
xlm = xle-t2 //calcul de la longitude moyenne (petit omega + m)

cir(6,:) = xlm

//calcul du jacobien de la transformation
if compute_jacob
  unsr = 1.0./r
  drsx = zeros(6,N)
  drsx(1:3,:) = (unsr.*.ones(3,1)).*pos_car //calcul des derivees de r
  drsx(4:6,:) = 0 //mise a zero de la partie non utile

  unsv = 1.0./v
  dvsx = zeros(6,N)
  dvsx(4:6,:)=(unsv.*.ones(3,1)).*vel_car //   ! calcul des derivees de v
  dvsx(1:3,:)= 0 //       ! mise a zero de la partie non utile

  deuxrv = 2.*r.*v
  dt1sx = unsm.*( drsx.*(v2.*.ones(6,1)) + (deuxrv.*.ones(6,1)).*dvsx ) //calcul des derivees de t1=r*v2/mu

  runmt1 = r.*unmt1.*unmt1
  dadx = (unmt1.*.ones(6,1)).*drsx + dt1sx.*(runmt1.*.ones(6,1))

  jacob1 = dadx //calcul des derivees du demi-grand axe

  drrp = hypermat([3,6,N])

  drrp(1,2,:)= vel_car(3,:)
  drrp(1,3,:)=-vel_car(2,:)
  drrp(1,5,:)=-pos_car(3,:)
  drrp(1,6,:)= pos_car(2,:)

  drrp(2,1,:)=-vel_car(3,:)    //calcul des derivees de w= produit vectoriel norme position*vitesse
  drrp(2,3,:)= vel_car(1,:)    //rrp(3) = produit vectoriel position * vitesse
  drrp(2,4,:)= pos_car(3,:)    //drrp(3,6) = derivees du produit vectoriel position * vitesse
  drrp(2,6,:)=-pos_car(1,:)

  drrp(3,1,:)= vel_car(2,:)
  drrp(3,2,:)=-vel_car(1,:)
  drrp(3,4,:)=-pos_car(2,:)
  drrp(3,5,:)= pos_car(1,:)

  // old code (not vectorized)
  //dan = zeros(6,N)
  //for i=1:N
  //  dan(:,i) = ( rrp(:,i)'*drrp(:,:,i) )'
  //end
  dan = drrp'*rrp;

  // old code (not vectorized)
  //dwdx = hypermat([3 6 N])
  //for i=1:N
  //  dwdx(:,:,i) = unsan(i) * ( drrp(:,:,i) - ((unsan(i)*unsan(i)*dan(:,i))'.*.ones(3,1)).*(rrp(:,i).*.ones(1,6)) )
  //end
  unsan_hm = hypermat([3,6,N] , [unsan.*.ones(1,18)]);
  dan_hm = hypermat([3,6,N] , [(dan' .*. ones(1,3))']);
  rrp_hm = hypermat([3,6,N] , [rrp.*.ones(1,6)]);
  dwdx = unsan_hm .* ( drrp - unsan_hm.^2 .* dan_hm .* rrp_hm);
  
  
  cotinc = ci./si //sinus de l'inclinaison non nulle car orbite non equatoriale
                  //remarque: cosinus de l'inclinaison = 0 si orbite polaire

  jacob4 = (cotinc.*.ones(6,1)) .* ...
           ( (w(1,:).*.ones(6,1)).*matrix(dwdx(1,:,:),6,N) + ...
           (w(2,:).*.ones(6,1)).*matrix(dwdx(2,:,:),6,N) ) - ...
           (si.*.ones(6,1)).*matrix(dwdx(3,:,:),6,N) //calcul des derivees de l'inclinaison

  jacob5 = ( (w(1,:).*.ones(6,1)).*matrix(dwdx(2,:,:),6,N) - ...
             (w(2,:).*.ones(6,1)).*matrix(dwdx(1,:,:),6,N) ...
           ) ./ (si2.*.ones(6,1))            // calcul des derivees de l'ascension droite du noeud ascendant

  usi2 = usi.^2

  dusi = -( (w(1,:).*.ones(6,1)).*matrix(dwdx(1,:,:),6,N) + ...
            (w(2,:).*.ones(6,1)).*matrix(dwdx(2,:,:),6,N) ...
          ) .* (usi2.*.ones(6,1))

  drhx = (usi.*.ones(6,1)) .* ( dusi.*(crh.*.ones(6,1)) - ...
                                (pos_car(1,:).*.ones(6,1)).*matrix(dwdx(2,:,:),6,N) + ...
                                (pos_car(2,:).*.ones(6,1)).*matrix(dwdx(1,:,:),6,N) )

  drlx = (usi.*.ones(6,1)) .* ( ...
             ( matrix(dwdx(3,:),6,N) + (w(3,:).*.ones(6,1)).*dusi ) .* (crl.*.ones(6,1)) - ...
             (w(3,:).*.ones(6,1)) .* ...
                ( (pos_car(1,:).*.ones(6,1)).*matrix(dwdx(1,:,:),6,N) + (pos_car(2,:).*.ones(6,1)).*matrix(dwdx(2,:,:),6,N) ) + ...
             (pos_car(3,:).*.ones(6,1)) .* ...
              ( (w(1,:).*.ones(6,1)).*matrix(dwdx(1,:,:),6,N) + (w(2,:).*.ones(6,1)).*matrix(dwdx(2,:,:),6,N) ) )


  drhx(1,:) = drhx(1,:) - usi.*w(2,:)
  drhx(2,:) = drhx(2,:) + usi.*w(1,:)
  drlx(1,:) = drlx(1,:) - w(3,:).*usi.*w(1,:)
  drlx(2,:) = drlx(2,:) - w(3,:).*usi.*w(2,:)
  drlx(3,:) = drlx(3,:) + si

  dsrrp = zeros(6,N)
  dsrrp(1:3,:)= vel_car  //derivees du produit scalaire position.vitesse
  dsrrp(4:6,:)= pos_car

  rdunsa = -0.5.*t2./a
  dt2sx = (unsra.*.ones(6,1)).*dsrrp + (rdunsa.*.ones(6,1)).*dadx   //derivees de t2

  deuxt1 = 2.0.*t1
  deuxt2=2.0.*t2
  r3 = r2.*r
  m2sr3=-2.0./r3
  c1sr2=c1./r2
  ac1=a.*c1
  tc2sr2=t2.*c2./r2
  at2c2=a.*t2.*c2
  coef=-0.5.*asr2.*t2./c2

  dc1x = dadx.*(c1sr2.*.ones(6,1)) + ...
         (ac1.*m2sr3.*.ones(6,1)).*drsx + ...
         (asr2.*.ones(6,1)).* ...
            ( dt1sx - (deuxt1.*.ones(6,1)).*dt1sx - (deuxt2.*.ones(6,1)).*dt2sx )

  dc2x = dadx.*(tc2sr2.*.ones(6,1)) + ...
         (at2c2.*m2sr3.*.ones(6,1)).*drsx + ...
         (ac2sr2.*.ones(6,1)).*dt2sx + ...
         (coef.*.ones(6,1)) .* ( (deuxt1.*.ones(6,1)).*dt1sx + (deuxt2.*.ones(6,1)).*dt2sx )

  dex = dc1x.*(rh.*.ones(6,1)) + ...
        (cof1.*.ones(6,1)).*drhx + ...
        dc2x.*(rl.*.ones(6,1)) + ...
        (cof2.*.ones(6,1)).*drlx //calcul des derivees du vecteur eccentricite

  dey = dc1x.*(rl.*.ones(6,1)) + (cof1.*.ones(6,1)).*drlx - dc2x.*(rh.*.ones(6,1)) - (cof2.*.ones(6,1)).*drhx

  jacob2 = dex
  jacob3 = dey

  beta2 = betaa.^2

  dla = ( drhx - (rh.*.ones(6,1)).*dadx./(a.*.ones(6,1)) ) ./ (a.*.ones(6,1))
  dmu = ( drlx - (rl.*.ones(6,1)).*dadx./(a.*.ones(6,1)) ) ./ (a.*.ones(6,1))
  dbeta = ( (t1.*.ones(6,1)).*dt1sx + dt2sx.*(t2.*.ones(6,1)) ) .* (beta2 ./c2.*.ones(6,1))
  dt2b = dt2sx.*(betaa.*.ones(6,1)) + (t2.*.ones(6,1)).*dbeta
  dcle = dla + dex - (t2beta.*.ones(6,1)).*dey - dt2b.*(ey.*.ones(6,1))
  dsle = dmu + dey + (t2beta.*.ones(6,1)).*dex + dt2b.*(ex.*.ones(6,1))
  un=cle.^2+sle.^2
  dle = ((cle.*.ones(6,1)).*dsle - (sle.*.ones(6,1)).*dcle) ./ (un.*.ones(6,1))

  jacob6 = dle - dt2sx //calcul des derivees de la longitude moyenne

  jacob = hypermat([6,6,N], [jacob1;jacob2;jacob3;jacob4;jacob5;jacob6]);
  jacob = jacob';

  if N==1 then jacob=jacob(:,:,1); end

end

endfunction
