//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [raan1,raan2]=CL_gm_beta2raan(betaa,inc,alpha_sun,delta_sun)
// Beta angle to right ascension of ascending node 
//
// Calling Sequence
// [raan1,raan2] = CL_gm_beta2raan(betaa,inc,alpha_sun,delta_sun)
//
// Description
// <itemizedlist><listitem>
// <p>Computes the right ascension of the ascending node (<b>raan</b>)
// from the beta angle (<b>betaa</b>), the Sun's right ascension 
// (<b>alpha_sun</b>) and declination (<b>delta_sun</b>), 
// and the inclination of the orbit (<b>inc</b>).</p>
// <p></p>
// <p>There are 2 solutions (<b>raan1</b>) and (<b>raan2</b>) such that: </p>
// <p>- raan2 = pi - raan1 + 2*alpha_sun </p>
// <p>- raan1 is in the interval [ alpha_sun - pi/2 ; alpha_sun + pi/2] modulo 2*pi</p>
// <p>- raan2 is in the interval [ alpha_sun + pi/2 ; alpha_sun + 3*pi/2] modulo 2*pi</p>
// <p></p>
// <p>Beta is the angle between the satellite's orbit plane and the Sun direction:</p>
// <inlinemediaobject><imageobject><imagedata fileref="beta.gif"/></imageobject></inlinemediaobject></listitem>
// <listitem>
// <p><b>Notes:</b></p>
// <p>Beta is a signed quantity. If it is positive then the Sun direction is less
// than 90 degrees from the orbit's angular momentum vector (vector perpendicular to the
// orbit plane and oriented according to right hand rule). </p></listitem>
// </itemizedlist>
//
// Parameters
// betaa: Beta angle [rad] (1xN)
// inc: Inclination of the orbit [rad] (1xN)
// alpha_sun: Sun right ascension [rad] (1xN)
// delta_sun: Sun declination [rad] (1xN)
// raan1: Right ascension of the ascending node (solution 1)  [rad] (1xN)
// raan2: Right ascension of the ascending node (solution 2) [rad] (1xN)
//
// See also
// CL_op_locTimeG50
// CL_gm_raan2beta
// CL_mod_moonSunG50
//
// Authors
// CNES - DCT/SB
//
// Examples
// // Get alpha_sun and delta_sun
// cjd = 20050;
// [r_sun,rs] = CL_mod_moonSunG50(cjd,'s');
// r_sph = CL_co_car2sph(r_sun);
// alpha_sun = r_sph(1);
// delta_sun = r_sph(2);
// betaa = %pi/4;
// inc = CL_deg2rad(98.7);
// [raan1,raan2] = CL_gm_beta2raan(betaa,inc,alpha_sun,delta_sun)

// Coherence des entrees

// Declarations:


// Code:

Nbetaa = size(betaa,2)
Ninc = size(inc,2)
Nalpha_sun = size(alpha_sun,2)
Ndelta_sun = size(delta_sun,2)
N = max(Nbetaa,Ninc,Nalpha_sun,Ndelta_sun)
coherence = (Nbetaa==1|Nbetaa==N)&(Ninc==1|Ninc==N)&(Nalpha_sun==1|Nalpha_sun==N)&(Ndelta_sun==1|Ndelta_sun==N)
if ~coherence then CL__error('bad size of input arguments'); end
if N~=1
  if Nbetaa==1 then betaa=betaa*ones(1,N); end
  if Ninc==1 then inc=inc*ones(1,N); end
  if Nalpha_sun==1 then alpha_sun=alpha_sun*ones(1,N); end
  if Ndelta_sun==1 then delta_sun=delta_sun*ones(1,N); end
end


raan1 = zeros(betaa);
raan2 = zeros(betaa);
f = zeros(betaa);
ii = find( sin(inc) ~= 0);
f(ii) = (sin(betaa(ii))-sin(delta_sun(ii)).*cos(inc(ii)))./(cos(delta_sun(ii)).*sin(inc(ii)))
raan1(ii) = CL_rMod(real(asin(f(ii))) + alpha_sun(ii),2*%pi);
raan2(ii) = CL_rMod(%pi - raan1(ii) + 2*alpha_sun(ii),2*%pi);

jj = find( sin(inc) == 0);
if( jj~=[])
  CL__error('Inclination equals 0 or pi : raan undefined');
end

// Si le resultat est complexe : c'est que beta demande est superieur au beta max...
if or (f > 1+%eps | f < -1-%eps)
   CL__error('Beta cannot be superior to inclination + declination');
end


endfunction
