//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [pos_car,vel_car,jacob]=CL_co_sph2car(pos_sph,vel_sph)
// Spherical coordinates to cartesian coordinates
//
// Calling Sequence
// [pos_car [,vel_car,jacob]] = CL_co_sph2car(pos_sph [,vel_sph])
//
// Description
// <itemizedlist><listitem>
// <p>Converts cartesian coordinates into spherical coordinates.</p>  
// <p> The input arguments can consist in position only, or position and velocity, in which case
// the derivatives of the spherical coordinates with respect to time are computed as well. </p> 
// <para/><inlinemediaobject><imageobject><imagedata fileref="spherical_coord.gif"/></imageobject></inlinemediaobject>
// <p></p></listitem>
// <listitem>
// <p><b>Notes:</b></p>
// <p> - The transformation jacobian is computed if the corresponding output argument exists. </p>
// </listitem>
// </itemizedlist>
//
// Parameters
// pos_sph : [lon;lat;r] Position vector in spherical coordinates [m,rad] (3xN)
// vel_sph : (optional) [d(lon)/dt;d(lat)/dt;d(r)/dt] Derivatives / time of spherical coordinates [rad/s,m/s] (3xN)
// pos_car: [X;Y;Z] Position vector in cartesian coordinates [m] (3xN)
// vel_car: (optional) [Vx,Vy,Vz] Velocity vector in cartesian coordinates [m/s] (3xN)
// jacob:  (optional) Transformation jacobian (6x6xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1) Mecanique Spatiale, Cnes - Cepadues Editions, Tome I, section 3.2.3 (Les reperes de l'espace et du temps, Relations entre les coordonnees)
// 2) CNES - MSLIB FORTRAN 90, Volume T (mt_geoc_car)
//
// See also
// CL_co_car2sph
// CL_co_car2ell
// CL_co_ell2car
//
// Examples
// // Example 1
// pos_sph = [0,0,%CL_eqRad;%pi/2,%pi/2,%CL_eqRad]';
// pos_car = CL_co_sph2car(pos_sph);
//
// // Example 2 :
// pos_sph = [0.090715;5.362077;6377951.7];
// vel_sph = [1.8332e-5;2.2060e-4;-181.488];
// [pos_car,vel_car,jacob1] = CL_co_sph2car(pos_sph,vel_sph);

// Declarations:


// Code:

[lhs,rhs] = argn();

select lhs
  case 1
    compute_velocity = %f
    compute_jacob = %f
  case 2
    compute_velocity = %t
    compute_jacob = %f
    if ~exists('vel_sph','local') then CL__error('give velocity as input argument'); end
  case 3
    compute_velocity = %t;
    compute_jacob = %t;
    if ~exists('vel_sph','local') then CL__error('give velocity as input argument'); end
  else
    CL__error('check number of output arguments')
end

// Distance
d = pos_sph(3,:);

// cos(lon),sin(lon),cos(lat),sin(lat)
clon = cos(pos_sph(1,:));
slon = sin(pos_sph(1,:));
clat = cos(pos_sph(2,:));
slat = sin(pos_sph(2,:));

clat_clon = clat.*clon;
clat_slon = clat.*slon;

// Tansform into cartesian coordinates
x = d.*clat_clon;
y = d.*clat_slon;
z = d.*slat;
pos_car = [x;y;z];

if compute_velocity
  slat_slon = slat.*slon;
  slat_clon = slat.*clon;
  lat_v = vel_sph(2,:);
  lon_v = vel_sph(1,:);
  d_v = vel_sph(3,:);

  vx = d_v.*clat_clon - z.*clon.*lat_v - y.*lon_v;
  vy = d_v.*clat_slon - z.*slon.*lat_v + x.*lon_v;
  vz = d_v.*slat + d.*clat.*lat_v;
  vel_car = [vx;vy;vz];
end

if compute_jacob
  N = size(pos_sph,2);
  jacob = zeros(6,6,N);
  //x partial derivative
  jacob(1,2,1:N) = -d.*slat_clon;
  jacob(1,1,1:N) = -y;
  jacob(1,3,1:N) = clat_clon;
  //y partial derivative
  jacob(2,2,1:N) = -d.*slat_slon;
  jacob(2,1,1:N) = x;
  jacob(2,3,1:N) = clat_slon;
  //z partial derivative
  jacob(3,2,1:N) = d.*clat;
  jacob(3,3,1:N) = slat;
  //vx partial derivative
  jacob(4,2,1:N) = -vz.*clon + z.*slon.*lon_v;
  jacob(4,1,1:N) = -vy;
  jacob(4,3,1:N) = -slat_clon.*lat_v - clat_slon.*lon_v;
  jacob(4,5,1:N) = -d.*slat_clon;
  jacob(4,4,1:N) = -y;
  jacob(4,6,1:N) = clat_clon;
  //vy partial derivative
  jacob(5,2,1:N) = -vz.*slon - z.*clon.*lon_v;
  jacob(5,1,1:N) = vx;
  jacob(5,3,1:N) = clat_clon.*lon_v - slat_slon.*lat_v;
  jacob(5,5,1:N) = -d.*slat_slon;
  jacob(5,4,1:N) = x;
  jacob(5,6,1:N) = clat_slon;
  //vz partial derivative
  jacob(6,2,1:N) = d_v.*clat - z.*lat_v;
  jacob(6,3,1:N) = clat.*lat_v;
  jacob(6,5,1:N) = d.*clat;
  jacob(6,6,1:N) = slat;

end

endfunction
