//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [path] = CL_path(name, directories)
// Looks for a file in directories
//
// Calling Sequence
// [path] = CL_path(name [,directories])
//
// Description
// <itemizedlist><listitem>
// <p>Looks for a file in a list of directories. Only the path of the first file found is returned.</p>
// <p>If <b>directories</b> is absent, the variable %CL_PATH is used if 
// it is defined. If %CL_PATH is not defined, the working directory is considered. </p>
// <p>An error is raised if the file is not found. </p>
// <p>Note : Wild card '*' can be used at the end of a directory to look in all sub-directories (only one level)</p>
// </listitem>
// </itemizedlist>
//
// Parameters
// name: (string) Name of file to be looked for (1x1)
// directories: (string, optional) Directories to be searched (1xN) (default is %CL_PATH or '.' 
// if %CL_PATH is not defined)
// path: (string) File path (1x1)
//
// Authors
// CNES - DCT/SB
//
// Examples
// dirlist = ['.', '/users/scripts', 'D:/data' , 'D:/data/*'];
// // Looks for script toto.sce in the given directories :
// CL_path('toto.sce', dirlist);
// // loads script using implicit directory list:
// %CL_PATH = ['.', '/users/scripts', 'D:/data']; 
// exec(CL_path('toto.sce'));
// // Absolute file path:
// abs_path = fullpath(CL_path('toto.sce'))

// Declarations:


// Code:
  if (~exists('directories', 'local'))
    if (~exists('%CL_PATH', 'nolocal'))
       %CL_PATH = ['.'];  
    end
    directories = %CL_PATH;
  end
  
  //if (~exists('verbose', 'local'))
    verbose = %f;
  //end
  
  if (size(directories,1) > 1) 
     CL__error("directories must be (1xN)."); 
  end
  
  // if a directory ends with *, add all sub-directories to the list of directories
  Nd = size(directories,2);
  directories_out = [];
  for k = 1 : Nd
    rep = directories(k);
    if (part(rep,length(rep)) == '*')
	files = listfiles(rep);
      if (files ~= [])
        sub_dirs = files(find(isdir(files)))';
        directories_out = [ directories_out , sub_dirs ];
      end
    else
      directories_out = [directories_out , directories(k)];
    end
  end
  
  // looks for the file
  found = 0;
  for rep = directories_out 
    if (length(rep) <> 0 & found == 0) 
      path = fullfile(rep, name);  // catenates directory and name
      [info, err] = fileinfo(path); 
      if (err == 0) // found! 
         found = 1;
         break; 
      end
    end
  end

  if (found == 0) 
    CL__error(sprintf("File ''%s'' not found.", name)); 
  end
  
  if (found == 1 & verbose) 
    fprintf(%io(2),"--- CelestLab Info: CL_path = ''%s'' \n", path); 
  end

endfunction


