/*  Screem:  object-wizard.c
 *
 *  A Wizard for building up CSS selector patterns
 * 
 *  (C) Copyright 2004 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>

#include <libgnome/gnome-i18n.h>
#include <libgnomeui/libgnomeui.h>
#include <libgnomevfs/gnome-vfs.h>

#include <gtk/gtk.h>

#include <glade/glade.h>

#include <string.h>

#include "screem-plugin.h"

#include "screem-window.h"
#include "screem-editor.h"

#include "fileops.h"

/* per wizard struct */
typedef struct {
	ScreemWindow *window;
	ScreemEditor *editor;

	gchar *type;
} ObjectWizard;


/* keep track of how many wizards we have */
static GList *wizards = NULL;

static GnomeThumbnailFactory *factory = NULL;

/* Required Plugin parts */

/* we don't want to try and load old plugins
   so this is a version symbol for screem to spot */
int screem_plugin_version = 4;

#define NAME "Insert Object Wizard"
#define AUTHOR "David A Knight (david@screem.org)"
#define PLUGIN_VERSION "1.0"
#define TAG NULL


G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module );
G_MODULE_EXPORT void g_module_unload( GModule *module );
G_MODULE_EXPORT void init( ScreemPlugin *plugin );
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view );
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view );

void object_wizard_display( GtkAction *action, gpointer user_data );
static void insert_object_markup( ObjectWizard *wizard,
		GtkWidget *widget );

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
}

G_MODULE_EXPORT void init( ScreemPlugin *plugin )
{
	plugin->name = NAME;
	plugin->author = AUTHOR;
	plugin->version = PLUGIN_VERSION;

	/* these can all be left out if not required */
	plugin->tag = TAG;

	g_module_symbol( plugin->module, "popup", 
			 (gpointer*)&plugin->popup );
	g_module_symbol( plugin->module, "add_ui", 
			 (gpointer*)&plugin->add_ui );
	g_module_symbol( plugin->module, "remove_ui", 
			 (gpointer*)&plugin->remove_ui );

	factory = gnome_thumbnail_factory_new( GNOME_THUMBNAIL_SIZE_NORMAL );
}

/* not needed if nothing is being added to the UI (unusual) */
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view )
{
	ObjectWizard *wizard;

	const gchar *ui = "\
<ui>\
<menubar>\
<menu action=\"Insert\">\
<menu action=\"Wizards\">\
<menuitem action=\"ObjectWizard\"/>\
</menu>\
</menu>\
</menubar>\
<toolbar name=\"Wizards Toolbar\">\
<toolitem action=\"ObjectWizard\" />\
</toolbar>\
</ui>";
	GtkAction *action;
	
	gchar *label;
	gchar *tip;
	GError *error;
	
	wizard = g_new0( ObjectWizard, 1 );
	wizard->window = SCREEM_WINDOW( window );
	wizard->editor = SCREEM_EDITOR( editor );
	
	label = g_strdup( _( "Object" ) );
	tip = g_strdup( _( "A Wizard to insert an object, such as a video clip, sound clip, image etc." ) );
	action = gtk_action_new( "ObjectWizard", label, tip,
			"Screem_Object" );
	g_signal_connect( G_OBJECT( action ), "activate",
			G_CALLBACK( object_wizard_display ), wizard );
	gtk_action_group_add_action( GTK_ACTION_GROUP( wizard->window->action_group ), action );
	g_free( label );
	g_free( tip );

	error = NULL;
	if( ! gtk_ui_manager_add_ui_from_string( GTK_UI_MANAGER( wizard->window->merge ),
						ui, strlen( ui ), &error ) ) {
		g_message( "%s ui error = %s", "object wizard",
				error->message );
		g_error_free( error );
	}

	wizards = g_list_append( wizards, wizard );
}

/* required it add_ui is present */
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view )
{
	GList *list;
	ObjectWizard *wizard;

	for( list = wizards; list; list = list->next ) {
		wizard = (ObjectWizard*)list->data;
		if( wizard->window == SCREEM_WINDOW( window ) ) {
			/* got it */
			break;
		}
	}
       
	g_return_if_fail( list != NULL );

	/* wizard is the one to erase */
	wizards = g_list_remove( wizards, wizard );
	g_free( wizard->type );
	g_free( wizard );
}

/* End of required section */

void object_wizard_display( GtkAction *action, gpointer user_data )
{
	ObjectWizard *wizard;
	ScreemSite *site;
	ScreemPage *page;
	GtkWidget *widget;
	GladeXML *xml;
	gint button;
	ScreemApplication *app;
	ScreemSession *session;
	const gchar *pathname;
	
	wizard = (ObjectWizard*)user_data;
	page = screem_window_get_document( wizard->window );

	app = SCREEM_APPLICATION( wizard->window->application );
	session = screem_application_get_session( app );
	
	if( page ) {
		site = screem_window_get_current( wizard->window );
		pathname = screem_site_get_pathname( site );
		
		xml = glade_xml_new( GLADE_PATH"/object-wizard.glade",
				"insert_object", NULL );

		widget = glade_xml_get_widget( xml, "data" );
		g_object_set_data( G_OBJECT( widget ), "wizard",
				wizard );
		if( pathname ) {
			gnome_file_entry_set_default_path( GNOME_FILE_ENTRY( widget ), pathname );
		}

		widget = glade_xml_get_widget( xml, "copy" );
		pathname = screem_page_get_pathname( page );
		gtk_widget_set_sensitive( widget, pathname != NULL );
		
		glade_xml_signal_autoconnect( xml );
		
		widget = glade_xml_get_widget( xml, "insert_object" );
		
		screem_session_restore_dialog( session, widget );
		
		button = gtk_dialog_run( GTK_DIALOG( widget ) );

		screem_session_store_dialog( session, widget );

		if( button == GTK_RESPONSE_APPLY ) {
			insert_object_markup( wizard, widget );		
		}
		
		widget = glade_xml_get_widget( xml, "insert_object" );
		gtk_widget_destroy( widget );
		g_object_unref( G_OBJECT( xml ) );
	}
}

void insert_object_href_changed( GtkWidget *entry )
{
	ObjectWizard *wizard;
	GladeXML *xml;
	const gchar *href;
	GnomeVFSFileInfo *info;
	gchar *uri;
	GdkPixbuf *pixbuf;
	GtkWidget *preview;
	
	gint width;
	gint height;
	
	wizard = g_object_get_data( G_OBJECT( entry ), "wizard" );
	xml = glade_get_widget_tree( entry );
		
	entry = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( entry ) );

	href = gtk_entry_get_text( GTK_ENTRY( entry ) );
	
	info = gnome_vfs_file_info_new();

	g_free( wizard->type );
	wizard->type = NULL;
	pixbuf = NULL;
	if( gnome_vfs_get_file_info( href, info, 
				GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
				GNOME_VFS_FILE_INFO_FOLLOW_LINKS ) ==
			GNOME_VFS_OK ) {
		wizard->type = g_strdup( info->mime_type );

		pixbuf = gnome_gdk_pixbuf_new_from_uri( href );
		if( pixbuf ) {
			width = gdk_pixbuf_get_width( pixbuf );
			height = gdk_pixbuf_get_height( pixbuf );
			g_object_unref( G_OBJECT( pixbuf ) );
		} else {
			width = 0;
			height = 0;
		}
		entry = glade_xml_get_widget( xml, "width" );
		gtk_spin_button_set_value( GTK_SPIN_BUTTON( entry ),
				width );
		entry = glade_xml_get_widget( xml, "height" );
		gtk_spin_button_set_value( GTK_SPIN_BUTTON( entry ),
				height );
		
		uri = gnome_thumbnail_factory_lookup( factory,
				href, info->mtime );
		if( uri ) {
			pixbuf = gdk_pixbuf_new_from_file( uri, NULL );	
			g_free( uri );
		} else {
	
			pixbuf = gnome_thumbnail_factory_generate_thumbnail( factory, href, info->mime_type );
			if( pixbuf ) {
				gnome_thumbnail_factory_save_thumbnail( factory, pixbuf, href, info->mtime );
			}
		}
	}
	preview = glade_xml_get_widget( xml, "preview" );
	gtk_image_set_from_pixbuf( GTK_IMAGE( preview ), pixbuf );
	entry = glade_xml_get_widget( xml, "thumb" );
	gtk_widget_set_sensitive( entry, pixbuf != NULL );
	if( pixbuf ) {
		g_object_unref( G_OBJECT( pixbuf ) );
	} else {
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( entry ),
				FALSE );
	}
	
	gnome_vfs_file_info_unref( info );
}

static void insert_object_markup( ObjectWizard *wizard,
		GtkWidget *widget )
{
	const gchar *fmt;
	const gchar *tfmt;
	GladeXML *xml;
	gchar *markup;

	gint width;
	gint height;
	const gchar *alt;
	const gchar *href;

	ScreemPage *page;
	const gchar *pathname;

	gchar *thumbnail;
	
	gchar *base;
	gchar *uri;
	gchar *thumb;
	gchar *tmp;

	gboolean copy;
	gboolean dothumb;
	GdkPixbuf *pixbuf;

	ScreemApplication *app;
	
	app = SCREEM_APPLICATION( wizard->window->application );
	
	fmt = "<object data=\"%s\" width=\"%i\" height=\"%i\" type=\"%s\">%s</object>";

	tfmt = "<a href=\"%s\"><img src=\"%s\" width=\"%i\" height=\"%i\" alt=\"%s\"></a>";
	
	page = screem_window_get_document( wizard->window );
	pathname = screem_page_get_pathname( page );
	
	xml = glade_get_widget_tree( widget );

	widget = glade_xml_get_widget( xml, "alt" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	alt = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "data" );
	widget = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( widget ) );
	href = gtk_entry_get_text( GTK_ENTRY( widget ) );

	if( ! wizard->type ) {
		wizard->type = g_strdup( "application/octet-stream" );
	}
	
	widget = glade_xml_get_widget( xml, "copy" );
	copy = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	thumbnail = gnome_thumbnail_path_for_uri( href,
			GNOME_THUMBNAIL_SIZE_NORMAL );

	if( copy ) {
		base = g_path_get_basename( href );
		thumb = g_strconcat( "thumb-", base, NULL );
		g_free( base );
	} else {
		thumb = g_strdup( thumbnail );
	}
	
	if( pathname ) {
		base = g_path_get_dirname( pathname );
	
		uri = relative_path( href, base );
	} else {
		base = NULL;
		uri = g_strdup( href );
	}
	
	widget = glade_xml_get_widget( xml, "thumb" );
	dothumb = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	if( dothumb ) {
		pixbuf = gdk_pixbuf_new_from_file( thumbnail, NULL );
		if( pixbuf ) {
			width = gdk_pixbuf_get_width( pixbuf );
			height = gdk_pixbuf_get_height( pixbuf );
			g_object_unref( G_OBJECT( pixbuf ) );
		} else {
			width = 0;
			height = 0;
		}
		markup = g_strdup_printf( tfmt, uri, thumb, width,
				height, alt );
	} else {
		widget = glade_xml_get_widget( xml, "width" );
		width = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
		widget = glade_xml_get_widget( xml, "height" );
		height = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );

		markup = g_strdup_printf( fmt, uri, width, height, 
				wizard->type, alt );
	}

	screem_editor_insert( wizard->editor, -1, markup );
	
	g_free( uri );
	g_free( markup );

	if( copy ) {
		uri = g_path_get_basename( href );
		tmp = g_build_path( G_DIR_SEPARATOR_S,
				base, uri, NULL );
		copy_file( href, tmp, 
				screem_application_file_op, app );
		g_free( tmp );
	
		if( dothumb ) {
			tmp = g_build_path( G_DIR_SEPARATOR_S,
					base, thumb, NULL );
			copy_file( thumbnail, tmp, 
					screem_application_file_op,
					app );
			g_free( tmp );
		}
		g_free( uri );
	}
	g_free( thumbnail );
	g_free( thumb );
	g_free( base );
}

