/*  Screem:  screem-application.c
 *
 *  The main screem application code, manages startup and generally looks
 *  after things
 *
 *  Copyright (C) 2001  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <errno.h>

#include <gmodule.h>

#ifdef HAVE_DBUS

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>

#endif

#include <gtk/gtk.h>

#include <gdk-pixbuf/gdk-pixbuf.h>

#include <glib/gi18n.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <unistd.h>

#include <libxml/xmlreader.h>

#ifdef HAVE_PYTHON
#include <Python.h>
#endif

#include "pageUI.h"

#include "screem-application.h"
#include "screem-file-browser.h"
#include "screem-hint.h"
#include "screem-window.h"

#include "screem-editor.h"

#include "screem-window-private.h" /* YUK! */

#include "screem-plugin.h"  /* YUK */


#include "screem-site-ui.h" /* YUK again */

#include "screem-helper.h"

#include "screem-tagtree.h"

#include "support.h"
#include "fileops.h"

#include "screem-session.h"
#include "screem-dtd-db.h"

#include "screem-encodings.h"

struct ScreemApplicationPrivate {
	GList *window_list;
	GList *loaded_sites;
	
	ScreemSession *session;
	ScreemDTDDB *dtddb;

	GtkTreeModel *encodings;
#ifdef HAVE_DBUS
	DBusConnection *conn;
#endif
	GSList *start_files;
};

static void screem_application_init( ScreemApplication *application );
static void screem_application_class_init( ScreemApplicationClass *application );
static void screem_application_finalize( GObject *object );

static void screem_application_destroyed_window( GObject *object,
						 ScreemApplication *application );
static gboolean screem_window_delete_event_callback( GtkWidget *widget,
						     GdkEvent *event,
						     gpointer user_data );

static void screem_application_add_icons( ScreemApplication *application );

static gboolean screem_application_load( ScreemApplication *application );

static void screem_application_load_helpers( ScreemApplication *application );
static void screem_application_helper_failed( ScreemHelper *helper,
		const gchar *msg, ScreemApplication *application );
static void screem_application_helper_error( ScreemHelper *helper,
					     gint status,
					     ScreemApplication *application );

static void screem_application_load_macros( ScreemApplication *application );

#ifdef HAVE_DBUS
static DBusHandlerResult screem_application_dbus_filter( DBusConnection *conn,
		DBusMessage *message,
		gpointer data );
#endif

#define SPLASH_TEXT( w, t ) gtk_label_set_text( GTK_LABEL( w ),t ); \
gdk_threads_leave(); while( g_main_context_iteration(NULL,FALSE) ){} \
gdk_threads_enter();

static void screem_application_offline_notify( GConfClient *client,
					       guint cnxn_id,
					       GConfEntry *entry,
					       gpointer data );


GList *screem_application_get_window_list( ScreemApplication *app )
{
	GList *list;

	g_return_val_if_fail( SCREEM_IS_APPLICATION( app ), NULL );
	
	list = app->priv->window_list;

	return list;
}

void screem_application_close_all_windows( ScreemApplication *app )
{
	gboolean close;
	GList *list;

	close = FALSE;
	if( app->priv->window_list ) {
		close = screem_application_save_sites( app );
	}
	
	if( close ) {

		g_object_set_data( G_OBJECT( app ), "confirmed",
				GINT_TO_POINTER( 1 ) );
	
		screem_session_save( app->priv->session, SESSION_LAST );
		while( app->priv->window_list ) {
			ScreemWindow *window;

			window = SCREEM_WINDOW( app->priv->window_list->data );
			for( list = app->helpers; list; list = list->next ) {
				screem_helper_remove_from_window( list->data, window );
			}
			screem_window_close( window );
		}
	}
}

static gboolean screem_window_delete_event_callback( GtkWidget *widget,
						     GdkEvent *event,
						     gpointer user_data )
{
	ScreemWindow *window;
	ScreemApplication *app;
	GList *list;
	gboolean close;
	gboolean last;
	gboolean confirmed;
	
	window = SCREEM_WINDOW( widget );

	g_object_get( G_OBJECT( window ), "app", &app, NULL);
	
	close = TRUE;
	last = FALSE;
		
	/* if this is the last window we should confirm saving */
	list = screem_application_get_window_list( app );
	
	confirmed = ( g_object_get_data( G_OBJECT( app ), 
					"confirmed" ) != NULL );
	if( confirmed ) {
		close = FALSE;
	}
	
	if( g_list_length( list ) == 1 ) {
		last = TRUE; 
		if( ! confirmed ) {
			close = screem_application_save_sites( app );
		}
	} 
	if( close ) {
		if( last && ( gtk_main_level() > 0 ) ) {
			screem_session_save( app->priv->session, SESSION_LAST );
		}
		for( list = app->helpers; list; list = list->next ) {
			screem_helper_remove_from_window( list->data, window );
		}
		screem_window_close( window );
	}
	
	g_object_unref( app );
		
	return TRUE;
}

static void
screem_application_destroyed_window( GObject *object, ScreemApplication *app )
{
	GList *list;
	ScreemPlugin *plugin;

	app->priv->window_list = g_list_remove( app->priv->window_list, 
						object );

	for( list = app->plugins; list; list = list->next ) {
		plugin = (ScreemPlugin*)list->data;
		if( plugin->remove_ui ) {
			plugin->remove_ui( GTK_WIDGET( object ), 
					NULL, NULL, NULL );
		}
	}

	/* if all windows are closed then exit */
	if( ( gtk_main_level() > 0 ) && ! app->priv->window_list ) {
		gtk_main_quit();
	}
}

ScreemWindow *screem_application_create_window( ScreemApplication *app )
{
	ScreemWindow *window;
	GList *list;

	window = SCREEM_WINDOW( g_object_new( screem_window_get_type(),
					      "app",
					      G_OBJECT( app ),
					      "app_id", "screem", NULL ) );
	g_signal_connect( G_OBJECT( window ), "delete_event",
			  G_CALLBACK( screem_window_delete_event_callback ),
			  NULL );
	g_signal_connect( G_OBJECT( window ), "destroy",
			  G_CALLBACK( screem_application_destroyed_window ),
			  app );

	app->priv->window_list = g_list_prepend( app->priv->window_list, window );

	list = screem_application_get_loaded_sites( app );
	for( ; list; list = list->next ) {
		screem_window_add_site( window, SCREEM_SITE( list->data ) );
	}

	for( list = app->plugins; list; list = list->next ) {
		ScreemPlugin *plugin = (ScreemPlugin*)list->data;

		if( plugin->add_ui ) {
			plugin->add_ui( GTK_WIDGET( window ), 
					window->details->editor,
					window->details->preview, 
					window->details->link_view );
		}
	}
	for( list = app->helpers; list; list = list->next ) {
		ScreemHelper *helper = (ScreemHelper*)list->data;
		screem_helper_add_to_window( helper, window );
	}

	return window;
}

ScreemApplication *screem_application_new()
{
	ScreemApplication *application;
	GType type;

	type = screem_application_get_type();

	application = SCREEM_APPLICATION( g_object_new( type, NULL ) );

	return application;
}

static GtkWidget *screem_application_splash( ScreemApplication *application,
					     GtkWindow **w,
					     gboolean close)
{
       	GtkWidget *label;
	static GtkWidget *window;
	GtkWidget *image;
	GtkWidget *box;

	if( ! close ) {
		window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title( GTK_WINDOW( window ),
				      _( "Loading - Screem" ) );
		gtk_window_set_position( GTK_WINDOW( window ), 
					 GTK_WIN_POS_CENTER_ALWAYS );
		gtk_window_set_type_hint( GTK_WINDOW( window ),
					GDK_WINDOW_TYPE_HINT_SPLASHSCREEN );
       
		image = gtk_image_new_from_file( SPLASH_DIR"/splash.png" );
		gtk_widget_show( image );		

		label = gtk_label_new( _( "Starting..." ) );
		gtk_widget_show( label );
		
		box = gtk_vbox_new( FALSE, 0 );
		gtk_box_pack_start( GTK_BOX( box ), image, TRUE, TRUE, 0 );
		gtk_box_pack_start( GTK_BOX( box ), label, TRUE, TRUE, 0 );

		gtk_widget_show( box );
		
		gtk_container_add( GTK_CONTAINER( window ), box );
		
		gtk_widget_show_all( window );
		
		if( w ) {
			*w = GTK_WINDOW( window );
		}
		
		return label;
	} else {
		gtk_widget_destroy( window );
		return NULL;
	}
}

void screem_application_startup( ScreemApplication *application,
				 const gchar *session,
				 const gchar **start_files )
{
	ScreemApplicationPrivate *priv;
	ScreemWindow *window;
	ScreemSite *site;
	ScreemPage *page;
	
	GtkWidget *splash_text;
	gchar *glade_path;

	gboolean show_hints;
	GList *windows;

	GtkWindow *splash;
	
#ifdef HAVE_DBUS
	DBusError error;
	DBusConnection *conn;
	DBusMessage *message;
	DBusMessageIter it;
#endif
	priv = application->priv;
	
#ifdef HAVE_DBUS
	dbus_g_thread_init();
	
	dbus_error_init( &error );
	priv->conn = conn = dbus_bus_get( DBUS_BUS_SESSION, &error );
	if( conn ) {
		dbus_connection_setup_with_g_main( conn, NULL );

		if( dbus_bus_service_exists( conn, "org.screem", 
					&error ) ) {
			message = dbus_message_new_method_call(
					"org.screem",
					"/org/screem/app",
					"org.screem.app",
					"Open" );
			if( start_files ) {
				while( *start_files ) {
					dbus_message_append_iter_init( message,
							&it );
					dbus_message_iter_append_string( &it,
							*start_files );
					start_files ++;
				}

				dbus_connection_send( conn, message, 0 );
				dbus_message_unref( message );
			}
			while( g_main_context_iteration( NULL, FALSE ) ) {}
			gtk_main_quit();
			return;

		} else {
			if( dbus_bus_acquire_service( conn, "org.screem",
				0, &error ) != -1 ) {
				/* acquired, listen in for 
				 * org.screem stuff */
				dbus_bus_add_match( conn,
						"type='method_call',interface='org.screem.app'", NULL );
				dbus_connection_add_filter( conn,
					screem_application_dbus_filter,
					application, NULL );
			} else {
				g_warning( "Can't acquire service:  %s: %s'", error.name, error.message);
			}
		}
	}
#endif

	if( start_files ) {
		while( *start_files ) {
			priv->start_files = g_slist_prepend( priv->start_files,
					g_strdup( *start_files ) );
			start_files ++;
		}
	}
	
	gdk_threads_enter();

	application->client = gconf_client_get_default();
	
	gconf_client_add_dir( application->client, "/apps/screem",
			      GCONF_CLIENT_PRELOAD_RECURSIVE, NULL );

	glade_path = screem_get_glade_path();
	g_free( glade_path );

	/* create splash screen */
	splash_text = screem_application_splash( application, &splash, FALSE );
	screem_set_cursor( GTK_WIDGET( splash ), GDK_WATCH );
	
	SPLASH_TEXT( splash_text, _( "Initialising..." ) );

#ifdef HAVE_PYTHON
	SPLASH_TEXT( splash_text, _( "Initialise Python..." ) );
	Py_Initialize();
#endif
	
	/* load syntax files */
	SPLASH_TEXT( splash_text, _( "Loading Syntax files..." ) );
	screem_application_load_syntax_tables();

	/* setup offline notification / status */
	application->offline = gconf_client_get_bool( application->client,
						      "/apps/screem/general/work_offline",
						      NULL );
	application->offline_notify = 
		gconf_client_notify_add( application->client,
					 "/apps/screem/general/work_offline", 
					 screem_application_offline_notify,
					 application, NULL, NULL );
	
	application->hint = GTK_WIDGET( screem_hint_new() );
	g_object_ref( G_OBJECT( application->hint ) );
	gtk_object_sink( GTK_OBJECT( application->hint ) );

	/* add icons */
	screem_application_add_icons( application );

	/* load plugins */
	if( g_module_supported() ) {
		SPLASH_TEXT( splash_text, _( "Loading Plugins..." ) );
		scan_plugins( application, splash_text );
	}

	/* load macros */
	SPLASH_TEXT( splash_text, _( "Loading Macros..." ) );
	screem_application_load_macros( application );

	/* load helpers */
	SPLASH_TEXT( splash_text, _( "Loading Helpers..." ) );
	screem_application_load_helpers( application );

	/* load DTDs */
	SPLASH_TEXT( splash_text, _( "Loading DTDs..." ) );
	application->priv->dtddb = screem_dtd_db_new();

	/* create the fake site for editing individual files */
	site = screem_site_new( G_OBJECT( application ) );
	screem_application_add_site( application, site );

	window = NULL;
	if( session ) {
		SPLASH_TEXT( splash_text, _( "Loading Session..." ) );

		screem_session_load( application->priv->session,
					session );
	}

	/* if we loaded a session that some how didn't have any windows
	 * then we still want to create a window, and an initial
	 * blank page */
	windows = screem_application_get_window_list( application );
	if( ! windows ) {
		/* create a dummy page for the fake site */
		page = screem_page_new( G_OBJECT( application ) );
		screem_page_set_data( page, NULL );

		screem_site_add_page( site, page );
		g_object_unref( page );

		/* create initial window */
		SPLASH_TEXT( splash_text, _( "Creating Interface..." ) );

		window = screem_application_create_window( application );
		screem_window_set_current( window, site );

		screem_window_set_document( window, page );

		gtk_widget_show( GTK_WIDGET( window ) );
	}
	
	screem_application_splash( application, NULL, TRUE );


	/* display hints if necessary */
	show_hints = gconf_client_get_bool( application->client,
					    "/apps/screem/general/show_tips",
					    NULL );
	if( show_hints ) {
		windows = screem_application_get_window_list( application );
		window = SCREEM_WINDOW( windows->data );
		gtk_window_set_transient_for(GTK_WINDOW(application->hint),
					     GTK_WINDOW(window));
							  
		gtk_widget_show( application->hint );
	}

	/* delay loading of tag trees so we can get the main window up
	   quicker */
	g_idle_add( (GSourceFunc)screem_tag_tree_load_trees, NULL );
	
	/* delay loading sites / files until the main loop has started up */
	g_idle_add( (GSourceFunc)screem_application_load, application );

	gdk_threads_leave();
}

void screem_application_close( ScreemApplication *app )
{
	if( app->priv->window_list ) {
		screem_session_save( app->priv->session, SESSION_LAST );
	}
}

static gboolean screem_application_load( ScreemApplication *application )
{
	ScreemApplicationPrivate *priv;
	GSList *start_files;
	gchar *sfile;
	gchar *mime_type;
	gchar *file;
	ScreemWindow *window;

	priv = application->priv;
	
	if( ! priv->window_list ) {
		/* no window created yet, this can occur when
		 * the Open dbus method has been called while
		 * screem is still starting up, don't do anything,
		 * as we add the idle handler again at the end of
		 * screem_application_startup() */
		return FALSE;
	}
       	window = priv->window_list->data;

	start_files = priv->start_files;
	
	/* process command line files / sites */
	for( start_files = priv->start_files; start_files;
			start_files = start_files->next ) {
		GnomeVFSURI *uri;
		gboolean isdir;

		sfile = (gchar*)start_files->data;
		uri = gnome_vfs_uri_new( sfile );
		
		/* is the filename local? */
		if( gnome_vfs_uri_is_local( uri ) ) {
			/* yes */
			/* HACK */
			if( *sfile == G_DIR_SEPARATOR ) {
				file = g_strconcat( "file://", 
						    sfile, NULL );
			} else {
				file = relative_to_full( sfile, NULL );
			}
		} else {
			file = g_strdup( sfile );
		}
		gnome_vfs_uri_unref( uri );

		isdir = FALSE;
		if( uri_exists( file, NULL ) ) {
			isdir = screem_uri_is_dir( file );
		}

		/* is it a project file or an html file? */
		mime_type = screem_get_mime_type( file, FALSE );
		if( isdir || 
		    ( mime_type &&
		      ! strcmp( mime_type, "application/x-screem" ) ) ) {
			gdk_threads_enter();
			screem_site_open_with_filename( window, application,
							file );
	      		gdk_threads_leave();
		} else {
			ScreemSite *site;
			gdk_threads_enter();
			site =screem_application_get_default_site(application);
			screem_page_open_with_filename( site, window, file );
	      		gdk_threads_leave();
		}
		g_free( mime_type );
		g_free( file );

		g_free( sfile );
	}
	g_slist_free( priv->start_files );
	priv->start_files = NULL;
	
	return FALSE;
}

GList *screem_application_get_loaded_sites( ScreemApplication *app )
{
	g_return_val_if_fail( SCREEM_IS_APPLICATION( app ), NULL );

	return app->priv->loaded_sites;
}

void screem_application_add_site( ScreemApplication *app, ScreemSite *site )
{
	GList *list;
	ScreemWindow *window;
	ScreemWindowDetails *details;
	const gchar *pathname;
	EggRecentItem *item;
	
	/* add to list of loaded sites */
	app->priv->loaded_sites = g_list_append( app->priv->loaded_sites, site );

	/* add to recent sites */
	if( ! screem_site_get_fake_flag( site ) ) {
		/* work on the first window, others will pick up
		   the changes via gconf */
		if( app->priv->window_list ) {
			window = app->priv->window_list->data;
			details = window->details;
			pathname = screem_site_get_pathname( site );
			item = egg_recent_item_new_from_uri( pathname );
			egg_recent_item_add_group( item,
					RECENT_SITE_GROUP );
			egg_recent_item_set_private( item, TRUE );
			egg_recent_model_add_full( details->recent_site_model,
					item );
			egg_recent_item_unref( item );
		}
	}

	for( list = screem_application_get_window_list( app ); list;
	     list = list->next ) {
		screem_window_add_site( SCREEM_WINDOW( list->data ), site );
	}
}

ScreemSite *screem_application_get_site( ScreemApplication *app,
					 const gchar *uri )
{
	ScreemSite *site;
	const GList *list;

	/* loaded_sites->next as the first one is guarenteed
	   to be the fake site */
	site = NULL;
	for( list = app->priv->loaded_sites->next; list; list = list->next ) {
		const gchar *pathname;

		site = SCREEM_SITE( list->data );
		pathname = screem_site_get_pathname( site );
		if( ! strcmp( uri, pathname ) ) {
			break;
		}
		site = NULL;
	}

	return site;
}

ScreemSite *screem_application_get_site_by_name( ScreemApplication *app,
						 const gchar *name )
{
	ScreemSite *site;
	const GList *list;

	/* loaded_sites->next as the first one is guarenteed
	   to be the fake site */
	site = NULL;
	for( list = app->priv->loaded_sites->next; list; list = list->next ) {
		const gchar *sname;

		site = SCREEM_SITE( list->data );
		sname = screem_site_get_name( site );
		if( ! strcmp( name, sname ) ) {
			break;
		}
		site = NULL;
	}

	return site;
}

ScreemSite *screem_application_get_default_site( ScreemApplication *app )
{
	/* default site is always first in the list */
	return SCREEM_SITE( app->priv->loaded_sites->data );
}

void screem_application_close_site( ScreemApplication *app, ScreemSite *site )
{
	GList *list;

	list = screem_application_get_loaded_sites( app );
	app->priv->loaded_sites = g_list_remove( list, site );

	for( list = screem_application_get_window_list( app ); list;
	     list = list->next ) {
		screem_window_remove_site( SCREEM_WINDOW( list->data ),
				site );
	}
}

void screem_application_close_all_sites( ScreemApplication *app )
{
	ScreemSite *site;

	while( app->priv->loaded_sites ) {
		site = SCREEM_SITE( app->priv->loaded_sites->data );
		screem_application_close_site( app, site );
		g_object_unref( site );
	}
}

gboolean screem_application_save_sites( ScreemApplication *app )
{
	GList *list;
	ScreemSite *site;
	gboolean ret;

	ret = TRUE;
	
	list = screem_application_get_loaded_sites( app );
	
	/* request confirmation to save any opened sites / pages */
	for( ; ret && list; list = list->next ) {
		site = SCREEM_SITE( list->data );

		ret = screem_site_save_confirmation( site );
	}

	return ret;
}

void screem_application_register_plugin( ScreemApplication *application, 
					 ScreemPlugin *plugin )
{
	application->plugins = g_list_prepend( application->plugins, plugin );
}

static void screem_application_load_features( GHashTable *features, 
						const gchar *path )
{
	xmlTextReaderPtr reader;
	gchar *feature;
	gboolean valid;
	GHashTable *ftable;
	gchar *tmp;	
	reader = xmlNewTextReaderFilename( path );
	feature = NULL;
	valid = FALSE;
	ftable = NULL;
	if( reader ) {
		while( xmlTextReaderRead( reader ) ) {
			xmlChar *name;
			xmlChar *type;

			name = xmlTextReaderName( reader );
			if( ! valid && ! strcmp( "screem", name ) ) {
				type = xmlTextReaderGetAttribute( reader,
								  "type" );
				valid = ( type && 
					  ! strcmp( "features", type ) );
				xmlFree( type );
			} else if( valid && ! strcmp( "feature", name ) ) {
				g_free( feature );
				type = xmlTextReaderGetAttribute( reader,
								  "type" );
				if( ! type ) {
					feature = NULL;
				} else {
					feature = g_strdup( type );
					ftable = g_hash_table_lookup( features, feature );
					if( ! ftable ) {
						ftable = g_hash_table_new( g_str_hash, g_str_equal );
						g_hash_table_insert( features, g_strdup( feature ), ftable );
					}
					xmlFree( type );
				}
			} else if( valid && feature &&
				   ! strcmp( "mime", name ) ) {
				type = NULL;
				if( xmlTextReaderRead( reader ) ) {
					type = xmlTextReaderValue( reader );
					type = g_strstrip( type );
				}
				if( ftable && *type ) {
					tmp = g_strdup( type );
					g_hash_table_insert( ftable, tmp, tmp );
				}
				if( type ) {
					xmlFree( type );
				}
			} 			
			xmlFree( name );
		}
		
		xmlFreeTextReader( reader );
	}
	g_free( feature );
}

GtkSourceLanguagesManager* screem_application_load_syntax_tables()
{
	static GtkSourceLanguagesManager *lm = NULL;
	static GHashTable *features;
	
	if( ! lm ) {
		gchar *dir;
		gchar *tmp;
		
		lm = gtk_source_languages_manager_new();
	
		features = g_hash_table_new( g_str_hash, g_str_equal );
		screem_application_load_features( features, 
					DATADIR"/screem/features.xml" );
		tmp = screem_get_dot_dir();
		dir = g_strconcat( tmp, G_DIR_SEPARATOR_S, "features.xml", 
				NULL );
		g_free( tmp );
		screem_application_load_features( features, dir );
		g_free( dir );

		g_object_set_data( G_OBJECT( lm ), "features", features );
	}

	return lm;
}

static void screem_application_add_icons( ScreemApplication *application )
{
	static struct ScreemStockPixmap {
		gchar const *stock_id;
		gchar const *filename;
		gchar const *ext;
	} const icons[] = {
		{ "Screem_Button", "button", ".xpm" },
		{ "Screem_Caption", "caption", ".xpm" },
		{ "Screem_Checkbutton", "checkbutton", ".xpm" },
		{ "Screem_Entry", "entry", ".xpm" },
		{ "Screem_FileEntry", "gnome-fileentry", ".xpm" },
		{ "Screem_Image", "stock_insert_image", ".png" },
		{ "Screem_Object", "stock_insert_object", ".png" },
		{ "Screem_Entity", "insert-symbol", ".png" },
		{ "Screem_Link", "add-link", ".png" },
		{ "Screem_Optionmenu", "optionmenu", ".xpm" },
		{ "Screem_Paragraph", "paragraphs", ".png" },
		{ "Screem_Pre", "pre", ".xpm" },
		{ "Screem_Radiobutton", "radiobutton", ".xpm" },
		{ "Screem_Sub", "sub", ".xpm" },
		{ "Screem_Sup", "sup", ".xpm" },
		{ "Screem_Table", "stock_insert_table", ".png" },
		{ "Screem_Td", "add_column", ".png" },
		{ "Screem_Text", "text", ".xpm" },
		{ "Screem_Th", "th", ".xpm" },
		{ "Screem_Tr", "add_row", ".png" },

		{ "Screem_Browser1", "", "" },
		{ "Screem_Browser2", "", "" },
		{ "Screem_Browser3", "", "" },

		{ "Screem_Site", "screem_site", ".png" },
		{ "Screem_Preview", "site_preview", ".png" },
		{ "Screem_LinkView", "site_structure", ".png" },
		{ "Screem_Resources", "resources", ".png" },

		{ "Screem_Todo", "todo", ".png" },

		{ "Screem_Bookmarks", "bookmarks-preferences", ".png" },
		{ "Screem_Bookmark", "bookmarks-open", ".png" },

		{ "Screem_Online", "stock_connect", ".png" },
		{ "Screem_Offline", "stock_disconnect", ".png" },

		{ "Screem_CVSCheckout", "cvs-checkout", ".png" },
		{ "Screem_CVSUpdate", "cvs-update", ".png" },
		{ "Screem_CVSAdd", "cvs-add", ".png" },
		{ "Screem_CVSRemove", "cvs-remove", ".png" },

		{ NULL, NULL, NULL }
	};
	static const GtkStockItem stockItems[] = {
		{ "Screem_Button", "<button>", 0, 0, 0 },
		{ "Screem_Caption", "<caption>", 0, 0, 0 },
		{ "Screem_Checkbutton", "Checkbox", 0, 0, 0 },
		{ "Screem_Entry", "text entry", 0, 0, 0 },
		{ "Screem_FileEntry", "file entry", 0, 0, 0 },
		{ "Screem_Image", "<img>", 0, 0, 0 },
		{ "Screem_Object", "<object>", 0, 0, 0 },
		{ "Screem_Entity", "Entity", 0, 0, 0 },
		{ "Screem_Link", "<a>", 0, 0, 0 },
		{ "Screem_Optionmenu", "<option>", 0, 0, 0 },
		{ "Screem_Paragraph", "<p>", 0, 0, 0 },
		{ "Screem_Pre", "<pre>", 0, 0, 0 },
		{ "Screem_Radiobutton", "Radiobutton", 0, 0, 0 },
		{ "Screem_Sub", "<sub>", 0, 0, 0 },
		{ "Screem_Sup", "<sup>", 0, 0, 0 },
		{ "Screem_Table", "<table>", 0, 0, 0 },
		{ "Screem_Td", "<td>", 0, 0, 0 },
		{ "Screem_Text", "<textarea>", 0, 0, 0 },
		{ "Screem_Th", "<th>", 0, 0, 0 },
		{ "Screem_Tr", "<tr>", 0, 0, 0 },

		{ "Screem_Browser1", "Browser 1", 0, 0, 0 },
		{ "Screem_Browser2", "Browser 2", 0, 0, 0 },
		{ "Screem_Browser3", "Browser 3", 0, 0, 0 },

		{ "Screem_Site", "", 0, 0, 0 },

		{ "Screem_Preview", "Preview", 0, 0, 0 },
		{ "Screem_LinkView", "Link View", 0, 0, 0 },
		{ "Screem_Resources", "Resources", 0, 0, 0 },

		{ "Screem_Todo", "Task List", 0, 0, 0 },

		{ "Screem_Bookmarks", "Edit Bookmarks...", 0, 0, 0 },
		{ "Screem_Bookmark", "", 0, 0, 0 },

		{ "Screem_Online", "", 0, 0, 0 },
		{ "Screem_Offline", "", 0, 0, 0 },

		{ "Screem_CVSCheckout", "", 0, 0, 0 },
		{ "Screem_CVSUpdate", "", 0, 0, 0 },
		{ "Screem_CVSAdd", "", 0, 0, 0 },
		{ "Screem_CVSRemove", "", 0, 0, 0 }

	};

	GtkIconFactory *factory;
	gint i;

	factory = gtk_icon_factory_new();

	for( i = 0; icons[ i ].stock_id; ++ i ) {
		GtkIconSet *set;
		GtkIconSource *src;
		GdkPixbuf *pixbuf;
		gchar *filename;
		gint j;
		gboolean wildcarded;
		static const gchar *sizes[] = {
			"", "-48", "-24", "-16", NULL
		};

		set = gtk_icon_set_new();
		wildcarded = FALSE;

		for( j = 0; sizes[ j ]; ++ j ) {
			filename = g_strconcat( PIXMAP_PATH, G_DIR_SEPARATOR_S,
						icons[ i ].filename,
						sizes[ j ],
						icons[ i ].ext,
						NULL );
			
			pixbuf = gdk_pixbuf_new_from_file( filename, NULL );
			if( pixbuf ) {
				src = gtk_icon_source_new();
				gtk_icon_source_set_pixbuf( src, pixbuf );
				if( ! wildcarded ) {
					wildcarded = TRUE;
				} else {
					gtk_icon_source_set_size_wildcarded( src, FALSE );
				}
				
				if( j == 1 ) {
					gtk_icon_source_set_size( src,
								  GTK_ICON_SIZE_DIALOG );
				} else if( j == 2 ) {
					gtk_icon_source_set_size( src,
								  GTK_ICON_SIZE_BUTTON );
				} else if( j == 3 ) {
					gtk_icon_source_set_size( src,
								  GTK_ICON_SIZE_MENU );
				}
				gtk_icon_set_add_source( set, src );
				g_object_unref( G_OBJECT( pixbuf ) );
				gtk_icon_source_free( src );
			}
			g_free( filename );
		}
		gtk_icon_factory_add( factory, icons[ i ].stock_id, set );
		
		gtk_icon_set_unref( set );
	}

	gtk_icon_factory_add_default( factory );

	gtk_stock_add( stockItems, G_N_ELEMENTS( stockItems ) );

	g_object_unref( G_OBJECT( factory ) );
}



static void screem_application_load_macros( ScreemApplication *application )
{
	GSList *macro_list;
	GSList *macro_action_list;
	GSList *list;
	GSList *alist;

	macro_list = gconf_client_get_list( application->client,
					    "/apps/screem/editor/macros",
					    GCONF_VALUE_STRING, NULL );
	macro_action_list = 
		gconf_client_get_list(application->client,
				      "/apps/screem/editor/macro_actions",
				      GCONF_VALUE_STRING, NULL );
	
	for( alist = macro_action_list, list = macro_list; list && alist; 
	     list = list->next, alist = alist->next ) {
		ScreemEditorMacro *macro = g_new0( ScreemEditorMacro, 1 );
	
		macro->key_name = g_strdup( list->data );
		macro->text = g_strdup( alist->data );
		screem_application_add_macro( application, macro );
	}
	if( macro_list )
		g_slist_free( macro_list );

	if( macro_action_list )
		g_slist_free( macro_action_list );
}

void screem_application_add_macro( ScreemApplication *application,
				   ScreemEditorMacro *macro )
{
	GList *macros;
	GSList *mlist;
	GSList *malist;
	GSList *list;
	GSList *alist;

	/* insert into macro list */
	for( macros = application->macros; macros; macros = macros->next ) {
		ScreemEditorMacro *mac = (ScreemEditorMacro*)macros->data;

		if( ! strcmp( mac->key_name, macro->key_name ) )
			break;
	}
	if( macros ) {
		macros->data = macro;
	} else {
		application->macros = g_list_append( application->macros,
						     macro );
	}

	/* store via gconf */
	mlist = gconf_client_get_list( application->client,
				       "/apps/screem/editor/macros",
				       GCONF_VALUE_STRING, NULL );
	malist = gconf_client_get_list( application->client,
					"/apps/screem/editor/macro_actions",
				       GCONF_VALUE_STRING, NULL );

	for( list = mlist, alist = malist; list && alist; 
	     list = list->next, alist = alist->next ) {
		const gchar *mname = (const gchar*)mlist->data;
		if( mname && ! strcmp( mname, macro->key_name ) )
			break;
	}
	if( list && alist ) {
		alist->data = g_strdup( macro->text );
	} else {
		mlist = g_slist_append( mlist, g_strdup( macro->key_name ) );
		malist = g_slist_append( malist, g_strdup( macro->text ) );
	}

	gconf_client_set_list( application->client,
			       "/apps/screem/editor/macros",
			       GCONF_VALUE_STRING, mlist, NULL );
	gconf_client_set_list( application->client,
			       "/apps/screem/editor/macro_actions",
			       GCONF_VALUE_STRING, malist, NULL );

	gconf_client_suggest_sync( application->client, NULL );

	/* cleanup */
	if( mlist ) {
		g_slist_free( mlist );
	}
	if( malist ) {
		g_slist_free( malist );
	}
}

void screem_application_remove_macro( ScreemApplication *application,
				      ScreemEditorMacro *macro )
{
	GList *macros;

	for( macros = application->macros; macros; macros = macros->next ) {
		ScreemEditorMacro *mac = (ScreemEditorMacro*)macros->data;

		if( ! strcmp( mac->key_name, macro->key_name ) )
			break;
	}
	if( macros ) {
		ScreemEditorMacro *mac = (ScreemEditorMacro*)macros->data;

		GSList *macros;
		GSList *actions;
		GSList *list;
		GSList *alist;

		g_free( mac->key_name );
		g_free( mac->text );
		g_free( mac );
		application->macros = g_list_remove( application->macros,
						     mac );

	
		/* now remove from gconf */
		macros = gconf_client_get_list( application->client,
						"/apps/screem/editor/macros",
						GCONF_VALUE_STRING, NULL );
		actions = gconf_client_get_list( application->client,
						 "/apps/screem/editor/macro_actions",
						 GCONF_VALUE_STRING, NULL );
		for(list = macros, alist = actions; list && alist;
		    list = list->next, alist = alist->next ) {
			const gchar *mname = (const gchar*)list->data;
			if( mname && ! strcmp( mname, macro->key_name ) )
				break;
		}
		if( list ) {
			g_free( list->data );
			macros = g_slist_remove( macros, list->data );
		}
		if( alist ) {
			g_free( alist->data );
			actions = g_slist_remove( actions, alist->data );
		}
		gconf_client_set_list( application->client,
				       "/apps/screem/editor/macros",
				       GCONF_VALUE_STRING, macros, NULL );
		gconf_client_set_list( application->client,
				       "/apps/screem/editor/macro_actions",
				       GCONF_VALUE_STRING, actions, NULL );

		gconf_client_suggest_sync( application->client, NULL );

		/* cleanup */
		if( macros ) {
			g_slist_free( macros );
		}
		if( actions ) {
			g_slist_free( actions );
		}
	}
}

ScreemDTDDB *screem_application_get_dtd_db( ScreemApplication *application )
{
	g_return_val_if_fail( SCREEM_IS_APPLICATION( application ),
				NULL );

	return application->priv->dtddb;
}

void screem_application_set_cursor( ScreemApplication *app,
		GdkCursorType type )
{
	GList *windows;

	for( windows = screem_application_get_window_list( app );
			windows; windows = windows->next ) {
		screem_set_cursor( GTK_WIDGET( windows->data ),
				type );
	}
}

void screem_application_file_op( GnomeVFSMonitorEventType type,
				const gchar *uri, gpointer data )
{
	ScreemApplication *app;
	ScreemFileBrowser *browser;
	gchar *dirname;
	GList *list;
	
	dirname = g_path_get_dirname( uri );
	app = SCREEM_APPLICATION( data );
	
	for( list = screem_application_get_loaded_sites( app );
			list; list = list->next ) {
		g_object_get( G_OBJECT( list->data ), "browser", &browser,
				NULL );
		screem_file_browser_file_mod( browser, dirname, uri, type );
		g_object_unref( browser );
	}
	g_free( dirname );
}

ScreemSession *screem_application_get_session( ScreemApplication *app )
{
	g_return_val_if_fail( SCREEM_IS_APPLICATION( app ), NULL );

	return app->priv->session;
}

GtkTreeModel *screem_application_get_encodings( ScreemApplication *app )
{
	g_return_val_if_fail( SCREEM_IS_APPLICATION( app ), NULL );

	return app->priv->encodings;
}

static void screem_application_helper_failed( ScreemHelper *helper,
		const gchar *msg, ScreemApplication *application )
{
	gchar *name;
	gchar *primary;

	g_object_get( G_OBJECT( helper ), "name", &name, NULL );
		
	primary = g_strdup_printf( _( "There was an error running the %s helper" ), name );

	screem_hig_alert( GTK_STOCK_DIALOG_ERROR,
			primary, msg, NULL );
	g_free( primary );
	g_free( name );
}

static void screem_application_helper_error( ScreemHelper *helper,
					     gint status,
					     ScreemApplication *application )
{
	gchar *name;
	gchar *primary;
	const gchar *secondary;

	if( status != 0 ) {
		g_object_get( G_OBJECT( helper ), "name", &name, NULL );
	
		primary = g_strdup_printf( "The %s helper returned errors", 
				name );
		secondary = _( "The error window will contain details of these" );
		screem_hig_alert( GTK_STOCK_DIALOG_WARNING,
				primary, secondary, NULL );
		g_free( primary );
		g_free( name );
	}
}

static gint helper_filter( const GnomeVFSFileInfo *info )
{
	return strcmp( "application/x-desktop", info->mime_type );
}

static void screem_application_load_helpers( ScreemApplication *app )
{
	gchar *path;
	gchar *tmp;
	GSList *helpers;
	GSList *list;
	ScreemHelper *helper;
	
	tmp = screem_get_dot_dir();
	path = g_build_path( G_DIR_SEPARATOR_S, tmp,
			"helpers", NULL );
	g_free( tmp );

	if( mkdir_recursive( path, 
			GNOME_VFS_PERM_USER_ALL, 
			screem_application_file_op, app ) ) {
		helpers = screem_vfs_scandir( path,
				helper_filter,
				(GCompareFunc)strcmp, FALSE );
		for( list = helpers; list; list = list->next ) {
			helper = screem_helper_new_from_file( list->data );
			if( helper ) {
				screem_application_add_helper( app,
						helper );
			}
			g_free( list->data );
		}
		g_slist_free( helpers );
	}

	g_free( path );
}

void screem_application_add_helper( ScreemApplication *application,
				    ScreemHelper *helper )
{
	GList *windows;
	ScreemWindow *window;

	/* add to all open windows */
	for( windows = application->priv->window_list; windows;
	     windows = windows->next ) {
		window = SCREEM_WINDOW( windows->data );
		screem_helper_add_to_window( helper, window );
	}
	application->helpers = g_list_append( application->helpers, 
			helper );

	/* hookup callbacks */
	g_signal_connect( G_OBJECT( helper ), "helper_failed",
			  G_CALLBACK( screem_application_helper_failed ),
			  application );
	g_signal_connect( G_OBJECT( helper ), "helper_error",
			  G_CALLBACK( screem_application_helper_error ),
			  application );
}

void screem_application_remove_helper( ScreemApplication *app,
				       ScreemHelper *helper )
{
	GList *hlist;
	gchar *name;
	gchar *dname;
	ScreemWindow *window;
	GList *windows;

	gchar *path;
	gchar *tmp;
	
	g_object_get( G_OBJECT( helper ), "name", &name, NULL );

	/* first try and find a matching helper in 
	   app->helpers */
	for( hlist = app->helpers; hlist; hlist = hlist->next ) {

		g_object_get( G_OBJECT( hlist->data ), 
				"name", &dname, NULL );

		if( ! strcmp( name, dname ) ) {
			g_free( dname );
			break;
		}
		g_free( dname );
	}
	tmp = screem_get_dot_dir();
	path = g_strconcat( tmp, G_DIR_SEPARATOR_S,
			"helpers", G_DIR_SEPARATOR_S,
			name, ".desktop", NULL );
	g_free( tmp );
	g_free( name );
	
	if( hlist ) {
		/* found it */
		helper = SCREEM_HELPER( hlist->data );
		app->helpers = g_list_remove( app->helpers,
				hlist->data );

		/* remove from all open windows */
		for( windows = app->priv->window_list; windows;
		     windows = windows->next ) {
			window = SCREEM_WINDOW( windows->data );
			screem_helper_remove_from_window( helper, 
					window );
		}

		delete_file( path, screem_application_file_op, app );
	}
	g_free( path );
}

void screem_application_exec_helpers( ScreemApplication *app,
		ScreemPage *page )
{
	const gchar *type;
	ScreemWindow *window;
	GList *windows;
	GList *docs;
	GList *helpers;
	ScreemHelper *helper;
	gboolean exec;
	gchar *etype;

	type = screem_page_get_mime_type( page );

	/* we need a window to output into, use the first one
	 * we find the page open in, or NULL otherwise, bit of
	 * a messy thing to do though */
	window = NULL;
	for( windows = app->priv->window_list; windows;
			windows = windows->next ) {
		window = SCREEM_WINDOW( windows->data );

		docs = screem_window_get_documents( window );
		if( g_list_find( docs, page ) ) {
			break;
		}
		window = NULL;
	}
	
	for( helpers = app->helpers; helpers; helpers = helpers->next ) {
		helper = SCREEM_HELPER( helpers->data );
		g_object_get( G_OBJECT( helper ),
				"exec_on_save", &exec,
				"exec_type", &etype,
				NULL );
		if( exec &&
			g_pattern_match_simple( etype, type ) ) {
			screem_helper_execute( helper, page,
					window );
		}
		g_free( etype );
	}
}

static void screem_application_offline_notify( GConfClient *client,
					       guint cnxn_id,
					       GConfEntry *entry,
					       gpointer data )
{
	ScreemApplication *application;

	application = SCREEM_APPLICATION( data );

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		gboolean state;

		state = gconf_value_get_bool( entry->value );

		if( application->offline != state ) {
			GList *list;

			application->offline = state;
	
			list = screem_application_get_window_list( application );
			g_list_foreach( list, 
					(GFunc)screem_window_toggle_online_status,
					NULL );
		}
	}
}

static gboolean screem_application_free_feature_type( gpointer key,
						gpointer value,
						gpointer data )
{
	g_assert( key == value );
	g_free( key );

	return FALSE;
}

static gboolean screem_application_free_features( gpointer key,
						  gpointer value,
						  gpointer data )
{
	GHashTable *table;

	table = (GHashTable*)value;
	
	g_hash_table_foreach( table,
			(GHFunc)screem_application_free_feature_type,
			NULL );
	
	g_hash_table_destroy( table );
	g_free( key );

	return FALSE;
}

#ifdef HAVE_DBUS
static DBusHandlerResult screem_application_dbus_filter( DBusConnection *conn,
		DBusMessage *message,
		gpointer data )
{
	ScreemApplication *app;
	ScreemApplicationPrivate *priv;
	DBusMessageIter it;
	gchar *uri;
	DBusHandlerResult res;
	gboolean add;
	
	app = SCREEM_APPLICATION( data );
	priv = app->priv;
	res = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	
	if( dbus_message_is_method_call( message, "org.screem.app",
				"Open" ) ) {
		dbus_message_iter_init( message, &it );
		
		add = ( ! priv->start_files ); 
		while( ( uri = dbus_message_iter_get_string( &it ) ) ) {
			priv->start_files = g_slist_prepend( priv->start_files, uri );	
			if( ! dbus_message_iter_next( &it ) ) {
				break;
			}
		}
		if( add ) {
			g_idle_add( (GSourceFunc)screem_application_load,
					app );
		}
		res = DBUS_HANDLER_RESULT_HANDLED;
	}
	return res;
}
#endif

/* G Object stuff */

#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void screem_application_class_init( ScreemApplicationClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );
	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_application_finalize;
}

static void screem_application_init( ScreemApplication *application )
{
	ScreemApplicationPrivate *priv;
	gint i;
	GtkTreeIter it;
	const gchar *trans;
	gchar *tmp;

	/* we properly init later by calling screem_application_startup */
	priv = application->priv = g_new0( ScreemApplicationPrivate, 1 );
	priv->session = screem_session_new( G_OBJECT( application ) );

	priv->encodings = GTK_TREE_MODEL( gtk_list_store_new( 2, 
				G_TYPE_STRING, G_TYPE_STRING ) );

	for( i = 0; i < screem_n_encodings; ++ i ) {
		gtk_list_store_append( GTK_LIST_STORE( priv->encodings ), &it );
		trans = gettext( screem_encodings[ i ].name );
		tmp = g_strconcat( trans, " (",
				screem_encodings[ i ].charset, ")", 
				NULL );
		gtk_list_store_set( GTK_LIST_STORE( priv->encodings ),
				&it,
				0, tmp,
				1, screem_encodings[ i ].charset,
				-1 );
		g_free( tmp );
	}
}

static void screem_application_finalize( GObject *object )
{
	ScreemApplication *application;
	ScreemApplicationPrivate *priv;
	GtkSourceLanguagesManager *lm;
	GHashTable *features;
	GList *list;
	ScreemPlugin *plugin;
	ScreemEditorMacro *macro;
	ScreemHelper *helper;
	
	application = SCREEM_APPLICATION( object );
	priv = application->priv;
#ifdef HAVE_DBUS
	if( priv->conn ) {
		dbus_connection_disconnect( priv->conn );
	}
#endif
	if( priv->start_files ) {
		g_slist_foreach( priv->start_files, (GFunc)g_free, NULL );
		g_slist_free( priv->start_files );
		priv->start_files = NULL;
	}
	
	screem_application_close_all_windows( application );

	screem_application_close_all_sites( application );

	if( application->offline_notify ) {
		gconf_client_notify_remove( application->client, 
				application->offline_notify );
	}

	if( priv->dtddb ) {
		g_object_unref( priv->dtddb );
	}

	if( application->client ) {
		g_object_unref( application->client );
	}

	if( priv->session ) {
		g_object_unref( priv->session );
	}

	lm = screem_application_load_syntax_tables();
	features = g_object_get_data( G_OBJECT( lm ), "features" );
	if( features ) {
		g_hash_table_foreach( features,
				(GHFunc)screem_application_free_features,
				NULL );
		g_hash_table_destroy( features );
		g_object_unref( G_OBJECT( lm ) );
	}

	if( priv->encodings ) {
		g_object_unref( priv->encodings );
	}
	
	for( list = application->plugins; list; list = list->next ) {
		plugin = (ScreemPlugin*)list->data;
		
		g_module_close( plugin->module );
		g_free( plugin );
	}
	if( application->plugins ) {
		g_list_free( application->plugins );
	}

	for( list = application->macros; list; list = list->next ) {
		macro = (ScreemEditorMacro*)list->data;

		g_free( macro->key_name );
		g_free( macro->text );
		g_free( macro );
	}
	if( application->macros ) {
		g_list_free( application->macros );
	}
	
	for( list = application->helpers; list; list = list->next ) {
		helper = (ScreemHelper*)list->data;	
	
		g_object_unref( helper );
	}
	if( application->helpers ) {
		g_list_free( application->helpers );
	}

	if( application->hint ) {
		g_object_unref( application->hint );
		gtk_widget_destroy( application->hint );
	}

	g_free( priv );

	screem_get_file_filter( NULL );

#ifdef HAVE_PYTHON
	Py_Finalize();
#endif
	
	G_OBJECT_CLASS( parent_class )->finalize( object );

	screem_file_browser_cleanup();
}

GType screem_application_get_type()
{

	static GType type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemApplicationClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_application_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemApplication ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_application_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemApplication",
					       &info, 0 );
	}

	return type;
}
