/*  Screem:  plugin.c,
 *  Handles loading of plugins
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <dirent.h>
#include <gmodule.h>
#include <gnome.h>

#include <sys/types.h>
#include <sys/wait.h>

#include "editor.h"
#include "htmlfuncs.h"
#include "plugin.h"
#include "site.h"
#include "page.h"
#include "guile.h"

extern Site *current_site;
extern GtkWidget *app;

typedef gchar*(*TagWiz)( gchar **wizard_name );
typedef void(*PopupWiz)( GtkMenuItem *menu, GList *data );
typedef struct TagWizard {
	gchar *tag;
	GModule *module;
	gchar *name;
	PopupWiz func;
} TagWizard;
static GList *tag_wizards = NULL;

typedef void(*Init)();

static gboolean destroy_attr_list( gpointer data );

static void load_plugins( GtkWidget *output, gchar *directory );

void scan_plugins( GtkWidget *output )
{
	const gchar *search_paths[ 7 ] = {
		PLUGIN_PATH,
		NULL,
                "/usr/lib/",
                "/usr/X11R6/lib",
		NULL,
                NULL
	};
	gint path = 0;
	gchar *directory = NULL;

	/* add these into search_paths here rather than in the declaration
	   as the values would be set at compile rather than runtime
	   otherwise */
	search_paths[ 1 ] = gnome_libdir_file( "" );
	search_paths[ 4 ] = g_getenv( "SCREEM_LIB_PATH" );

	/* global paths to look for the plugins are searched in order,
	   if plugins are found in one of them no more are checked */
	do {
		g_free( directory );
                directory = g_strdup_printf( "%sscreem%cplugins%c",
                                             search_paths[ path ],
                                             G_DIR_SEPARATOR,
                                             G_DIR_SEPARATOR );
                path ++;
        } while( ( search_paths[ path ] ) &&
                 ( ! g_file_test( directory, G_FILE_TEST_ISDIR ) ) );

	/* now load the global plugins */
	load_plugins( output, directory );

	g_free( directory );

	/* now load from the users $HOME/.screem directory */
	directory = g_strdup_printf( "%s%c.screem", g_get_home_dir(),
				     G_DIR_SEPARATOR );

	if( g_file_test( directory, G_FILE_TEST_ISDIR ) )
		load_plugins( output, directory );

	g_free( directory );
	g_free( (gchar*)search_paths[ 1 ] );
}

/* load the shared library plugins */
static void load_plugins( GtkWidget *output, gchar *directory )
{
        struct dirent **namelist;
       	gint number;
	gint position;
	gint length;
	gchar *file;
	gchar *module;

	GModule *self;
	GModule *mod;

	Init init;

	TagWiz tagwiz;
	PopupWiz popupwiz;
	gchar *wizname;
	TagWizard *tw;

	g_return_if_fail( directory != NULL );

	self = g_module_open( NULL, G_MODULE_BIND_LAZY );

	number = scandir( directory, &namelist, 0, alphasort );

	for( position = 0; position < number; position ++ ) {  
		file = namelist[ position ]->d_name;
                length = strlen( file );
		if( length < 5 ) {
			free( namelist[ position ] );
			continue;
		}
		if( ! strcmp( ".scm", (file + length  - 4 ) ) ) {
				/* we have a script plugin */
			module = g_strdup_printf( "%s%s", 
						  directory, file );
			screem_guile_script_load( module );
			g_free( module );
		} else if( length < 4 ) {
			free( namelist[ position ] );
			continue;
		}
                if( ! strcmp( ".so", ( file + length - 3 ) ) ) {
                        /* we've got a plugin */
                        module = g_strdup_printf( "%s%s", directory, file );

			gtk_text_insert(GTK_TEXT( output ), 0, 0, 0, "\t", 1 );
			gtk_text_insert(GTK_TEXT( output ), 0, 0, 0, file,-1 );
			gtk_text_insert(GTK_TEXT( output ), 0, 0, 0, "\n", 1 );
			while( gtk_events_pending() )
				gtk_main_iteration();

                        mod = g_module_open( module, G_MODULE_BIND_LAZY );
                        if( mod ) {
                                if( ! g_module_symbol( mod, "init", 
                                                       (gpointer*)&init ) ) {
                                        g_print( _("Module has no init() function\n" ) );
                                        g_module_close( mod );
                                } else {
                                        init();
					/* grab the possible tag its used for*/
					g_module_symbol( mod, "html_tag",
							 (gpointer*)&tagwiz );
					g_module_symbol( mod, "popup_wizard",
							 (gpointer*)&popupwiz);
					if( tagwiz ) {
						tw = g_new( TagWizard, 1 );
						tw->tag = tagwiz( &wizname );
						tw->module = mod;
						tw->name = wizname;
						tw->func = popupwiz;
						tag_wizards = g_list_append( tag_wizards, tw );
					}
				}
                        } else {
                                g_print( _("Error loading module %s\n%s"),
                                         module, g_module_error() );
                        }
                        g_free( module );
                }
		free( namelist[ position ] );
	}

}

/* register a plugin as a wizard for a particular tag */
GtkWidget *tag_plugin_wizard( gchar *tag, gint start, gint len )
{
	GtkWidget *menu = NULL;
	GList *list;
	TagWizard *tw;
	gchar *temp;

	GList *data = NULL;

	for( list = tag_wizards; list; list = list->next ) {
		tw = (TagWizard*)list->data;
		if( ! strcmp( tw->tag, "sitecopy_hack" ) &&
		    ! strcasecmp( tw->tag, tag ) )
			break;
	}

	if( list ) {
		/* build up data list */
		data = g_list_append( data, (gpointer)start );
		data = g_list_append( data, (gpointer)len );
		/* we want to get all the attributes and separate them
		   into name and value */
		temp = screem_editor_get_text( start, len - start);
		data = screem_html_build_attributes_list( temp, data );
		g_free( temp );

		/* we found a wizard for this tag */
		temp = g_strdup_printf( _("%s Wizard"), tw->name );
		menu = gtk_menu_item_new_with_label( temp );
		gtk_widget_show( menu );
		gtk_signal_connect_full( GTK_OBJECT( menu ), "activate",
					 GTK_SIGNAL_FUNC( tw->func ), 
					 NULL, data,
					 (GtkDestroyNotify) destroy_attr_list,
					 0, 0 );
		g_free( temp );
	}

	return menu;
}

/* hack - get the sitecopy upload wizard GModule */
GModule *screem_plugin_get_sitecopy()
{
	GList *list;
	TagWizard *tw;

	for( list = tag_wizards; list; list = list->next ) {
		tw = (TagWizard*)list->data;
		if( ! strcmp( tw->name, "sitecopy" ) )
			return tw->module;
	}

	return NULL;
}

static gboolean destroy_attr_list( gpointer data )
{
	GList *list;

	list = (GList*)data;
   	for( list = list->next->next; list; list = list->next )
		g_free( list->data );

	g_list_free( (GList*)data );

	return TRUE;
}
