/*  Screem:  siteDruid.c,
 *  A site druid for guiding users through the process of creating a site
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>

#include <glade/glade.h>

#include "fileops.h"
#include "preferences.h"
#include "site.h"
#include "siteUI.h"
#include "siteTemplate.h"
#include "siteTreeUI.h"

extern GtkWidget *app;
extern Site *current_site;
extern GList *loaded_sites;
extern Preferences *cfg;


static GladeXML *xml;
static Site *site;
static gchar *template;

/*
 * screem_site_druid:
 * @site:  the site struct to use
 *
 * opens the site druid dialog
 *
 * return values: none
 */
void screem_site_druid( Site *s )
{
	xml = glade_xml_new( cfg->glade_path, "site_druid" );

	glade_xml_signal_autoconnect( xml );

	site = s;
}

void screem_site_druid_cancel( GtkWidget *widget )
{
	GtkWidget *window;

	window = glade_xml_get_widget( xml, "site_druid" );

	gtk_widget_destroy( window );
	screem_site_destroy( site );
	g_free( template );
}

void screem_site_druid_finish( GnomeDruidPage *widget, GtkWidget *druid )
{
	GtkWidget *window;
	FILE *file;
      	const gchar *ptemplate;
	const gchar *default_file;

	/* create the site */
	if( screem_site_create( site ) ) {
		ptemplate = screem_site_get_template_path( site );
		default_file = cfg->mpage->default_filename;

		if( template ) {
			build_site_with_template( site, template );
			chdir( screem_site_get_pathname( site ) );
		} else if( ptemplate ) {
			/* copy the template */
			copy_file( ptemplate, default_file );
		} else if( overwrite( default_file ) ) {
			file = fopen( default_file, "w" );
			fclose( file );
		}
		/* add this site to the list of loaded sites */
		loaded_sites = g_list_append( loaded_sites, site );
		/* build the combo options */
		screem_build_sites_combo_list();
		/* set the combo entry to be the name of the current site */
		widget = gtk_object_get_data( GTK_OBJECT( app ), "sitecombo" );
		gtk_entry_set_text( GTK_ENTRY( GTK_COMBO( widget )->entry ),
				    screem_site_get_name( site ) );
		current_site = site;

		/* create the site file tree */
		refresh_file_tree();
	}

	window = glade_xml_get_widget( xml, "site_druid" );
	gtk_widget_destroy( window );
}

void screem_site_druid_details_prepare( GnomeDruidPage *druid_page,
                                        GtkWidget *druid )
{
	GtkWidget *widget;
	GtkWidget *widget2;
	const gchar *data;
	gboolean active;

	widget = glade_xml_get_widget( xml, "use_template" );
	widget2 = glade_xml_get_widget( xml, "site_template" );
	active = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	gtk_widget_set_sensitive( widget2, active );

	data = screem_site_get_cvs_root( site );
	widget = glade_xml_get_widget( xml, "use_cvs" );
	widget2 = glade_xml_get_widget( xml, "cvsroot" );

	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
				      (gboolean) data );
	gtk_widget_set_sensitive( widget2, (gboolean) data );
	if( data )
		gtk_entry_set_text( GTK_ENTRY( widget2 ), data );
	else
		gtk_entry_set_text( GTK_ENTRY( widget2 ), "" );

	data = screem_site_get_template_path( site );
	widget = glade_xml_get_widget( xml, "use_page_template" );
	widget2 = glade_xml_get_widget( xml, "page_template" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
				      (gboolean) data );
	gtk_widget_set_sensitive( widget2, (gboolean) data );

	if( data )
		gtk_entry_set_text( GTK_ENTRY( widget2 ), data );
	else
		gtk_entry_set_text( GTK_ENTRY( widget2 ), "" );
}

gboolean screem_site_druid_details( GnomeDruidPage *druid_page, 
				    GtkWidget *druid )
{
	GtkWidget *widget;
	gchar *data;
     	gchar cwd[ 16384 ];
	const gchar *data1;
	const gchar *data2;

	getcwd( cwd, 16384 );

	widget = glade_xml_get_widget( xml, "site_name" );
	data = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_name( site, data );

	widget = glade_xml_get_widget( xml, "site_path" );
       	data = gtk_entry_get_text( GTK_ENTRY( widget ) );

	if( ! g_path_is_absolute( data ) ) {
		data = g_strconcat( cwd, G_DIR_SEPARATOR_S, data, NULL );
		screem_site_set_pathname( site, data );
		g_free( data );
	} else
		screem_site_set_pathname( site, data );

	widget = glade_xml_get_widget( xml, "use_template" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "site_template" );
		template = gtk_entry_get_text( GTK_ENTRY( widget ) );
	} else
		template = NULL;

	widget = glade_xml_get_widget( xml, "use_cvs" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "cvsroot" );
		data = gtk_entry_get_text( GTK_ENTRY( widget ) );
		screem_site_set_cvs_root( site, data );
	} else
		screem_site_set_cvs_root( site, NULL );
       
	widget = glade_xml_get_widget( xml, "use_page_template" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "page_template" );
		data = gtk_entry_get_text( GTK_ENTRY( widget ) );
		screem_site_set_template_path( site, data );
	} else
		screem_site_set_template_path( site, NULL );

	data1 = screem_site_get_pathname( site );
	data2 = screem_site_get_name( site );
	if( ( strlen( data1 ) == 0 ) || ( strlen( data2 ) == 0 ) ) {
		glade_xml_new( cfg->glade_path, "smessage1" );
		return TRUE;
	}

	return FALSE;
}

void screem_site_druid_use_template( GtkToggleButton *toggle )
{
	GtkWidget *widget;
	gboolean active;

	widget = glade_xml_get_widget( xml, "site_template" );
	active = gtk_toggle_button_get_active( toggle );
	gtk_widget_set_sensitive( widget, active );
}

void screem_site_druid_use_cvs( GtkToggleButton *toggle )
{
	GtkWidget *widget;
	gboolean active;

	widget = glade_xml_get_widget( xml, "cvsroot" );
	active = gtk_toggle_button_get_active( toggle );
	gtk_widget_set_sensitive( widget, active );
}

void screem_site_druid_use_page_template( GtkToggleButton *toggle )
{
	GtkWidget *widget;
	gboolean active;

	widget = glade_xml_get_widget( xml, "page_template" );
	active = gtk_toggle_button_get_active( toggle );
	gtk_widget_set_sensitive( widget, active );
}

gboolean screem_site_druid_publish1( GnomeDruidPage *druid_page, 
				     GtkWidget *druid )
{
	GtkWidget *widget;
	UploadMethods method;
	gchar *data;
	gint port;

	method = LOCAL;
	widget = glade_xml_get_widget( xml, "ftp" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		method = FTP;
	widget = glade_xml_get_widget( xml, "webdav" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		method = WEBDAV;
	screem_site_set_remote_method( site, method );

	widget = glade_xml_get_widget( xml, "hostname" );
	data = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_remote_url( site, data );

	widget = glade_xml_get_widget( xml, "port" );
	port = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );

	widget = glade_xml_get_widget( xml, "remote_path" );
	data = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_remote_path( site, data );

	widget = glade_xml_get_widget( xml, "baseuri" );
	data = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_http_url( site, data );

	widget = glade_xml_get_widget( xml, "username" );
	data = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_remote_user( site, data );

	widget = glade_xml_get_widget( xml, "password" );
	data = gtk_entry_get_text( GTK_ENTRY( widget ) );
	screem_site_set_remote_pass( site, data );

	return FALSE;
}

gboolean screem_site_druid_publish2( GnomeDruidPage *druid_page, 
				     GtkWidget *druid )
{
	GtkWidget *widget;
	GtkWidget *widget2;
	gboolean flag;
	SitePerms sp;
	SiteSymlinks sl;

	widget = glade_xml_get_widget( xml, "passiveftp" );
	flag = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	screem_site_set_passive_ftp( site, flag );

	widget = glade_xml_get_widget( xml, "nodelete" );
	flag = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	screem_site_set_no_delete( site, flag );

	widget = glade_xml_get_widget( xml, "checkmoved" );
	flag = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	screem_site_set_check_moved( site, flag );

	widget = glade_xml_get_widget( xml, "nooverwrite" );
	flag = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	screem_site_set_no_overwrite( site, flag );

	sp = PERMS_IGNORE;
	widget = glade_xml_get_widget( xml, "perm_exec" );
	widget2 = glade_xml_get_widget( xml, "perm_all" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		sp = PERMS_EXEC;
	else if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget2 ) ) )
		sp = PERMS_ALL;

	sl = SYM_IGNORE;
	widget = glade_xml_get_widget( xml, "link_follow" );
	widget2 = glade_xml_get_widget( xml, "link_maintain" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
		sl = SYM_FOLLOW;
	else if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget2 ) ) )
		sl = SYM_MAINTAIN;

	return FALSE;
}
