/*  Screem:  siteTreeUI.c,
 *  The UI side of the site tree, handling user interaction such as clicking,
 *  and drag and drop
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <dirent.h>
#include <gnome.h>
#include <gnome-xml/debugXML.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <glade/glade.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <pwd.h>
#include <grp.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs-mime.h>
#endif

#include "cvs.h"
#include "editor.h"
#include "fileops.h"
#include "page.h"
#include "pageUI.h"
#include "preview.h"
#include "preferences.h"
#include "site.h"
#include "siteTree.h"
#include "siteTreeUI.h"
#include "siteUI.h"
#include "support.h"
#include "tag_tree.h"
#include "xml.h"

#include "dir-open.xpm"
#include "dir-close.xpm"

extern Site *current_site;
extern GtkWidget *app;
extern Preferences *cfg;

extern GList *icon_list;

static void build_file_tree( GtkWidget *tree, Site *site, xmlDocPtr doc );
static void add_node( Site *site, GtkCTree *t, xmlDocPtr doc, xmlNodePtr node,
		      GtkCTreeNode *parent );

static void tree_item_clicked(GtkCTree *tree, GtkCTreeNode *row, gint column);

void tree_open_file( void );
void tree_delete_file( void );
void tree_rename_file( void );

static GdkPixmap *folder_open = NULL;
static GdkPixmap *folder_close = NULL;
static GdkBitmap *folder_open_mask = NULL;
static GdkBitmap *folder_close_mask = NULL;

static GnomeUIInfo tree_dnd_menu[] = {
        { GNOME_APP_UI_ITEM, N_("_Move here"), N_("Move file"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("_Copy here"), N_("Copy file"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_SEPARATOR,
        { GNOME_APP_UI_ITEM, N_("Cancel drag"), N_("Cancel drag"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_END
};

static void load_folder_icons()
{
	GdkPixbuf *pixbuf;

	if( ! folder_open ) {
		pixbuf = gdk_pixbuf_new_from_xpm_data( (const char**)
						       DIRECTORY_OPEN_XPM );
		if( ! pixbuf ) {
			g_warning( _( "Failed to create folder open icon" ) );
			return;
		}
		gdk_pixbuf_render_pixmap_and_mask( pixbuf, &folder_open,
						   &folder_open_mask,
						   100 );
		gdk_pixbuf_unref( pixbuf );
		pixbuf = gdk_pixbuf_new_from_xpm_data( (const char**)
						       DIRECTORY_CLOSE_XPM );
		if( ! pixbuf ) {
			g_warning( _( "Failed to create folder close icon" ) );
			return;
		}
		gdk_pixbuf_render_pixmap_and_mask( pixbuf, &folder_close,
						   &folder_close_mask,
						   100 );
		gdk_pixbuf_unref( pixbuf );
	}
}

static void build_file_tree( GtkWidget *tree, Site *site, xmlDocPtr doc )
{
	xmlNodePtr node;
 	GtkCTreeNode *tnode;
	
	g_return_if_fail( tree != NULL );
	g_return_if_fail( doc != NULL );
	
	load_folder_icons();

	node = doc->root;
	/* not interested in the xml version element */
	node = node->childs;

	gtk_clist_freeze( GTK_CLIST( tree ) );
	add_node( site, GTK_CTREE( tree ), doc, node, NULL );
	
	/* now sort the tree */
	tnode = gtk_ctree_node_nth( GTK_CTREE( tree ), 0 );
	gtk_ctree_sort_recursive( GTK_CTREE( tree ), tnode );
	gtk_ctree_expand( GTK_CTREE( tree ), tnode );

	gtk_clist_thaw( GTK_CLIST( tree ) );
}


/*
 * refresh_file_tree:
 *
 * Rebuilds the file tree
 *
 * return values: none
 */
void refresh_file_tree()
{
	static Site *site = NULL;
	xmlDocPtr doc;
	GtkWidget *tree;

	const gchar *site_path;

     	tree = gtk_object_get_data( GTK_OBJECT( app ), "file_tree" );

	/* fake sites can't refresh file trees */
	if( screem_site_get_fake_flag( current_site ) ) {
	       	gtk_clist_clear( GTK_CLIST( tree ) );
		return;
	}

	site_path = screem_site_get_pathname( current_site );

	doc = build_directory_tree( current_site, site_path, NULL );

	g_return_if_fail( doc != NULL );

	identify_pages( current_site, doc, doc->root->childs );

	if( site != current_site )
		gtk_clist_clear( GTK_CLIST( tree ) );
	build_file_tree( tree, current_site, doc );

	xmlFreeDoc( doc );

	site = current_site;
}

static GtkCTreeNode* add_node_do( Site *site, 
				  GtkCTree* tree, GtkCTreeNode *parent,
				  const gchar *path, gchar *mime_type )
{
	struct stat s;
	gboolean is_dir;
	gchar *temp_path;
	gchar *root;
	gchar flags[ 3 ] = { '.', '.', '.' };
	gchar *text[ 3 ] = { NULL, NULL, NULL };
	
	GdkPixmap *open = folder_open;
        GdkBitmap *open_mask = folder_open_mask;
        GdkPixmap *close = folder_close;
        GdkBitmap *close_mask = folder_close_mask;
	Icons *icon = NULL;

	GtkCTreeNode *node;
	GtkCTreeNode *exist;
	GtkCTreeNode *die;
	GtkCTreeNode *already;
	gchar *exist_path;
	
	if( ! path )
		return parent;
	
	if( stat( path, &s ) < 0 )
		return parent;
	
	is_dir = S_ISDIR( s.st_mode );

	/* get root path, and set flags if necessary */
	if( site )
		root = g_strdup( screem_site_get_pathname( site ) );
	else
		root = g_strdup( RESOURCE_PATH );

	/* bypass the root path for the file, which is either the site
	   pathname, or the resource folder pathname */

	if( is_dir ) {
		temp_path = path + strlen( root );
		temp_path[ strlen( temp_path ) - 1 ] = '\0';
		text[ 0 ] = g_strconcat( G_DIR_SEPARATOR_S,
					 g_basename( temp_path ), NULL );
		temp_path -= strlen( root );
	} else {
		text[ 0 ] = g_strdup( g_basename( path ) );
		icon = icon_from_mime_type( path, mime_type );
		open = icon->open;
		open_mask = icon->open_mask;
		close = open;
		close_mask = open_mask;
	}

	if( site ) {
		if( screem_site_is_excluded( site, path ) )
			flags[ 0 ] = 'E';
		if( screem_site_is_ignored( site, path ) )
			flags[ 1 ] = 'I';
		if( screem_site_is_ascii( site, path ) )
			flags[ 2 ] = 'A';
		text[ 1 ] = g_strdup_printf( "%c%c%c", 
					     flags[ 0 ], flags[ 1 ],
					     flags[ 2 ] );
	}

	/* do we already have this node? */
     	exist = parent;
	if( exist )
		exist = GTK_CTREE_ROW( exist )->children;
	else
		exist = gtk_ctree_node_nth( tree, 0 );
   	already = NULL;
	while( exist ) {
		exist_path = gtk_ctree_node_get_row_data( tree, exist );
		/* stat it to check if it exists, if not we remove the node */
		if( exist_path && stat( exist_path, &s ) < 0 ) {
			die = exist;
			exist = GTK_CTREE_ROW( exist )->sibling;
			gtk_ctree_remove_node( tree, die );
		} else {
			/* it exists, is it the one we are trying to add? */
			if( exist_path && ! strcmp( exist_path, path ) )
				already = exist;
			exist = GTK_CTREE_ROW( exist )->sibling;
		}
	}

	/* add it if we need to */
	if( already )
		node = already;
	else if( strcmp( mime_type, "application/x-screem" ) ) {
		node = gtk_ctree_insert_node( tree, parent, NULL, text, 3,
					      close, close_mask,
					      open, open_mask,
					      (gboolean)parent, FALSE );
		gtk_ctree_node_set_row_data( tree, node, 
					     g_strdup( path ) );
	} else
		node = parent;
	
	g_free( root );
	g_free( text[ 0 ] );
	g_free( text[ 1 ] );

	return node;
}

static void add_node( Site *site, GtkCTree *tree, 
		      xmlDocPtr doc, xmlNodePtr node,
		      GtkCTreeNode *parent )
{
	GtkCTreeNode *this;
	gchar *mime_type;
	gchar *entry;

    	g_return_if_fail( tree != NULL );

        /* add the node */
        entry = xmlNodeListGetString( doc, node->childs, 1 );

	mime_type = xml_get_value( node, "type" );

	this = add_node_do( site, tree, parent, entry, mime_type );
	g_free( entry );
	g_free( mime_type );

	/* has children? */
        if( node->childs )
                add_node( site, tree, doc, node->childs, this );

        /* move on to the next sibling */
        if( node->next )
                add_node( site, tree, doc, node->next, parent );
}

void tree_clicked( GtkWidget *widget, GdkEventButton *event,
		   gpointer data )
{
	GtkCTreeNode *node = NULL;
	GladeXML *xml;
        GtkWidget *menu;
	gchar *path;
	GtkWidget *check;
        gint row = -1;
        gint col;

	if( ! current_site )
		return;

	gtk_clist_get_selection_info( GTK_CLIST( widget ),
                                      event->x, event->y, &row, &col );

	if( row != - 1 )
                node = gtk_ctree_node_nth( GTK_CTREE( widget ), ( guint )row );

	/* set the node that was clicked on */
        if( node ) {
                gtk_object_set_data( GTK_OBJECT( widget ),
                                     "drag_start", GINT_TO_POINTER( row ) );
        } else {
                gtk_object_set_data( GTK_OBJECT( widget ),
                                     "drag_start", NULL);
	}

	/* what button was clicked? */
	if( event->button == 1 ) {
                if( event->type == GDK_2BUTTON_PRESS ) /* double click */
                        tree_item_clicked( GTK_CTREE( widget ), node, 1 );
	} else if( event->button == 3 ) {
                /* select the line underneath the mouse */
                xml =  glade_xml_new( cfg->glade_path, "file_tree_menu" );
		menu = glade_xml_get_widget( xml, "file_tree_menu" );

		if( node ) {
                        gtk_ctree_select( GTK_CTREE( widget ), node );
			path = gtk_ctree_node_get_row_data( GTK_CTREE( widget ), node );
			gtk_object_set_data( GTK_OBJECT( menu ), "path",path );

			/* find upload flags for the file/directory */
			screem_set_upload_flags( current_site, xml, path );
		} else {
			check = glade_xml_get_widget( xml, "delete" );
			change_state( check, NULL );
			check = glade_xml_get_widget( xml, "rename" );
			change_state( check, NULL );
			check = glade_xml_get_widget( xml, "properties" );
			change_state( check, NULL );
			check = glade_xml_get_widget( xml, "upload_flags" );
			change_state( check, NULL );
		}
		glade_xml_signal_autoconnect( xml );
		gtk_object_set_data( GTK_OBJECT( menu ), "tree", widget );
		gtk_object_set_data( GTK_OBJECT( menu ), "node", node );
		gtk_object_set_data( GTK_OBJECT( widget ), "node", node );
                gnome_popup_menu_do_popup_modal( menu, 0, 0, event, 0 );
                gtk_widget_destroy( menu );
        }
}

void exclude_file( GtkWidget *widget )
{
	GladeXML *xml;
	GtkWidget *menu;
	gboolean active;
	gchar *path;
	
	GtkWidget *tree;
	GtkWidget *node;
	gchar *text;
	gchar flags[ 3 ] = { '.', '.', '.' };

	xml = glade_get_widget_tree( widget );
	menu = glade_xml_get_widget( xml, "file_tree_menu" );

	tree = gtk_object_get_data( GTK_OBJECT( menu ), "tree" );
	node = gtk_object_get_data( GTK_OBJECT( menu ), "node" );
	path = gtk_object_get_data( GTK_OBJECT( menu ), "path" );

	active = screem_site_is_excluded( current_site, path );

	if( ! active )
		screem_site_add_exclude( current_site, path );
	else
		screem_site_remove_exclude( current_site, path );

	if( screem_site_is_excluded( current_site, path ) )
		flags[ 0 ] = 'E';
	if( screem_site_is_ignored( current_site, path ) )
		flags[ 1 ] = 'I';
	if( screem_site_is_ascii( current_site, path ) )
		flags[ 2 ] = 'A';
	
	text = g_strdup_printf( "%c%c%c", flags[ 0 ], flags[ 1 ], flags[ 2 ] );
	gtk_ctree_node_set_text( GTK_CTREE( tree ), GTK_CTREE_NODE( node ),
				 1, text );
	g_free( text );
}

void ascii_file( GtkWidget *widget )
{
	GladeXML *xml;
	GtkWidget *menu;
	gboolean active;
	gchar *path;

	GtkWidget *tree;
	GtkWidget *node;
	gchar *text;
	gchar flags[ 3 ] = { '.', '.', '.' };

	xml = glade_get_widget_tree( widget );
	menu = glade_xml_get_widget( xml, "file_tree_menu" );

	tree = gtk_object_get_data( GTK_OBJECT( menu ), "tree" );
	node = gtk_object_get_data( GTK_OBJECT( menu ), "node" );
	path = gtk_object_get_data( GTK_OBJECT( menu ), "path" );

	active = screem_site_is_ascii( current_site, path );

	if( ! active )
		screem_site_add_ascii( current_site, path );
	else
		screem_site_remove_ascii( current_site, path );

	if( screem_site_is_excluded( current_site, path ) )
		flags[ 0 ] = 'E';
	if( screem_site_is_ignored( current_site, path ) )
		flags[ 1 ] = 'I';
	if( screem_site_is_ascii( current_site, path ) )
		flags[ 2 ] = 'A';
	
	text = g_strdup_printf( "%c%c%c", flags[ 0 ], flags[ 1 ], flags[ 2 ] );
	gtk_ctree_node_set_text( GTK_CTREE( tree ), GTK_CTREE_NODE( node ),
				 1, text );
	g_free( text );
}

void ignore_file( GtkWidget *widget )
{
	GladeXML *xml;
	GtkWidget *menu;
	gboolean active;
	gchar *path;


	GtkWidget *tree;
	GtkWidget *node;
	gchar *text;
	gchar flags[ 3 ] = { '.', '.', '.' };

	xml = glade_get_widget_tree( widget );
	menu = glade_xml_get_widget( xml, "file_tree_menu" );

	tree = gtk_object_get_data( GTK_OBJECT( menu ), "tree" );
	node = gtk_object_get_data( GTK_OBJECT( menu ), "node" );
	path = gtk_object_get_data( GTK_OBJECT( menu ), "path" );

	active = screem_site_is_ignored( current_site, path );

	if( ! active )
		screem_site_add_ignore( current_site, path );
	else
		screem_site_remove_ignore( current_site, path );

	if( screem_site_is_excluded( current_site, path ) )
		flags[ 0 ] = 'E';
	if( screem_site_is_ignored( current_site, path ) )
		flags[ 1 ] = 'I';
	if( screem_site_is_ascii( current_site, path ) )
		flags[ 2 ] = 'A';
	
	text = g_strdup_printf( "%c%c%c", flags[ 0 ], flags[ 1 ], flags[ 2 ] );
	gtk_ctree_node_set_text( GTK_CTREE( tree ), GTK_CTREE_NODE( node ),
				 1, text );
	g_free( text );
}

static void tree_item_clicked( GtkCTree *tree, GtkCTreeNode *row, gint column )
{
	Page *page = NULL;
	static Page *prev;
	gchar *path = NULL;
	const gchar *mime_type;
	const gchar *mime_exec;
	gchar **params;
	gint num = 0;

	if( ! current_site )
		return;

	path = gtk_ctree_node_get_row_data( tree, row );

	if( ! path )
		return;

	if( g_file_test( path, G_FILE_TEST_ISDIR ) )
		return;

	/* locate the page with the matching path */
	page = screem_site_locate_page( current_site, path );

#ifndef HAVE_GNOME_VFS
	mime_type = gnome_mime_type( path );
#else
	mime_type = gnome_vfs_mime_type_from_name( path );
#endif

	/* if the path wasn't a page then we open it in the program
	   determined by its mime type */
	if( ! page ) {
		/* FIXME: use gnome vfs is possible */
		mime_exec = gnome_mime_program( mime_type );
		if( ! mime_exec )
			return;
		/* replace an occurance of %f in mime_exec with the
		   filename to open */
		params = g_strsplit( mime_exec, " ", 255 );
                while( params[ num ] ) {
                        if( ! strcmp( params[ num ], "%f" ) )
                                params[ num ] = g_strdup( path );
                        num ++;
                }
                path = g_strjoinv( " ", params );
                g_strfreev( params );
		
		/* now execite the progra, */
		gnome_execute_shell( NULL, path );

		g_free( path );

		prev = NULL;

		return;
	}

	if( page == prev )
		return;

	/* buffer any previous page that may be open in the editor */
	if( screem_site_get_current_page( current_site ) )
		screem_editor_buffer_text();

	screem_page_insert( page );
}


void tree_set_dnd_data( GtkWidget *widget, GdkDragContext *context,
			GtkSelectionData *selectionData, guint info,
			guint time, gpointer data)
{
	GtkCTreeNode *node;
        gchar *path = NULL;
        gchar *text = NULL;
	gchar *dir = NULL;
	gint row;

        if( ! GTK_CLIST( widget )->selection )
                return;
	
	if( GTK_IS_CTREE( widget ) ) {
		node = ((GList*)(GTK_CLIST(widget)->selection))->data;
		if( node )
			path = gtk_ctree_node_get_row_data( GTK_CTREE(widget),
							    node );
	} else {
		row = (gint)gtk_object_get_data(GTK_OBJECT(widget), "drag_start");

		dir = gtk_object_get_data( GTK_OBJECT( widget ), "current_dir" );
		gtk_clist_get_text( GTK_CLIST( widget ), row, 1, &path );
		path = g_strconcat( dir, G_DIR_SEPARATOR_S, path, NULL );
		/* FIXME: Mem Leak */
	}
	
        if( path && ( *path != '\0' ) ) {
		switch( info ) {
                case TARGET_URI_LIST:
                        text = g_strconcat( "file:", path, "\r\n", NULL );
                        break;
                }
		
                gtk_selection_data_set( selectionData, selectionData->target,
                                        8, text, strlen( text ) );
                g_free( text );
        } else {
                gtk_selection_data_set( selectionData, selectionData->target,
                                        8, NULL, 0 );
	}
}

void tree_drag_begin( GtkWidget *widget, GdkDragContext *context,
		      gpointer data )
{
	GtkCTreeNode *node;
        gpointer key;
        
        key = gtk_object_get_data( GTK_OBJECT( widget ), "drag_start" );
	
	g_return_if_fail( key != NULL );
       
	if( GTK_IS_CTREE( widget ) ) {
		node = gtk_ctree_node_nth( GTK_CTREE( widget ), 
					   GPOINTER_TO_INT( key ));
		g_return_if_fail( node != NULL );
		gtk_ctree_select( GTK_CTREE( widget ), node );
	} else {
		gtk_clist_select_row( GTK_CLIST( widget ), 
				      GPOINTER_TO_INT( key ), 0 );
}
}

gboolean tree_drag_motion( GtkWidget *widget, 
			   GdkDragContext *context, 
			   gint x, gint y, guint time,
			   gpointer data )
{
	GdkDragAction action;
	
        GtkWidget *source;

        source = gtk_drag_get_source_widget( context );

        if( ( source ) &&
            ( context->suggested_action != GDK_ACTION_ASK ) )
                action = GDK_ACTION_MOVE; 
        else
                action = context->suggested_action;

        gdk_drag_status( context, action, time );

	return TRUE;
}

void tree_drop_data( GtkWidget *widget, GdkDragContext *context,
		     gint x, gint y, GtkSelectionData *selectionData,
		     guint info, guint time )
{

	Site *site;
	GtkWidget *popup;
	gint item;
	gboolean move;
	gchar *text;
	gint row;
	gint col;
	GtkCTreeNode *node = NULL;
	gchar *path;
    	gchar *file;
	gboolean res;

	gboolean is_dir;
	gboolean is_dir2;

	const gchar *site_path;
	Page *page;

	site = current_site;

	g_return_if_fail( site != NULL );

	site_path = screem_site_get_pathname( site );

	/* if its a middle drag we need to ask the user what to do */
	if( context->action == GDK_ACTION_ASK ) {
		popup = gnome_popup_menu_new( tree_dnd_menu );
		item = gnome_popup_menu_do_popup_modal( popup, 0, 0, 0, 0 );
		switch( item ) {
		case 0:
			context->action = GDK_ACTION_MOVE;
			break;
		case 1:
			context->action = GDK_ACTION_COPY;
			break;
		default:
			return;
			break;
		}
	}

	/* set flag if we are performing a move */
	move = ( context->action == GDK_ACTION_MOVE );

	/* id the drop type */
	switch( info ) {
	case TARGET_URI_LIST:
		text = selectionData->data + strlen( "file:" );
		text[ strlen( text ) - 2 ] = 0;
		is_dir = g_file_test( text, G_FILE_TEST_ISDIR );

		gtk_clist_get_selection_info( GTK_CLIST( widget ), x, y,
					      &row, &col );
		/* id the node we dropped onto */
		if( row != -1 && GTK_IS_CTREE( widget ) ) {
			node = gtk_ctree_node_nth( GTK_CTREE( widget ),
						   ( guint )row );
			g_return_if_fail( node != NULL );
			path = 	(gchar*)
				gtk_ctree_node_get_row_data( GTK_CTREE(widget),
							     node );
		} else {
			g_return_if_fail( row != -1 );
			path = gtk_clist_get_row_data( GTK_CLIST(widget),row );
		}

		g_return_if_fail( path != NULL );

		/* if node isn't a directory node then get its parent
		   ( which will be a directory ) */
		is_dir2 = g_file_test( path, G_FILE_TEST_ISDIR );
		if( (! is_dir2) && GTK_IS_CTREE( widget ) ) {
			node = GTK_CTREE_ROW( node )->parent;
			/* node should never be NULL */
			g_return_if_fail( node != NULL );
			/* get the path for the parent */
			path = (gchar*)
				gtk_ctree_node_get_row_data( GTK_CTREE(widget),
							     node );
		} else if( ! is_dir2 ) {
			/* browse clist, rather than site ctree */
			path = (gchar*)gtk_object_get_data( GTK_OBJECT(widget),
							    "current_dir" );
		}

		g_return_if_fail( path != NULL );

		/* check to see if text and path are the same */
		if( ! strcmp( text, path ) )
			return;

		file = g_basename( text );
		path = g_strconcat( path, G_DIR_SEPARATOR_S, file, NULL );

		/* check to see if text and path are the same now */
		if( ! strcmp( text, path ) ) {
			g_free( path );
			return;
		}

    		if( is_dir )
			res = copy_dir( text, path, FALSE );
		else
			res = copy_file( text, path );

		if( ! res ) {
			/* we failed */
			g_free( path );
			return;
		}

		screem_site_rename_pages_with_path( site, text, path );

		/* update any links etc if we performed a move */
		if( move ) {
			screem_editor_buffer_text();
			screem_site_file_change( site, text, path );
			if( is_dir )
				delete_dir( text );
			else
				delete_file( text );

			if( ( page = screem_site_get_current_page( site ) ) ) {
				/* insert the possibly changed page */
				screem_editor_display_page( page );

			}
		}
		g_free( path );

		/* copy/move performed, update the tree */
		refresh_file_tree();

		break;
	}

}

void tree_open_file()
{
	GtkWidget *file_tree;
        GtkCTreeNode *node;

	file_tree = gtk_object_get_data( GTK_OBJECT( app ), "file_tree" );

	node = gtk_object_get_data( GTK_OBJECT( file_tree ), "node" );

        if( ! node )
                return;

	tree_item_clicked( GTK_CTREE( file_tree ), node, 0 );
}

void tree_delete_file()
{
	GtkWidget *file_tree;
	GtkWidget *box;
        const gchar *file = _("Delete selected file");
        const gchar *dir =  _("Delete selected directory");
        const gchar *message;

        gpointer key;
        GtkCTreeNode *node;
        gchar *path;
	gint button;
	gboolean status;
	Site *site;

	site = current_site;

	g_return_if_fail( site != NULL );

	file_tree = gtk_object_get_data( GTK_OBJECT( app ), "file_tree" );

        key = gtk_object_get_data( GTK_OBJECT( file_tree ),
                                   "drag_start" );
        if( ! key )
                return;
        node = gtk_ctree_node_nth(GTK_CTREE( file_tree ), 
                                  GPOINTER_TO_INT( key ) );
        if( ! node )
                return;

        path = (gchar*)gtk_ctree_node_get_row_data( GTK_CTREE( file_tree ),
                                                    node );

        if( g_file_test( path,G_FILE_TEST_ISDIR ) )
                message = dir;
        else
                message = file;

        box = gnome_message_box_new( message,
                                     GNOME_MESSAGE_BOX_QUESTION,
                                     GNOME_STOCK_BUTTON_YES,
                                     GNOME_STOCK_BUTTON_NO, NULL );
        gnome_dialog_set_parent( GNOME_DIALOG( box ), GTK_WINDOW( app ) );
        gnome_dialog_set_default( GNOME_DIALOG( box ), 0 );

	button = gnome_dialog_run_and_close( GNOME_DIALOG( box ) );

	if( button == 1 )
		return;

	/* delete confirmed */
	if( ! g_file_test( path, G_FILE_TEST_ISDIR ) )
                status = delete_file( path );
        else
                status = delete_dir( path );

	if( ! status ) {
		/* failed to delete */
		return;
	}

	/* perform cvs delete? */
	if( screem_site_get_cvs_root( site ) ) {
		box = gnome_message_box_new( _("perform CVS delete?"),
					     GNOME_MESSAGE_BOX_QUESTION,
					     GNOME_STOCK_BUTTON_YES,
					     GNOME_STOCK_BUTTON_NO, NULL );
		gnome_dialog_set_parent( GNOME_DIALOG( box ), 
					 GTK_WINDOW( app ) );
		gnome_dialog_set_default( GNOME_DIALOG( box ), 0 );
		if( gnome_dialog_run_and_close( GNOME_DIALOG( box ) ) == 0 ) {
			/* yes */
			 cvs_delete_with_filename( path );
		}
	}

	screem_site_purge_pages_with_path( current_site, path );

	refresh_file_tree();
}

void tree_rename_file()
{
	GtkWidget *file_tree;
	GtkWidget *dialog;
        GtkWidget *table;
        GtkWidget *label;
        GtkWidget *entry;

	GtkWidget *box;

        const gchar *file = _("Old file");
        const gchar *file2 = _("New file");
        const gchar *dir =  _("Old directory");
        const gchar *dir2 = _("New directory");
        const gchar *fileTitle = _("Rename file");
        const gchar *dirTitle = _("Rename directory");
        const gchar *title;
        const gchar *message;
        const gchar *message2;

	const gchar *site_path;
 
        GtkCTreeNode *node;
        gchar *path;
	gint button;
	Site *site;
	gchar *new_path;
	gchar *temp;
	gboolean status;
	Page *cpage;

	gboolean is_dir;

	site = current_site;

	g_return_if_fail( site != NULL );

	site_path = screem_site_get_pathname( site );

	file_tree = gtk_object_get_data( GTK_OBJECT( app ), "file_tree" );

	node = gtk_object_get_data( GTK_OBJECT( file_tree ), "node" );
  
        if( ! node )
                return;

        path = (gchar*)gtk_ctree_node_get_row_data( GTK_CTREE( file_tree ),
                                                    node );

        if( g_file_test( path, G_FILE_TEST_ISDIR ) ) {
                message = dir;
                message2 = dir2;
                title = dirTitle;
        } else {
                message = file;
                message2 = file2;
                title = fileTitle;
        }

        table = gtk_table_new( 2, 2, FALSE );

        label = gtk_label_new( message );
        gtk_table_attach( GTK_TABLE( table ), label, 0, 1, 0, 1,
                          GTK_FILL, 0, GNOME_PAD, GNOME_PAD );
        entry = gtk_entry_new();
        gtk_entry_set_editable( GTK_ENTRY( entry ), FALSE );
        /* we chop off the site_path though */
        gtk_entry_set_text( GTK_ENTRY( entry ), path + strlen( site_path ) );
        gtk_table_attach( GTK_TABLE( table ), entry, 1, 2, 0, 1,
                          GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

        label = gtk_label_new( message2 );
        gtk_table_attach( GTK_TABLE( table ), label, 0, 1, 1, 2,
                          GTK_FILL, 0, GNOME_PAD, GNOME_PAD );
        entry = gnome_file_entry_new( NULL, NULL );
        gtk_table_attach( GTK_TABLE( table ), entry, 1, 2, 1, 2,
                          GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

        dialog = gnome_dialog_new( title,
                                   GNOME_STOCK_BUTTON_OK,
                                   GNOME_STOCK_BUTTON_CANCEL,
                                   NULL );
        gtk_object_set_data( GTK_OBJECT( dialog ), "entry", entry );
        gtk_box_pack_start( GTK_BOX( GNOME_DIALOG( dialog )->vbox ),
                            table, TRUE, TRUE, GNOME_PAD );
        gnome_dialog_set_default( GNOME_DIALOG( dialog ), 0 );

	gtk_widget_show_all( table );
	button = gnome_dialog_run( GNOME_DIALOG( dialog ) );

	/* rename ok'ed */
	entry = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( entry ) );
	new_path = g_strdup_printf( "%s%s", site_path,
				    gtk_entry_get_text( GTK_ENTRY( entry ) ) );

	gtk_widget_destroy( dialog );

	if( button == 1 ) {
		g_free( new_path );
		return;
	}

	is_dir = g_file_test( path, G_FILE_TEST_ISDIR );

	/* if it is a dir then we must have a G_DIR_SEPARATOR on the end */
	if( is_dir && new_path[ strlen( new_path ) - 1 ] != G_DIR_SEPARATOR ) {
		temp = new_path;
		new_path = g_strconcat( temp, G_DIR_SEPARATOR_S, NULL );
		g_free( temp );
	}

	if( ! is_dir )
		status = copy_file( path, new_path );
	else
		status = copy_dir( path, new_path, FALSE );

	if( ! status ) {
		/* rename failed */
		g_free( new_path );
		return;
	}

	screem_site_rename_pages_with_path( site, path, new_path );

	/* update any links */
	cpage = screem_site_get_current_page( site );
	if( cpage )
		screem_editor_buffer_text();
       
	screem_site_file_change( site, path, new_path );

	if( ! is_dir )
		delete_file( path );
	else
		delete_dir( path );

	if( cpage ) {
		/* we had a page open in the editor, show
		   the possible changes to it */
		screem_editor_display_page( cpage );
	}

	/* perform cvs delete + cvs add ? */
	if( screem_site_get_cvs_root( site ) ) {
		box = gnome_message_box_new( _("Rename in CVS?"),
					     GNOME_MESSAGE_BOX_QUESTION,
					     GNOME_STOCK_BUTTON_YES,
					     GNOME_STOCK_BUTTON_NO, NULL );
		gnome_dialog_set_parent( GNOME_DIALOG( box ), 
					 GTK_WINDOW( app ) );
		gnome_dialog_set_default( GNOME_DIALOG( box ), 0 );
		if( gnome_dialog_run_and_close( GNOME_DIALOG( box ) ) == 0 ) {
			/* yes */
			 cvs_delete_with_filename( path );
			 cvs_add_with_filename( new_path );
		}
	}
	g_free( new_path );

	/* add the new node */
	refresh_file_tree();
}

void screem_browse_list_show_dir( gchar *dir )
{
	GtkWidget *list;
	DIR *d;
	struct dirent *entry = NULL;
	gchar *name;
	Icons *icon;
	const gchar *mime_type;
	gchar *item[ 2 ] = { NULL, NULL };
	gint row;
	gboolean is_dir;

	gchar cwd[ 16384 ];
	gchar cwd2[ 16384 ];

	g_return_if_fail( dir != NULL );

	list = gtk_object_get_data( GTK_OBJECT( app ), "browse" );

	load_folder_icons();

	getcwd( cwd, 16384 );

	gtk_object_set_data( GTK_OBJECT( list ), "current_dir", 
			     g_strdup( dir ) );
	d = opendir( dir );
	if( d ) {
		chdir( dir );
		getcwd( cwd2, 16384 );
		dir = cwd2;
		gtk_clist_freeze( GTK_CLIST( list ) );
		gtk_clist_clear( GTK_CLIST( list ) );

		gtk_clist_set_sort_column( GTK_CLIST( list ), 1 );
		gtk_clist_set_sort_type( GTK_CLIST( list ), GTK_SORT_ASCENDING );
		gtk_clist_set_auto_sort( GTK_CLIST( list ), TRUE );

		while( ( entry = readdir( d ) ) ) {
			name = entry->d_name;
			if( ! strcmp( ".", name ) ) {
				name = dir;
				continue;;
			} else if( strcmp( "..", name ) && name[ 0 ] == '.' ) {
				/* don't show hidden files */
				continue;
			}
			item[ 1 ] = g_strdup( name );
			is_dir = g_file_test( name, G_FILE_TEST_ISDIR );
			if( ! is_dir ) {
#ifndef HAVE_GNOME_VFS
				mime_type = gnome_mime_type( name );
#else
				mime_type = gnome_vfs_mime_type_from_name( name );
#endif
				icon = icon_from_mime_type( NULL, mime_type );
				row = gtk_clist_append( GTK_CLIST( list ), 
							item );
				gtk_clist_set_pixmap( GTK_CLIST( list ),
						      row, 0, icon->open,
						      icon->open_mask );
			} else {
				if( *item[1] != '/' ) {
					g_free( item[ 1 ] );
					item[ 1 ] = g_strconcat( "/", name, NULL );
				}
				row = gtk_clist_insert( GTK_CLIST( list ), 0,
							item );
				gtk_clist_set_pixmap( GTK_CLIST( list ),
						      row, 0, folder_close,
						      folder_close_mask );
			}
			g_free( item[ 1 ] );
		}
		gtk_clist_columns_autosize( GTK_CLIST( list ) );
		/* switch auto sort off so we can show the cwd name
		   as the top node */
		gtk_clist_set_auto_sort( GTK_CLIST( list ), FALSE );
		item[ 1 ] = dir;
		row = gtk_clist_prepend( GTK_CLIST( list ), item );
		gtk_clist_set_pixmap( GTK_CLIST( list ), row, 0,
				      folder_close, folder_close_mask );
		gtk_clist_thaw( GTK_CLIST( list ) );
		closedir( d );
	}

	chdir( cwd );
}

void browse_list_clicked( GtkWidget *widget, GdkEventButton *event,
			  gpointer data )
{
        gint row = -1;
        gint col;

	gchar *path;
	gchar *dir;

	gtk_clist_get_selection_info( GTK_CLIST( widget ),
                                      event->x, event->y, &row, &col );

	if( row != - 1 ) {
		gtk_object_set_data( GTK_OBJECT( widget ),
                                     "drag_start", GINT_TO_POINTER( row ) );
        } else {
                gtk_object_set_data( GTK_OBJECT( widget ),
                                     "drag_start", NULL);
	}

	/* what button was clicked? */
	if( event->button == 1 && row ) {
		if( event->type == GDK_2BUTTON_PRESS ) {
			dir = gtk_object_get_data( GTK_OBJECT( widget ), "current_dir" );
			gtk_clist_get_text( GTK_CLIST( widget ), row, 1, &path );
			path = g_strconcat( dir, G_DIR_SEPARATOR_S, path, NULL );

			if( g_file_test( path, G_FILE_TEST_ISDIR ) )
				screem_browse_list_show_dir( path );
			else if( screem_page_is_file_page( path ) ){
				/* switch to fake site and open */
				screem_site_switch_to_site(_("Single Pages"));
				screem_page_open_with_filename( path );
			}
			g_free( path );
		}
	} 
}


void resource_list_display( GtkWidget *tree )
{
	const gchar *resource_root = RESOURCE_PATH;
	xmlDocPtr doc;
      	gint number;
	gint pos;
        struct dirent **namelist;
	gchar *path;
	gchar *name;

	number = scandir( resource_root, &namelist, 0, alphasort );

	if( number < 0 )
		return;

	gtk_clist_clear( GTK_CLIST( tree ) );

	for( pos = 0; pos < number; pos ++ ) {
		name = namelist[ pos ]->d_name;
		if( strlen( name ) > 2 ) {
			path = g_strconcat( resource_root, name, NULL );
			doc = build_directory_tree( NULL, path, NULL );
			build_file_tree( tree, NULL, doc );
			xmlFreeDoc( doc );
			g_free( path );
		}
		free( namelist[ pos ] );
	}

	/* now add the tag tree in as a resource */
	gtk_clist_freeze( GTK_CLIST( tree ) );
	screem_tag_tree_build( tree, cfg->tag_tree );
	gtk_clist_thaw( GTK_CLIST( tree ) );
}

void tree_file_info()
{
	GtkWidget *file_tree;
        GtkCTreeNode *node;
	gchar *path;

	file_tree = gtk_object_get_data( GTK_OBJECT( app ), "file_tree" );

	node = gtk_object_get_data( GTK_OBJECT( file_tree ), "node" );

        if( ! node )
                return;

        path = (gchar*)gtk_ctree_node_get_row_data( GTK_CTREE( file_tree ),
                                                    node );

	file_information( path );
}

void file_information( const gchar *filename )
{
	struct stat s;
	GladeXML *xml;
	GtkWidget *widget;
	GtkWidget *i;
	gchar *temp;
	Site *site;
	Icons *icon;
	gchar *t;
	const gchar *mime_type;
	GdkPixmap *open = NULL;
	GdkBitmap *open_mask = NULL;

	uid_t user_id = geteuid();
	struct passwd *userinfo;
	struct group *groupinfo;
	
	site = current_site;

	if( stat( filename, &s ) < 0 )
		return;

	xml = glade_xml_new( cfg->glade_path, "file_info" );

	widget = glade_xml_get_widget( xml, "local" );
	gtk_label_set( GTK_LABEL( widget ), filename );

	widget = glade_xml_get_widget( xml, "mime" );
	i = glade_xml_get_widget( xml, "file_icon" );

	if( S_ISDIR(s.st_mode) ) {
		mime_type = "Directory";
		gtk_pixmap_set( GTK_PIXMAP( i ), folder_close,
				folder_close_mask );
	} else {
#ifndef HAVE_GNOME_VFS
		mime_type = gnome_mime_type( filename );
#else
		mime_type = gnome_vfs_mime_type_from_name( filename );
#endif
		icon = icon_from_mime_type( filename, mime_type );
		if( icon->pixbuf ) {
			gdk_pixbuf_render_pixmap_and_mask( icon->pixbuf,
							   &open, &open_mask,
							   100 );
			gtk_pixmap_set( GTK_PIXMAP( i ), open, open_mask );
		}
	}
	gtk_label_set( GTK_LABEL( widget ), mime_type );

	widget = glade_xml_get_widget( xml, "mod" );
	t = g_strdup( ctime( &s.st_mtime ) );
	t = g_strchomp( t );
	gtk_label_set( GTK_LABEL( widget ), t );

	widget = glade_xml_get_widget( xml, "change" );
	t = g_strdup( ctime( &s.st_ctime ) );
	t = g_strchomp( t );
	gtk_label_set( GTK_LABEL( widget ), t );

	widget = glade_xml_get_widget( xml, "access" );
	t = g_strdup( ctime( &s.st_atime ) );
	t = g_strchomp( t );
	gtk_label_set( GTK_LABEL( widget ), t );

	widget = glade_xml_get_widget( xml, "uid" );
	temp = g_strdup_printf( "%i", s.st_uid );
	userinfo = getpwuid( s.st_uid );
	if( userinfo ) {
		gtk_entry_set_text( GTK_ENTRY( widget ), userinfo->pw_name );
	} else {
		gtk_entry_set_text( GTK_ENTRY( widget ), temp );
	}
	g_free( temp );

	widget = glade_xml_get_widget( xml, "gid" );
	temp = g_strdup_printf( "%i", s.st_gid );
	groupinfo = getgrgid( s.st_gid );
	if( groupinfo ) {
		gtk_entry_set_text( GTK_ENTRY( widget ), groupinfo->gr_name );
	} else {
		gtk_entry_set_text( GTK_ENTRY( widget ), temp );
	}
	g_free( temp );

	widget = glade_xml_get_widget( xml, "size" );
	temp = g_strdup_printf( "%i KBytes (%i bytes)", 
				((int)s.st_size) / 1024,
				(int)s.st_size );
	gtk_label_set( GTK_LABEL( widget ), temp );
	g_free( temp );

	widget = glade_xml_get_widget( xml, "file_exclude" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
				      screem_site_is_excluded(site,filename) );
	widget = glade_xml_get_widget( xml, "file_ignore" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
				      screem_site_is_ignored(site, filename) );
	widget = glade_xml_get_widget( xml, "file_ascii" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
				      screem_site_is_ascii( site, filename ) );

	widget = glade_xml_get_widget( xml, "file_info" );
	gtk_widget_show( widget );

	gnome_dialog_run_and_close( GNOME_DIALOG( widget ) );
}

/* Process a directory name, and if one was given, create it. */
void create_dir_callback( gchar *string, gpointer data )
{
	if( ! string ) {
		screem_show_warning( _("No directory name specified.") );
		return;
	} else if( ! mkdir_recursive( string ) ) {
		screem_show_warning( _( "Directory could not be created." ) );
		return;
	}
	/* Directory was created just fine, refresh the file view */
	refresh_file_tree();
}
 
/* Request a directory name from the user in a gnome-dialog */
void screem_site_create_dir()
{
	gnome_app_request_string( GNOME_APP( app ),
				  _( "Enter the directory name to create:" ),
				  (GnomeStringCallback)create_dir_callback,
				  NULL );
}
