#include "../include/os.h"

/* Compile this module only for Windows. */
#ifdef __MSW__
#include <stdio.h>

#include <windows.h>	// Header File For Windows
#include <gl\gl.h>	// Header File For The OpenGL32 Library
#include <gl\glu.h>	// Header File For The GLu32 Library
#include <gl\glaux.h>	// Header File For The Glaux Library

#include "gw.h"
#include "stategl.h"


long GWWGetCurMilliTime(void);
int GWStrCaseCmp(const char *s1, const char *s2);
Boolean GWParseGeometry(
	const char *geometry,
	int *x, int *y, int *width, int *height
);
int GWGetKeyValueFromWParam(gw_display_struct *display, WPARAM wParam);
void GWSetWindowIconFile(
        gw_display_struct *display,
        const char *path
);
void *GWCreateWindow(
        gw_display_struct *display,
        int x, int y,
        int width, int height,
        int depth,              /* Color depth in bits. */
        const char *title   
);

gw_display_struct *GWInit(int argc, char **argv);
LRESULT CALLBACK WndProc(
        HWND hWnd,      // Handle For This Window 
        UINT uMsg,      // Message For This Window
        WPARAM wParam,  // Additional Message Information
        LPARAM lParam   // Additional Message Information
);
void GWManage(gw_display_struct *display);
void GWShutdown(gw_display_struct *display);

void GWOutputMessage(
	gw_display_struct *display,
	int type,
	const char *subject,
	const char *message,
	const char *help_details
);
int GWConfirmation(
	gw_display_struct *display,
	int type,
	const char *subject,
        const char *message,
        const char *help_details
);
int GWConfirmationSimple(
	gw_display_struct *display,
	const char *message
);

void GWSetDrawCB(
	gw_display_struct *display,
	void (*func)(void *),
	void *data
);
void GWSetResizeCB(
	gw_display_struct *display,
        void (*func)(void *, int, int, int, int),
        void *data
);
void GWSetKeyboardCB(
	gw_display_struct *display,
	void (*func)(void *, int, Boolean),
	void *data
);
void GWSetPointerCB(
	gw_display_struct *display,
	void (*func)(void *, int, int, int),
	void *data
);
void GWSetVisibilityCB(
	gw_display_struct *display,
	void (*func)(void *, int),
	void *data
);
void GWSetSaveYourselfCB(
        gw_display_struct *display,
        void (*func)(void *),
        void *data
);
void GWSetCloseCB(
	gw_display_struct *display,
	void (*func)(void *, void *),
	void *data
);
void GWSetTimeoutCB(
	gw_display_struct *display,
	void (*func)(void *),
        void *data
);

void GWPostRedraw(gw_display_struct *display);
void GWSwapBuffer(gw_display_struct *display);

void GWKeyboardAutoRepeat(gw_display_struct *display, Boolean b);

void GWSetPointerCursor(
        gw_display_struct *display,
        int cursor      /* One of GWPointerCursor*. */
);

void GWSetInputBusy(gw_display_struct *display);
void GWSetInputReady(gw_display_struct *display);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))

/*
 * Array of 256 unsigned bytes for recording key states.
 */
#define GW_KEY_STATE_TABLE_MAX_KEYS	256
static unsigned char key_state[256];

/*
 *	Windows key value to GW key value lookup table.
 *
 *	Used in GWGetKeyValueFromWParam(), even number index values
 *	are Windows key values and odd number index values are
 *	GW key values.
 */
#define WINDOWS_TO_GW_KEYTABLE	\
{ \
	VK_MENU, GWKeyAlt, \
	VK_LMENU, GWKeyAlt, \
	VK_RMENU, GWKeyAlt, \
	VK_CONTROL, GWKeyCtrl, \
	VK_LCONTROL, GWKeyCtrl, \
	VK_RCONTROL, GWKeyCtrl, \
	VK_SHIFT, GWKeyShift, \
	VK_LSHIFT, GWKeyShift, \
	VK_RSHIFT, GWKeyShift, \
 \
	VK_NUMPAD0, GWKeyInsert, \
	VK_NUMPAD1, GWKeyEnd, \
	VK_NUMPAD2, GWKeyDown, \
	VK_NUMPAD3, GWKeyPageDown, \
	VK_NUMPAD4, GWKeyLeft, \
	VK_NUMPAD5, GWKeyCenter, \
	VK_NUMPAD6, GWKeyRight, \
	VK_NUMPAD7, GWKeyHome, \
	VK_NUMPAD8, GWKeyUp, \
	VK_NUMPAD9, GWKeyPageUp, \
	VK_MULTIPLY, '*', \
	VK_ADD, '+', \
	VK_SEPARATOR, '=', \
	VK_SUBTRACT, '-', \
	VK_DECIMAL, '.', \
	VK_DIVIDE, '/', \
 \
	VK_F1, GWKeyF1, \
	VK_F2, GWKeyF2, \
	VK_F3, GWKeyF3, \
	VK_F4, GWKeyF4, \
	VK_F5, GWKeyF5, \
	VK_F6, GWKeyF6, \
	VK_F7, GWKeyF7, \
	VK_F8, GWKeyF8, \
	VK_F9, GWKeyF9, \
	VK_F10, GWKeyF10, \
	VK_F11, GWKeyF11, \
	VK_F12, GWKeyF12, \
	VK_F13, GWKeyF13, \
	VK_F14, GWKeyF14, \
	VK_F15, GWKeyF15, \
	VK_F16, GWKeyF16, \
	VK_F17, GWKeyF17, \
	VK_F18, GWKeyF18, \
	VK_F19, GWKeyF19, \
	VK_F20, GWKeyF20, \
 \
	VK_SPACE, ' ', \
	VK_ESCAPE, 0x1b, \
	VK_UP, GWKeyUp, \
	VK_DOWN, GWKeyDown, \
	VK_LEFT, GWKeyLeft, \
	VK_RIGHT, GWKeyRight, \
/*	VK_CENTER, GWKeyCenter, */ \
	VK_HOME, GWKeyHome, \
	VK_END, GWKeyEnd, \
	VK_PRIOR, GWKeyPageUp, \
	VK_NEXT, GWKeyPageDown, \
 \
	VK_TAB, '\t', \
	VK_RETURN, '\n', \
	VK_BACK, GWKeyBackSpace, \
	VK_DELETE, GWKeyDelete, \
	VK_INSERT, GWKeyInsert, \
 \
	VK_PAUSE, GWKeyPause, \
	VK_SCROLL, GWKeyScrollLock, \
	VK_PRINT, GWKeySysReq, \
 \
/* No VK_* definations in Windows for these values, match as ASCII values. */ \
	(unsigned char)'.', '.', \
	(unsigned char)'\'', '\'', \
	(unsigned char)',', ',', \
	(unsigned char)'-', '-', \
	(unsigned char)'_', '_', \
	(unsigned char)'+', '+', \
	(unsigned char)'=', '=', \
	(unsigned char)'/', '/', \
 \
	0, 0	/* Windows key value of 0 marks end of table. */ \
}



/*
 *	Returns the current time in milliseconds, used for time
 *	stamps on event message returns.
 */

long GWWGetCurMilliTime(void)
{
        SYSTEMTIME t;   /* Current time of day structure. */

        GetSystemTime(&t);
        return(
            (time_t)(
                (((((t.wHour * 60.0) + t.wMinute) * 60) + t.wSecond) * 1000) +
                t.wMilliseconds
            )
        );
}

/*
 *	Basically strcasecmp.
 */
int GWStrCaseCmp(const char *s1, const char *s2)
{
	if((s1 == NULL) ||
           (s2 == NULL)
	)
	    return(1);	/* False. */

	while((*s1) && (*s2))
	{
	    if(toupper(*s1) != toupper(*s2))
		return(1);	/* False. */

	    s1++;
	    s2++;
	}
	if((*s1) == (*s2))
	    return(0);	/* True. */
	else
	    return(1);	/* False. */
}

/*
 *	Parses the given geometry string which is of the format
 *	"WxH+X+Y" (variables are uppercase). Alternate geometry
 *	string format "WxH" is also acceptable and the returns for
 *	x and y will be set to 0 and 0 (respectivly) in such case.
 *
 *	Returns True on success and False on error.
 */
Boolean GWParseGeometry(
	const char *geometry,
	int *x, int *y, int *width, int *height
)
{
	const char *s = geometry;

	if(x != NULL)
	    (*x) = 0;
	if(y != NULL)
	    (*y) = 0;
	if(width != NULL)
	    (*width) = 0;
	if(height != NULL)
	    (*height) = 0;

	if(s == NULL)
	    return(False);

	/* Get width. */
	if(width != NULL)
	    (*width) = atoi(s);

	/* Seek to dimensions deliminator 'x'. */
	s = (const char *)strchr(s, 'x');
	if(s == NULL)
	{
	    /* Must have complete dimensions, so this would be an
	     * error.
	     */
	    return(False);
	}
	else
	{
	    /* Seek past deliminator. */
	    s += 1;
	}

	/* Get height. */
	if(height != NULL)
	    (*height) = atoi(s);

	/* Now check if coordinates are given. */
	while(((*s) != '\0') && ((*s) != '+') && ((*s) != '-'))
	    s++;
	if((*s) == '\0')
	{
	    /* No coordinates in the given geometry string,
	     * return success though.
	     */
	    return(True);
	}
	else
	{
	    /* Seek past deliminator (but not if its '-'). */
	    if((*s) == '+')
		s += 1;
	}

	/* Get x coordinate. */
	if(x != NULL)
	    (*x) = atoi(s);

	/* Seek to next coordinate. */
	s += 1;
	while(((*s) != '\0') && ((*s) != '+') && ((*s) != '-'))
	    s++;
	if((*s) == '\0')
	{
	    /* Second coordinate not given, this is an error. */
	    return(False);
	}
	else
	{
	    /* Seek past deliminator (but not if its '-'). */
	    if((*s) == '+')
		s += 1;
	}

	/* Get y coordinate. */
	if(y != NULL)
	    (*y) = atoi(s);

	return(True);
}

/*
 *	Returns a GW key value from the given wParam.
 */
int GWGetKeyValueFromWParam(gw_display_struct *display, WPARAM wParam)
{
	const unsigned int keytable[] = WINDOWS_TO_GW_KEYTABLE;
	const unsigned int *kt_ptr = &(keytable[0]);


	if(display == NULL)
	    return('\0');

	while((*kt_ptr) != 0)
	{
	    if(*kt_ptr++ == wParam)
		return((int)*kt_ptr);

	    kt_ptr++;
	}

	/* No match, return as character literal (make sure lower case). */
	return(tolower((int)wParam));
}


/*
 *	Loads an icon for the toplevel window (does nothing for now).
 */
void GWSetWindowIconFile(
        gw_display_struct *display,
        const char *path
)
{
	return;
}


/*
 *	Returns a created window and set up for OpenGL drawing.
 *	If the device or rendering contexts are not set in
 *	the given display structure then they will
 *	be initialized.
 */
void *GWCreateWindow(
	gw_display_struct *display,
	int x, int y,
	int width, int height,
	int depth,		/* Color depth in bits. */
	const char *title
)
{
	void *w;
	GLuint PixelFormat;	// Holds The Results After Searching For A Match
	WNDCLASS wc;		// Windows Class Structure
	DWORD dwExStyle;	// Window Extended Style
	DWORD dwStyle;	// Window Style
	RECT WindowRect;	// Grabs Rectangle Upper Left / Lower Right Values


	/* Set window size (left and top are not positions). */
	WindowRect.left = (long)0;
	WindowRect.right = (long)width;
	WindowRect.top = (long)0;
	WindowRect.bottom = (long)height;

	/* Begin setting up window class. */
	/* Redraw on resize and own dc for the window. */
	wc.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;

	/* Set WndProc() as callback for message (event) handling. */
	wc.lpfnWndProc = (WNDPROC)WndProc;

	/* No extra window class data. */
	wc.cbClsExtra = 0;

	/* Extra window data for client data. */
	wc.cbWndExtra = 4;

	/* Set the instance. */
	wc.hInstance = (HINSTANCE)display->pid;

	/* Load default icon. */
	wc.hIcon = LoadIcon(NULL, IDI_WINLOGO);

	/* Load arrow pointer. */
	wc.hCursor = LoadCursor(NULL, IDC_ARROW);

	/* No background. */
	wc.hbrBackground = NULL;

	/* No menu. */
	wc.lpszMenuName = NULL;

	/* Class name. */
	wc.lpszClassName = "OpenGL";

	/* Try to register the window class. */
	if(!RegisterClass(&wc))
	{
	    MessageBox(
		NULL,
		"Failed To Register The Window Class.",
		"ERROR",
		MB_OK | MB_ICONEXCLAMATION
	    );
	    return(NULL);
	}

	/* Use full screen mode? */
	if(display->fullscreen)
	{
	    /* Set up device mode (for specifying full screen
	     * parameters).
	     */
	    DEVMODE dmScreenSettings;

	    /* Reset device mode structure values. */
	    memset(
		&dmScreenSettings,
		0x00,
		sizeof(dmScreenSettings)
	    );

	    /* Set values for device mode structure. */
	    dmScreenSettings.dmSize = sizeof(dmScreenSettings);

	    dmScreenSettings.dmPelsWidth = width;
	    dmScreenSettings.dmPelsHeight = height;
	    dmScreenSettings.dmBitsPerPel = depth;

	    dmScreenSettings.dmFields = DM_BITSPERPEL | DM_PELSWIDTH |
	                                DM_PELSHEIGHT;

	    /* Try To Set Selected Mode And Get Results.
	     * NOTE: CDS_FULLSCREEN Gets Rid Of Start Bar.
	     */
	    if(ChangeDisplaySettings(
		&dmScreenSettings,
		CDS_FULLSCREEN
	    ) != DISP_CHANGE_SUCCESSFUL)
	    {
		/* If The Mode Fails, Offer Two Options.  Quit Or Use
		 * Windowed Mode.
		 */
		if(MessageBox(
		    NULL,
"The requested fullscreen mode is not supported by\n\
your video card. Use Windowed Mode instead?",
		    "OpenGL",
		    MB_YESNO | MB_ICONEXCLAMATION
		) == IDYES)
		{
		    /* Windowed Mode selected. */
		    display->fullscreen = FALSE;
		}
		else
		{
		    /* User does not want windowed mode. */
		    MessageBox(
			NULL,
			"The application will now be closed.",
			"ERROR",
			MB_OK | MB_ICONSTOP
		    );
		    return(NULL);
		}
	    }
	}

	/* Still in fullscreen mode? */
	if(display->fullscreen)
	{
	    dwExStyle = WS_EX_APPWINDOW;	// Window Extended Style
	    dwStyle = WS_POPUP;			// Windows Style

	    /* Hide cursor while in full screen mode? */
	    if(display->fullscreen_hide_cursor)
		ShowCursor(FALSE);
	}
	else
	{
	    /* Window extended style. */
	    dwExStyle = WS_EX_APPWINDOW | WS_EX_WINDOWEDGE;
	    dwStyle = WS_OVERLAPPEDWINDOW;	// Windows Style
	}

	/* Adjust window to actual requested size. */
	AdjustWindowRectEx(&WindowRect, dwStyle, FALSE, dwExStyle);


	/* Create the window. */
	w = CreateWindowEx(
	    dwExStyle,		// Extended Style For The Window
	    "OpenGL",		// Class Name
	    title,		// Window Title
	    dwStyle |		// Defined Window Style
	    WS_CLIPSIBLINGS |	// Required Window Style
	    WS_CLIPCHILDREN,	// Required Window Style
	    x, y,		// Window Position
	    WindowRect.right-WindowRect.left,	// Calculate Window Width
	    WindowRect.bottom-WindowRect.top,	// Calculate Window Height
	    NULL,		// No Parent Window
	    NULL,		// No Menu
	    (HINSTANCE)display->pid,	// Instance
	    display		/* Client data. */
	);
	if(!w)
	{
	    MessageBox(
		NULL,
		"Window Creation Error.",
		"ERROR",
		MB_OK | MB_ICONEXCLAMATION
	    );
	    return(NULL);
	}

	/* Set up pixel format descriptor, this defines the per
	 * pixel specifications.
	 */
	static PIXELFORMATDESCRIPTOR pfd =
	{
	    sizeof(PIXELFORMATDESCRIPTOR),	/* Size of this structure. */
	    1,				/* Version number. */
	    PFD_DRAW_TO_WINDOW | 	/* Format must support window. */
	    PFD_SUPPORT_OPENGL |	/* Format must support OpenGL. */
	    PFD_DOUBLEBUFFER,		/* Must support double buffer. */
	    PFD_TYPE_RGBA,		/* Request RGBA format. */
	    ((depth > 16) ? 24 : depth),	/* Color depth. */
	    0, 0, 0, 0, 0, 0,		/* Ignore colors bits. */
	    8,				/* Alpha bits. */
	    0,				/* Alpha shift. */
	    0,				/* Accumulation buffer bits. */
	    0, 0, 0, 0,			/* Accumulation bits (RGBA). */
	    16,				/* Z depth buffer bits. */
	    8,				/* Stencil buffer bits. */
	    0,				/* Auxiliary buffer bits. */
	    PFD_MAIN_PLANE,		/* Main drawing layer. */
	    0,				/* Reserved. */
	    0, 0, 0			/* Layer masks ignored. */
	};

	/* Did we get a device context? */
	if(display->dc == NULL)
	{
	    display->dc = GetDC((HWND)w);
	    if(display->dc == NULL)
	    {
	        MessageBox(
		    NULL,
		    "Can't Create A GL Device Context.",
		    "ERROR",
		    MB_OK | MB_ICONEXCLAMATION
	        );
	        return(NULL);
	    }
	}

	/* Select pixel format descriptor. */
	PixelFormat = ChoosePixelFormat(
	    (HDC)display->dc,
	    &pfd
	);
	if(PixelFormat)
	{
	    display->has_double_buffer = True;
	    display->alpha_channel_bits = 8;
	}
	else
	{
	    display->has_double_buffer = False;
	    display->alpha_channel_bits = 0;
	    MessageBox(
		NULL,
		"Cannot Find A Suitable PixelFormat.",
		"ERROR",
		MB_OK | MB_ICONEXCLAMATION
	    );
	    return(NULL);
	}
/* Need to try other pixel formats? */

	/* Able to set pixel format? */
	if(!SetPixelFormat(
	    (HDC)display->dc,
	    PixelFormat,
	    &pfd
	))
	{
	    MessageBox(
		NULL,
		"Can't Set The PixelFormat.",
		"ERROR",
		MB_OK | MB_ICONEXCLAMATION
	    );
	    return(NULL);
	}

	/* Are we able to get a rendering context? */
	if(display->rc == NULL)
	{
	    display->rc = wglCreateContext((HDC)display->dc);
	    if(display->rc == NULL)
	    {
	        MessageBox(
		    NULL,
		    "Can't Create A GL Rendering Context.",
		    "ERROR",
		    MB_OK | MB_ICONEXCLAMATION
	        );
	        return(NULL);
	    }
	}

	/* Try to activate rendering context. */
	if(!wglMakeCurrent(
	    (HDC)display->dc,
	    (HGLRC)display->rc
	))
	{
	    MessageBox(
		NULL,
		"Can't Activate The GL Rendering Context.",
		"ERROR",
		MB_OK | MB_ICONEXCLAMATION
	    );
	    return(NULL);
	}

	ShowWindow((HWND)w, SW_SHOW);	/* Show the window. */
	SetForegroundWindow((HWND)w);	// Slightly Higher Priority
	SetFocus((HWND)w);			// Sets Keyboard Focus To The Window

	return(w);
}

/*
 *	Initializes the graphics wrapper.
 */
gw_display_struct *GWInit(int argc, char **argv)
{
	int i, no_windows = 0;
	const char *arg_ptr;
	int x = 0, y = 0;
	int width = 320, height = 240;
	int depth = 16;
	const char *title = "Untitled";
	gw_display_struct *display;
	double aspect_offset = 0.0;
	Boolean fullscreen = False;


	/* Parse arguments. */
	for(i = 0; i < argc; i++)
	{
	    arg_ptr = (const char *)argv[i];
	    if(arg_ptr == NULL)
		continue;

	    /* Window title. */
	    if(!GWStrCaseCmp(arg_ptr, "--title") ||
               !GWStrCaseCmp(arg_ptr, "-title")
	    )
	    {
		i++;
		if(i < argc)
		{
		    arg_ptr = (const char *)argv[i];
		    title = arg_ptr;
		}
		else
		{
		    fprintf(
			stderr,
			"%s: Requires argument.\n",
			argv[i - 1]
		    );
		}
	    }
	    /* Use full screen mode? */
	    else if(!GWStrCaseCmp(arg_ptr, "--fullscreen") ||
		    !GWStrCaseCmp(arg_ptr, "-fullscreen") ||
                    !GWStrCaseCmp(arg_ptr, "--full_screen") ||
                    !GWStrCaseCmp(arg_ptr, "-full_screen")
	    )
	    {
		fullscreen = True;
	    }
	    /* Position and size of toplevel window. */
	    else if(!GWStrCaseCmp(arg_ptr, "--geometry") ||
		    !GWStrCaseCmp(arg_ptr, "-geometry")
	    )
	    {
		i++;
		if(i < argc)
		{
		    arg_ptr = (const char *)argv[i];
		    GWParseGeometry(arg_ptr, &x, &y, &width, &height);
		}
		else
		{
		    fprintf(
			stderr,
			"%s: Requires argument.\n",
			argv[i - 1]
		    );
		}
	    }
	    /* Aspect offset. */
            else if(!GWStrCaseCmp(arg_ptr, "--aspect_offset") ||
                    !GWStrCaseCmp(arg_ptr, "-aspect_offset") ||
                    !GWStrCaseCmp(arg_ptr, "--aspectoffset") ||
                    !GWStrCaseCmp(arg_ptr, "-aspectoffset")
            )
            {    
                i++;
                if(i < argc)
                {
		    aspect_offset = atof(argv[i]);
                }
                else
                {
                    fprintf(
                        stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
            }
            /* No window, this is a special argument to specify that no
             * windows (or dialogs) be created. 
             */
            else if(!GWStrCaseCmp(arg_ptr, "--no_windows") ||
                    !GWStrCaseCmp(arg_ptr, "-no_windows") ||
                    !GWStrCaseCmp(arg_ptr, "--nowindows") ||
                    !GWStrCaseCmp(arg_ptr, "-nowindows")
            )
            {
                no_windows = 1;
            }
	}

	/* Going to full screen? */
	if(fullscreen)
	{
	    /* Need to adjust width to a suitable value. */
	    if(width <= 320)
		width = 320;
	    else if(width <= 640)
		width = 640;
	    else if(width <= 800)
		width = 800;
	    else if(width <= 1024)
		width = 1024;
	    else if(width <= 1152)
		width = 1152;
	    else
		width = 320;

	    /* Set appropriate height. */
	    height = (int)(width * 0.75);
	}



	/* Allocate a new graphics wrapper display structure. */
	display = (gw_display_struct *)calloc(
	    1,
	    sizeof(gw_display_struct)
	);
	if(display == NULL)
            return(display);

	/* Reset device context. */
	display->dc = NULL;

	/* Reset rendering context. */
	display->rc = NULL;

	/* Get HINSTANCE. */
	display->pid = GetModuleHandle(NULL);

	/* Keyboard auto repeat. */
	display->keyboard_autorepeat = True;

	/* Set aspect offset. */
	display->aspect_offset = aspect_offset;

	/* Full screen? Note that the call to GWCreateWindow() will
	 * do the actual switching to full screen.
	 */
	display->fullscreen = fullscreen;

	display->fullscreen_hide_cursor = False;

	/* Reset desktop window values. */
	display->root = NULL;
	display->root_width = 0;
	display->root_height = 0;

	/* Clear key state table. */
	memset(
	    &key_state,
	    0x00,
	    GW_KEY_STATE_TABLE_MAX_KEYS * sizeof(unsigned char)
	);

	if(!no_windows)
	{
	    /* Create window. */
	    display->toplevel = GWCreateWindow(
	        display,
	        0, 0,
	        width, height,
	        depth,
	        title
	    );
	    display->width = width;
	    display->height = height;
	}

        /* Reset all other OpenGL states on record. */
        StateGLResetAll(&display->state_gl);

	/* Reset other OpenGL states. */
/*	glShadeModel(GL_SMOOTH);
	glClearColor(0.0f, 0.0f, 0.0f, 0.5f);
	glClearDepth(1.0f);
	StateGLEnable(&display->state_gl, GL_DEPTH_TEST);
	StateGLDepthFunc(&display->state_gl, GL_LEQUAL);
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
 */

	/* Set pixel storage format (for rastering font bitmaps). */
        glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
        glPixelStorei(GL_PACK_ALIGNMENT, 1);

	/* Set other GL defaults that we want. Some of these might default
	 * to on but we want them off to improve performance.
	 */
	StateGLDisable(&display->state_gl, GL_DITHER);

	return(display);
}

/*
 *	Windows event management (`processing').
 */
LRESULT CALLBACK WndProc(
	HWND hWnd,	/* Window ID. */
	UINT uMsg,	/* Message for the window. */
	WPARAM wParam,	/* Message values. */
	LPARAM lParam
)
{
	gw_display_struct *dpy = NULL;


	/* Handle by message type. */
	switch(uMsg)
	{
	  /* Create message. */
	  case WM_CREATE:
	    /* Get pointer to display structure. */
	    dpy = (gw_display_struct *)(
		((CREATESTRUCT *)lParam)->lpCreateParams
	    );
	    SetWindowLong(hWnd, 0, (LONG)dpy);
	    return(0);

	  /* Activate message. */
	  case WM_ACTIVATE:
	    dpy = (gw_display_struct *)GetWindowLong(hWnd, 0);
	    if(dpy != NULL)
	    {
		if(dpy->func_visibility != NULL)
		{
	            /* HIWORD(wParam) if true means program is minimized. */
		    if(HIWORD(wParam))
			dpy->func_visibility(
			    dpy->func_visibility_data,
			    GWVisibilityFullyObscured
			);
		    else
			dpy->func_visibility(
                            dpy->func_visibility_data,
                            GWVisibilityUnobscured
                        );
		}
	    }
	    return(0);

	  /* Alt-tab. */
	  case WM_KILLFOCUS:
            dpy = (gw_display_struct *)GetWindowLong(hWnd, 0);
	    if(dpy != NULL)
	    {
		/* Is our toplevel window going out of focus and
		 * are we in full screen mode?
		 */
		if((dpy->toplevel == (void *)hWnd) &&
                   dpy->fullscreen
		)
                {
		    /* Set toplevel window back to GUI mode. */
                    ChangeDisplaySettings(NULL, 0);
                    ShowCursor(TRUE);

		    dpy->fullscreen = False;
		}
	    }
	    return(0);

	  /* Redraw. */
	  case WM_PAINT:
	    dpy = (gw_display_struct *)GetWindowLong(hWnd, 0);
	    if(dpy != NULL)
	    {
	        if(dpy->toplevel == (void *)hWnd)
		{
		    if(dpy->func_draw != NULL)
                        dpy->func_draw(
			    dpy->func_draw_data
			);
		}
	    }
	    break;

	  /* System command. */
	  case WM_SYSCOMMAND:
	    /* Handle by system call type. */
            dpy = (gw_display_struct *)GetWindowLong(hWnd, 0);
	    switch(wParam)
	    {
	      /* Screen saver wanting to start. */
	      case SC_SCREENSAVE:
		return(0);
	        break;

	      /* Monitor entering power save. */
	      case SC_MONITORPOWER:
		return(0);
		break;
	    }
	    break;

	  /* Close window. */
	  case WM_CLOSE:
            dpy = (gw_display_struct *)GetWindowLong(hWnd, 0);
	    PostQuitMessage(0);	/* Send quit message. */
	    return(0);

	  /* Key press. */
	  case WM_KEYDOWN:
	    /* wParam is a 0 to 255 value indicating the key value.
	     */
            dpy = (gw_display_struct *)GetWindowLong(hWnd, 0);
	    if(dpy != NULL)
	    {
		int i, k;
		Boolean actually_changed = False;

		i = (int)wParam;
		if(i >= GW_KEY_STATE_TABLE_MAX_KEYS)
		    i = GW_KEY_STATE_TABLE_MAX_KEYS - 1;
		if(i < 0)
		    i = 0;

		/* Check if keyboard auto repeat is off. */
		if(dpy->keyboard_autorepeat)
		{
		    if(!(key_state[i]))
			actually_changed = True;
		}
		else
		{
		    if(key_state[i])
			return(0);
		    else
			actually_changed = True;
		}
		key_state[i] = 0x01;

		/* Get GW key value from wParam. */
		k = GWGetKeyValueFromWParam(dpy, wParam);

		/* Skip if key is a modifier and state did not actually change. */
		if(!actually_changed)
		{
		    if((k == GWKeyAlt) ||
                       (k == GWKeyCtrl) ||
		       (k == GWKeyShift)
		    )
			return(0);
		}

	        if(dpy->func_keyboard != NULL)
		{
                    long t = (long)GWWGetCurMilliTime();

                    dpy->func_keyboard(
                        dpy->func_keyboard_data,
                        k,
                        True,	/* Key press. */
			t
                    );
		}
	    }
	    return(0);

	  /* Key release. */
	  case WM_KEYUP:
	    /* wParam is a 0 to 255 value indicating the key value.
	     */
            dpy = (gw_display_struct *)GetWindowLong(hWnd, 0);
	    if(dpy != NULL)
	    {
		int i, k;
		Boolean actually_changed = False;

		i = (int)wParam;
		if(i >= GW_KEY_STATE_TABLE_MAX_KEYS)
		    i = GW_KEY_STATE_TABLE_MAX_KEYS - 1;
		if(i < 0)
		    i = 0;

		/* Check if keyboard auto repeat is off. */
		if(dpy->keyboard_autorepeat)
		{
		    if(key_state[i])
			actually_changed = True;
		}
		else
		{
		    if(key_state[i])
			actually_changed = True;
		    else
			return(0);
		}
		key_state[i] = 0x00;

		/* Get GW key value from wParam. */
		k = GWGetKeyValueFromWParam(dpy, wParam);

		/* Skip if key is a modifier and state did not actually change. */
		if(!actually_changed)
		{
		    if((k == GWKeyAlt) ||
                       (k == GWKeyCtrl) ||
		       (k == GWKeyShift)
		    )
			return(0);
		}

		if(dpy->func_keyboard != NULL)
		{
                    long t = (long)GWWGetCurMilliTime();
 
                    dpy->func_keyboard(
                        dpy->func_keyboard_data,
			k,
			False, t
		    );
		}
	    }
	    return(0);

	  /* Move. */
	  case WM_MOVE:
	    /* Ignore this. */
	    break;

	  /* Resize. */
	  case WM_SIZE:
	    /* LoWord = Width, HiWord = Height. */
            dpy = (gw_display_struct *)GetWindowLong(hWnd, 0);
	    if(dpy != NULL)
	    {
	        if(dpy->toplevel == (void *)hWnd)
		{
		    dpy->width = LOWORD(lParam);
                    dpy->height = HIWORD(lParam);

		    /* Call resize callback. */
		    if(dpy->func_resize != NULL)
                        dpy->func_resize(
			    dpy->func_resize_data,
			    0, 0,
			    dpy->width,
			    dpy->height
			);

                    /* Call redraw callback as well. */
                    if(dpy->func_draw != NULL)
                        dpy->func_draw(
                            dpy->func_draw_data
                        ); 

		}
	    }
	    return(0);

	  /* Pointer button press. */
	  case WM_LBUTTONDOWN:
            dpy = (gw_display_struct *)GetWindowLong(hWnd, 0);
	    if(dpy != NULL)
	    {
	        if(dpy->toplevel == (void *)hWnd)
		{
		    if(dpy->func_pointer != NULL)
		    {
			int x = (int)((unsigned int)lParam & 0x0000ffff);
			int y = (int)(((unsigned int)lParam & 0xffff0000) >> 16);
			long t = (long)GWWGetCurMilliTime();

			dpy->func_pointer(
			    dpy->func_pointer_data,
			    x, y, GWEventTypeButtonPress, 1, t
			);
		    }
		}
	    }
	    return(0);
          case WM_RBUTTONDOWN:
            dpy = (gw_display_struct *)GetWindowLong(hWnd, 0);
            if(dpy != NULL)
            {
                if(dpy->toplevel == (void *)hWnd)
                {
                    if(dpy->func_pointer != NULL)
                    {
                        int x = (int)((unsigned int)lParam & 0x0000ffff);
                        int y = (int)(((unsigned int)lParam & 0xffff0000) >> 16);
                        long t = (long)GWWGetCurMilliTime();

                        dpy->func_pointer(
                            dpy->func_pointer_data,
                            x, y, GWEventTypeButtonPress, 3, t
                        );
                    } 
                }
            }
            return(0);

	  /* Pointer button release. */
	  case WM_LBUTTONUP:
            dpy = (gw_display_struct *)GetWindowLong(hWnd, 0);
	    if(dpy != NULL)
	    {
	        if(dpy->toplevel == (void *)hWnd)
		{
		    if(dpy->func_pointer != NULL)
		    {
			int x = (int)((unsigned int)lParam & 0x0000ffff);
			int y = (int)(((unsigned int)lParam & 0xffff0000) >> 16);
                        long t = (long)GWWGetCurMilliTime();

			dpy->func_pointer(
			    dpy->func_pointer_data,
			    x, y, GWEventTypeButtonRelease, 1, t
			);
		    }
		}
	    }
	    return(0);
          case WM_RBUTTONUP:
            dpy = (gw_display_struct *)GetWindowLong(hWnd, 0);
            if(dpy != NULL)
            {
                if(dpy->toplevel == (void *)hWnd)  
                {
                    if(dpy->func_pointer != NULL)
                    {
                        int x = (int)((unsigned int)lParam & 0x0000ffff);
                        int y = (int)(((unsigned int)lParam & 0xffff0000) >> 16);
			long t = (long)GWWGetCurMilliTime();

                        dpy->func_pointer(
                            dpy->func_pointer_data,
                            x, y, GWEventTypeButtonRelease, 3, t
                        ); 
                    }
                }
            }
            return(0);

	  /* Pointer motion. */
	  case WM_MOUSEMOVE:
           dpy = (gw_display_struct *)GetWindowLong(hWnd, 0);
	    if(dpy != NULL)
	    {
	        if(dpy->toplevel == (void *)hWnd)
		{
		    if(dpy->func_pointer != NULL)
		    {
			int x = (int)((unsigned int)lParam & 0x0000ffff);
			int y = (int)(((unsigned int)lParam & 0xffff0000) >> 16);
                        long t = (long)GWWGetCurMilliTime();

			dpy->func_pointer(
			    dpy->func_pointer_data,
			    x, y, GWEventTypePointerMotion, 0, t
			);
		    }
		}
	    }
	    return(0);
	}

	/* Pass All Unhandled Messages To DefWindowProc */
	return(DefWindowProc(hWnd, uMsg, wParam, lParam));
}

/*
 *	Graphics wrapper management.
 */
void GWManage(gw_display_struct *display)
{
	MSG msg;

	if(display == NULL)
	    return;

	/* Update modifier key states. */
	display->alt_key_state = ((GetAsyncKeyState(VK_MENU)) ? True : False);
	display->ctrl_key_state = ((GetAsyncKeyState(VK_CONTROL)) ? True : False);
	display->shift_key_state = ((GetAsyncKeyState(VK_SHIFT)) ? True : False);

	/* Manage event messages. */
	if(PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
	{
	    switch(msg.message)
	    {
	      case WM_QUIT:
		if(display->func_close != NULL)
		    display->func_close(
		        display->func_close_data,
		        (void *)msg.hwnd
		    );
		break;
				      
	      default:
		TranslateMessage(&msg);	// Translate message
		DispatchMessage(&msg);	// Dispatch message
		break;
	    }
	}
	else
	{
	    /* Did not get any new messages, call client timeout function. */
	    if(display->func_timeout != NULL)
		display->func_timeout(
		    display->func_timeout_data
		);
	}

	return;
}

/*
 *	Graphics wrapper shutdown.
 */
void GWShutdown(gw_display_struct *display)
{
        if(display == NULL)
            return;

	/* Are we in full screen mode? */
	if(display->fullscreen)
	{
	    /* Switch back to windowed mode. */
	    ChangeDisplaySettings(NULL, 0);
	    ShowCursor(TRUE);
	}

	/* Do we have a rendering context? */
	if(display->rc)
	{
	    /* Are We Able To Release The DC And RC Contexts? */
	    if(!wglMakeCurrent(NULL, NULL))
	    {
		MessageBox(
		    NULL,
		    "Release Of DC And RC Failed.",
		    "SHUTDOWN ERROR",
		    MB_OK | MB_ICONINFORMATION
		);
	    }

	    /* Are we able to delete the RC? */
	    if(!wglDeleteContext((HGLRC)display->rc))
	    {
		MessageBox(
		    NULL,
		    "Release Rendering Context Failed.",
		    "SHUTDOWN ERROR",
		    MB_OK | MB_ICONINFORMATION
		);
	    }
	    display->rc = NULL;
	}

	/* Are we able to release the DC? */
	if((display->dc != NULL) &&
	   !ReleaseDC((HWND)display->toplevel,
	    (HDC)display->dc
	    )
	)
	{
	    MessageBox(
		NULL,
		"Release Device Context Failed.",
		"SHUTDOWN ERROR",MB_OK | MB_ICONINFORMATION
	    );
	    display->dc = NULL;
	}

	/* Are we able to destroy the window? */
	if((display->toplevel != NULL) &&
	   !DestroyWindow((HWND)display->toplevel)
	)
	{
	    MessageBox(
		NULL,
		"Could Not Release hWnd.",
		"SHUTDOWN ERROR",
		MB_OK | MB_ICONINFORMATION
	    );
	    display->toplevel = NULL;										// Set hWnd To NULL
	}

	/* Are we able to unregister class? */
	if(!UnregisterClass("OpenGL", (HINSTANCE)display->pid))
	{
	    MessageBox(
		NULL,
		"Could Not Unregister Class.",
		"SHUTDOWN ERROR",
		MB_OK | MB_ICONINFORMATION
	    );
	    display->pid = NULL;
	}

	/* Deallocate display structure. */
	free(display);

	return;
}

/*
 *	Output message in dialog form.
 */
void GWOutputMessage(
        gw_display_struct *display,
        int type,       /* One of GWOutputMessageType*. */
        const char *subject,
        const char *message,
        const char *help_details
)
{
	UINT uType = MB_OK | MB_ICONINFORMATION;


	if(display == NULL)
	    return;

	switch(type)
	{
	  case GWOutputMessageTypeGeneral:
	    uType = MB_OK | MB_ICONINFORMATION;
	    break;

	  case GWOutputMessageTypeWarning:
	    uType = MB_OK | MB_ICONWARNING;
	    break;

	  case GWOutputMessageTypeError:
	    uType = MB_OK | MB_ICONERROR;
	    break;

	  case GWOutputMessageTypeQuestion:
            uType = MB_OK | MB_ICONQUESTION;
            break;
	}

	MessageBox(
	    NULL,	/* Window handle. */
	    message,	/* Text. */
	    subject,	/* Caption. */
	    uType
	);

	return;
}

/*
 *	Map confirmation dialog and block client execution.
 */
int GWConfirmation(
        gw_display_struct *display,
        int type,       /* One of GWOutputMessageType*. */
        const char *subject,   
        const char *message,
        const char *help_details
)
{
/* Need to work on this. */
	return(GWConfirmationNotAvailable);
}

/*
 *	Simplified version of GWConfirmation().
 */
int GWConfirmationSimple(
        gw_display_struct *display,
        const char *message
)
{
	return(GWConfirmation(
	    display,
	    GWOutputMessageTypeQuestion,
	    "Confirmation",
	    message,
	    NULL
	));
}

/* Set draw callback function, needs to be set before the resize function! */
void GWSetDrawCB(
        gw_display_struct *display,
        void (*func)(void *),
	void *data
)
{
	if(display != NULL)
	{
	    display->func_draw = func;
	    display->func_draw_data = data;
	}
	return;
}


/* Set resize callback function.

   Reminder, when resize callback is set, a call must be made to it
   immediatly because under Windows, the callback needs
   to be set after the Windows is created and Windows may not send
   a resize request later.

   So this function needs to be set after the draw function.
 */
void GWSetResizeCB(
        gw_display_struct *display, 
        void (*func)(void *, int, int, int, int),
        void *data
)
{
	if(display != NULL)
	{
	    display->func_resize = func;
	    display->func_resize_data = data;

	    /* Now call the resize function. */
	    if(display->func_resize != NULL)
		display->func_resize(
		    display->func_resize_data,
		    0, 0,
		    display->width, display->height
		);
	}
	return;
}

/* Set keyboard callback function. */
void GWSetKeyboardCB(
        gw_display_struct *display,
        void (*func)(void *, int, Boolean, long),
        void *data
)
{
	if(display != NULL)
	{
	    display->func_keyboard = func;
	    display->func_keyboard_data = data;
	}
	return;
}

/* Set pointer callback function. */
void GWSetPointerCB(
	gw_display_struct *display,
        void (*func)(void *, int, int, int, int, long),
        void *data
)
{
	if(display != NULL)
	{
	    display->func_pointer = func;
	    display->func_pointer_data = data;
	}
	return;
}

/* Set visibility callback function. */
void GWSetVisibilityCB(
        gw_display_struct *display,
        void (*func)(void *, int),
        void *data
)
{
	if(display != NULL)
	{
	    display->func_visibility = func;
	    display->func_visibility_data = data;

	    /* Need to send visibility notify upon setting of function,
	     * assume visibility unobscured.
	     */
	    if(display->func_visibility != NULL)
	    {
		display->func_visibility(
		    display->func_visibility_data,
		    GWVisibilityUnobscured
		);
	    }
	}
	return;
}

/* Set save yourself callback function. */
void GWSetSaveYourselfCB(
        gw_display_struct *display,
        void (*func)(void *),
        void *data
)
{
        if(display != NULL)
	{
            display->func_save_yourself = func;
            display->func_save_yourself_data = data;
	}
	return;
}

/* Set close callback function. */
void GWSetCloseCB(
        gw_display_struct *display,
        void (*func)(void *, void *),
        void *data
)
{
	if(display != NULL)
	{
	    display->func_close = func;
	    display->func_close_data = data;
	}
	return;
}

/* Set timeout callback function. */
void GWSetTimeoutCB(
        gw_display_struct *display,
        void (*func)(void *),
        void *data
)
{
	if(display != NULL)
	{
	    display->func_timeout = func;
	    display->func_timeout_data = data;
	}
	return;
}

/*
 *	Sends a redraw event (calls redraw callback function immediatly).
 */
void GWPostRedraw(gw_display_struct *display)
{
	if(display != NULL)
	{
	    if(display->func_draw != NULL)
	    {
		display->func_draw(
		    display->func_draw_data
		);
	    }
	}

	return;
}

/*
 *	Swaps buffers.
 */
void GWSwapBuffer(gw_display_struct *display)
{
	if(display == NULL)
	    return;

	if(display->dc != NULL)
	    SwapBuffers((HDC)display->dc);

	return;
}

/*
 *	Turns keyboard auto repeat on/off.
 */
void GWKeyboardAutoRepeat(gw_display_struct *display, Boolean b)
{
	if(display != NULL)
	    display->keyboard_autorepeat = b;

	return;
}

/*
 *	Sets the pointer cursor to the one specified by cursor.
 */
void GWSetPointerCursor(   
        gw_display_struct *display,
        int cursor      /* One of GWPointerCursor*. */
)
{
        if(display == NULL)
            return;

	return;
}

/*
 *	Sets pointer cursor as busy and ignores input.
 */
void GWSetInputBusy(gw_display_struct *display)
{
	if(display == NULL)
	    return;

	return;
}

/*
 *      Sets pointer cursor as ready and resumes allowing input.
 */
void GWSetInputReady(gw_display_struct *display)
{
	if(display == NULL)
	    return;

	return;
}


#endif	/* __MSW__ */
