/*
 *                          SAR File Format IO
 *
 *	For mission and scenery files, loads data from those files into
 *	an array of structures for easy manipulation.
 */

#ifndef SARFIO_H
#define SARFIO_H

#include <sys/types.h>
#include "sar.h"


/*
 *	File format codes:
 */
#define SAR_FILE_FORMAT_SCENE		1
#define SAR_FILE_FORMAT_MISSION		2


/*
 *	File format parameter and value type codes:
 */
#define SAR_PARM_NAME			10	/* Of respective file. */
#define SAR_PARM_DESCRIPTION		11	/* Of respective file. */
#define SAR_PARM_PLAYER_MODEL_FILE	12
#define SAR_PARM_WEATHER		13
#define SAR_PARM_TIME_OF_DAY		14

#define SAR_PARM_REGISTER_LOCATION	20
#define SAR_PARM_SCENE_GPS		21
#define SAR_PARM_SCENE_MAP              22
#define SAR_PARM_SCENE_ELEVATION	23
#define SAR_PARM_SCENE_CANT		24
#define SAR_PARM_SCENE_GROUND_FLAGS	25
#define SAR_PARM_SCENE_GROUND_TILE	26

#define SAR_PARM_TEXTURE_BASE_DIRECTORY 30
#define SAR_PARM_TEXTURE_LOAD		31

#define SAR_PARM_MISSION_TYPE		40
#define SAR_PARM_MISSION_SCENE_FILE	41
#define SAR_PARM_MISSION_TIME_LEFT	42
#define SAR_PARM_MISSION_BEGIN_AT	43
#define SAR_PARM_MISSION_ARRIVE_AT	44
#define SAR_PARM_MISSION_MESSAGE_SUCCESS	45
#define SAR_PARM_MISSION_MESSAGE_FAIL		46
#define SAR_PARM_MISSION_HUMANS_TALLY	47
#define SAR_PARM_MISSION_ADD_INTERCEPT	48

#define SAR_PARM_NEW_OBJECT		50
#define SAR_PARM_NEW_HELIPAD		51
#define SAR_PARM_NEW_RUNWAY		52
#define SAR_PARM_NEW_HUMAN		53
#define SAR_PARM_NEW_FIRE		54
#define SAR_PARM_NEW_SMOKE		55
#define SAR_PARM_NEW_PREMODELED		56
#define SAR_PARM_SELECT_OBJECT_BY_NAME	57
#define SAR_PARM_MODEL_FILE		58
#define SAR_PARM_RANGE			59
#define SAR_PARM_RANGE_FAR		60
#define SAR_PARM_TRANSLATE		61
#define SAR_PARM_TRANSLATE_RANDOM	62
#define SAR_PARM_ROTATE			63
#define SAR_PARM_NO_DEPTH_TEST		64
#define SAR_PARM_POLYGON_OFFSET		65
#define SAR_PARM_CONTACT_BOUNDS_SPHERICAL	66
#define SAR_PARM_CONTACT_BOUNDS_CYLENDRICAL	67
#define SAR_PARM_CONTACT_BOUNDS_RECTANGULAR	68
#define SAR_PARM_GROUND_ELEVATION	69

#define SAR_PARM_OBJECT_NAME		70
#define SAR_PARM_OBJECT_MAP_DESCRIPTION	71
#define SAR_PARM_FUEL			72
#define SAR_PARM_HITPOINTS		73
#define SAR_PARM_ENGINE_STATE		74
#define SAR_PARM_PASSENGERS		75	/* And crew. */
#define SAR_PARM_HUMAN_MESSAGE_ENTER	76
#define SAR_PARM_HUMAN_REFERANCE	77


/*
 *	File format parameter and value structures:
 *
 *	All structures have the first member to be an int named type
 *	to indicate what kind of structure it is.
 */
/* SAR_PARM_NAME */
typedef struct {

	int type;
	char *name;

} sar_parm_name_struct;

/* SAR_PARM_DESCRIPTION */
typedef struct {

        int type;
        char *description;

} sar_parm_description_struct;

/* SAR_PARM_PLAYER_MODEL_FILE */
typedef struct {

        int type;
        char *file;

} sar_parm_player_model_file_struct;

/* SAR_PARM_WEATHER */
typedef struct {

	int type;
	char *weather_preset_name;

} sar_parm_weather_struct;

/* SAR_PARM_TIME_OF_DAY */
typedef struct {
   
        int type;
	double tod;	/* Time of day (in seconds) since midnight. */

} sar_parm_time_of_day_struct;


/* SAR_PARM_REGISTER_LOCATION */
typedef struct {

        int type;
	char *name;
	sar_position_struct pos;
	sar_direction_struct dir;

} sar_parm_register_location_struct;

/* SAR_PARM_SCENE_GPS */
typedef struct {

        int type;
	double dms_x_offset, dms_y_offset;	/* In degrees. */
	double planet_radius;			/* In meters. */

} sar_parm_scene_gps_struct;

/* SAR_PARM_SCENE_MAP */
typedef struct {

        int type;
        char *file;		/* Scene map texture file. */
        double width, height;	/* In meters. */

} sar_parm_scene_map_struct;

/* SAR_PARM_SCENE_ELEVATION */
typedef struct {

        int type;
	double elevation;	/* In meters. */

} sar_parm_scene_elevation_struct;

/* SAR_PARM_SCENE_CANT */
typedef struct {

        int type;
	double cant;		/* Radians to be applied (added) to readout
				 * angles.
				 */

} sar_parm_scene_cant_struct;

/* SAR_PARM_SCENE_GROUND_FLAGS */
typedef struct {

        int type;
	sar_obj_flags_t flags;

} sar_parm_ground_flags_struct;

/* SAR_PARM_SCENE_GROUND_TILE */
typedef struct {

        int type;
	int tile_width, tile_height;		/* In meters. */
	double close_range;			/* In meters. */
	sar_color_struct color;			/* Far solid color. */
	char *texture_name;

} sar_parm_scene_ground_tile_struct;


/* SAR_PARM_TEXTURE_BASE_DIRECTORY */
typedef struct {

        int type;
	char *directory;

} sar_parm_texture_base_directory_struct;

/* SAR_PARM_TEXTURE_LOAD */
typedef struct {

        int type;
	char *name;
	char *file;
	double priority;

} sar_parm_texture_load_struct;


/* SAR_PARM_MISSION_TYPE */
typedef struct { 
        
        int type;
        int mission_type;       /* One of MISSION_TYPE_*. */

} sar_parm_mission_type_struct;

/* SAR_PARM_MISSION_SCENE_FILE */
typedef struct {

        int type;
        char *file;

} sar_parm_mission_scene_file_struct;

/* SAR_PARM_MISSION_TIME_LEFT */
typedef struct {

        int type;
        int time_left_type;     /* One of MISSION_TIME_LEFT_*. */
        double time_left;       /* In seconds. */

} sar_parm_mission_time_left_struct;

/* SAR_PARM_MISSION_BEGIN_AT */
typedef struct {

        int type;
	char *name;		/* Name of object to begin at. */

} sar_parm_mission_begin_at_struct;

/* SAR_PARM_MISSION_ARRIVE_AT */
typedef struct {

        int type;
        char *name;		/* Name of object to arrive at. */

} sar_parm_mission_arrive_at_struct;

/* SAR_PARM_MISSION_MESSAGE_SUCCESS */
typedef struct {

        int type;
        char *message;

} sar_parm_mission_message_success_struct;

/* SAR_PARM_MISSION_MESSAGE_FAIL */
typedef struct {

        int type;
        char *message;

} sar_parm_mission_message_fail_struct;

/* SAR_PARM_MISSION_HUMANS_TALLY */
typedef struct {

        int type;
	int total, need_rescue;

} sar_parm_mission_humans_tally_struct;

/* SAR_PARM_MISSION_ADD_INTERCEPT */
typedef struct {

        int type;

	/* Referance code:
	 *	0 = Standard
	 *	2 = Begin at location
	 *	3 = Arrive at location
	 */
	int ref_code;

	sar_position_struct pos;
	double radius;
	double urgency;

	char *name;

} sar_parm_mission_add_intercept_struct;


/* SAR_PARM_NEW_OBJECT */
typedef struct {

        int type;
	int object_type;	/* One of SAR_OBJ_TYPE_*. */

} sar_parm_new_object_struct;

/* SAR_PARM_NEW_HELIPAD */
typedef struct {

        int type;

	char *style;
	double length, width;	/* In meters. */
	double recession;	/* In meters. */
	char *label;
	sar_obj_flags_t flags;
	char *ref_obj_name;
	sar_position_struct ref_offset;	/* Rel to ref object. */
        sar_direction_struct ref_dir;	/* Rel to ref object. */

} sar_parm_new_helipad_struct;

/* SAR_PARM_NEW_RUNWAY */
typedef struct {

        int type;
	double length, width;   /* In meters. */
	double range;		/* Visual range in meters. */
	char *north_label, *south_label;
        sar_obj_flags_t flags;

} sar_parm_new_runway_struct;

/* SAR_PARM_NEW_HUMAN */
typedef struct {

        int type;
	char *type_name;
	sar_obj_flags_t flags;

} sar_parm_new_human_struct;

/* SAR_PARM_NEW_FIRE */
typedef struct {

	int type;
	double	radius,		/* Size in meters. */
		height;		/* Height above base in meters. */

} sar_parm_new_fire_struct;

/* SAR_PARM_NEW_SMOKE */
typedef struct {

	int type;
	sar_position_struct offset;
	double	radius_start,	/* In meters. */
		radius_max,	/* In meters. */
		radius_rate;	/* Can be -1.0 for autocalculate. */
	int hide_at_max;
	time_t respawn_int;	/* Respawn interval in ms. */
	int total_units;
	int color_code;		/* 0 = light grey/white
				 * 1 = medium grey
				 * 2 = dark/black
				 */
} sar_parm_new_smoke_struct;

/* SAR_PARM_NEW_PREMODELED */
typedef struct {

        int type;
	char *model_type;

	/* Arguments (not including the model type). */
	char **argv;
	int argc;

} sar_parm_new_premodeled_struct;

/* SAR_PARM_SELECT_OBJECT_BY_NAME */
typedef struct {

	int type;
	char *name;

} sar_parm_select_object_by_name;

/* SAR_PARM_MODEL_FILE */
typedef struct {

        int type;
	char *file;

} sar_parm_model_file_struct;

/* SAR_PARM_RANGE */
typedef struct {

        int type;
	double range;		/* In meters. */

} sar_parm_range_struct;

/* SAR_PARM_RANGE_FAR */
typedef struct {

        int type;
        double range_far;	/* In meters. */

} sar_parm_range_far_struct;

/* SAR_PARM_TRANSLATE */
typedef struct {

        int type;
	sar_position_struct translate;

} sar_parm_translate_struct;

/* SAR_PARM_TRANSLATE_RANDOM */
typedef struct {

	int type;
	double radius_bound;	/* In meters. */
	double z_bound;		/* In meters. */

} sar_parm_translate_random_struct;

/* SAR_PARM_ROTATE */
typedef struct {

        int type;
	sar_direction_struct rotate;

} sar_parm_rotate_struct;

/* SAR_PARM_NO_DEPTH_TEST */
typedef struct {

        int type;

} sar_parm_no_depth_test_struct;

/* SAR_PARM_POLYGON_OFFSET */
typedef struct {

        int type;

	/* Offset flags, can contain any of the following:
	 * SAR_OBJ_FLAG_POLYGON_OFFSET, 
	 * SAR_OBJ_FLAG_POLYGON_OFFSET_REVERSE, and/or
	 * SAR_OBJ_FLAG_POLYGON_OFFSET_WRITE_DEPTH
	 */
	sar_obj_flags_t flags;

} sar_parm_polygon_offset_struct;

/* SAR_PARM_CONTACT_BOUNDS_SPHERICAL */
typedef struct {

        int type;
	double radius;	/* In meters. */

} sar_parm_contact_bounds_spherical_struct;

/* SAR_PARM_CONTACT_BOUNDS_CYLENDRICAL */
typedef struct {

        int type;
	double radius;			/* In meters. */
	double height_min, height_max;	/* In meters. */

} sar_parm_contact_bounds_cylendrical_struct;

/* SAR_PARM_CONTACT_BOUNDS_RECTANGULAR */
typedef struct {

        int type;
	double	x_min, x_max,		/* In meters. */
		y_min, y_max,
		z_min, z_max;

} sar_parm_contact_bounds_rectangular_struct;

/* SAR_PARM_GROUND_ELEVATION */
typedef struct {

        int type;
	double elevation;		/* In meters. */

} sar_parm_ground_elevation_struct;

/* SAR_PARM_OBJECT_NAME */
typedef struct {

        int type;
        char *name;

} sar_parm_object_name_struct;

/* SAR_PARM_OBJECT_MAP_DESCRIPTION */
typedef struct {

        int type;
        char *description;

} sar_parm_object_map_description_struct;

/* SAR_PARM_FUEL */
typedef struct {

        int type;
	double fuel, fuel_max;	/* In kg. */

} sar_parm_fuel_struct;

/* SAR_PARM_HITPOINTS */
typedef struct {

        int type;
	double hitpoints, hitpoints_max;

} sar_parm_hitpoints_struct;

/* SAR_PARM_ENGINE_STATE */
typedef struct {

        int type;
	int state;		/* One of SAR_ENGINE_STATE_*. */

} sar_parm_engine_state_struct;

/* SAR_PARM_PASSENGERS */
typedef struct {

        int type;
	int	passengers,
		passengers_max;	/* Can be -1 for leave as is. */

} sar_parm_passengers_struct;

/* SAR_PARM_HUMAN_MESSAGE_ENTER */
typedef struct {

        int type;
	char *message;

} sar_parm_human_message_enter_struct;

/* SAR_PARM_HUMAN_REFERANCE */
typedef struct {

        int type;

	/* Can contain one of the following flags;
	 * SAR_HUMAN_FLAG_RUN_TOWARDS or SAR_HUMAN_FLAG_RUN_AWAY
	 */
	sar_obj_flags_t flags;

	/* Name of referance object to run to or run away from. */
        char *referance_name;

} sar_parm_human_referance_struct;



extern void *SARParmNew(int type);

extern void SARParmDelete(void *p);
extern void SARParmDeleteAll(void ***ptr, int *total);

extern int SARParmLoadFromFile(
        const char *filename, int file_format,
        void ***parm, int *total_parms,
	int filter_parm_type,
        void *client_data,
        int (*progress_func)(void *, long, long)
);



#endif	/* SARFIO_H */
