#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "menu.h"
#include "simutils.h"
#include "sar.h"


/* Menu handling utilities and common procedures. */
void SARMenuButtonCBSelectMenu(
	sar_core_struct *core_ptr,
	const char *name
);
sar_menu_list_struct *SARMenuGetList(
	sar_menu_struct *menu_ptr, int skip, int *list_num
);
sar_menu_spin_struct *SARMenuGetSpin(
	sar_menu_struct *menu_ptr, int skip, int *spin_num
);
sar_menu_list_item_struct *SARMenuListCBGetItem(
	sar_menu_struct *menu_ptr,
	int skip
);
char *SARMenuGetCurrentMenuName(sar_core_struct *core_ptr);

/* Particular menu callbacks. */
void SARMenuCBBeginMission(sar_core_struct *core_ptr);
void SARMenuCBMissionMap(sar_core_struct *core_ptr);
void SARMenuCBMissionBrief(sar_core_struct *core_ptr);
void SARMenuCBBeginFreeFlight(sar_core_struct *core_ptr);
void SARMenuCBMapOp(sar_core_struct *core_ptr, int id_code);

/* Menu widget/objects callback. */
void SARMenuButtonCB(
	void *object, void *client_data,
        int id_code
);
void SARMenuListSelectCB(
        void *object, void *client_data,
	int item_num, void *item_ptr
);
void SARMenuSpinSelectCB(
	void *object, void *client_data,
	int id_code, char *value
);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *      Default spacings (in pixels):
 *      
 *      These values should match and coenciding values in menu.c. 
 */
#define DEF_XMARGIN             5
#define DEF_YMARGIN             5

#define DEF_WIDTH               100
#define DEF_HEIGHT              100


/*
 *	Common callback for menu buttons, sets the current menu on
 *	indicated on the core structure to that matching the given name.
 *
 *	If name is NULL then the current menu on the core structure is set
 *	to -1, indicating no menu is selected and implying simulation is
 *	being displayed.
 */
void SARMenuButtonCBSelectMenu(sar_core_struct *core_ptr, const char *name)
{
	int new_menu_num, old_menu_num;


	if(core_ptr == NULL)
	    return;

	/* Is given name NULL? */
	if(name == NULL)
	{
	    /* Set current menu on core to -1. */
	    core_ptr->cur_menu = -1;

	    /* Redraw. */
/* Comment this out for now, redrawing would be done since we're entering
   simulation and that's redrawn rather frequently.
            GWPostRedraw(core_ptr->display);
 */
	    return;
	}

	/* Record old (current) menu number. */
	old_menu_num = core_ptr->cur_menu;

	/* Get menu number to change to by matching the menu who's name
	 * is the same as the given name.
	 */
	new_menu_num = SARMatchMenuByName(core_ptr, name);
	/* Got match and matched menu number is different from the one
	 * currently selected?
	 */
	if((new_menu_num > -1) && (new_menu_num != core_ptr->cur_menu))
	{
	    /* Mark input as busy while deallocating/allocating resources
	     * during changing menus.
	     */
            GWSetInputBusy(core_ptr->display);

	    /* Is there a menu currently selected? */
	    if(SARIsMenuAllocated(core_ptr, old_menu_num))
	    {
		sar_menu_struct *old_menu = core_ptr->menu[old_menu_num];

		/* Unload background image on the old menu? */
/*		if(option.prioritize_memory) */
		if(1)
		    SARMenuUnloadBackgroundImage(old_menu);
	    }

	    /* Is the new menu to be switched to allocated on the core
	     * structure?
	     */
            if(SARIsMenuAllocated(core_ptr, new_menu_num))
	    {
		sar_menu_struct *new_menu = core_ptr->menu[new_menu_num];

		/* Load background image on new menu if the option
		 * menu_backgrounds is enabled and the new menu does not
		 * have a background image loaded yet.
		 */
		if(option.menu_backgrounds && (new_menu->bg_image == NULL))
                    SARMenuLoadBackgroundImage(new_menu);

		/* Reset any progress bars on the new menu. */
                SARMenuSetProgressSimple(
		    core_ptr->display,
		    new_menu,
		    0,
		    0
		);

		/* Update number of newly selected menu on the core
		 * structure.
		 */
	        core_ptr->cur_menu = new_menu_num;
	    }

	    /* Redraw. */
	    GWPostRedraw(core_ptr->display);

	    /* Mark input as ready. */
	    GWSetInputReady(core_ptr->display);
	}
}

/*
 *	Returns the pointer to the list object on the menu,
 *	skipping the indicated number of lists.
 *
 *	Can return NULL for no match.
 */
sar_menu_list_struct *SARMenuGetList(
	sar_menu_struct *menu_ptr, int skip, int *list_num
)     
{
        int i;
        void *ptr;
        sar_menu_list_struct *list_ptr;


        if(menu_ptr == NULL)
            return(NULL);

	if(list_num != NULL)
	    *list_num = -1;
 
        for(i = 0; i < menu_ptr->total_objects; i++)
        {       
            ptr = menu_ptr->object[i];
            if(ptr == NULL)
                continue;

            if((*(int *)ptr) != SAR_MENU_OBJECT_TYPE_LIST)
                continue;

            /* Get pointer to list object. */
            list_ptr = ptr;

            /* Skip this list object? */
            if(skip > 0)
            {
                skip--;
                continue;
            }
	    else
	    {
		if(list_num != NULL)
		    *list_num = i;
                return(list_ptr);
	    }
	}

	return(NULL);
}

/*
 *      Returns the pointer to the spin object on the menu,
 *      skipping the indicated number of spin objects.
 *
 *	Can return NULL for no match.
 */
sar_menu_spin_struct *SARMenuGetSpin(
        sar_menu_struct *menu_ptr, int skip, int *spin_num
)
{
        int i;
        void *ptr;
	sar_menu_spin_struct *spin_ptr;


        if(menu_ptr == NULL)
            return(NULL);

        if(spin_num != NULL)
            *spin_num = -1;

        for(i = 0; i < menu_ptr->total_objects; i++)
        {
            ptr = menu_ptr->object[i];
            if(ptr == NULL)
                continue;
 
            if((*(int *)ptr) != SAR_MENU_OBJECT_TYPE_SPIN)
                continue;

            /* Get pointer to spin object. */
            spin_ptr = ptr;

            /* Skip this spin object? */
            if(skip > 0)
            {
                skip--;
                continue;
            }
            else
            {
                if(spin_num != NULL)
                    *spin_num = i;

                return(spin_ptr);
            }
        }

        return(NULL);
}

/*
 *	Returns the pointer to the selected list item on the list object
 *	on the menu, skipping the indicated number of list objects.
 */
sar_menu_list_item_struct *SARMenuListCBGetItem(
	sar_menu_struct *menu_ptr,
	int skip
)
{
	int i, sel_num;
	void *ptr;
	sar_menu_list_struct *list_ptr;


	if(menu_ptr == NULL)
	    return(NULL);

	for(i = 0; i < menu_ptr->total_objects; i++)
	{
	    ptr = menu_ptr->object[i];
	    if(ptr == NULL)
		continue;

	    if((*(int *)ptr) != SAR_MENU_OBJECT_TYPE_LIST)
                continue;

	    /* Get pointer to list object. */
	    list_ptr = ptr;

	    /* Skip this list object? */
	    if(skip > 0)
	    {
		skip--;
		continue;
	    }

	    /* Get selected item number. */
	    sel_num = list_ptr->selected_item;

	    /* Return pointer to selected item if valid. */
	    if((sel_num < 0) || (sel_num >= list_ptr->total_items))
		return(NULL);
	    else
		return(list_ptr->item[sel_num]);
	}

	return(NULL);
}

/*
 *	Returns the pointer to the name of the currently selected
 *	menu or NULL on failure.
 */
char *SARMenuGetCurrentMenuName(sar_core_struct *core_ptr)
{
	int cur_menu;
	sar_menu_struct *menu_ptr;


	if(core_ptr == NULL)
	    return(NULL);

	cur_menu = core_ptr->cur_menu;
	if(SARIsMenuAllocated(core_ptr, cur_menu))
	    menu_ptr = core_ptr->menu[cur_menu];
	else
	    return(NULL);

	return(menu_ptr->name);
}


/*
 *	Begin mission.
 */
void SARMenuCBBeginMission(sar_core_struct *core_ptr)
{
        int i;
        sar_menu_struct *menu_ptr;
        sar_menu_list_item_struct *item_ptr;
        char *filename;
        sar_menu_list_item_data_struct *li_data_ptr;
	sar_progress_cb_struct cb_data;

  
        if(core_ptr == NULL)
            return;

        /* Get pointer to menu Mission. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_MISSION);
        menu_ptr = ((i < 0) ? NULL : core_ptr->menu[i]);

        /* Get pointer to selected item on first list object. */
        item_ptr = SARMenuListCBGetItem(menu_ptr, 0);
        li_data_ptr = ((item_ptr == NULL) ? NULL : item_ptr->client_data);
        filename = ((li_data_ptr == NULL) ? NULL : li_data_ptr->filename);
        free(core_ptr->cur_mission_file);
        core_ptr->cur_mission_file = StringCopyAlloc(filename);


        /* Switch to load simulation progress menu. */
        SARMenuButtonCBSelectMenu(
            core_ptr, SAR_MENU_NAME_LOADING_SIMULATION
        );

        GWSetInputBusy(core_ptr->display);

        /* Begin loading mission. */
        if(core_ptr->scene == NULL)
        {
            core_ptr->scene = (sar_scene_struct *)calloc(
                1, sizeof(sar_scene_struct)
            );
            if(core_ptr->scene == NULL)
            {
		/* Scene allocation failed, switch back to main menu,
		 * mark input as ready, and return.
		 */
                SARMenuButtonCBSelectMenu(
                    core_ptr, SAR_MENU_NAME_MAIN
                );
                GWSetInputReady(core_ptr->display);
                return;
            }
        }


	/* Unload mission (incase it's already loaded/allocated). */
	SARMissionDestroy(core_ptr->mission);
	core_ptr->mission = NULL;

        /* Unload scene (incase it's already loaded/allocated). */
        SARSceneDestroy(
            core_ptr,
            core_ptr->scene,
            &core_ptr->object,
            &core_ptr->total_objects
        );

        /* Delete draw map object names list. */
        SARDrawMapObjNameListDelete(
            &core_ptr->drawmap_objname,
            &core_ptr->total_drawmap_objnames
        );


	/* Set up load progress callback data. */
	memset(&cb_data, 0x00, sizeof(sar_progress_cb_struct));
	cb_data.core_ptr = core_ptr;
	cb_data.coeff_offset = 0.0;
	cb_data.coeff_range = 1.0;
	cb_data.can_abort = False;

	/* Load mission. */
	core_ptr->mission = SARMissionLoadFromFile(
	    core_ptr,
	    core_ptr->cur_mission_file,
	    (void *)&cb_data, SARSceneLoadProgressCB
	);
	if(core_ptr->mission == NULL)
	{
            /* Failed to load scene. */
            SARSceneDestroy(
                core_ptr,
                core_ptr->scene,
                &core_ptr->object,
                &core_ptr->total_objects
            );
            free(core_ptr->scene); 
            core_ptr->scene = NULL;

	    /* Switch back to main menu. */
            SARMenuButtonCBSelectMenu(
                core_ptr, SAR_MENU_NAME_MAIN
            );
	}
        else
        {
            /* Successfully loaded scene, switch off menus. */
            SARMenuButtonCBSelectMenu(core_ptr, NULL);

            /* Update sound on scene. */
            SARSceneSoundUpdate(
                core_ptr,
                option.engine_sounds,
                option.event_sounds,
                option.voice_sounds,
                option.music
            );
        }

	/* Need to reset timmings since all this loading probably
	 * took a long time and if the lapsed_millitime is too long
	 * simulations and other timmings can get thrown out of wack.
	 */
	SARResetTimmersCB(core_ptr, SARGetCurMilliTime());

	/* Reset other values before starting simulation. */
        core_ptr->display_help = 0;
	time_compression = 1.0;

        GWSetInputReady(core_ptr->display);

	return;
}

/*
 *	Mission map.
 */
void SARMenuCBMissionMap(sar_core_struct *core_ptr)
{
	int i;
        sar_menu_struct *menu_ptr;
        sar_menu_list_item_struct *item_ptr;
        char *filename;
        sar_menu_list_item_data_struct *li_data_ptr;


        if(core_ptr == NULL)
            return;


        /* Get pointer to mission menu. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_MISSION);
        menu_ptr = ((i < 0) ? NULL : core_ptr->menu[i]);

        /* Get pointer to selected item on first list object. */
        item_ptr = SARMenuListCBGetItem(menu_ptr, 0);
        li_data_ptr = ((item_ptr == NULL) ? NULL : item_ptr->client_data);
        filename = ((li_data_ptr == NULL) ? NULL : li_data_ptr->filename);


        /* Get pointer to mission map menu. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_MISSION_MAP);
        menu_ptr = ((i < 0) ? NULL : core_ptr->menu[i]);

	/* Load mission map from mission file. */
	SARMissionLoadMenuMap(
	    core_ptr, menu_ptr, filename
	);

	/* Switch to mission map menu. */
        SARMenuButtonCBSelectMenu(core_ptr, SAR_MENU_NAME_MISSION_MAP);

	return;
}

/*
 *	Mission to briefing.
 */
void SARMenuCBMissionBrief(sar_core_struct *core_ptr)
{
        int i;
        sar_menu_struct *menu_ptr;
        sar_menu_list_item_struct *item_ptr;
        char *filename;
        sar_menu_list_item_data_struct *li_data_ptr;
	char *mission_desc;


        if(core_ptr == NULL)
            return;


        /* Get pointer to mission menu. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_MISSION);
        menu_ptr = ((i < 0) ? NULL : core_ptr->menu[i]);

        /* Get pointer to selected item on first list object. */
        item_ptr = SARMenuListCBGetItem(menu_ptr, 0);
        li_data_ptr = ((item_ptr == NULL) ? NULL : item_ptr->client_data);
        filename = ((li_data_ptr == NULL) ? NULL : li_data_ptr->filename);


        /* Get pointer to mission brief menu. */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_MISSION_BRIEF);
        menu_ptr = ((i < 0) ? NULL : core_ptr->menu[i]);


	/* Get mission description (briefing). */
	mission_desc = SARMissionLoadDescription(filename);

        /* Set text for message box on briefing menu. */
        SARMenuSetMessageBoxSimple(
            core_ptr->display,
            menu_ptr,
            mission_desc,
            0
        );

        /* Switch to mission brief menu. */
        SARMenuButtonCBSelectMenu(core_ptr, SAR_MENU_NAME_MISSION_BRIEF);

	return;
}

/*
 *	Begin free flight.
 */
void SARMenuCBBeginFreeFlight(sar_core_struct *core_ptr)
{
	int i, status;
	sar_scene_struct *scene;
	sar_menu_struct *menu_ptr;
	sar_menu_list_item_struct *item_ptr;
	const char *filename;
	sar_menu_list_item_data_struct *li_data_ptr;
	int spin_num, spin_sel_item;
	sar_menu_spin_struct *spin_ptr;
	sar_position_struct *start_pos = NULL;
	sar_direction_struct *start_dir = NULL;
	const char *weather_preset_name = NULL;
	sar_progress_cb_struct cb_data;


	if(core_ptr == NULL)
	    return;


	/* Get pointer to menu Free Flight: Select Scenery */
	i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_FREE_FLIGHT);
	menu_ptr = ((i < 0) ? NULL : core_ptr->menu[i]);

	/* Get pointer to selected item on first list object. */
	item_ptr = SARMenuListCBGetItem(menu_ptr, 0);
	li_data_ptr = ((item_ptr == NULL) ? NULL : item_ptr->client_data);
	filename = (const char *)((li_data_ptr == NULL) ?
	    NULL : li_data_ptr->filename
	);
	free(core_ptr->cur_scene_file);
	core_ptr->cur_scene_file = StringCopyAlloc(filename);


        /* Get pointer to selected item on second list object. */
        item_ptr = SARMenuListCBGetItem(menu_ptr, 1);
        li_data_ptr = ((item_ptr == NULL) ? NULL : item_ptr->client_data);
	if(li_data_ptr != NULL)
	{
            start_pos = &li_data_ptr->pos;
	    start_dir = &li_data_ptr->dir;
	}


        /* Get pointer to menu Free Flight: Select Aircraft */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_FREE_FLIGHT_AIRCRAFT);
        menu_ptr = ((i < 0) ? NULL : core_ptr->menu[i]);

        /* Get pointer to selected item on first list object. */
        item_ptr = SARMenuListCBGetItem(menu_ptr, 0);
        li_data_ptr = ((item_ptr == NULL) ? NULL : item_ptr->client_data);
        filename = (const char *)((li_data_ptr == NULL) ?
	    NULL : li_data_ptr->filename
	);
        free(core_ptr->cur_player_model_file);
        core_ptr->cur_player_model_file = StringCopyAlloc(filename);


	/* Get pointer to menu Free Flight: Weather */
        i = SARMatchMenuByName(core_ptr, SAR_MENU_NAME_FREE_FLIGHT_WEATHER);
        menu_ptr = ((i < 0) ? NULL : core_ptr->menu[i]);

	spin_ptr = SARMenuGetSpin(
	    menu_ptr, 0, &spin_num
	);
	weather_preset_name = (const char *)SARMenuSpinGetCurrentValue(
	    menu_ptr, spin_num, &spin_sel_item
	);


	/* Check inputs. */
	/* Is player model file valid? */
	if(core_ptr->cur_player_model_file == NULL)
	{
	    GWOutputMessage(
		core_ptr->display,
		GWOutputMessageTypeError,
		"Aircraft not selected",
		"No aircraft selected.",
		"You need to select an aircraft to fly."
	    );
	    return;
	}

	/* Begin switching to free flight simulation. */

	/* Switch to load simulation progress menu. */
	SARMenuButtonCBSelectMenu(
	    core_ptr, SAR_MENU_NAME_LOADING_SIMULATION
        );

        GWSetInputBusy(core_ptr->display);

	/* Allocate scene structure as needed. */
	if(core_ptr->scene == NULL)
	{
            core_ptr->scene = (sar_scene_struct *)calloc(
                1, sizeof(sar_scene_struct)
            );
            if(core_ptr->scene == NULL)
            {
		SARMenuButtonCBSelectMenu(
                    core_ptr, SAR_MENU_NAME_MAIN
                );
                GWSetInputReady(core_ptr->display);
                return;
            }
	}

	/* Set up load progress callback data. */
	memset(&cb_data, 0x00, sizeof(sar_progress_cb_struct));
	cb_data.core_ptr = core_ptr;
	cb_data.coeff_offset = 0.0;
	cb_data.coeff_range = 1.0;
	cb_data.can_abort = True;

	/* Load scene. */
	scene = core_ptr->scene;
        status = SARSceneLoadFromFile(
            core_ptr, scene, core_ptr->cur_scene_file,
	    weather_preset_name,
            (void *)&cb_data, SARSceneLoadProgressCB
        );

	/* Done loading scene, check for errors. */
	if(status)
	{
	    /* Failed to load scene. */
	    SARSceneDestroy(
                core_ptr, scene,
		&core_ptr->object,
		&core_ptr->total_objects
	    );  
	    free(scene);
	    core_ptr->scene = NULL;

	    SARMenuButtonCBSelectMenu(
                core_ptr, SAR_MENU_NAME_MAIN
            );	    
	}
	else
	{
	    /* Successfully loaded scene. */

	    /* Add player object to scene. */
	    SARSceneAddPlayerObject(
		core_ptr, scene,
		core_ptr->cur_player_model_file,
		start_pos, start_dir
	    );

	    /* Switch off menus. */
	    SARMenuButtonCBSelectMenu(core_ptr, NULL);

	    /* Update sound on scene. */
	    SARSceneSoundUpdate(
		core_ptr,
		option.engine_sounds,
		option.event_sounds,
                option.voice_sounds,
		option.music
	    );
	}

        /* Need to reset timmings since all this loading probably
         * took a long time and if the lapsed_millitime is too long
         * simulations and other timmings can get thrown out of wack.
         */
        SARResetTimmersCB(core_ptr, SARGetCurMilliTime());

        /* Reset other values before starting simulation. */
        core_ptr->display_help = 0;
        time_compression = 1.0;

        GWSetInputReady(core_ptr->display);
}


/*
 *	Map menu button callback.
 */
void SARMenuCBMapOp(sar_core_struct *core_ptr, int id_code)
{
	gw_display_struct *display;
	int i, ptype, map_num = -1;
	int width, height, w, h;
	void *p;
        sar_menu_struct *m;
	sar_menu_map_struct *map_ptr = NULL;

	double mtop_coeff;
	int	scroll_inc = 20;	/* Scroll increment in pixels. */
	int	tex_w, tex_h,	/* All these units in pixels. */
		scroll_x_min, scroll_x_max,
		scroll_y_min, scroll_y_max;


	if(core_ptr == NULL)
	    return;

	display = core_ptr->display;
	if(display == NULL)
	    return;

        /* Get current menu (assuming it has a map object on it. */
	m = SARGetCurrentMenuPtr(core_ptr);
	if(m == NULL)
	    return;

        /* Get first map object on the menu. */
        for(i = 0; i < m->total_objects; i++)
        {
            p = m->object[i];
            if(p == NULL)
                continue;
            else
                ptype = (*(int *)p);

            if(ptype == SAR_MENU_OBJECT_TYPE_MAP)
            {
                map_num = i;
                map_ptr = (sar_menu_map_struct *)p;
                break;
            }
        }
        if(map_ptr == NULL)
            return;

/* Calculates scroll bounds by first fetching latest mtop_coeff
 * value and updates variables tex_w, tex_h, scroll_x_min, scroll_y_min,
 * scroll_x_max, and scroll_y_max.
 */
#define GET_SCROLL_BOUNDS	\
{ \
 mtop_coeff = map_ptr->m_to_pixels_coeff; \
 \
 /* Get texture size in pixels. */ \
 tex_w = (int)(map_ptr->bg_tex_width * mtop_coeff); \
 tex_h = (int)(map_ptr->bg_tex_height * mtop_coeff); \
 \
 /* Calculate scroll bounds in pixels. */ \
 scroll_x_min = (w / 2) - (tex_w / 2); \
 scroll_x_max = (tex_w / 2) - (w / 2); \
 \
 scroll_y_min = (h / 2) - (tex_h / 2); \
 scroll_y_max = (tex_h / 2) - (h / 2); \
}

	/* Get bounds of display. */
        width = display->width;
        height = display->height;

	/* Get bounds of map object. */
        w = (int)(map_ptr->width * width);
        if(w <= 0)
            w = (int)(width - (2 * DEF_XMARGIN));
        if(w <= 0)
            w = DEF_XMARGIN;

        if(map_ptr->height > 0)
            h = (int)(map_ptr->height * height);
        else
            h = DEF_HEIGHT;
        if(h <= 0)
            h = DEF_YMARGIN;


	/* Calculate scroll bounds. */
	GET_SCROLL_BOUNDS

	/* Handle by id code. */
	switch(id_code)
	{
	  case SAR_OP_MISSION_MAP_LEFT: case SAR_OP_MISSION_LOG_MAP_LEFT:
/*
	    map_ptr->scroll_x = CLIP(
		map_ptr->scroll_x + scroll_inc,
		scroll_x_min, scroll_x_max
	    );
 */
	    map_ptr->scroll_x = map_ptr->scroll_x + scroll_inc;
	    break;

          case SAR_OP_MISSION_MAP_RIGHT: case SAR_OP_MISSION_LOG_MAP_RIGHT:
/*
            map_ptr->scroll_x = CLIP(
                map_ptr->scroll_x - scroll_inc,
                scroll_x_min, scroll_x_max
            );
 */
	    map_ptr->scroll_x = map_ptr->scroll_x - scroll_inc;
            break;

          case SAR_OP_MISSION_MAP_UP: case SAR_OP_MISSION_LOG_MAP_UP:
/*
            map_ptr->scroll_y = CLIP(
                map_ptr->scroll_y - scroll_inc,
                scroll_y_min, scroll_y_max
            );
 */
	    map_ptr->scroll_y = map_ptr->scroll_y - scroll_inc;
            break;

          case SAR_OP_MISSION_MAP_DOWN: case SAR_OP_MISSION_LOG_MAP_DOWN:
/*
            map_ptr->scroll_y = CLIP(
                map_ptr->scroll_y + scroll_inc,
                scroll_y_min, scroll_y_max
            );
 */
	    map_ptr->scroll_y = map_ptr->scroll_y + scroll_inc;
            break;

          case SAR_OP_MISSION_MAP_ZOOM_IN: case SAR_OP_MISSION_LOG_MAP_ZOOM_IN:
	    /* This portion of code came from menumap.c's drag zoom
	     * event handling.
	     */
	    {
                    double scroll_x_coeff, scroll_y_coeff;
                    double prev_m_to_pixels_coeff =
                        map_ptr->m_to_pixels_coeff;

                    /* Calculate new zoom. */
                    map_ptr->m_to_pixels_coeff = CLIP(
                        map_ptr->m_to_pixels_coeff +
                            (scroll_inc *
                            map_ptr->m_to_pixels_coeff / 100),
                        0.002,          /* Zoomed out max. */
                        0.050           /* Zoomed in max. */
                    );

                    /* Need to calculate last scroll position coeff
                     * relative to the previous zoom dimension, then
                     * set the new scroll position by multiplying
                     * the scroll position coeff just calculated by
                     * the new zoom dimension.
                     */
                    if(map_ptr->bg_tex_width > 0)
                        scroll_x_coeff = (double)map_ptr->scroll_x /
                            (map_ptr->bg_tex_width *
                                prev_m_to_pixels_coeff);
                    else
                        scroll_x_coeff = 0.0;
                    map_ptr->scroll_x = (int)(map_ptr->bg_tex_width *
                        map_ptr->m_to_pixels_coeff * scroll_x_coeff);

                    if(map_ptr->bg_tex_height > 0)
                        scroll_y_coeff = -(double)map_ptr->scroll_y /
                            (map_ptr->bg_tex_height *
                                prev_m_to_pixels_coeff);
                    else
                        scroll_y_coeff = 0.0;

                    map_ptr->scroll_y = -(int)(map_ptr->bg_tex_height *
                        map_ptr->m_to_pixels_coeff * scroll_y_coeff);

                    /* Need to calculate scroll bounds again. */
                    GET_SCROLL_BOUNDS

                    /* Reclip scroll bounds. */
/* Its safe to scroll anywhere, plus we can see offmap markings.
                    map_ptr->scroll_x = CLIP(
                        map_ptr->scroll_x,
                        scroll_x_min, scroll_x_max
                    );
                    map_ptr->scroll_y = CLIP(
                        map_ptr->scroll_y,
                        scroll_y_min, scroll_y_max
                    );
 */
	    }
	    break;

          case SAR_OP_MISSION_MAP_ZOOM_OUT: case SAR_OP_MISSION_LOG_MAP_ZOOM_OUT:
            /* This portion of code came from menumap.c's drag zoom  
             * event handling.
             */
            {
                    double scroll_x_coeff, scroll_y_coeff;
                    double prev_m_to_pixels_coeff =
                        map_ptr->m_to_pixels_coeff;

                    /* Calculate new zoom. */
                    map_ptr->m_to_pixels_coeff = CLIP(
                        map_ptr->m_to_pixels_coeff -
                            (scroll_inc *
                            map_ptr->m_to_pixels_coeff / 100),
                        0.002,          /* Zoomed out max. */
                        0.050           /* Zoomed in max. */
                    );

                    /* Need to calculate last scroll position coeff
                     * relative to the previous zoom dimension, then
                     * set the new scroll position by multiplying
                     * the scroll position coeff just calculated by
                     * the new zoom dimension.
                     */
                    if(map_ptr->bg_tex_width > 0)
                        scroll_x_coeff = (double)map_ptr->scroll_x /
                            (map_ptr->bg_tex_width *
                                prev_m_to_pixels_coeff);
                    else
                        scroll_x_coeff = 0.0;
                    map_ptr->scroll_x = (int)(map_ptr->bg_tex_width *
                        map_ptr->m_to_pixels_coeff * scroll_x_coeff);

                    if(map_ptr->bg_tex_height > 0)
                        scroll_y_coeff = -(double)map_ptr->scroll_y /
                            (map_ptr->bg_tex_height * 
                                prev_m_to_pixels_coeff);
                    else
                        scroll_y_coeff = 0.0;

                    map_ptr->scroll_y = -(int)(map_ptr->bg_tex_height *
                        map_ptr->m_to_pixels_coeff * scroll_y_coeff);

                    /* Need to calculate scroll bounds again. */   
                    GET_SCROLL_BOUNDS

                    /* Reclip scroll bounds. */
/* Its safe to scroll anywhere, plus we can see offmap markings.
                    map_ptr->scroll_x = CLIP(
                        map_ptr->scroll_x,
                        scroll_x_min, scroll_x_max
                    );
                    map_ptr->scroll_y = CLIP(
                        map_ptr->scroll_y,
                        scroll_y_min, scroll_y_max
                    );
 */
            }
            break;
	}

	/* Redraw map object. */
	SARMenuDrawObject(
	    display, m, map_num
	);
	GWSwapBuffer(display);

#undef GET_SCROLL_BOUNDS
}


/*
 *	Menu button callback.
 */
void SARMenuButtonCB(
	void *object, void *client_data, int id_code
)
{
	sar_core_struct *core_ptr = (sar_core_struct *)client_data;


	if((core_ptr == NULL) ||
           (object == NULL)
	)
	    return;

	switch(id_code)
	{
	  case SAR_OP_MENU_TO_MAIN:
	    SARMenuButtonCBSelectMenu(core_ptr, SAR_MENU_NAME_MAIN);
	    break;

	  case SAR_OP_MENU_EXIT:
            runlevel = 1;
            break;

	  case SAR_OP_MENU_TO_FREE_FLIGHT:
            SARMenuButtonCBSelectMenu(core_ptr, SAR_MENU_NAME_FREE_FLIGHT);
	    break;

	  case SAR_OP_MENU_TO_FREE_FLIGHT_AIRCRAFT:
            SARMenuButtonCBSelectMenu(core_ptr, SAR_MENU_NAME_FREE_FLIGHT_AIRCRAFT);
            break;

          case SAR_OP_MENU_TO_FREE_FLIGHT_WEATHER:
            SARMenuButtonCBSelectMenu(core_ptr, SAR_MENU_NAME_FREE_FLIGHT_WEATHER);
            break;

	  case SAR_OP_MENU_TO_FREE_FLIGHT_BEGIN:
            SARMenuCBBeginFreeFlight(core_ptr);
            break;

	  case SAR_OP_MENU_TO_MISSION:
            SARMenuButtonCBSelectMenu(core_ptr, SAR_MENU_NAME_MISSION);
	    break;

          case SAR_OP_MENU_TO_MISSION_MAP:
	    SARMenuCBMissionMap(core_ptr);
            break;

          case SAR_OP_MENU_TO_MISSION_BRIEF:
	    SARMenuCBMissionBrief(core_ptr);
            break;

	  case SAR_OP_MENU_TO_MISSION_BEGIN:
	    SARMenuCBBeginMission(core_ptr);
	    break;

	  case SAR_OP_MISSION_MAP_LEFT:
	  case SAR_OP_MISSION_MAP_RIGHT:
	  case SAR_OP_MISSION_MAP_UP:
	  case SAR_OP_MISSION_MAP_DOWN:
	  case SAR_OP_MISSION_MAP_ZOOM_IN:
          case SAR_OP_MISSION_MAP_ZOOM_OUT:
          case SAR_OP_MISSION_LOG_MAP_LEFT:
          case SAR_OP_MISSION_LOG_MAP_RIGHT:
          case SAR_OP_MISSION_LOG_MAP_UP:
          case SAR_OP_MISSION_LOG_MAP_DOWN:
          case SAR_OP_MISSION_LOG_MAP_ZOOM_IN:
          case SAR_OP_MISSION_LOG_MAP_ZOOM_OUT:
            SARMenuCBMapOp(core_ptr, id_code);   
            break;

	  case SAR_OP_MENU_TO_CAMPAIGN:
/*
            SARMenuButtonCBSelectMenu(core_ptr, SAR_MENU_NAME_CAMPAIGN);
 */
            GWOutputMessage(
                core_ptr->display, GWOutputMessageTypeWarning,
                "Campaign Support Not Finished",
 "Sorry, campaign support is still under construction.",
"The coding work for campaign support is still under construction\n\
and not yet playable, please choose to either fly a mission or\n\
free flight."
            );

	    break;

	  case SAR_OP_MENU_TO_OPTIONS:
	    SARFetchOptionsToMenus(core_ptr);
	    SARMenuButtonCBSelectMenu(core_ptr, SAR_MENU_NAME_OPTIONS);
	    break;

          case SAR_OP_MENU_TO_OPTIONS_SIMULATION:
            SARMenuButtonCBSelectMenu(core_ptr, SAR_MENU_NAME_OPTIONS_SIMULATION);
            break;

          case SAR_OP_MENU_TO_OPTIONS_CONTROLLER:
            SARMenuButtonCBSelectMenu(core_ptr, SAR_MENU_NAME_OPTIONS_CONTROLLER);
            break;

	  case SAR_OP_MENU_TO_OPTIONS_CONTROLLER_JS_BTN:
	    SARMenuButtonCBSelectMenu(
		core_ptr, SAR_MENU_NAME_OPTIONS_CONTROLLER_JS_BTN
	    );
            break;

          case SAR_OP_MENU_TO_OPTIONS_GRAPHICS:
            SARMenuButtonCBSelectMenu(core_ptr, SAR_MENU_NAME_OPTIONS_GRAPHICS);
            break;

          case SAR_OP_MENU_TO_OPTIONS_SOUNDS:
            SARMenuButtonCBSelectMenu(core_ptr, SAR_MENU_NAME_OPTIONS_SOUND);
            break;


	  default:
	    fprintf(stderr,
 "SARMenuButtonCB(): Internal error: Unsupported menu operation code `%i'.\n",
		id_code
	    );
	    break;
	}

	return;
}


/*
 *	Menu standard list object callback.
 */
void SARMenuListSelectCB(
	void *object, void *client_data,
	int item_num, void *item_ptr
)
{
	char *cur_menu_name;
	sar_core_struct *core_ptr;
	sar_menu_struct *cur_menu_ptr;
	sar_menu_list_struct *list_ptr;
	sar_menu_list_item_data_struct *item_data_ptr;


	list_ptr = object;
	if(list_ptr == NULL)
	    return;

	item_data_ptr = client_data;
	if(item_data_ptr == NULL)
	    return;

	core_ptr = item_data_ptr->core_ptr;
	if(core_ptr == NULL)
	    return;


	/* Get current selected menu and name. */
	cur_menu_ptr = SARGetCurrentMenuPtr(core_ptr);
	if(cur_menu_ptr == NULL)
	    return;

	cur_menu_name = cur_menu_ptr->name;
	if(cur_menu_name == NULL)
	    return;



	/* Mission: Select Mission */
	if(!strcasecmp(cur_menu_name, SAR_MENU_NAME_MISSION))
	{
	    option.last_selected_mission = item_num;
	}
	/* Free Flight: Select Scenery */
        else if(!strcasecmp(cur_menu_name, SAR_MENU_NAME_FREE_FLIGHT))
        {
	    if(item_data_ptr->filename != NULL)
	    {
		int loc_list_num;
		sar_menu_list_struct *loc_list_ptr;

		/* Specified filename implies this is the item data from
		 * the scenery list (not the locations list on the same
		 * menu).
		 */

		/* Get pointer to second list (skip 1). */
		loc_list_ptr = SARMenuGetList(cur_menu_ptr, 1, &loc_list_num);
		if(loc_list_ptr != NULL)
		{
		    /* Load registered locations from scene, any
		     * existing locations on the list will be deleted.
		     */
		    SARSceneLoadLocationsToList(
		        core_ptr,
		        cur_menu_ptr, loc_list_ptr, loc_list_num,
		        item_data_ptr->filename
		    );

		    /* Select first item on locations list. */
		    if(loc_list_ptr->total_items > 0)
		        loc_list_ptr->selected_item = 0;
		}

		SARMenuDrawAll(core_ptr->display, cur_menu_ptr);
		GWPostRedraw(core_ptr->display);
	    }

            option.last_selected_ffscene = item_num;
	}
        /* Free Flight: Select Aircraft */
        else if(!strcasecmp(cur_menu_name, SAR_MENU_NAME_FREE_FLIGHT_AIRCRAFT))
        {

            option.last_selected_ffaircraft = item_num;
	}

	return;
}

/*
 *	Menu standard spin object callback.
 */
void SARMenuSpinSelectCB(
	void *object, void *client_data,
	int id_code, char *value
)
{
        char *cur_menu_name;
        sar_core_struct *core_ptr = client_data;
        sar_menu_spin_struct *spin_ptr = object;
        sar_menu_struct *cur_menu_ptr;


        if((core_ptr == NULL) ||
           (spin_ptr == NULL)
        )
            return;


        /* Get current selected menu and name. */
        cur_menu_ptr = SARGetCurrentMenuPtr(core_ptr);
        if(cur_menu_ptr == NULL)
            return;

        cur_menu_name = cur_menu_ptr->name;
        if(cur_menu_name == NULL)
            return;

        /* Mission: Select Mission */
        if(!strcasecmp(cur_menu_name, SAR_MENU_NAME_FREE_FLIGHT_WEATHER))
        {
            option.last_selected_ffweather = spin_ptr->cur_value;
        }

	return;
}
