#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>

#ifdef HAVE_Y
# include <Y2/Y.h>	/* We get Y_H defined by #including this. */
# include <Y2/Ylib.h>
#endif	/* HAVE_Y */

#include "../include/string.h"
#include "../include/disk.h"

#include "sound.h"


#ifdef Y_H
/* Default values for Y. */
# ifndef DEF_Y_CONNECT_ARG
#  define DEF_Y_CONNECT_ARG		"127.0.0.1:9433"
# endif
# ifndef DEF_Y_AUDIO_MODE_NAME
#  define DEF_Y_AUDIO_MODE_NAME		"Default"
# endif
#endif	/* Y_H */


snd_recorder_struct *SoundInit(
        int type,
        const char *connect_arg,
        const char *start_arg
);
int SoundManageEvents(snd_recorder_struct *recorder);
void SoundShutdown(snd_recorder_struct *recorder);

int SoundChangeMode(
	snd_recorder_struct *recorder, const char *mode_name
);

snd_play_struct *SoundStartPlay(
	snd_recorder_struct *recorder,
	const char *object,
	double volume_left, double volume_right,
	int sample_rate,
	snd_flags_t options
);
void SoundStartPlayVoid(
	snd_recorder_struct *recorder,
	const char *object,
	double volume_left, double volume_right,
	int sample_rate,
	snd_flags_t options
);

void SoundPlayMute(
	snd_recorder_struct *recorder,
	snd_play_struct *snd_play,
	int mute
);
void SoundChangePlayVolume(
	snd_recorder_struct *recorder,
	snd_play_struct *snd_play,
	double volume_left, double volume_right
);
void SoundChangePlaySampleRate(
	snd_recorder_struct *recorder,
	snd_play_struct *snd_play,
	int sample_rate
);

void SoundStopPlay(
	snd_recorder_struct *recorder,
	snd_play_struct *snd_play
);

int SoundChangeBackgroundMusic(
	snd_recorder_struct *recorder,
	const char *object,
	snd_flags_t options
);
void SoundStopBackgroundMusic(snd_recorder_struct *recorder);



/*
 *	Initializes sound server and returns a newly allocated
 *	recorder structure.
 */
snd_recorder_struct *SoundInit(
	int type,
	const char *connect_arg,
	const char *start_arg
)
{
	snd_recorder_struct *recorder = NULL;


	/* Allocate recorder structure. */
	recorder = (snd_recorder_struct *)calloc(
	    1, sizeof(snd_recorder_struct)
	);
	if(recorder == NULL)
	    return(NULL);


	/* Initialize by which sound server type: */
	switch(type)
	{
	  case SNDSERV_TYPE_Y:
#ifdef Y_H
	    if(connect_arg == NULL)
		connect_arg = (const char *)getenv("RECORDER");
	    if(connect_arg == NULL)
		connect_arg = DEF_Y_CONNECT_ARG;

	    /* Connect to sound server. */
	    recorder->con = (void *)YOpenConnection(
		start_arg,
		connect_arg
	    );
	    if(recorder->con == NULL)
	    {
		free(recorder);
		return(NULL);
	    }

	    /* Record sound server type. */
	    recorder->type = type;
#else
	    fprintf(stderr,
		"Y support not compiled.\n"
	    );
            free(recorder);   
            return(NULL);
#endif	/* Y_H */
	    break;

	  default:
	    fprintf(stderr,
 "Unsupported sound server type `%i'.\n",
		type
	    );
	    free(recorder);
	    return(NULL);
	    break;
	}

	return(recorder);
}


/*
 *	Manages sound server events.
 *
 *	Returns the number of events recieved or -1 on error.
 *
 *	If -1 is returned, then the recorder pointer value should
 *	be considered invalid and no longer used.
 */
int SoundManageEvents(snd_recorder_struct *recorder)
{
	int events_handled = 0;


	if(recorder == NULL)
	    return(-1);

	switch(recorder->type)
        {
          case SNDSERV_TYPE_Y:
#ifdef Y_H
            if(recorder->con != NULL)
            {
		YEvent event;

                while(YGetNextEvent(
		    (YConnection *)recorder->con,
                    &event,
                    False		/* Do not block. */
                ) > 0)
                {
                    events_handled++;

                    switch(event.type)
                    {
                      case YDisconnect: case YShutdown:
			/* Y server has disconnected or shutdown, the
			 * connection structure needs to be closed,
			 * deallocated, and set to NULL.
			 */
			YCloseConnection(
			    (YConnection *)recorder->con, False
			);
			recorder->con = NULL;

if(event.type == YShutdown)
 printf("Lost connection to Y server: Y server has shut down.\n");
else
 printf("Lost connection to Y server: Y server has disconnected us.\n");

			/* Deallocate our local recorder structure. */
			free(recorder);
			recorder = NULL;

			return(-1);
			break;

                      /* Sound object play has stopped. */
                      case YSoundObjectKill:
/* Check if this is the background mood music. */
/*
                        if(option.music)
                        {   
                            if((yevent.kill.yid != YIDNULL) &&
                               (yevent.kill.yid == (YID)sound.bkg_playid)
                            ) 
                                sound.bkg_playid = NULL;
                        }
 */
                        break; 

                      case YMixerChannel:
                        break;
                    }
		}
	    }
#endif  /* Y_H */
            break;

	  default:
	    free(recorder);
	    return(-1);
            break;
        }

	return(events_handled);
}

/*
 *      Shuts down the sound server.
 */
void SoundShutdown(snd_recorder_struct *recorder)
{
	if(recorder == NULL)
	    return;

        switch(recorder->type)
        {
          case SNDSERV_TYPE_Y:
#ifdef Y_H
            if(recorder->con != NULL)
            {
                /* Kill previous background music play. */
/*
                if(sound.bkg_playid != NULL)
                {
                    YDestroyPlaySoundObject(
                        (YConnection *)sound.con_data,
                        (YID)sound.bkg_playid
                    );
                    sound.bkg_playid = NULL;
                }
 */
                /* Close connection to server. */
                YCloseConnection(
                    (YConnection *)recorder->con,
                    False
                );
            }
#endif  /* Y_H */
            break;

	  default:
	    break;
	}

	/* Deallocate recorder structure. */
	free(recorder);

	return;
}



/*
 *	Changes audio mode to the one specified by mode_name.
 *
 *	Returns 0 on success or -1 on error.
 */
int SoundChangeMode(
        snd_recorder_struct *recorder, const char *mode_name
)
{
        if(recorder == NULL)
            return(-1);

        switch(recorder->type)
        {
          case SNDSERV_TYPE_Y:
#ifdef Y_H
	    if(recorder->con != NULL)
	    {
		YAudioModeValuesStruct **ptr, *yaudio_mode;
                int i, total_yaudio_modes;

		int sample_rate = 0;
		int sample_size = 0;
		int channels = 0;
		int fragment_size_bytes = 0;


                if(mode_name == NULL)
                    mode_name = DEF_Y_AUDIO_MODE_NAME;

                /* Get listing of available Audio mode. */
                ptr = YGetAudioModes(
                    (YConnection *)recorder->con,
                    &total_yaudio_modes
                );

                /* Check if specified Audio mode is in list. */
                for(i = 0; i < total_yaudio_modes; i++)  
                {
		    yaudio_mode = ptr[i];
                    if(yaudio_mode == NULL)
                        continue;

		    if(yaudio_mode->name == NULL)
			continue;

                    /* Audio mode names match? */
                    if(!strcasecmp(yaudio_mode->name, mode_name))
		    {
			sample_rate = yaudio_mode->sample_rate;
			sample_size = yaudio_mode->sample_size;
			channels = yaudio_mode->sample_rate;
			fragment_size_bytes = yaudio_mode->fragment_size_bytes;
                        break;
		    }
                }
                /* Free Audio modes listing. */
                YFreeAudioModesList(ptr, total_yaudio_modes);

	        /* Audio mode name in list? */
	        if(i >= total_yaudio_modes)
		{
		    /* Could not match audio mode name from server's list
		     * of audio mode names. So we need to return -1 here.
		     */
		    return(-1);
		}

	        /* Change Audio mode. */
                if(YChangeAudioModePreset(
                    (YConnection *)recorder->con,
                    mode_name		/* Audio mode name. */
                ) < 0)
		    return(-1);

		/* Record new Audio values on our recorder structure. */
		recorder->sample_rate = sample_rate;
		recorder->sample_size = sample_size;
		recorder->channels = channels;
		recorder->bytes_per_cycle = fragment_size_bytes;
	    }
#endif	/* Y_H */
            break;

	  default:
            break;
        }

	return(0);
}


/*
 *	Plays a sound object by given object path. Returns a dynamically
 *	allocated structure containing the play information or NULL on failure.
 */
snd_play_struct *SoundStartPlay(
        snd_recorder_struct *recorder,
        const char *object,	/* Full path to sound object. */
        double volume_left,     /* Volume, from 0.0 to 1.0. */
        double volume_right,
        int sample_rate,        /* Applied sample rate, can be 0. */
        snd_flags_t options     /* Any of SND_PLAY_OPTION_*. */
)
{
	snd_play_struct *snd_play = NULL;


	if((recorder == NULL) ||
           (object == NULL)
	)
	    return(NULL);

        switch(recorder->type)
        {
          case SNDSERV_TYPE_Y:
#ifdef Y_H
	    if(recorder->con != NULL)
	    {
		YID yid;
		YEventSoundPlay value;


		/* Set up sound play values. */
		value.flags = YPlayValuesFlagPosition |
                              YPlayValuesFlagTotalRepeats |
                              YPlayValuesFlagVolume |
                              YPlayValuesFlagSampleRate;
		value.position = 0;
		value.total_repeats = ((options & SND_PLAY_OPTION_REPEATING) ?
		    -1 : 1
		);
		value.left_volume = ((options & SND_PLAY_OPTION_MUTE) ?
		    0 : volume_left
		);
		value.right_volume = ((options & SND_PLAY_OPTION_MUTE) ?
		    0 : volume_right
		);
		value.sample_rate = sample_rate;

		/* Start playing sound object. */
                yid = YStartPlaySoundObject(
		    (YConnection *)recorder->con,
		    object,
		    &value
		);
		if(yid == YIDNULL)
		    return(NULL);

		/* Allocate sound play structure. */
		snd_play = (snd_play_struct *)malloc(sizeof(snd_play_struct));
		if(snd_play == NULL)
		    return(NULL);

		/* Record values. */
		snd_play->data = (void *)yid;
                snd_play->volume_left = volume_left;
                snd_play->volume_right = volume_right;
                snd_play->sample_rate = sample_rate;
		snd_play->options = options;
	    }
#endif	/* Y_H */
            break;

          default:
            break;
        }

	return(snd_play);
}

/*
 *	Same as SoundStartPlay() except no referance is returned and there
 *	is no way to control the sound object being played after its started
 *	playing.
 */
void SoundStartPlayVoid(
        snd_recorder_struct *recorder,
        const char *object,	/* Full path to sound object. */
        double volume_left,     /* Volume, from 0.0 to 1.0. */
        double volume_right,
        int sample_rate,        /* Applied sample rate, can be 0. */
        snd_flags_t options     /* Any of SND_PLAY_OPTION_*. */
)
{
        if((recorder == NULL) ||
           (object == NULL)
        )
            return;

        switch(recorder->type)
        {
          case SNDSERV_TYPE_Y:
#ifdef Y_H
            if(recorder->con != NULL)
            {
                YID yid;
                YEventSoundPlay value;


                /* Set up sound play values. */
                value.flags = YPlayValuesFlagPosition |
                              YPlayValuesFlagTotalRepeats |
                              YPlayValuesFlagVolume |
                              YPlayValuesFlagSampleRate;
                value.position = 0;
                value.total_repeats = ((options & SND_PLAY_OPTION_REPEATING) ?
                    -1 : 1
                );
                value.left_volume = ((options & SND_PLAY_OPTION_MUTE) ?
                    0 : volume_left
                );
                value.right_volume = ((options & SND_PLAY_OPTION_MUTE) ?
                    0 : volume_right
                );
                value.sample_rate = sample_rate;
            
                /* Start playing sound object. */
                yid = YStartPlaySoundObject(
                    (YConnection *)recorder->con,
                    object,
                    &value
                );
                if(yid == YIDNULL)
                    return;
            }
#endif  /* Y_H */
            break;

          default:
            break;
        }

        return;
}

/*
 *	Mutes or unmutes a sound object already playing.
 */
void SoundPlayMute(
        snd_recorder_struct *recorder,
        snd_play_struct *snd_play,
	int mute
)
{
        if((recorder == NULL) ||
           (snd_play == NULL)
        )
            return;

	if(mute)
	    snd_play->options |= SND_PLAY_OPTION_MUTE;
	else
	    snd_play->options &= ~SND_PLAY_OPTION_MUTE;

	SoundChangePlayVolume(
	    recorder, snd_play,
	    snd_play->volume_left, snd_play->volume_right
	);

	return;
}

/*
 *	Changes the volume of the sound object already playing.
 */
void SoundChangePlayVolume(
        snd_recorder_struct *recorder,
        snd_play_struct *snd_play,
        double volume_left,     /* Volume, from 0.0 to 1.0. */
        double volume_right
)
{
        if((recorder == NULL) || 
           (snd_play == NULL)
        )
            return;

	/* Change in volume? */
	if((volume_left == snd_play->volume_left) &&
           (volume_right == snd_play->volume_right)
	)
	    return;

        switch(recorder->type)   
        {
          case SNDSERV_TYPE_Y:
#ifdef Y_H
            if((recorder->con != NULL) &&
               (snd_play->data != NULL)
	    )
            {
                YEventSoundPlay value;


                /* Set up sound play values. */
                value.flags = YPlayValuesFlagVolume;
                value.left_volume = ((snd_play->options & SND_PLAY_OPTION_MUTE) ?
		    0 : volume_left
		);
                value.right_volume = ((snd_play->options & SND_PLAY_OPTION_MUTE) ?
		    0 : volume_right
		);

		/* Change play sound object values. */
                YSetPlaySoundObjectValues(
		    (YConnection *)recorder->con,
		    (YID)snd_play->data,
		    &value
		);

		/* Record values. */
                snd_play->volume_left = volume_left;
                snd_play->volume_right = volume_right;
            }
#endif  /* Y_H */
            break;

          default:
            break;
        }

        return;
}

/*
 *      Changes the sample rate of the sound object already playing.
 */
void SoundChangePlaySampleRate(
        snd_recorder_struct *recorder,
        snd_play_struct *snd_play,
        int sample_rate         /* Applied sample rate, can be 0. */
)
{
        if((recorder == NULL) ||
           (snd_play == NULL)
        )
            return;

        /* Change in applied sample rate? */
        if(sample_rate == snd_play->sample_rate)
            return;

        switch(recorder->type)
        {
          case SNDSERV_TYPE_Y:
#ifdef Y_H
            if((recorder->con != NULL) &&
               (snd_play->data != NULL)
            )
            {
                YEventSoundPlay value;


		if(sample_rate < 0)
                    sample_rate = 0;

                /* Set up sound play values. */
                value.flags = YPlayValuesFlagSampleRate;
                value.sample_rate = sample_rate;

                /* Change play sound object values. */
                YSetPlaySoundObjectValues(
                    (YConnection *)recorder->con,
                    (YID)snd_play->data,
                    &value
                );

		/* Record values. */
		snd_play->sample_rate = sample_rate;
            }
#endif  /* Y_H */
            break;

          default:
            break;
        }

        return;
}

/*
 *	Stops a sound object already playing.
 *
 *	The pointer snd_play should not be referanced again after calling
 *	this function.
 */
void SoundStopPlay(
        snd_recorder_struct *recorder,
        snd_play_struct *snd_play
)
{
#define DO_FREE_PLAY_STRUCT	\
if(snd_play != NULL) \
{ \
 free(snd_play); \
 snd_play = NULL; \
}

	if(recorder == NULL)
	{
	    DO_FREE_PLAY_STRUCT
	    return;
	}

	if(snd_play == NULL)
	    return;

        switch(recorder->type)
        {       
          case SNDSERV_TYPE_Y:
#ifdef Y_H
            if((recorder->con != NULL) &&
               ((YID)snd_play->data != YIDNULL)
            )
            {
		YDestroyPlaySoundObject(
                    (YConnection *)recorder->con,
                    (YID)snd_play->data
		);
	    }
#endif	/* Y_H */
	    break;

	  default:
	    break;
	}

	DO_FREE_PLAY_STRUCT
#undef DO_FREE_PLAY_STRUCT

	return;
}


/*
 *	Under construction.
 */
int SoundChangeBackgroundMusic(
        snd_recorder_struct *recorder,
        const char *object,
        snd_flags_t options
)
{
        if(recorder == NULL)
            return(-1);

        switch(recorder->type)
        {
          case SNDSERV_TYPE_Y:
#ifdef Y_H
            if(recorder->con != NULL)
            {

            }
#endif  /* Y_H */ 
            break;
        }

        return(0);
} 

/*
 *	Under construction.
 */
void SoundStopBackgroundMusic(snd_recorder_struct *recorder)
{
	if(recorder == NULL)
	    return;

        switch(recorder->type)
        {
          case SNDSERV_TYPE_Y:
#ifdef Y_H
	    if(recorder->con != NULL)
	    {


            }
#endif  /* Y_H */
            break;
	}

	return;
}
