#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "../include/os.h"

#ifdef __MSW__
# include <windows.h>
#else
# include <sys/time.h>
# include <unistd.h>
#endif

#include <GL/gl.h>
#include "../include/disk.h"
#include "../include/strexp.h"
#include "../include/string.h"
#include "gw.h"

#include "menu.h"
#include "messages.h"
#include "textinput.h"
#include "sarreality.h"
#include "obj.h"
#include "simmanage.h"
#include "weather.h"
#include "gctl.h"
#include "sound.h"
#include "mission.h"
#include "texturelistio.h"

#include "sar.h"
#include "config.h"

#include "font6x10.h"
#include "font7x14.h"
#include "fontbanner.h"
#include "fontmenu.h"


void SARHandleSignal(int s);

void SARUpdateBackgroundMusic(sar_core_struct *core_ptr);

int SARInitGCTL(sar_core_struct *core_ptr);

void SARTextInputCBSendMessage(void *client_data, const char *buffer);
void SARTextInputCBQuitSimulation(void *client_data, const char *buffer);

void SARDrawCB(void *ptr);
void SARKeyBoardCB(void *ptr, int c, Boolean state, long t);
void SARPointerCB(void *ptr, int x, int y, int state, int btn_num, long t);
void SARReshapeCB(void *ptr, int x, int y, int width, int height);
void SARVisibilityCB(void *ptr, int v);
void SARSaveYourselfCB(void *ptr);
void SARCloseCB(void *ptr, void *data);
void SARResetTimmersCB(sar_core_struct *core_ptr, time_t t_new);

sar_core_struct *SARInit(int argc, char **argv);
void SARManage(void *ptr);
void SARShutdown(sar_core_struct *core_ptr);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))

#define RADTODEG(r)     ((r) * 180 / PI)
#define DEGTORAD(d)     ((d) * PI / 180)


static int segfault_count;


sar_option_struct option;
sar_dname_struct dname;
sar_fname_struct fname;
sar_next_struct next;

double debug_value;

int runlevel;

time_t	cur_millitime,
	cur_systime,
	lapsed_millitime;

double	time_compensation,
	time_compression;


/*
 *	Records visibility of window.
 */
static int is_visable = 0;




/*
 *	Signal handler.
 */
void SARHandleSignal(int s)
{
#ifdef __MSW__

#else
	switch(s)
	{
          case SIGINT:
          case SIGTERM:  
          case SIGKILL:  
            runlevel = 1;
            break;

	  case SIGSEGV:
	    runlevel = 1;
	    segfault_count++;
            fprintf(
                stderr,
                "%s triggered a segmentation fault (%i times)\n",
                PROG_NAME,
                segfault_count
            );
	    if(segfault_count >= 3)
	    {
		fprintf(stderr,
		    "%s attempting immediate process exit().\n",
		    PROG_NAME
		);
		exit(1);
	    }
	    break;
	}
#endif
}

/*
 *	Updates the background music.
 */
void SARUpdateBackgroundMusic(sar_core_struct *core_ptr)
{
	if(core_ptr == NULL)
	    return;

/* Need to work on this. */



}

/*
 *	(Re)initialize game controller. Global options gctl_controller_type
 *	and gctl_options will be used as the specified game controller and
 *	its parameters.
 */
int SARInitGCTL(sar_core_struct *core_ptr)
{
	int status;
	gctl_struct *gctl;


	if(core_ptr == NULL)
	    return(-1);

	/* Allocate gctl structure as needed. */
	if(core_ptr->gctl == NULL)
	{
	    core_ptr->gctl = (gctl_struct *)calloc(1, sizeof(gctl_struct));
	    if(core_ptr->gctl == NULL)
		return(-1);

	    /* Note, GCtlInit() will attempt to shutdown the current
	     * controller by the current type in the gctl struct, however
	     * the pointers to the resources will be NULL or marked as
	     * uninitialized so nothing will actually be shutdown.
	     */
	}

	gctl = core_ptr->gctl;
        status = GCtlInit(
	    gctl,
	    option.gctl_controller_type,
	    option.gctl_options
	);
        if(status)
        {
	    switch(option.gctl_controller_type)
	    {
	      case GCTL_CONTROLLER_KEYBOARD:
                GWOutputMessage(
                    core_ptr->display, GWOutputMessageTypeError,
                    "Error Initializing Keyboard",
"Cannot initialize keyboard as game controller.",
"Could not initialize the keyboard as the game controller,\n\
your operating system or current working enviroment may not\n\
allow a keyboard to function as the game controller. Please\n\
try a different game controller type or contact authors for\n\
additional assistance."
                );
		break;

	      case GCTL_CONTROLLER_JOYSTICK:
                GWOutputMessage(
                    core_ptr->display, GWOutputMessageTypeError,
                    "Error Initializing Joystick(s)",
"One or more joystick(s) could not be initialized\n\
as the game controller. Please verify that your\n\
joystick(s) are plugged in and that your joystick\n\
driver or modules have been installed and loaded.",
"One or more joystick(s) could not be initialized\n\
as the game controller. Please verify that your\n\
joystick(s) are plugged in and that your joystick\n\
driver(s) or module(s) have been installed and loaded.\n\
In addition, make sure that your joysticks have been\n\
caliberated properly. Contact authors for additional\n\
assistance as needed."
                );
                break;
	    }
	    return(-1);
        }
        else
        {
            /* Set joystick button mappings. */
            gctl->js0_btn_rotate = option.js0_btn_rotate;   
            gctl->js0_btn_air_brakes = option.js0_btn_air_brakes;
            gctl->js0_btn_wheel_brakes = option.js0_btn_wheel_brakes;
            gctl->js0_btn_zoom_in = option.js0_btn_zoom_in;
            gctl->js0_btn_zoom_out = option.js0_btn_zoom_out;
	    gctl->js0_btn_hoist_up = option.js0_btn_hoist_up;
            gctl->js0_btn_hoist_down = option.js0_btn_hoist_down;

            gctl->js1_btn_rotate = option.js1_btn_rotate;
            gctl->js1_btn_air_brakes = option.js1_btn_air_brakes;
            gctl->js1_btn_wheel_brakes = option.js1_btn_wheel_brakes;
            gctl->js1_btn_zoom_in = option.js1_btn_zoom_in;   
            gctl->js1_btn_zoom_out = option.js1_btn_zoom_out; 
            gctl->js1_btn_hoist_up = option.js1_btn_hoist_up; 
            gctl->js1_btn_hoist_down = option.js1_btn_hoist_down;

	    return(0);
        }
}

/*
 *      Tempory function to handle send message callback.
 */
void SARTextInputCBSendMessage(void *client_data, const char *buffer)
{
        sar_core_struct *core_ptr = (sar_core_struct *)client_data;
        if((core_ptr == NULL) || (buffer == NULL))
            return;

	SARMessageAdd(
	    core_ptr->scene,
	    buffer
	);
}

/*
 *	Quick simulation text input callback.
 */
void SARTextInputCBQuitSimulation(void *client_data, const char *buffer)
{
	sar_core_struct *core_ptr = (sar_core_struct *)client_data;
	if((core_ptr == NULL) || (buffer == NULL))
	    return;

	if(toupper(*buffer) == 'Y')
	    SARSimEnd(core_ptr);
}

/*
 *	Redraw callback.
 */
void SARDrawCB(void *ptr)
{
	int cur_menu;
        sar_core_struct *core_ptr = (sar_core_struct *)ptr;
        if(core_ptr == NULL)
            return;

	/* Get current menu number. */
	cur_menu = core_ptr->cur_menu;

	/* Is current menu valid? */
	if(SARIsMenuAllocated(core_ptr, cur_menu))
	{
	    /* Go to 2D and draw menu. */
	    GWOrtho2D(core_ptr->display);
	    SARMenuDrawAll(
                core_ptr->display,
                core_ptr->menu[cur_menu]
	    );
	    GWSwapBuffer(core_ptr->display);
	}
	else
	{
	    /* Draw scene. */
	    SARDraw(core_ptr);
	}
}

/*
 *	Keyboard callback.
 */
void SARKeyBoardCB(void *ptr, int c, Boolean state, long t)
{
        sar_core_struct *core_ptr = (sar_core_struct *)ptr;
        sar_menu_struct *menu_ptr;
        if(core_ptr == NULL)
            return;

	/* First check to see if the text input prompt is mapped by
	 * checking if the text input callback function is set (not
	 * NULL).
	 */
        if(core_ptr->text_input_cb != NULL)
        {
	    /* Pass key event to text input prompt handler if it is
	     * a key press.
	     */
            if(state)
                SARTextInputHandleKey(core_ptr, c);

	    /* Return, do not continue. This is so that the key event
	     * is not passed to any other key event handlers.
	     */
            return;
        }


	/* Get pointer to currently selected menu on core structure. */
        menu_ptr = SARGetCurrentMenuPtr(core_ptr);

	/* Is the selected menu pointer NULL? */
        if(menu_ptr == NULL)
        {
	    /* Menu pointer is NULL, implying that no menu is selected. */

	    /* We must be in simulation so pass key event to simulation
	     * key event handler. Note that this function will not post
	     * a redraw.
	     */
	    SARKey(core_ptr, c, state, t); 
        }
        else
        {
	    /* A menu is selected, pass key event to menu system key event
	     * handler.
	     */
            SARMenuManageKey(core_ptr->display, menu_ptr, c, state);
	}
}

/*
 *	Pointer callback.
 */
void SARPointerCB(void *ptr, int x, int y, int state, int btn_num, long t)
{
        sar_core_struct *core_ptr = (sar_core_struct *)ptr;
	sar_menu_struct *menu_ptr;
	if(core_ptr == NULL)
	    return;

	menu_ptr = SARGetCurrentMenuPtr(core_ptr);
	if(menu_ptr == NULL)
	{
	    /* No in game pointer event management yet. */

	}
	else
	{
	    /* Forward event to menu system's pointer event management
	     * function.
	     */
	    SARMenuManagePointer(
		core_ptr->display, menu_ptr,
		x, y, state, btn_num
	    );
	}

	return;
}

/*
 *	Resize callback, updates the last size of the toplevel window.
 *
 *	If width or height are less than 1 the last glViewport() setting
 *	for width or height will be called again.
 */
void SARReshapeCB(void *ptr, int x, int y, int width, int height)
{
	sar_core_struct *core_ptr = (sar_core_struct *)ptr;
	gw_display_struct *dpy;

	if(core_ptr == NULL)
	    return;

	dpy = core_ptr->display;
	if(dpy == NULL)
	    return;

	/* Change width and height if specified. */
        if((width > 0) && (height > 0))
        {
            /* Update last positions of toplevel window on global
	     * options.
	     */
            option.last_x = x;
            option.last_y = y;
            option.last_width = width;
            option.last_height = height;
        }
	else
	{
	    /* Use previous size specified on the display structure. */
	    width = dpy->width;
	    height = dpy->height;
	}

	/* Set new view port dimensions. */
        glViewport(0, 0, width, height);

	return;
}


/*
 *	Visibility change callback.
 */
void SARVisibilityCB(void *ptr, int v)
{
	switch(v)
	{
	  case GWVisibilityFullyObscured:
	    is_visable = 0;
	    break;

	  default:
	    is_visable = 1;
	    break;
	}

	return;
}

/*
 *	Save yourself callback.
 */
void SARSaveYourselfCB(void *ptr)
{
        sar_core_struct *core_ptr = (sar_core_struct *)ptr;
        if(core_ptr == NULL)
            return;



	return;
}

/*
 *	Close window callback.
 */
void SARCloseCB(void *ptr, void *data)
{
        sar_core_struct *core_ptr = (sar_core_struct *)ptr;
        if(core_ptr == NULL)
            return;

	/* Is a menu selected? */
	if(core_ptr->cur_menu > -1)
	{
	    /* Switch global runlevel to 1, causing the program to begin
	     * shutting down.
	     */
	    runlevel = 1;
	}
	else
	{
	    /* Just end simulation, going back to the menus. */
            SARSimEnd(core_ptr);
	}

	return;
}

/*
 *	Resets global timmers to zero and updates the global
 *	variable cur_millitime to the value of t_new.
 *
 *	Global variable lapsed_millitime will be reset to 0 and
 *	time_compensation will be set to 1.0.
 *
 *	Timmings on core structure, scene, objects, and other related
 *	resources will also be reset.
 */
void SARResetTimmersCB(sar_core_struct *core_ptr, time_t t_new)
{
	int i, n;
	gctl_struct *gc;
	sar_scene_struct *scene;
	sar_cloud_bb_struct *cloud_bb_ptr;
	sar_light_struct *light_ptr;
	sar_object_struct *obj_ptr;
	sar_object_aircraft_struct *obj_aircraft_ptr;
        sar_object_ground_struct *obj_ground_ptr;
	sar_object_human_struct *obj_human_ptr;
	sar_object_smoke_struct *obj_smoke_ptr;
	sar_object_fire_struct *obj_fire_ptr;
	sar_object_explosion_struct *obj_explosion_ptr;
	sar_mission_struct *mission = NULL;


	if(core_ptr == NULL)
	    return;


	/* Set global variable cur_millitime to t_new. */
	cur_millitime = t_new;

	/* Reset lapsed and time compensation. */
	lapsed_millitime = 0;
	time_compensation = 1.0;


	/* Reset global next timmers. */
	memset(&next, 0x00, sizeof(sar_next_struct));


	/* Reset game controller timmers. */
	gc = core_ptr->gctl;
	if(gc != NULL)
	{
	    gc->last_updated = 0;

	    gc->heading_kb_last = 0;
	    gc->pitch_kb_last = 0;
	    gc->bank_kb_last = 0;
            gc->throttle_kb_last = 0;
            gc->hat_x_kb_last = 0;
            gc->hat_y_kb_last = 0;

	    gc->zoom_in_kb_last = 0;
	    gc->zoom_out_kb_last = 0;

	    gc->hoist_up_kb_last = 0;
	    gc->hoist_down_kb_last = 0;

	    gc->air_brakes_kb_last = 0;
	    gc->wheel_brakes_kb_last = 0;
	}

	/* Reset timmings on scene. */
	scene = core_ptr->scene;
	if(scene != NULL)
	{
	    scene->message_display_untill = 0;
	    scene->camera_ref_title_display_untill = 0;

	    /* Cloud `billboard' objects lightning timmers. */
	    for(i = 0; i < scene->total_cloud_bbs; i++)
	    {
		cloud_bb_ptr = scene->cloud_bb[i];
		if(cloud_bb_ptr == NULL)
		    continue;

		cloud_bb_ptr->lightning_last = 0;
	    }
	}

	/* Reset timmings on mission. */
	mission = core_ptr->mission;
	if(mission != NULL)
	{
	    mission->next_check = 0;
	}


	/* Reset timmers on objects. */
	for(i = 0; i < core_ptr->total_objects; i++)
	{
	    obj_ptr = core_ptr->object[i];
	    if(obj_ptr == NULL)
		continue;

	    /* Any object with a defined life span needs to die whenever
	     * timmers are reset.
	     */
	    if(obj_ptr->life_span > 0)
		obj_ptr->life_span = 1;

	    switch(obj_ptr->type)
	    {
	      case SAR_OBJ_TYPE_AIRCRAFT:
		obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
		if(obj_aircraft_ptr != NULL)
		{
		    obj_aircraft_ptr->next_engine_on = 0;
		}
		break;

	      case SAR_OBJ_TYPE_GROUND:
		obj_ground_ptr = (sar_object_ground_struct *)obj_ptr->data;
		break;

	      case SAR_OBJ_TYPE_HUMAN:
		obj_human_ptr = (sar_object_human_struct *)obj_ptr->data;
		break;

	      case SAR_OBJ_TYPE_SMOKE:
		obj_smoke_ptr = (sar_object_smoke_struct *)obj_ptr->data;
		if(obj_smoke_ptr != NULL)
		{
		    obj_smoke_ptr->respawn_next = 0;
		}
		break;

	      case SAR_OBJ_TYPE_FIRE:
		obj_fire_ptr = (sar_object_fire_struct *)obj_ptr->data;
		if(obj_fire_ptr != NULL)
		{
		    obj_fire_ptr->next_frame_inc = 0;
		}
		break;

              case SAR_OBJ_TYPE_EXPLOSION:
                obj_explosion_ptr = (sar_object_explosion_struct *)obj_ptr->data;
		if(obj_explosion_ptr != NULL)
		{
		    obj_explosion_ptr->next_frame_inc = 0;
		}
                break;

/* Add handling for other types of objects which have timming in their
 * substructures.
 */
	    }

	    /* Lights. */
	    for(n = 0; n < obj_ptr->total_lights; n++)
	    {
		light_ptr = obj_ptr->light[n];
		if(light_ptr != NULL)
		{
		    light_ptr->next_off = 0;
		    light_ptr->next_on = 0;
		}
	    }

/* Add other object common timming values that need to be reset here. */

	}

	return;
}


/*
 *	SAR initialize.
 */
sar_core_struct *SARInit(int argc, char **argv)
{
	int i;
	char *strptr;
	const char *cstrptr;
#ifndef __MSW__
	char *strptr2;
#endif
	sar_color_struct *color;
	sar_core_struct *core_ptr;
	gw_display_struct *dpy;
	const char *font_name = NULL;
	Boolean direct_rendering = True;
	Boolean fullscreen = False;
	Boolean startup_no_sound = False;
	const char *sound_server_connect_arg = NULL;
	Boolean notify_install_local_error = False,
		notify_install_local_success = False;
	char cwd[PATH_MAX];
	char tmp_path[PATH_MAX + NAME_MAX];
	int strc;
	char **strv;
	double aspect_offset = 0.0;
	char geometry_string[GW_GEOMETRY_STRING_MAX];


        /* Set up signals to watch for. */
        signal(SIGINT, SARHandleSignal); 
        signal(SIGTERM, SARHandleSignal);
        signal(SIGSEGV, SARHandleSignal);

	/* Get current working directory. */
	getcwd(cwd, PATH_MAX);
	cwd[PATH_MAX - 1] = '\0';

	(*geometry_string) = '\0';


	/* Reset globals. */
	cur_millitime = 0;
	lapsed_millitime = 0;
	time_compensation = 1.0;
	time_compression = 1.0;


	/* Get local game directory. */
#ifdef __MSW__
	strncpy(dname.local_data, cwd, PATH_MAX);
#else	/* All else assume UNIX. */
	strptr = getenv("HOME");
	strptr2 = PrefixPaths(
	    ((strptr == NULL) ? cwd : strptr),
	    SAR_DEF_LOCAL_DATA_DIR
	);
	strncpy(
	    dname.local_data,
	    ((strptr2 == NULL) ? cwd : strptr2),
	    PATH_MAX
	);
#endif
	dname.local_data[PATH_MAX - 1] = '\0';

	/* Get global game directory. */
#ifdef __MSW__
        strncpy(dname.global_data, cwd, PATH_MAX);
#else	/* All else assume UNIX. */
	strptr = getenv(SAR_DEF_ENV_GLOBAL_DIR);
	if((strptr == NULL) ? 0 : !ISPATHABSOLUTE(strptr))
	    fprintf(
		stderr,
		SAR_DEF_ENV_GLOBAL_DIR ": Value must be an absolute path.\n"
	    );
	strncpy(
            dname.global_data,
	    (strptr == NULL) ? SAR_DEF_GLOBAL_DATA_DIR : strptr,
            PATH_MAX
        );
#endif
	dname.global_data[PATH_MAX - 1] = '\0';

	/* Set default path to preferances (rc) file. */
	strptr = PrefixPaths(dname.local_data, SAR_DEF_RC_FILE);
	strncpy(
	    fname.preferances,
	    ((strptr == NULL) ? SAR_DEF_RC_FILE : strptr),
	    PATH_MAX + NAME_MAX
	);
	fname.preferances[PATH_MAX + NAME_MAX - 1] = '\0';

	/* Set default global textures list file. */
	strptr = PrefixPaths(dname.global_data, SAR_DEF_TEXTURES_FILE);
	strncpy(
            fname.textures,
            (strptr == NULL) ? SAR_DEF_TEXTURES_FILE : strptr,
            PATH_MAX + NAME_MAX
        );
        fname.textures[PATH_MAX + NAME_MAX - 1] = '\0';

	/* Set default global human data presets file. */
	strptr = PrefixPaths(dname.global_data, SAR_DEF_HUMAN_FILE);
        strncpy(
            fname.human,
            (strptr == NULL) ? SAR_DEF_HUMAN_FILE : strptr,
            PATH_MAX + NAME_MAX
        );
        fname.human[PATH_MAX + NAME_MAX - 1] = '\0';

	/* Set default global weather data presets file. */
	strptr = PrefixPaths(dname.global_data, SAR_DEF_WEATHER_FILE);
        strncpy(
            fname.weather,
            (strptr == NULL) ? SAR_DEF_WEATHER_FILE : strptr,
            PATH_MAX + NAME_MAX
        );
        fname.weather[PATH_MAX + NAME_MAX - 1] = '\0';
        



	/* Reset options to defaults. */
	option.menu_backgrounds = True;
	option.console_quiet = False;
	option.internal_debug = False;
	option.runtime_debug = False;
	option.prioritize_memory = False;

	option.textured_ground = True;
	option.textured_clouds = True;
	option.textured_objects = True;
	option.atmosphere = True;
	option.dual_pass_depth = True;
	option.prop_wash = True;
	option.smoke_trails = True;
	option.gl_polygon_offset_factor = -1.9;
	option.gl_shade_model = GL_SMOOTH;
	option.visibility_max = 4;

	option.engine_sounds = False;
        option.event_sounds = False;
        option.voice_sounds = False;
        option.music = False;

	option.aircraft_engine_sound_distance = SAR_DEF_AIRCRAFT_ENGINE_SOUND_DISTANCE;

	color = &option.hud_color;
	color->a = 1.0;
	color->r = 1.0;
        color->g = 1.0;
        color->b = 1.0;

	color = &option.message_color;
        color->a = 1.0;
        color->r = 1.0;
        color->g = 1.0;
        color->b = 1.0;

	option.explosion_frame_int = SAR_DEF_EXPLOSION_FRAME_INT;
	option.splash_frame_int = SAR_DEF_SPLASH_FRAME_INT;
	option.crash_explosion_life_span = SAR_DEF_CRASH_EXPLOSION_LIFE_SPAN;
	option.fuel_tank_life_span = SAR_DEF_FUEL_TANK_LIFE_SPAN;

	option.rotor_wash_vis_coeff = SAR_DEF_ROTOR_WASH_VIS_COEFF;

        option.gctl_controller_type = GCTL_CONTROLLER_KEYBOARD;

	option.js0_btn_rotate = 3;
	option.js0_btn_air_brakes = 6;
	option.js0_btn_wheel_brakes = 0;
	option.js0_btn_zoom_in = 2;
	option.js0_btn_zoom_out = 1;
	option.js0_btn_hoist_up = 5;
	option.js0_btn_hoist_down = 4;

        option.js1_btn_rotate = 3;
        option.js1_btn_air_brakes = 6;
        option.js1_btn_wheel_brakes = 0;
        option.js1_btn_zoom_in = 2;
        option.js1_btn_zoom_out = 1;
        option.js1_btn_hoist_up = 5;
        option.js1_btn_hoist_down = 4;

/*
	option.gctl_options =	GCTL_OPT_JS0_INIT |
				GCTL_OPT_JS0_PITCH | GCTL_OPT_JS0_BANK |
				GCTL_OPT_JS0_HEADING | GCTL_OPT_JS0_THROTTLE |
				GCTL_OPT_JS0_HAT;
 */
	option.gctl_options = 0;

	option.hoist_contact_expansion_coeff = 1.0;
	option.damage_resistance_coeff = 1.0;
	option.flight_physics = FLIGHT_PHYSICS_REALISTIC;	/* Make it hard. */

	option.last_selected_mission = 0;
	option.last_selected_ffscene = 0;
	option.last_selected_ffaircraft = 0;
	option.last_selected_ffweather = 0;
	option.last_x = 0;
	option.last_y = 0;
	option.last_width = SAR_DEF_WINDOW_WIDTH;
	option.last_height = SAR_DEF_WINDOW_HEIGHT;


	/* Parse arguments. */
#ifdef __MSW__
/* Windows needs to parse arguments starting from 0. */
	for(i = 0; i < argc; i++)
#else
	for(i = 1; i < argc; i++)
#endif
	{
	    cstrptr = (const char *)argv[i];
	    if(cstrptr == NULL)
		 continue;

            /* Run time debug. */
            if(strcasepfx(cstrptr, "--runtime_debug") ||
               strcasepfx(cstrptr, "-runtime_debug")
            )
            {
		option.runtime_debug = True;
            }
	    /* Help. */
	    else if(strcasepfx(cstrptr, "--he") ||
                    strcasepfx(cstrptr, "-he") ||
                    strcasepfx(cstrptr, "-?") ||
                    strcasepfx(cstrptr, "/h") ||
                    strcasepfx(cstrptr, "/?") ||
                    strcasepfx(cstrptr, "?")
	    )
	    {
		printf(PROG_USAGE_MESG);
		return(NULL);
	    }
	    /* Version. */
	    else if(!strcasecmp(cstrptr, "--version") ||
                    !strcasecmp(cstrptr, "-version")
	    )
	    {
		/* Print program version and copyright. */
		printf("%s Version %s\n%s\n",
		    PROG_NAME, PROG_VERSION,
		    PROG_COPYRIGHT
		);
		return(NULL);
	    }
            /* RC file. */
            else if(strcasepfx(cstrptr, "--rcfile") ||
                    strcasepfx(cstrptr, "-rcfile") ||
                    strcasepfx(cstrptr, "-f")
	    )
	    {
		i++;
		if(i < argc)
		{
		    char *strptr3;


		    strptr3 = PathSubHome(argv[i]);
                    strncpy(
			tmp_path,
			((strptr3 == NULL) ? argv[i] : strptr3),
			PATH_MAX + NAME_MAX
		    );
		    tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';

		    if(!ISPATHABSOLUTE(tmp_path))
		    {
			strptr3 = PrefixPaths(cwd, tmp_path);
			if(strptr3 != NULL)
			{
			    strncpy(tmp_path, strptr3, PATH_MAX + NAME_MAX);
                            tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';
			}
		    }

		    strncpy(fname.preferances, tmp_path, PATH_MAX + NAME_MAX);
		    fname.preferances[PATH_MAX + NAME_MAX - 1] = '\0';
		}
		else
		{
		    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
		}
	    }
	    /* Control. */
	    else if(strcasepfx(cstrptr, "--con") ||
                    strcasepfx(cstrptr, "-c")
            )
	    {
		i++;
                if(i < argc)
                {   
                    cstrptr = (const char *)argv[i];
		    if(cstrptr == NULL)
			continue;

		    /* Set control type, initialize later. */
		    if(strcasepfx(cstrptr, "j"))
			option.gctl_controller_type = GCTL_CONTROLLER_JOYSTICK;
		    else if(strcasepfx(cstrptr, "k"))
                        option.gctl_controller_type = GCTL_CONTROLLER_KEYBOARD;
		    else
			fprintf(stderr,
			    "%s: Unsupported argument `%s'\n",
			    argv[i - 1], cstrptr
			);
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
	    }
	    /* Software rendering? */
            else if(!strcasecmp(cstrptr, "--software_rendering") ||
                    !strcasecmp(cstrptr, "-software_rendering") ||
                    !strcasecmp(cstrptr, "--softwarerendering") ||
                    !strcasecmp(cstrptr, "-softwarerendering") ||
                    !strcasecmp(cstrptr, "--software") ||
                    !strcasecmp(cstrptr, "-software")
            )
            {
		direct_rendering = False;
	    }
            /* Display (connection to X server). */
            else if(strcasepfx(cstrptr, "--dis") ||
                    strcasepfx(cstrptr, "-dis") ||
                    strcasepfx(cstrptr, "--dpy") ||
                    strcasepfx(cstrptr, "-dpy")
            )
            {
                i++;
                if(i < argc)
                {
/* Need to work on this. */
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
            }
	    /* Full screen. */
	    else if(!strcasecmp(cstrptr, "--full_screen") ||
                    !strcasecmp(cstrptr, "-full_screen") ||
                    !strcasecmp(cstrptr, "--fullscreen") ||
                    !strcasecmp(cstrptr, "-fullscreen")
            )
            {
		fullscreen = True;
            }
	    /* Font (font name for XFonts, X only). */
            else if(!strcasecmp(cstrptr, "--font") ||
                    !strcasecmp(cstrptr, "-font") ||
                    !strcasecmp(cstrptr, "--fn") ||
                    !strcasecmp(cstrptr, "-fn")
            )
            {
                i++;
                if(i < argc)
                {
		    font_name = (const char *)argv[i];
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
	    }
	    /* Geometry (position and size of toplevel window). */
	    else if(!strcasecmp(cstrptr, "--geometry") ||
                    !strcasecmp(cstrptr, "-geometry")
            )
            {
                i++;
                if(i < argc)
                {
		    strncpy(geometry_string, argv[i], GW_GEOMETRY_STRING_MAX);
		    geometry_string[GW_GEOMETRY_STRING_MAX - 1] = '\0';
                }   
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
            }
	    /* Aspect offset. */
	    else if(!strcasecmp(cstrptr, "--aspect_offset") ||
                    !strcasecmp(cstrptr, "-aspect_offset") ||
                    !strcasecmp(cstrptr, "--aspectoffset") ||
                    !strcasecmp(cstrptr, "-aspectoffset")
	    )
	    {
                i++;
                if(i < argc)
                {
		    aspect_offset = atof(argv[i]);
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
	    }
	    /* Recorder. */
	    else if(strcasepfx(cstrptr, "--rec") ||
                    strcasepfx(cstrptr, "-rec")
	    )
	    {
		i++;
		if(i < argc)
		{
		    sound_server_connect_arg = argv[i];
		}
		else
		{
		    fprintf(stderr,
			"%s: Requires argument.\n",
			argv[i - 1]
		    );
		}
	    }
            /* No sound. */
            else if(!strcasecmp(cstrptr, "--no_sound") ||
                    !strcasecmp(cstrptr, "--nosound") ||
                    !strcasecmp(cstrptr, "-no_sound") ||
                    !strcasecmp(cstrptr, "-nosound")
            )
            {
                startup_no_sound = True;
            }
	    /* No menu backgrounds. */
	    else if(!strcasecmp(cstrptr, "--nomenubackgrounds") ||
                    !strcasecmp(cstrptr, "--nomenubackground") ||
                    !strcasecmp(cstrptr, "--nomenubkg") ||
                    !strcasecmp(cstrptr, "--nomenubg") ||
                    !strcasecmp(cstrptr, "-nomenubackgrounds") ||
                    !strcasecmp(cstrptr, "-nomenubackground") ||
                    !strcasecmp(cstrptr, "-nomenubkg") ||
                    !strcasecmp(cstrptr, "-nomenubg")
	    )
	    {
		option.menu_backgrounds = False;
	    }

	}

	/* Check if program has been `installed' globally. */
	if(!SARIsInstalledGlobal(&dname, &fname))
	{
	    /* Not globally installed, critical error. */
	    fprintf(
		stderr,
 "Cannot find global installation of %s in %s.\n",
 		PROG_NAME, dname.global_data
	    );
	    fprintf(
                stderr,
 "Please verify that you have installed this program correctly. If you\n\
have installed the global data in a non-standard location, then you should\n\
set the environment variable `%s' to reffer to the\n\
correct directory.\n",
		SAR_DEF_ENV_GLOBAL_DIR
	    );
	    return(NULL);
	}
	/* Same check as above, cept for local installation. */
	if(!SARIsInstalledLocal(&dname, &fname))
	{
	    /* Not locally installed, so install from global. */
	    if(SARDoInstallLocal(&dname, &fname, &option))
	    {
	        fprintf(stderr,
 "Cannot complete installation of data files for %s to %s.\n",
		    PROG_NAME, dname.local_data
		);
		notify_install_local_error = True;
	    }
	    else
	    {
		notify_install_local_success = True;
	    }
	}

	/* Load preferances and configuration from file. */
	if(SAROptionsLoadFromFile(&option, fname.preferances))
	{
	    fprintf(stderr,
 "%s: Warning: Error occured while loading preferances.\n",
		fname.preferances
	    );
	}


	/* Allocate core structure. */
	core_ptr = (sar_core_struct *)calloc(1, sizeof(sar_core_struct));
	if(core_ptr == NULL)
	    return(NULL);

	/* Record program file name. */
	if(argc > 0)
	{
	    cstrptr = (const char *)argv[0];
	    if(cstrptr != NULL)
	    {
		free(core_ptr->prog_file_full_path);
		core_ptr->prog_file_full_path = strdup(cstrptr);

		cstrptr = strrchr(cstrptr, DIR_DELIMINATOR);
		if(cstrptr != NULL)
		{
		    free(core_ptr->prog_file);
		    core_ptr->prog_file = strdup(cstrptr + 1);
		}
		else
		{
		    free(core_ptr->prog_file);
                    core_ptr->prog_file = strdup(core_ptr->prog_file_full_path);
		}
	    }
	}



        /* Set up arguments for initializing graphics wrapper. */
	strc = 10;
	strv = (char **)calloc(strc, sizeof(char *));
	if(strv == NULL)
	{
	    SARShutdown(core_ptr);
	    return(NULL);
	}
	strv[0] = strdup((direct_rendering) ?
	    "--hardware_rendering" : "--software_rendering"
	);
	strv[1] = strdup("--geometry");
	if((*geometry_string) == '\0')
	    sprintf(geometry_string, "%ix%i%s%i%s%i",
		option.last_width,
		option.last_height,
		((option.last_x < 0) ? "" : "+"),
		option.last_x,
		((option.last_y < 0) ? "" : "+"),
		option.last_y
	    );
	strv[2] = strdup(geometry_string);
	strv[3] = strdup("--title");
	strv[4] = strdup(PROG_NAME);
	strv[5] = strdup("--icon_path");
	strv[6] = strdup(SAR_DEF_SAR_ICON_FILE);
	strv[7] = strdup("--icon_name");
	strv[8] = strdup(PROG_NAME);
	strv[9] = ((fullscreen) ? strdup("--full_screen") : strdup("--windowed"));
	if(aspect_offset != 0.0)
	{
	    char num_str[80];
	    sprintf(num_str, "%f", aspect_offset);
            strc += 2;
            strv = (char **)realloc(strv, strc * sizeof(char *));
            if(strv == NULL)
            {
                SARShutdown(core_ptr);
                return(NULL);
            }
            strv[strc - 2] = strdup("--aspect_offset");
            strv[strc - 1] = strdup(num_str);
	}
	if(option.runtime_debug)
	{
	    strc += 1;
	    strv = (char **)realloc(strv, strc * sizeof(char *));
	    if(strv == NULL)
            {
                SARShutdown(core_ptr);
                return(NULL);
            }
	    strv[strc - 1] = strdup("--gw_debug");
	}
        if(font_name != NULL)
        {
            strc += 2;
            strv = (char **)realloc(strv, strc * sizeof(char *));
            if(strv == NULL)
            {
                SARShutdown(core_ptr);
                return(NULL);
            }
            strv[strc - 2] = strdup("--font");
	    strv[strc - 1] = strdup(font_name);
        }

	/* Initialize graphics erapper. */
        core_ptr->display = GWInit(strc, strv);

	/* Deallocate arguments used in initializing graphics wrapper. */
	StringFreeArray(strv, strc);
	strv = NULL;
	strc = 0;

	/* Failed to initialize graphics wrapper? */
        if(core_ptr->display == NULL)
	{
	    SARShutdown(core_ptr);
            return(NULL);
	}
	dpy = core_ptr->display;


	/* Check if we have OpenGL 1.1 or newer. */
	if((dpy->gl_version_major < 1) ?
	    1 : ((dpy->gl_version_major == 1) ?
		(dpy->gl_version_minor < 1) : 0)
	)
	{
	    fprintf(stderr,
"%s requires OpenGL version 1.1 or newer, current version is %i.%i.\n",
		PROG_NAME, dpy->gl_version_major, dpy->gl_version_minor
	    );
	}

	/* Check if we have alpha bits. */
	if(dpy->alpha_channel_bits <= 0)
	{
	    fprintf(
		stderr,
"%s requires an alpha channel, current alpha channel is 0 bits in size.\n",
                PROG_NAME
            );
	}

        /* Set font data referances. */
        option.hud_font = font_6x10;		/* For HUD text. */
        option.message_font = font_7x14;	/* For standard/misc messages. */
	option.menu_font = font_menu;		/* For menus (except for menu values. */
	option.banner_font = font_banner;	/* For the sticky banner messages. */

        /* Set graphics wrapper options. */
        GWSetDrawCB(dpy, SARDrawCB, core_ptr);
        GWSetKeyboardCB(dpy, SARKeyBoardCB, core_ptr);
        GWSetPointerCB(dpy, SARPointerCB, core_ptr);
        GWSetResizeCB(dpy, SARReshapeCB, core_ptr);
        GWSetVisibilityCB(dpy, SARVisibilityCB, core_ptr);
	GWSetSaveYourselfCB(dpy, SARSaveYourselfCB, core_ptr);
        GWSetCloseCB(dpy, SARCloseCB, core_ptr);
        GWSetTimeoutCB(dpy, SARManage, core_ptr);


	/* Set audio mode name for changing of sound server audio mode. */
	free(core_ptr->audio_mode_name);
	core_ptr->audio_mode_name = strdup("PlayStereo11025");

	/* Initialize sound wrapper. */
	if(startup_no_sound ||
	   (!option.engine_sounds &&
            !option.event_sounds &&
            !option.voice_sounds
           )
	)
	{
	    /* Do not initialize sound, mark all sound resources as
	     * not available.
	     */

	    core_ptr->recorder = NULL;

	    option.engine_sounds = False;
	    option.event_sounds = False;
	    option.voice_sounds = False;
	    option.music = False;
	}
	else
	{
	    core_ptr->recorder = SoundInit(
	        SNDSERV_TYPE_Y,
	        sound_server_connect_arg,	/* Connect argument. */
		NULL				/* Start argument. */
	    );
	    if(core_ptr->recorder == NULL)
	    {
	        fprintf(
		    stderr,
PROG_NAME " could not connect to sound server,\n\
verify that the sound server is running and check\n\
settings in options->sound.\n"
		);

		option.engine_sounds = False;
		option.event_sounds = False;
		option.voice_sounds = False;
		option.music = False;
	    }
	    else
	    {
		/* Change Y audio mode. */
	        SoundChangeMode(
		    core_ptr->recorder, core_ptr->audio_mode_name
		);

		/* Update recorder address. */
		if(sound_server_connect_arg != NULL)
		{
		    free(core_ptr->recorder_address);
		    core_ptr->recorder_address = strdup(sound_server_connect_arg);
		}
	    }
	}


	/* Initialize game controller, options need to be set
	 * up properly prior to calling this.
	 */
	SARInitGCTL(core_ptr);


	/* Load global texture referance names list (these contain
	 * only the referance names and file names to the textures.
	 * They will be loaded when the scene is loaded.
	 */
	SARTextureListLoadFromFile(
	    fname.textures,
	    &core_ptr->texture_list,
	    &core_ptr->total_texture_list
	);

	/* Initialize human data presets resources. */
	core_ptr->human_data = SARHumanPresetsInit(core_ptr);
        if(core_ptr->human_data == NULL)
        {
            fprintf(   
                stderr,
                "Error initializing human presets data.\n"
	    );
        }
        /* Load human preset data from file. */
        SARHumanLoadFromFile(
            core_ptr->human_data,
            fname.human
        );

	/* Initialize weather data presets resources. */
	core_ptr->weather_data = SARWeatherPresetsInit(core_ptr);
	if(core_ptr->weather_data == NULL)
	{
	    fprintf(
		stderr,
		"Error initializing weather presets data.\n"
	    );
	}
	/* Load weather preset data from file. */
	SARWeatherLoadFromFile(
	    core_ptr->weather_data,
	    fname.weather
	);


	/* Reset GL states for menu system. */
	SARMenuGLStateReset(core_ptr->display);

        /* Create all menu system resources and each menu. */
        SARBuildMenus(core_ptr);

        /* Force select the main menu. */
        core_ptr->cur_menu = -1;
        SARMenuButtonCBSelectMenu(core_ptr, SAR_MENU_NAME_MAIN);

        /* No startup mission. */
        core_ptr->mission = NULL;


	/* Print messages as needed. */
	if(notify_install_local_error)
	{
	    char mesg[1024];

	    sprintf(mesg,
"Cannot install local data files to:\n%s",
		dname.local_data
	    );

	    GWOutputMessage(
		dpy, GWOutputMessageTypeError,
		"Installation Error",
		mesg,
"There was a problem installing local data files, make sure that\n\
global data for this program has been installed properly and that\n\
your home directory exists and has has write permission for this program"
	    );
	}
	else if(notify_install_local_success)
	{
	    char mesg[1024];

	    sprintf(mesg,
"Local data files for this program have been\ninstalled in:\n\n%s",
		dname.local_data
	    );

	    GWOutputMessage(
		dpy, GWOutputMessageTypeGeneral,
		"Local Data Files Installed",
		mesg,
"The data files needed by this program have been installed, this\n\
occured because the program needs them and they were not detected\n\
to exist before."
	    );
	}

	return(core_ptr);
}

/*
 *	SAR management, this is called once per loop as the timeout
 *	function.
 *
 *	The graphics wrapper normally calls this function on timeouts.
 */
void SARManage(void *ptr)
{
	int status;
	int cur_menu;
	time_t t_new;
	sar_scene_struct *scene;
	sar_core_struct *core_ptr = (sar_core_struct *)ptr;
	if(core_ptr == NULL)
	    return;

        /* Update global timming. */
        t_new = SARGetCurMilliTime();	/* Get current time in ms. */
	/* Cinderella check. */
        if(t_new < cur_millitime)
        {
            /* Timming cycled, reset timmers. */
	    SARResetTimmersCB(core_ptr, t_new);
        }
        else
        {
	    /* Calculate lapsed ms from last loop. */
            lapsed_millitime = t_new - cur_millitime;

	    /* Calculate time compensation coeff. */
            time_compensation = (double)(
                (double)lapsed_millitime / (double)CYCLE_LAPSE_MS
            );
	    /* Sanitize time compensation coeff. */
	    if(time_compensation > 1000.0)
		time_compensation = 1000.0;
            if(time_compensation < 0.0)
                time_compensation = 0.0;

	    /* Set new current time in ms. */
	    cur_millitime = t_new;
        }

	/* Get current systime seconds. */
	cur_systime = time(NULL);


	/* Get new game controller positions. */
	GCtlUpdate(core_ptr->gctl);


	/* Check if a current menu is allocated (hence selected). */
	cur_menu = core_ptr->cur_menu;
	if(SARIsMenuAllocated(core_ptr, cur_menu))
	{
	    /* A menu is selected, this implies we are presenting the
	     * user with menus.
	     */
/*
	    sar_menu_struct *menu_ptr = core_ptr->menu[cur_menu];
 */

	    /* Since the graphics wrapper calls this function, it
	     * manages the menu system for us and there nothing we
	     * need to do here.
	     */
	}
	else
	{
	    /* No menu selected, this implies we are in game and
	     * as such we need to handle simulation updates and
	     * redraw.
	     */
	    scene = core_ptr->scene;

	    SARSimUpdateScene(core_ptr, scene);
	    SARSimUpdateSceneObjects(core_ptr, scene);

	    /* Redraw all objects. */
	    if(is_visable)
	        GWPostRedraw(core_ptr->display);
	}

	/* Manage sound events. */
	if(core_ptr->recorder != NULL)
	{
	    if(SoundManageEvents(core_ptr->recorder) < 0)
		core_ptr->recorder = NULL;
	}


	/* Manage mission. */
	status = SARMissionManage(core_ptr);
	/* Check mission manage result. */
	switch(status)
	{
	  case 1:
	    /* Mission ended with success, end simulation and tabulate
	     * mission results.
	     */
	    SARSimEnd(core_ptr);
	    break;

	  case 2:
	    /* Mission over and failed. */
	    break;

	  case -1:
	    /* Mission management error. */
	    break;

	  default:
	    /* Nothing eventful. */
	    break;
	}
}

/*
 *	Deallocates all resources in the given core structure and 
 *	deallocates the structure itself.
 */
void SARShutdown(sar_core_struct *core_ptr)
{
	int i, n, p;
	char *strptr;
	sar_menu_struct *menu_ptr;
	void *ptr;
	sar_menu_list_struct *list_ptr;
        sar_menu_map_struct *map_ptr;
	sar_image_struct **img;


	if(core_ptr == NULL)
	    return;

	/* Save preferances and configuration. */
	strptr = fname.preferances;
	if((*strptr) != '\0')
	{
	    if(SAROptionsSaveToFile(&option, strptr))
	    {
		fprintf(
		    stderr,
"%s: Warning: Error occured while saving configuration.\n",
		    strptr
		);
	    }
	}


	/* Text input. */
	free(core_ptr->text_input_prompt);
	core_ptr->text_input_prompt = NULL;

	free(core_ptr->text_input_buffer);
	core_ptr->text_input_buffer = NULL;

	core_ptr->text_input_cb = NULL;


	/* Global texture file referance list. */
	SARTextureListDeleteAll(
	    &core_ptr->texture_list,
	    &core_ptr->total_texture_list
	);

	/* File names. */
	free(core_ptr->cur_mission_file);
	core_ptr->cur_mission_file = NULL;

        free(core_ptr->cur_player_model_file);
        core_ptr->cur_player_model_file = NULL;

        free(core_ptr->cur_scene_file);
        core_ptr->cur_scene_file = NULL;


	/* Menus. */
	for(i = 0; i < core_ptr->total_menus; i++)
	{
	    menu_ptr = core_ptr->menu[i];
	    if(menu_ptr == NULL)
		continue;

	    /* Delete client data on each list item on each list
	     * on each menu.
	     */
	    for(n = 0; n < menu_ptr->total_objects; n++)
	    {
		ptr = menu_ptr->object[n];
		if(ptr == NULL)
		    continue;

		switch(*(int *)ptr)
		{
		  case SAR_MENU_OBJECT_TYPE_LIST:
		    list_ptr = ptr;
		    /* Make sure we delete any client_data structures
		     * in any of menu list object's items.
		     */
		    for(p = 0; p < list_ptr->total_items; p++)
			SARDeleteListItemData(list_ptr->item[p]);
		    break;

		  case SAR_MENU_OBJECT_TYPE_MAP:
		    map_ptr = ptr;

		    /* Delete background image since it's not shared
		     * but menu system thinks it is.
		     */
/*
		    SARImageDestroy(map_ptr->bg_image);
		    map_ptr->bg_image = NULL;
 */
		    break;
		}
	    }

	    SARMenuDestroy(menu_ptr);
	}

	free(core_ptr->menu);
	core_ptr->menu = NULL;
	core_ptr->cur_menu = 0;
	core_ptr->total_menus = 0;


	/* Menu shared data. */
#define DO_DESTROY_IMAGE	\
{ \
 if((*img) != NULL) \
 { \
  SARImageDestroy(*img); \
  (*img) = NULL; \
 } \
}
	if(core_ptr->menu_list_bg_img != NULL)
	{
	    for(i = 0; i < 9; i++)
	    {
		img = &core_ptr->menu_list_bg_img[i];
		DO_DESTROY_IMAGE
	    }
	    free(core_ptr->menu_list_bg_img);
	    core_ptr->menu_list_bg_img = NULL;
	}

	img = &core_ptr->menu_button_armed_img;
	DO_DESTROY_IMAGE
	img = &core_ptr->menu_button_unarmed_img;
	DO_DESTROY_IMAGE
        img = &core_ptr->menu_button_highlighted_img;
	DO_DESTROY_IMAGE
	img = &core_ptr->menu_button_label_img;
	DO_DESTROY_IMAGE

	img = &core_ptr->menu_label_bg_img;
	DO_DESTROY_IMAGE

	img = &core_ptr->menu_switch_bg_img;
	DO_DESTROY_IMAGE
	img = &core_ptr->menu_switch_off_img;
	DO_DESTROY_IMAGE
	img = &core_ptr->menu_switch_on_img;
	DO_DESTROY_IMAGE

        img = &core_ptr->menu_spin_label_img;
	DO_DESTROY_IMAGE
	img = &core_ptr->menu_spin_value_img;
	DO_DESTROY_IMAGE

	img = &core_ptr->menu_progress_bg_img;
	DO_DESTROY_IMAGE
	img = &core_ptr->menu_progress_fg_img;
	DO_DESTROY_IMAGE

	img = &core_ptr->menu_button_pan_up_armed_img;
	DO_DESTROY_IMAGE
	img = &core_ptr->menu_button_pan_up_unarmed_img;
	DO_DESTROY_IMAGE
	img = &core_ptr->menu_button_pan_down_armed_img;
	DO_DESTROY_IMAGE
	img = &core_ptr->menu_button_pan_down_unarmed_img;
	DO_DESTROY_IMAGE
	img = &core_ptr->menu_button_pan_left_armed_img;
	DO_DESTROY_IMAGE
	img = &core_ptr->menu_button_pan_left_unarmed_img;
	DO_DESTROY_IMAGE
	img = &core_ptr->menu_button_pan_right_armed_img;
	DO_DESTROY_IMAGE
	img = &core_ptr->menu_button_pan_right_unarmed_img;
	DO_DESTROY_IMAGE

        img = &core_ptr->menu_button_zoom_in_armed_img;
        DO_DESTROY_IMAGE
        img = &core_ptr->menu_button_zoom_in_unarmed_img;
        DO_DESTROY_IMAGE
        img = &core_ptr->menu_button_zoom_out_armed_img;
        DO_DESTROY_IMAGE
        img = &core_ptr->menu_button_zoom_out_unarmed_img;
        DO_DESTROY_IMAGE

        img = &core_ptr->menumap_helipad_img;
	DO_DESTROY_IMAGE
        img = &core_ptr->menumap_intercept_img;
	DO_DESTROY_IMAGE
        img = &core_ptr->menumap_helicopter_img;
	DO_DESTROY_IMAGE
        img = &core_ptr->menumap_victim_img;
	DO_DESTROY_IMAGE
        img = &core_ptr->menumap_boat_img;
	DO_DESTROY_IMAGE

#undef DO_DESTROY_IMAGE

        /* Delete mission structure (if any). Note that mission should
	 * have been properly ended and deallocated before shutting
	 * down.
	 */
        SARMissionDestroy(core_ptr->mission);
        core_ptr->mission = NULL;

	/* Delete scene. */
        SARSceneDestroy(
	    core_ptr,
	    core_ptr->scene,
	    &core_ptr->object,
	    &core_ptr->total_objects
	);
	free(core_ptr->scene);
	core_ptr->scene = NULL;

	/* Delete draw map object names list. */
	SARDrawMapObjNameListDelete(
	    &core_ptr->drawmap_objname,
	    &core_ptr->total_drawmap_objnames
	);

	/* Human presets data resources. */
	SARHumanPresetsShutdown(core_ptr->human_data);
	core_ptr->human_data = NULL;

        /* Weather presets data resources. */
        SARWeatherPresetsShutdown(core_ptr->weather_data);
        core_ptr->weather_data = NULL;


        /* Controller. */
        GCtlShutdown(core_ptr->gctl);
	free(core_ptr->gctl);
	core_ptr->gctl = NULL;

	/* Music file referances list. */
	SARMusicListDeleteAll(&core_ptr->music_ref, &core_ptr->total_music_refs);

	/* Sound wrapper. */
        SoundShutdown(core_ptr->recorder);
        core_ptr->recorder = NULL;

	free(core_ptr->audio_mode_name);
	core_ptr->audio_mode_name = NULL;

	free(core_ptr->recorder_address);
	core_ptr->recorder_address = NULL;

	/* Graphics wrapper. */
	GWShutdown(core_ptr->display);
	core_ptr->display = NULL;

	/* Program name. */
	free(core_ptr->prog_file_full_path);
	core_ptr->prog_file_full_path = NULL;
	free(core_ptr->prog_file);
	core_ptr->prog_file = NULL;

	/* Deallocate core structure itself. */
	free(core_ptr);
}

#ifdef NEED_WINMAIN
int WINAPI WinMain(
	HINSTANCE hInstance,		// Instance
	HINSTANCE hPrevInstance,	// Previous Instance
	LPSTR lpCmdLine,		// Command Line Parameters
	int nCmdShow			// Window Show State
)
#else
int main(int argc, char **argv)
#endif
{
	sar_core_struct *sar_core;
#ifdef NEED_WINMAIN
	int argc = 0;
	char **argv = NULL;
#endif


	/* Reset globals. */
	debug_value = 0.0;
	segfault_count = 0;


#ifdef NEED_WINMAIN
	/* Explode command line arguments. */
	if(lpCmdLine != NULL)
	    argv = strexp(lpCmdLine, &argc);
#endif

	/* Initialize program. */
        sar_core = SARInit(argc, argv);
	if(sar_core == NULL)
	    return(1);

	/* Maintain program. */
        runlevel = 2;
        while(runlevel >= 2)
        {
#ifdef __MSW__
	    /* No sleeping for Windows. */
#else
            usleep(1000);
#endif
            GWManage(sar_core->display);
        }

	/* Shutdown program. */
	SARShutdown(sar_core);
	sar_core = NULL;

#ifdef NEED_WINMAIN
	/* Free command line arguments. */
	StringFreeArray(argv, argc);
#endif

	return(0);
}
