#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "../include/os.h"
#include "../include/string.h"

#include "obj.h"
#include "messages.h"
#include "mission.h"
#include "simutils.h"
#include "sar.h"
#include "config.h"


#ifdef __MSW__
static double rint(double x);
#endif	/* __MSW__ */
static int SARMissionGetObjectPassengers(
	sar_core_struct *core_ptr,
	sar_object_struct *obj_ptr
);

static char *SARMissionFailedMessage(sar_mission_struct *mission);
static char *SARMissionSuccessMessage(sar_mission_struct *mission);

static Boolean SARMissionProcessGCCNameMatch(
        sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_object_struct **ptr, int total,
        int *gcc_list, int gcc_list_total,
        const char *name
);

void SARMissionPrintStats(
	sar_core_struct *core_ptr,
	sar_scene_struct *scene,
        sar_mission_struct *mission,
        sar_object_struct *obj_ptr
);

static void SARMissionLandNotifyArrive(
	sar_core_struct *core_ptr,
	sar_mission_struct *mission,
        sar_object_struct *obj_ptr,
	int *list, int list_total
);
static void SARMissionLandNotifyRescue(
	sar_core_struct *core_ptr,
	sar_mission_struct *mission,
	sar_object_struct *obj_ptr,
	int *list, int list_total
);

void SARMissionManageArrive(
	sar_core_struct *core_ptr, sar_mission_struct *mission
);
void SARMissionManageRescue(
	sar_core_struct *core_ptr, sar_mission_struct *mission
);

void SARMissionPassengersLeaveNotify(
	sar_core_struct *core_ptr, sar_object_struct *obj_ptr,
	int passengers_left
);

void SARMissionDestroyNotify(
	sar_core_struct *core_ptr, sar_object_struct *obj_ptr
);
void SARMissionLandNotify(
	sar_core_struct *core_ptr, sar_object_struct *obj_ptr,
	int *list, int list_total
);
int SARMissionManage(sar_core_struct *core_ptr);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))


/*
 * 	Interval between mission end check and going back to
 *	menus.
 */
#define DEF_MISSION_END_INT	5000



#ifdef __MSW__
static double rint(double x)
{
	if((double)((double)x - (int)x) > (double)0.5)
	    return((double)((int)x + (int)1));
	else
	    return((double)((int)x));
}
#endif	/* __MSW__ */

/*
 *	Returns the number of passengers on the object, depending on the
 *	object type it may not have any objects. Does not include crew
 *	or any fixed on-board `individuals'.
 *
 *	For any object that has a hoist and rescue basket, any passengers
 *	in the rescue basket are NOT counted.
 *
 *	The object is assumed valid.
 */
static int SARMissionGetObjectPassengers(
        sar_core_struct *core_ptr, 
        sar_object_struct *obj_ptr
)
{
	int *passengers;

	if(obj_ptr == NULL)
	    return(0);

        if(SARObjGetOnBoardPtr(
            obj_ptr,
            NULL, &passengers, NULL,
            NULL,
            NULL, NULL 
        ))
            return(0);

	if(passengers != NULL)
	     return(MAX(*passengers, 0));
	else
	    return(0);
}

/*
 *	For processing ground contact check hit list given by gcc_list,
 *	which is an array of object numbers (any special object numbers
 *	encountered will be ignored).
 *
 *	Returns True on match or False on no match or error.
 */
static Boolean SARMissionProcessGCCNameMatch(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
	sar_object_struct **ptr, int total,
	int *gcc_list, int gcc_list_total,
        const char *name
)
{
	int i, obj_num;
	sar_object_struct *obj_ptr;

	if((name == NULL) || (gcc_list == NULL) || (ptr == NULL))
	    return(False);

        /* Check if landed at right place, itterate through list of   
         * ground contact objects for matching name.
         */
        for(i = 0; i < gcc_list_total; i++)
        {
	    obj_num = gcc_list[i];
            obj_ptr = SARObjGetPtr(ptr, total, obj_num);
            if((obj_ptr == NULL) ? 1 : (obj_ptr->name == NULL))
                continue;

            /* Names match? */
            if(!strcasecmp(obj_ptr->name, name))
                break;
        }
        /* No match? */
        if(i >= gcc_list_total)
            return(False);
	else
	    return(True);
}

/*
 *	Returns pointer to failed message on mission structure or
 *	NULL on error.
 */
static char *SARMissionFailedMessage(sar_mission_struct *mission)
{
	sar_mobj_arrive_struct *arrive;
	sar_mobj_rescue_struct *rescue;

	if(mission == NULL)
	    return(NULL);

	switch(mission->type)
	{
	  case MISSION_TYPE_ARRIVE:
	    arrive = (sar_mobj_arrive_struct *)mission->objective;
	    if(arrive != NULL)
		return(arrive->message_fail);
	    break;

          case MISSION_TYPE_RESCUE:
            rescue = (sar_mobj_rescue_struct *)mission->objective;
            if(rescue != NULL)
                return(rescue->message_fail);
            break;
/* Add support for other mission objective types here. */
	}

	return(NULL);
}

/*
 *      Returns pointer to success message on mission structure or
 *      NULL on error.
 */
static char *SARMissionSuccessMessage(sar_mission_struct *mission)
{
        sar_mobj_arrive_struct *arrive;
        sar_mobj_rescue_struct *rescue;

        if(mission == NULL)
            return(NULL);

        switch(mission->type)
        {
          case MISSION_TYPE_ARRIVE:
            arrive = (sar_mobj_arrive_struct *)mission->objective;
            if(arrive != NULL)
                return(arrive->message_success);
            break;

          case MISSION_TYPE_RESCUE:
            rescue = (sar_mobj_rescue_struct *)mission->objective;
            if(rescue != NULL)
                return(rescue->message_success);
            break;
/* Add support for other mission objective types here. */
        }

        return(NULL);
}


/*
 *	Prints mission stats to the scene's message output with respect
 *	to the given object and mission structures.
 */
void SARMissionPrintStats(
	sar_core_struct *core_ptr, sar_scene_struct *scene,
        sar_mission_struct *mission,
        sar_object_struct *obj_ptr
)
{
	int	passengers = 0, 	/* Current on board. */
		passengers_max = 0;	/* Maximum capacity. */
	sar_object_aircraft_struct *obj_aircraft_ptr;
	char text[1024];


	/* First print object statistics if object is valid. */
	if(obj_ptr != NULL)
	{
	    /* Handle by object type. */
	    switch(obj_ptr->type)
	    {
	      case SAR_OBJ_TYPE_AIRCRAFT:
		obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
		if(obj_aircraft_ptr != NULL)
		{
		    passengers = obj_aircraft_ptr->passengers;
		    passengers_max = obj_aircraft_ptr->passengers_max;
		}
		break;
	    }

	    /* Format passengers on board string. */
	    sprintf(
		text,
		"%s: %i(%i)",
		SAR_MESG_PASSENGERS,
		passengers,
		passengers_max
	    );
	    SARMessageAdd(scene, text);
	}

	/* Mission status. */
	if(mission != NULL)
	{
	    int more_to_find = 0;
	    sar_mobj_arrive_struct *arrive_objective;
	    sar_mobj_rescue_struct *rescue_objective;
#define tmp_name_len	80
	    char tmp_name[tmp_name_len];


            /* Handle by mission type. */
            switch(mission->type)
            {
              case MISSION_TYPE_ARRIVE:
		arrive_objective = mission->objective;
		if(arrive_objective == NULL)
		    break;

		strncpy(
		    tmp_name,
		    ((arrive_objective->arrive_at_name == NULL) ?
			"unknown" : arrive_objective->arrive_at_name
		    ),
		    tmp_name_len
		);
		tmp_name[tmp_name_len - 1] = '\0';

		/* Print arrive mission status. */
		switch(mission->state)
		{
		  case MISSION_STATE_IN_PROGRESS:
		    sprintf(text,
			"%s %s",
			SAR_MESG_MISSION_IN_PROGRESS_ENROUTE, tmp_name
		    );
		    SARMessageAdd(scene, text);

		    /* Is there a time limit? */
		    if(arrive_objective->time_left > 0.0)
		    {
			sprintf(text,
                            "%s %s",
			    SAR_MESG_TIME_LEFT,
                            SARDeltaTimeString(
				core_ptr,
				(time_t)arrive_objective->time_left
			    )
                        );
                        SARMessageAdd(scene, text);
		    }
		    break;

		  case MISSION_STATE_ACCOMPLISHED:
                    strcpy(text, SAR_MESG_MISSION_OBJECTIVE_ACHIEVED);
		    SARMessageAdd(scene, text);
                    break;

                  case MISSION_STATE_FAILED:
                    strcpy(text, SAR_MESG_MISSION_OBJECT_FAILED);
                    SARMessageAdd(scene, text);
                    break;
		}
		break;

	      case MISSION_TYPE_RESCUE:
                rescue_objective = mission->objective;
                if(rescue_objective == NULL)
                    break;

		/* Print rescue mission status. */
                switch(mission->state)
                {
                  case MISSION_STATE_IN_PROGRESS:
		    /* Print how many more to find, remember to
		     * calculate this by; how_need_rescue -
		     * already_rescued - those_onboard
		     */
		    more_to_find = MAX(
                        rescue_objective->humans_need_rescue -
                        rescue_objective->humans_rescued -
                        passengers,
                        0
                    );
		    if(more_to_find > 0)
                        sprintf(
			    text,
                            "%s, %i %s",
			    SAR_MESG_MISSION_RESCUE_IN_PROGRESS,
			    more_to_find,
			    SAR_MESG_MISSION_MORE_TO_FIND
			);
		    else
                        sprintf(
                            text,
                            "%s, %s",
                            SAR_MESG_MISSION_RESCUE_IN_PROGRESS,
			    SAR_MESG_MISSION_ALL_FOUND
                        );
                    SARMessageAdd(scene, text);

		    /* Is there a time limit? */
                    if(rescue_objective->time_left > 0.0)
                    {
			/* Handle by time limit type. */
			switch(rescue_objective->time_left_type)
			{
			  case MISSION_TIME_LEFT_TYPE_RESCUE_ALL:
			    sprintf(text,
				"%s %s %s",
				SAR_MESG_TIME_LEFT,
				SARDeltaTimeString(
				    core_ptr,
				    (time_t)rescue_objective->time_left
				),
				SAR_MESG_MISSION_TO_GET_ALL_TO_SAFETY
			    );
			    break;

			  case MISSION_TIME_LEFT_TYPE_PICKUP_ALL:
			    sprintf(text,
				"%s %s %s",
				SAR_MESG_TIME_LEFT,
                                SARDeltaTimeString(
				    core_ptr,
                                    (time_t)rescue_objective->time_left
                                ),
				SAR_MESG_MISSION_TO_PICK_UP_ALL
			    );
			    break;

			  default:
			    strcpy(
				text,
 "SARMissionPrintStats(): Unsupported time limit type!"
			    );
			    break;
			}
                        SARMessageAdd(scene, text);
                    }
                    break;

                  case MISSION_STATE_ACCOMPLISHED:
                    strcpy(text, SAR_MESG_MISSION_OBJECTIVE_ACHIEVED);
                    SARMessageAdd(scene, text);
                    break;

                  case MISSION_STATE_FAILED:
                    strcpy(text, SAR_MESG_MISSION_OBJECT_FAILED);
                    SARMessageAdd(scene, text);
                    break;
                }
		break;
	    }
#undef tmp_name_len
        }

	return;
}


/*
 *	Handle land notify arrive mission.
 */
static void SARMissionLandNotifyArrive(
	sar_core_struct *core_ptr, sar_mission_struct *mission,
        sar_object_struct *obj_ptr,
	int *list, int list_total
)
{
        void *ptr = mission->objective;
        sar_mobj_arrive_struct *objective_ptr;
	sar_scene_struct *scene = core_ptr->scene;


	if(scene == NULL)
	    return;

        if((*(int *)ptr) != MISSION_TYPE_ARRIVE)
            return;
        else
            objective_ptr = ptr;

        /* Not landed at right place? */
	if(!SARMissionProcessGCCNameMatch(
	    core_ptr, scene,
	    core_ptr->object, core_ptr->total_objects,
	    list, list_total,
	    objective_ptr->arrive_at_name
	))
	    return;

        /* Mission accomplished? Check if mission is in progress and
	 * the given arrive object pointer is the player object.
	 */
        if((mission->state == MISSION_STATE_IN_PROGRESS) &&
           (scene->player_obj_ptr == obj_ptr)
        )
        {
	    const char *mesg_ptr = objective_ptr->message_success;

	    /* Mark mission progress state as accomplished. */
            mission->state = MISSION_STATE_ACCOMPLISHED;

	    /* Set mission accomplished banner. */
	    SARBannerMessageAppend(scene, NULL);
	    SARBannerMessageAppend(scene,
		SAR_MESG_MISSION_ACCOMPLISHED_BANNER
	    );
	    SARBannerMessageAppend(
                scene,
		((mesg_ptr != NULL) ? mesg_ptr :
		    SAR_MESG_MISSION_ARRIVE_COMPLETE
		)
            );
	    /* Do not set continue banner message for mission accomplished. */

	    /* Schedual next mission check in 5 seconds which will
	     * detect acomplished state.
	     */
	    mission->next_check = cur_millitime + DEF_MISSION_END_INT;
        }

	return;
}

/*
 *	Handle land notify rescue mission:
 */
static void SARMissionLandNotifyRescue(
	sar_core_struct *core_ptr,
	sar_mission_struct *mission,
        sar_object_struct *obj_ptr,
	int *list, int list_total
)
{
	int passengers_onboard = 0;
	sar_scene_struct *scene = core_ptr->scene;
	sar_object_aircraft_struct *obj_aircraft_ptr;
	void *ptr = mission->objective;
        sar_mobj_rescue_struct *objective_ptr;


	if(scene == NULL)
            return;

        if((*(int *)ptr) != MISSION_TYPE_RESCUE)
            return;
        else
            objective_ptr = ptr;

        /* Not landed at right place? */
        if(!SARMissionProcessGCCNameMatch(   
            core_ptr, scene,
            core_ptr->object, core_ptr->total_objects,
            list, list_total,
            objective_ptr->arrive_at_name
        ))
            return;

	/* Get number of passengers onboard by object type and unload
	 * passengers from that object.
	 */
	switch(obj_ptr->type)
	{
	  case SAR_OBJ_TYPE_AIRCRAFT:
	    obj_aircraft_ptr = (sar_object_aircraft_struct *)obj_ptr->data;
	    if(obj_aircraft_ptr != NULL)
	    {
		/* Get number of passengers on board. */
		passengers_onboard = obj_aircraft_ptr->passengers;

		/* Unload passengers. */
/* We may need to do a more prettier unload.
 * and also check for *when* passengers are actually unloaded.
 */
		SARSimOpPassengersUnloadAll(scene, obj_ptr);
	    }

	    break;
/* Add support for other object types that have passengers. */

	}

	/* Add humans rescued to mission objective structure. */
	objective_ptr->humans_rescued += passengers_onboard;

	/* Mission accomplished? Check if mission is in progress and
	 * if all victims have been rescued. Do not check if this is
	 * the player object since that does not matter for this type of
	 * mission objective.
	 */
	if((objective_ptr->humans_rescued >=
		objective_ptr->humans_need_rescue) &&
	   (mission->state == MISSION_STATE_IN_PROGRESS)
	)
	{
	    const char *mesg_ptr = objective_ptr->message_success;

	    /* Mark mission progress state as accomplished. */
	    mission->state = MISSION_STATE_ACCOMPLISHED;

            /* Set mission accomplished banner. */
            SARBannerMessageAppend(scene, NULL);
            SARBannerMessageAppend(scene,
                SAR_MESG_MISSION_ACCOMPLISHED_BANNER
            );
            SARBannerMessageAppend(
                scene,
                ((mesg_ptr != NULL) ? mesg_ptr :
                    SAR_MESG_MISSION_RESCUE_COMPLETE
                )
            );   
            /* Do not set continue banner message for mission accomplished. */

            /* Schedual next mission check in 5 seconds which will
             * detect acomplished state.
             */
            mission->next_check = cur_millitime + DEF_MISSION_END_INT;
	}

	return;
}

/*
 *	Manage arrive at mission.
 */
void SARMissionManageArrive(
        sar_core_struct *core_ptr,
        sar_mission_struct *mission
)
{
	void *ptr = mission->objective;
	sar_scene_struct *scene = core_ptr->scene;
        sar_mobj_arrive_struct *objective_ptr;


	if((scene == NULL) || (ptr == NULL))
	    return;

        if((*(int *)ptr) != MISSION_TYPE_ARRIVE)
            return;
        else
            objective_ptr = ptr;

	/* Is there a time limit? */
	if(objective_ptr->time_left > 0.0)
	{
	    /* Reduce time left. */
	    objective_ptr->time_left -= (double)(
		(double)lapsed_millitime * time_compression / 1000
	    );

	    /* Time limit exceeded? */
	    if(objective_ptr->time_left <= 0.0)
	    {
		const char *mesg_ptr = objective_ptr->message_fail;

		/* Mark mission progress state as failed. */
		mission->state = MISSION_STATE_FAILED;

                /* Set mission failed banner. */
                SARBannerMessageAppend(scene, NULL);
                SARBannerMessageAppend(scene,
                    SAR_MESG_MISSION_FAILED_BANNER
                );
                SARBannerMessageAppend(
                    scene,
		    (mesg_ptr != NULL) ? mesg_ptr :
			SAR_MESG_MISSION_ARRIVE_FAILED_TIME
		);
		/* Set continue message. */
		SARBannerMessageAppend(
                    scene,
		    SAR_MESG_MISSION_POST_FAILED_BANNER
		);

                /* Schedual next mission check in 5 seconds which will
                 * detect failed state.
                 */
                mission->next_check = cur_millitime + DEF_MISSION_END_INT;
	    }
	}	/* Is there a time limit? */

	return;
}

/*
 *	Manage rescue mission.
 */
void SARMissionManageRescue(
	sar_core_struct *core_ptr, sar_mission_struct *mission
)
{
        void *ptr = mission->objective;
        sar_mobj_rescue_struct *objective_ptr;
	sar_scene_struct *scene = core_ptr->scene;


        if((scene == NULL) || (ptr == NULL))
            return;

	if((*(int *)ptr) != MISSION_TYPE_RESCUE)
	    return;
	else
	    objective_ptr = ptr;

	/* Is there a time limit? */
        if(objective_ptr->time_left > 0.0)
        {
	    /* Reduce time left. */
            objective_ptr->time_left -= (double)(
		(double)lapsed_millitime * time_compression / 1000
            );

	    /* Time limit exceeded? */
            if(objective_ptr->time_left <= 0.0)
            {
		int passengers;
		const char *mesg_ptr = objective_ptr->message_fail;

		/* Check by time limit type. */
		switch(objective_ptr->time_left_type)
		{
		  case MISSION_TIME_LEFT_TYPE_RESCUE_ALL:
		    /* Need to rescue all and get all of them to safety
		     * within the time limit.
		     */

		    /* Mark mission progress state as failed. */
		    mission->state = MISSION_STATE_FAILED;

                    /* Set mission failed banner. */
                    SARBannerMessageAppend(scene, NULL);
                    SARBannerMessageAppend(scene,
                        SAR_MESG_MISSION_FAILED_BANNER
                    );
                    SARBannerMessageAppend(
                        scene,
			(mesg_ptr != NULL) ? mesg_ptr :
			    SAR_MESG_MISSION_RESCUE_FAILED_TIME
		    );
                    /* Set continue message. */
                    SARBannerMessageAppend(
                        scene,
                        SAR_MESG_MISSION_POST_FAILED_BANNER
                    );

		    /* Schedual next mission check in 5 seconds which will
		     * detect mission failed state.
		     */
		    mission->next_check = cur_millitime + DEF_MISSION_END_INT;
		    break;

		  case MISSION_TIME_LEFT_TYPE_PICKUP_ALL:
		    /* Just need to make sure all humans are picked up. */

		    /* Get number of passengers from scene structure
		     * specified player object.
		     */
		    if(scene == NULL)
			passengers = 0;
		    else
			passengers = SARMissionGetObjectPassengers(
			    core_ptr,
			    scene->player_obj_ptr
			);
		    /* Have not picked up and/or rescued all victims? */
		    if((passengers + objective_ptr->humans_rescued) <
			objective_ptr->humans_need_rescue
		    )
		    {
                        /* Did not pick up and/or rescue all victims in
			 * time limit.
			 */

			/* Mark mission progress state as failed. */
			mission->state = MISSION_STATE_FAILED;

                        /* Set mission failed banner. */
                        SARBannerMessageAppend(scene, NULL);
                        SARBannerMessageAppend(scene,
                            SAR_MESG_MISSION_FAILED_BANNER
                        );
                        SARBannerMessageAppend(
                            scene,
                            (mesg_ptr != NULL) ? mesg_ptr :
                                SAR_MESG_MISSION_RESCUE_FAILED_TIME
                        );
                        /* Set continue message. */
                        SARBannerMessageAppend(
                            scene,
                            SAR_MESG_MISSION_POST_FAILED_BANNER
                        );

			/* Schedual next mission check in 5 seconds which will
			 * detect mission failed state.
			 */
			mission->next_check = cur_millitime + DEF_MISSION_END_INT;
		    }
		    break;

		}	/* Check by time limit type. */

	    }	/* Time limit exceeded? */
        }	/* Is there a time limit? */

	return;
}


/*
 *	Passengers leave notify, this function should be called
 *	whenever the given object safely lets off passengers.
 *
 *	The value passengers_left indicates how many passengers
 *	left the given object (since the object's current passenger
 *	count is not checked).
 */
void SARMissionPassengersLeaveNotify(
	sar_core_struct *core_ptr, sar_object_struct *obj_ptr,
	int passengers_left
)
{
        sar_mission_struct *mission;
            
            
        if((core_ptr == NULL) ||
           (obj_ptr == NULL)
        )
            return;

        mission = core_ptr->mission;
        if(mission == NULL)
            return;

/* This is not implmented yet, when it does however it will check if
 * the number of passengers let off meets the mission requirements
 * (if this is a rescue mission) and in which case then mark the
 * mission as successfully ended.
 */


	return;
}


/*
 *	Destroy notify. This function should be called whenever an
 *	object is *about* to be destroyed.
 */
void SARMissionDestroyNotify(
        sar_core_struct *core_ptr, sar_object_struct *obj_ptr
)
{
        sar_mission_struct *mission;
	sar_scene_struct *scene;


        if((core_ptr == NULL) || (obj_ptr == NULL))
            return;

        mission = core_ptr->mission;
        if(mission == NULL)
            return;

	scene = core_ptr->scene;
	if(scene == NULL)
	    return;

	/* Is this the player object that has been destroyed? */
	if(scene->player_obj_ptr == obj_ptr)
	{
	    const char *mesg_ptr;

	    /* Mark mission progress state as failed. */
            mission->state = MISSION_STATE_FAILED;

	    /* Set mission failed banner. */
	    SARBannerMessageAppend(scene, NULL);
	    SARBannerMessageAppend(scene,
		SAR_MESG_MISSION_FAILED_BANNER
	    );

	    /* Get mission failed message. */
	    mesg_ptr = SARMissionFailedMessage(mission);
	    if(mesg_ptr != NULL)
		SARBannerMessageAppend(scene, mesg_ptr);

	    /* Set continue message. */
	    SARBannerMessageAppend(
		scene,
		SAR_MESG_MISSION_POST_FAILED_BANNER
	    );

            /* Schedual next mission check in 5 seconds which will
             * detect failed state.
             */
            mission->next_check = cur_millitime + DEF_MISSION_END_INT;
	}
}

/*
 *	Land notify, this function should be called whenever an object
 *	has landed.
 *
 *      The given list should be a hit list of object index numbers of
 *      the surrounding area where the given object landed, it will not
 *      be deallocated by this function.
 */
void SARMissionLandNotify(
	sar_core_struct *core_ptr, sar_object_struct *obj_ptr,
	int *list, int list_total
)
{
	sar_mission_struct *mission;


	if((core_ptr == NULL) ||
           (obj_ptr == NULL)
	)
	    return;

	mission = core_ptr->mission;
	if(mission == NULL)
	    return;

	/* Handle by mission type. */
	switch(mission->type)
	{
	  case MISSION_TYPE_ARRIVE:
	    SARMissionLandNotifyArrive(
		core_ptr, mission, obj_ptr, list, list_total
	    );
	    break;

          case MISSION_TYPE_RESCUE:
            SARMissionLandNotifyRescue(
		core_ptr, mission, obj_ptr, list, list_total
	    );
            break;
	}

	return;
}

/*
 *	Manages mission, this function should be called once per loop.
 *
 *	Return codes are as follows:
 *
 *	-1	Error.
 *	0	Nothing happened or not time to manage.
 *	1	Mission end success.
 *	2	Mission end failure.
 */
int SARMissionManage(sar_core_struct *core_ptr)
{
        sar_mission_struct *mission;
        
           
        if(core_ptr == NULL)
	    return(-1);

	mission = core_ptr->mission;
	if(mission == NULL)
	    return(-1);

	/* Update time spent on mission. */
	mission->time_spent += (double)(
	    (double)lapsed_millitime * time_compression / 1000
	);

	/* Not time to manage mission yet? */
	if(mission->next_check > cur_millitime)
	{
	    /* Skip and return success. */
	    return(0);
	}
	else
	{
	    /* Yes its time to check mission status. */

	    /* Schedual next time to check mission.
	     *
	     * Note that mission->next_check may be modified again by
	     * one of the mission manage handler functions farther
	     * below.
	     */
	    mission->next_check = cur_millitime + mission->check_int;
	}

	/* Mission no longer in progress? */
	if(mission->state != MISSION_STATE_IN_PROGRESS)
	{
	    /* Check the resulting state of the mission. */
	    switch(mission->state)
	    {
	      case MISSION_STATE_ACCOMPLISHED:
		return(1);
		break;

	      case MISSION_STATE_FAILED:
		return(2);
		break;

	      default:
		/* Unsupported mission state, return error. */
		return(-1);
		break;
	    }
	}

	/* Mission still in progress, handle by mission type. */
	switch(mission->type)
	{
	  case MISSION_TYPE_ARRIVE:
	    SARMissionManageArrive(core_ptr, mission);
	    break;

	  case MISSION_TYPE_RESCUE:
	    SARMissionManageRescue(core_ptr, mission);
	    break;

/* Add support for managing other types of mission objectives here. */

	}

	return(0);
}
