#include <sys/types.h>

#include "../include/os.h"
#ifdef __MSW__
# include <windows.h>
#endif
#include <GL/gl.h>
#include <GL/glu.h>

#include "gw.h"
#include "stategl.h"
#include "obj.h"
#include "sar.h"

#include "sardrawdefs.h"


extern void SARVertexBoxNS(double x_len, double y_len, double z_len);
extern void SARVertexBoxBaseNS(double x_len, double y_len, double z_len);


void SARDrawHumanIterate(
        gw_display_struct *display, sar_scene_struct *scene,
        double height, double mass,
        sar_obj_flags_t flags,          /* Human flags. */
        sar_color_struct *palette,      /* Human colors. */
        int water_ripple_tex_num,
        sar_grad_anim_t anim_pos
);
void SARDrawHuman(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_human_struct *obj_human_ptr
);


/*
 *	Called by SARDrawHuman() to draw one human.
 *
 *	All inputs assumed valid.
 */
void SARDrawHumanIterate(
	gw_display_struct *display, sar_scene_struct *scene,
	double height, double mass,
	sar_obj_flags_t flags,		/* Human flags. */
	sar_color_struct *palette,	/* Human colors. */
	int water_ripple_tex_num,
	sar_grad_anim_t anim_pos
)
{
	sar_color_struct *c;
	GLenum shade_model_mode;
	StateGLBoolean texture_2d_state;
	double anim_coeff = (double)anim_pos /
	    (double)((sar_grad_anim_t)-1);

	double torso_angle;
	double left_bisep_angle, right_bisep_angle;
	double left_shoulder_angle, right_shoulder_angle;
	double left_trisep_angle, right_trisep_angle;
	double left_hip_to_thigh_angle, right_hip_to_thigh_angle;
	double left_knee_to_calv_angle, right_knee_to_calv_angle;

	double thigh_len = 0.3;
	double calv_len = 0.4;
	double torso_len = 0.8;
	double bisep_len = 0.3;
	double trisep_len = 0.4;
	double base_to_torso = 0.0;	/* Ground to bottom of torso
					 * (calculated later).
					 */
	double streatcher_height = 0.8;	/* Ground to bed of streatcher. */

#define SET_COLOR	\
{ \
 glColor4d(c->r, c->g, c->b, c->a); \
}

	/* Record previous GL_TEXTURE_2D state and disable
	 * GL_TEXTURE_2D.
	 */
	texture_2d_state = display->state_gl.texture_2d;
	StateGLDisable(&display->state_gl, GL_TEXTURE_2D);
	V3DTextureSelect(NULL);

        /* Set shade model dependant of global options. */
        shade_model_mode = display->state_gl.shade_model_mode;
        StateGLShadeModel(&display->state_gl, option.gl_shade_model);


	/* Calculate angles of body parts. Note that all zero radians
	 * indicates human is standing upright with hands to side.
	 */
	if(flags & SAR_HUMAN_FLAG_LYING)
	{
            torso_angle = 1.5 * PI;
	    left_shoulder_angle = 0.0 * PI;
	    right_shoulder_angle = 0.0 * PI;
            left_bisep_angle = 0.0 * PI;
	    right_bisep_angle = 0.0 * PI;
            left_trisep_angle = 0.0 * PI;
            right_trisep_angle = 0.0 * PI;
            left_hip_to_thigh_angle = 0.0 * PI;
            right_hip_to_thigh_angle = 0.0 * PI;
            left_knee_to_calv_angle = 0.0 * PI;
            right_knee_to_calv_angle = 0.0 * PI;

            base_to_torso = thigh_len + calv_len;

	    if(flags & SAR_HUMAN_FLAG_ON_STREATCHER)
	    {
		/* Some translating will be done later. */
	    }
	}
        else if(flags & SAR_HUMAN_FLAG_SIT)
        {
	    /* Sitting base at tush. */
            torso_angle = 0.0 * PI;
            left_shoulder_angle = 0.0 * PI;
            right_shoulder_angle = 0.0 * PI;
            left_bisep_angle = 0.0 * PI;
            right_bisep_angle = 0.0 * PI;
            left_trisep_angle = 1.75 * PI;
            right_trisep_angle = 1.75 * PI;
            left_hip_to_thigh_angle = 1.5 * PI;
            right_hip_to_thigh_angle = 1.5 * PI;
            left_knee_to_calv_angle = 0.5 * PI;
            right_knee_to_calv_angle = 0.5 * PI;

            base_to_torso = 0;
        }
        else if(flags & SAR_HUMAN_FLAG_SIT_DOWN)
        {
            torso_angle = 0.0 * PI;
            left_shoulder_angle = 0.0 * PI;
            right_shoulder_angle = 0.0 * PI;
            left_bisep_angle = 0.0 * PI;
            right_bisep_angle = 0.0 * PI;
            left_trisep_angle = 1.75 * PI;
            right_trisep_angle = 1.75 * PI;
            left_hip_to_thigh_angle = 1.5 * PI;
            right_hip_to_thigh_angle = 1.5 * PI;
            left_knee_to_calv_angle = 0.0 * PI;
            right_knee_to_calv_angle = 0.0 * PI;

            base_to_torso = 0;
	}
        else if(flags & SAR_HUMAN_FLAG_SIT_UP)
        {
	    /* Sitting base at feet. */
            torso_angle = 0.0 * PI;
            left_shoulder_angle = 0.0 * PI;
            right_shoulder_angle = 0.0 * PI;
            left_bisep_angle = 0.0 * PI;
            right_bisep_angle = 0.0 * PI;
            left_trisep_angle = 1.75 * PI;
            right_trisep_angle = 1.75 * PI;
            left_hip_to_thigh_angle = 1.5 * PI;
            right_hip_to_thigh_angle = 1.5 * PI; 
            left_knee_to_calv_angle = 0.5 * PI;
            right_knee_to_calv_angle = 0.5 * PI;
        
            base_to_torso = calv_len;
	}
	else if(flags & SAR_HUMAN_FLAG_DIVER_CATCHER)
	{
	    /* Diver catching victim, used for hoist rescue end drawing. */
	    torso_angle = 0.0 * PI;
	    left_shoulder_angle = 0.1 * PI;
	    right_shoulder_angle = 1.9 * PI;
	    left_bisep_angle = 1.8 * PI;
	    right_bisep_angle = 1.8 * PI;
	    left_trisep_angle = 1.8 * PI;   
	    right_trisep_angle = 1.8 * PI;
	    left_hip_to_thigh_angle = 1.9 * PI;
	    right_hip_to_thigh_angle = 1.9 * PI;
	    left_knee_to_calv_angle = 0.2 * PI; 
	    right_knee_to_calv_angle = 0.2 * PI;

	    /* If gripped it implies on end of rescue hoist rope. */
	    if(flags & SAR_HUMAN_FLAG_GRIPPED)
		base_to_torso = torso_len;
	    else if(flags & SAR_HUMAN_FLAG_IN_WATER)
		base_to_torso = torso_len * -0.92;
	    else
		base_to_torso = thigh_len + calv_len;
	}
	/* Run should be checked last since lying and setting have
	 * precidence.
	 */
        else if(flags & SAR_HUMAN_FLAG_RUN)
        {
            if(anim_coeff > 0.5)
                anim_coeff = (1.0 - anim_coeff) / 0.5;
            else
                anim_coeff = anim_coeff / 0.5;

            torso_angle = 0.1 * PI;
            left_shoulder_angle = 0.0 * PI;
            right_shoulder_angle = 0.0 * PI;
            if(flags & SAR_HUMAN_FLAG_PUSHING)
            {
                left_bisep_angle = 1.7 * PI;
                right_bisep_angle = 1.7 * PI;
                left_trisep_angle = 1.6 * PI;
                right_trisep_angle = 1.6 * PI;
            }
            else
            {
                left_bisep_angle = (1.7 + (0.35 * (1.0 - anim_coeff))) * PI;
                right_bisep_angle = (1.7 + (0.35 * anim_coeff)) * PI;
                left_trisep_angle = (1.6 + (0.15 * (1.0 - anim_coeff))) * PI;
                right_trisep_angle = (1.6 + (0.15 * anim_coeff)) * PI;
            }
            left_hip_to_thigh_angle = (0.2 - (0.5 * anim_coeff)) * PI;
            right_hip_to_thigh_angle = (0.2 - (0.5 * (1.0 - anim_coeff))) * PI;
            left_knee_to_calv_angle = 0.2 * PI;
            right_knee_to_calv_angle = 0.2 * PI;

            base_to_torso = thigh_len + calv_len;
	}
	/* All else assume just standing. */
	else
	{
	    if(flags & SAR_HUMAN_FLAG_IN_WATER)
	    {
		/* Standing in water, arms fore and moving. */
		if(anim_coeff > 0.5)
		    anim_coeff = (1.0 - anim_coeff) / 0.5;
		else
		    anim_coeff = anim_coeff / 0.5;

                torso_angle = 0.0 * PI;
                left_shoulder_angle = (0.4 * anim_coeff) * PI;
                right_shoulder_angle = (2 - (0.4 * anim_coeff)) * PI;
                left_bisep_angle = 1.5 * PI;
                right_bisep_angle = 1.5 * PI;
                left_trisep_angle = 0.0 * PI;
	        right_trisep_angle = 0.0 * PI;
                left_hip_to_thigh_angle = 0.0 * PI;
                right_hip_to_thigh_angle = 0.0 * PI;
                left_knee_to_calv_angle = 0.0 * PI;
                right_knee_to_calv_angle = 0.0 * PI;

		/* Since in water, move torso down. */
		base_to_torso = torso_len * -0.92;
	    }
	    else
	    {
		/* Standing on ground. */
		if((flags & SAR_HUMAN_FLAG_NEED_RESCUE) &&
		   (flags & SAR_HUMAN_FLAG_AWARE)
		)
		{
                 /* Need rescue and aware that rescuer is
		  * near by, move arms fore and wave.
		  */
                 if(anim_coeff > 0.5)
                    anim_coeff = (1.0 - anim_coeff) / 0.5;
                 else
                    anim_coeff = anim_coeff / 0.5;

                 torso_angle = 0.0 * PI;
                 left_shoulder_angle = (0.7 + (0.4 * anim_coeff)) * PI; 
                 right_shoulder_angle = (1.3 - (0.4 * anim_coeff)) * PI;
                 left_bisep_angle = 0.0 * PI; 
                 right_bisep_angle = 0.0 * PI; 
                 left_trisep_angle = 0.0 * PI; 
                 right_trisep_angle = 0.0 * PI;
                 left_hip_to_thigh_angle = 0.0 * PI; 
                 right_hip_to_thigh_angle = 0.0 * PI;  
                 left_knee_to_calv_angle = 0.0 * PI; 
                 right_knee_to_calv_angle = 0.0 * PI;

                 base_to_torso = thigh_len + calv_len;
		}
		else
		{
		 /* Just standing on ground. */
                 torso_angle = 0.0 * PI;
                 left_shoulder_angle = 0.0 * PI;
                 right_shoulder_angle = 0.0 * PI;
                 left_bisep_angle = 0.0 * PI;
                 right_bisep_angle = 0.0 * PI;
                 left_trisep_angle = 0.0 * PI;
                 right_trisep_angle = 0.0 * PI;
                 left_hip_to_thigh_angle = 0.0 * PI;
                 right_hip_to_thigh_angle = 0.0 * PI;
                 left_knee_to_calv_angle = 0.0 * PI;
                 right_knee_to_calv_angle = 0.0 * PI;

		 base_to_torso = thigh_len + calv_len;
		}
	    }
	}

	/* Human appendage length and rotation values have now been
	 * set up.
	 */

	/* Draw streatcher? */
	if(flags & SAR_HUMAN_FLAG_ON_STREATCHER)
	{
	    double x = 0.35, y = 1.05;
	    double h = streatcher_height - 0.1, hl = 0.1;

	    glColor4d(0.8, 0.8, 0.8, 1.0);

	    /* Bed of streatcher. */
	    glBegin(GL_QUADS);
	    {
		glNormal3d(0.0, 1.0, 0.0);
		glVertex3d(x, h, -y);
                glVertex3d(-x, h, -y);
                glVertex3d(-x, h, y);
                glVertex3d(x, h, y);

                glNormal3d(0.0, -1.0, 0.0);
                glVertex3d(-x, h, -y);
                glVertex3d(x, h, -y);
                glVertex3d(x, h, y);
                glVertex3d(-x, h, y);
	    }
	    glEnd();

            glColor4d(0.7, 0.7, 0.7, 1.0);

	    /* Left grids. */
            glBegin(GL_LINE_LOOP);
	    {
                glNormal3d(-1.0, 0.0, 0.0);
                glVertex3d(-x, hl, -y);
                glVertex3d(-x, h, y);
                glVertex3d(-x, hl, y);
                glVertex3d(-x, h, -y);
            }
            glEnd();

            /* Right grids. */
            glBegin(GL_LINE_LOOP);  
            {
                glNormal3d(1.0, 0.0, 0.0);
                glVertex3d(x, hl, -y);
                glVertex3d(x, h, y);
                glVertex3d(x, hl, y);
                glVertex3d(x, h, -y);
            }
            glEnd();


            glColor4d(0.1, 0.1, 0.1, 1.0);

	    /* Simple square wheels. */
            glBegin(GL_QUADS);
            {
                glNormal3d(-1.0, 0.0, 0.0);
                glVertex3d(-x, 0, -y);
                glVertex3d(-x, 0, -(y - 0.1));
                glVertex3d(-x, hl, -(y - 0.1));
                glVertex3d(-x, hl, -y);
            }
            glEnd();

            glBegin(GL_QUADS);        
            {
                glNormal3d(-1.0, 0.0, 0.0);
                glVertex3d(-x, hl, y);
                glVertex3d(-x, hl, (y - 0.1));
                glVertex3d(-x, 0, (y - 0.1)); 
                glVertex3d(-x, 0, y);
            }
            glEnd();

            glBegin(GL_QUADS);
            {
                glNormal3d(1.0, 0.0, 0.0);
                glVertex3d(x, 0, -y);
                glVertex3d(x, hl, -y);
                glVertex3d(x, hl, -(y - 0.1));
                glVertex3d(x, 0, -(y - 0.1));
            }
            glEnd();

            glBegin(GL_QUADS);
            {
                glNormal3d(1.0, 0.0, 0.0);
                glVertex3d(x, 0, (y - 0.1));
                glVertex3d(x, hl, (y - 0.1));
                glVertex3d(x, hl, y);
                glVertex3d(x, 0, y);
            }
            glEnd();
	}

        /* Begin drawing human. */

	/* Torso. */
        glPushMatrix();
        {
            /* Check if on streatcher, if so translate up. */
            if(flags & SAR_HUMAN_FLAG_ON_STREATCHER)  
            {
		/* Push on extra matrix and move up (matrix poped later). */
		glPushMatrix();
                glTranslated(0, streatcher_height, -(base_to_torso + 0.2));
            }

	    /* Move to base of torso (the tush). */
            glRotated(-SFMRadiansToDegrees(torso_angle), 1.0, 0.0, 0.0);
            glTranslated(0, base_to_torso, 0);

	    /* Move to center of torso and draw torso. */
            glPushMatrix();
            {
		/* Hips of torso. */
		c = &palette[SAR_HUMAN_COLOR_HIPS];
		SET_COLOR
                glBegin(GL_QUADS);
                {
	            SARVertexBoxBaseNS(
			0.45 * torso_len,
			0.25 * torso_len,
			0.10 * torso_len
		    );
                }
                glEnd();


		glTranslated(0, 0.10 * torso_len, 0);

                /* Upper torso. */
                c = &palette[SAR_HUMAN_COLOR_TORSO];
                SET_COLOR
                glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(
                        0.45 * torso_len,
                        0.25 * torso_len,
                        0.90 * torso_len
                    );
                }
                glEnd();
	    }
	    glPopMatrix();
	    /* Back to base of torso. */

	    /* Draw left leg. */
            glPushMatrix();
            {
		/* Thigh. */
		glRotated(-SFMRadiansToDegrees(left_hip_to_thigh_angle), 1.0, 0.0, 0.0);
                glTranslated(-0.16 * torso_len, 0.0, 0.0);

                c = &palette[SAR_HUMAN_COLOR_LEGS];
                SET_COLOR

                glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(
                        0.5 * thigh_len,
                        0.5 * thigh_len,
                        -thigh_len
                    );
                }
                glEnd();

		/* Calv. */
                glTranslated(0, -thigh_len, 0);
		glRotated(-SFMRadiansToDegrees(left_knee_to_calv_angle), 1.0, 0.0, 0.0);

		/* Use same color as for thighs. */

                glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(
                        0.3 * calv_len,
                        0.3 * calv_len,
                        -calv_len
                    );
                }
                glEnd();

		/* Feet. */
                glTranslated(0.0, -calv_len, -0.05);

                c = &palette[SAR_HUMAN_COLOR_FEET];
                SET_COLOR

                glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(
                        0.1,
                        0.2,
                        -0.1
                    );
                }
                glEnd();
	    }
            glPopMatrix();
            /* Back to base of torso. */

            /* Draw right leg. */
            glPushMatrix();
            {
                /* Thigh. */
		glRotated(-SFMRadiansToDegrees(right_hip_to_thigh_angle), 1.0, 0.0, 0.0);
                glTranslated(0.16 * torso_len, 0.0, 0.0);

                c = &palette[SAR_HUMAN_COLOR_LEGS];
                SET_COLOR

                glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(
                        0.5 * thigh_len,
                        0.5 * thigh_len,
                        -thigh_len
                    );  
                }
                glEnd();   
                
                /* Calv. */
                glTranslated(0.0, -thigh_len, 0.0);
		glRotated(-SFMRadiansToDegrees(right_knee_to_calv_angle), 1.0, 0.0, 0.0);

		/* Use same color as for thighs. */

                glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(
                        0.3 * calv_len,
                        0.3 * calv_len,
                        -calv_len
                    );
                }
                glEnd();

                /* Feet. */
                glTranslated(0.0, -calv_len, -0.05);

                c = &palette[SAR_HUMAN_COLOR_FEET];
                SET_COLOR

                glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(
                        0.1,
                        0.2,
                        -0.1
                    );
                }
                glEnd();
            }
            glPopMatrix();
            /* Back to base of torso. */

	    /* Move to shoulders. */
	    glTranslated(0, 0.95 * torso_len, 0);

            /* Draw left arm. */
            glPushMatrix();
            {
                /* Bisep and shoulder. */
		glTranslated(
		    (-0.225 * torso_len) + (-0.25 * bisep_len),
		    0,
		    0
		);
                glRotated(-SFMRadiansToDegrees(left_bisep_angle), 1.0, 0.0, 0.0);
		glRotated(-SFMRadiansToDegrees(left_shoulder_angle), 0.0, 0.0, 1.0);

                c = &palette[SAR_HUMAN_COLOR_ARMS];
                SET_COLOR

                glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(
                        0.5 * bisep_len,
                        0.5 * bisep_len,
                        -bisep_len
                    );
                }
                glEnd();

                /* Trisep. */
                glTranslated(0.0, -bisep_len, 0.0);
		glRotated(-SFMRadiansToDegrees(left_trisep_angle), 1.0, 0.0, 0.0);

                glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(
                        0.28 * trisep_len,
                        0.28 * trisep_len,
                        -trisep_len
                    );
                }
                glEnd();


                /* Hand. */
                glTranslated(0.0, -trisep_len, 0.0);

                c = &palette[SAR_HUMAN_COLOR_HANDS];
                SET_COLOR

                glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(0.2, 0.1, -0.2);
                }
                glEnd();
	    }
            glPopMatrix();
            /* Back to shoulders. */

	    /* Draw right arm. */
            glPushMatrix();
            {
                /* Bisep. */
                glTranslated(
		    (0.225 * torso_len) + (0.25 * bisep_len),
		    0.0,
		    0.0
		);
		glRotated(-SFMRadiansToDegrees(right_bisep_angle), 1.0, 0.0, 0.0);
		glRotated(-SFMRadiansToDegrees(right_shoulder_angle), 0.0, 0.0, 1.0);

                c = &palette[SAR_HUMAN_COLOR_ARMS];
                SET_COLOR

                glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(
                        0.5 * bisep_len,
                        0.5 * bisep_len,
                        -bisep_len
                    );
                }
                glEnd();

                /* Trisep. */
                glTranslated(0.0, -bisep_len, 0.0);
		glRotated(-SFMRadiansToDegrees(right_trisep_angle), 1.0, 0.0, 0.0);

                glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(
                        0.28 * trisep_len,
                        0.28 * trisep_len,
                        -trisep_len
                    );
                }
                glEnd();


		/* Hand. */
                glTranslated(0.0, -trisep_len, 0.0);

                c = &palette[SAR_HUMAN_COLOR_HANDS];
                SET_COLOR

		glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(0.2, 0.1, -0.2);
                }
                glEnd();
            }
            glPopMatrix();
            /* Back to shoulders. */


            /* Head. */
            glTranslated(0.0, 0.05 * torso_len, 0.0);

            c = &palette[SAR_HUMAN_COLOR_FACE];
            SET_COLOR

            glBegin(GL_QUADS);
            {
                SARVertexBoxBaseNS(0.18, 0.18, 0.25);
            }
            glEnd();


	    /* Hair. */
	    glPushMatrix();
	    {
                glTranslated(0, 0.25, 0);

                c = &palette[SAR_HUMAN_COLOR_HAIR];
                SET_COLOR

                glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(0.18, 0.18, 0.1);
                }
                glEnd();
	    }
	    glPopMatrix();
	    /* Back of head hair. */
            glPushMatrix();
            {
                glTranslated(0.0, 0.05, -(-0.09 - 0.03));
                glBegin(GL_QUADS);
                {
                    SARVertexBoxBaseNS(0.18, 0.06, 0.25);
                }
                glEnd();
            }
            glPopMatrix();


            /* Check if on streatcher, if so we need to pop one matrix
	     * level that was added for translation up on to streatcher.
	     */
            if(flags & SAR_HUMAN_FLAG_ON_STREATCHER)
            {
                glPopMatrix();
            }
        }
        glPopMatrix();

	/* Re-enable GL_TEXTURE_2D as needed. */
	if(texture_2d_state)
	    StateGLEnable(&display->state_gl, GL_TEXTURE_2D);

	/* Draw water ripples if in water. */
	if(flags & SAR_HUMAN_FLAG_IN_WATER)
	{
	    GLboolean depth_mask_flag;
	    int tex_num, frame_num;
	    v3d_texture_ref_struct *t;
	    double hr = 5.0;	/* Half ripple radius. */


	    tex_num = water_ripple_tex_num;
            if(SARIsTextureAllocated(scene, tex_num) &&
	       option.textured_objects
	    )
	    {
		t = scene->texture_ref[tex_num];

		/* Select frame by animation, re-update anim_coeff. */
		anim_coeff = (double)anim_pos /
                    (double)((sar_grad_anim_t)-1);

		frame_num = (int)(anim_coeff * t->total_frames);
		if(frame_num >= t->total_frames)
		    frame_num = t->total_frames - 1;
		if(frame_num < 0)
		    frame_num = 0;
		V3DTextureSelectFrame(t, frame_num);

		glColor4d(1.0, 1.0, 1.0, 0.5);

                StateGLEnable(&display->state_gl, GL_BLEND);
                StateGLDisable(&display->state_gl, GL_ALPHA_TEST);
                depth_mask_flag = display->state_gl.depth_mask_flag;
                StateGLDepthMask(&display->state_gl, GL_FALSE);
		StateGLEnable(&display->state_gl, GL_POLYGON_OFFSET_FILL);
		StateGLBlendFunc(
		    &display->state_gl, GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA
		);
		StateGLPolygonOffset(
		    &display->state_gl,
		    (GLfloat)option.gl_polygon_offset_factor, -1.0
		);

		glBegin(GL_QUADS);
		{
		    glNormal3d(0, 1, 0);
                    glTexCoord2d(0, 1 - 0);
                    glVertex3d(-hr, 0, hr);
                    glTexCoord2d(1, 1 - 0);
                    glVertex3d(hr, 0, hr);
                    glTexCoord2d(1, 1 - 1);
                    glVertex3d(hr, 0, -hr);
                    glTexCoord2d(0, 1 - 1);
                    glVertex3d(-hr, 0, -hr);
		}
		glEnd();

                StateGLDisable(&display->state_gl, GL_POLYGON_OFFSET_FILL);
                StateGLDisable(&display->state_gl, GL_BLEND);
		StateGLDepthMask(&display->state_gl, depth_mask_flag);
                StateGLEnable(&display->state_gl, GL_ALPHA_TEST);
	    }
	}

        /* Restore to previous shade model. */
        StateGLShadeModel(&display->state_gl, shade_model_mode);

#undef SET_COLOR

	return;
}


/*
 *      Draws a human specified by the object obj_ptr.
 *
 *      The human substructure obj_human_ptr should be the one on
 *      the object.
 *
 *      All inputs are assumed valid.
 */
void SARDrawHuman(
        gw_display_struct *display, sar_scene_struct *scene,
        sar_object_struct *obj_ptr,
        sar_object_human_struct *obj_human_ptr
)
{
	/* Draw main human model for this call first, passing the
	 * given human object structure's values directly.
	 */
	SARDrawHumanIterate(
	    display, scene,
	    obj_human_ptr->height, obj_human_ptr->mass,
	    obj_human_ptr->flags, obj_human_ptr->color,
	    obj_human_ptr->water_ripple_tex_num,
	    obj_human_ptr->anim_pos
	);

	/* Draw assisting humans if any and as appropriate.
	 * Do not draw assisting humans if this human is being gripped
	 * (ie in a rescue basket).
	 */
	if((obj_human_ptr->assisting_humans > 0) &&
	   !(obj_human_ptr->flags & SAR_HUMAN_FLAG_GRIPPED)
	)
	{
	    int i;
	    sar_obj_flags_t flags;
	    double s = 0.8;	/* Left/right spacing. */

	    /* Draw all assisting human objects, left and right. */
	    for(i = 0; i < obj_human_ptr->assisting_humans; i++)
	    {
		glPushMatrix();
		{
		    /* Translate to the side and a bit behind. */
		    glTranslated(s, 0.0, 0.2);

		    /* Set up `filtered' human flags that will be
		     * passed to the human draw itteration function.
		     * This is so that we don't pass all the flags, 
		     * because in some situations the human may be
		     * lying on a streatcher and the assisting humans
		     * need to be running.
		     */
		    flags = 0;
		    if(obj_human_ptr->flags & SAR_HUMAN_FLAG_ALERT)
		        flags |= SAR_HUMAN_FLAG_ALERT;
                    if(obj_human_ptr->flags & SAR_HUMAN_FLAG_AWARE)
                        flags |= SAR_HUMAN_FLAG_AWARE;
                    if(obj_human_ptr->flags & SAR_HUMAN_FLAG_IN_WATER)
                        flags |= SAR_HUMAN_FLAG_IN_WATER;
                    if(obj_human_ptr->flags & SAR_HUMAN_FLAG_RUN)
                        flags |= SAR_HUMAN_FLAG_RUN;
                    if(obj_human_ptr->flags & SAR_HUMAN_FLAG_PUSHING)
                        flags |= SAR_HUMAN_FLAG_PUSHING;

		    /* Draw our human object as an assisting human, using
		     * the filtered human flags and the 
		     * assisting_human_color color palette.
		     */
		    SARDrawHumanIterate(
			display, scene,
			obj_human_ptr->height, obj_human_ptr->mass,
			flags, obj_human_ptr->assisting_human_color,
			obj_human_ptr->water_ripple_tex_num,
			obj_human_ptr->anim_pos
		    );
		}
		glPopMatrix();

		/* Shift spacing to other side. */
		s *= -1.0;
	    }
	}

	return;
}
