/*  selectwm - X11 window manager selector
 *
 *  Copyright (C) 1999  Luc Dufresne - ldufresne@altern.org
 *                      26, rue des Comices
 *                      59650 Villeneuve d'Ascq
 *                      FRANCE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  See the file COPYING
 */

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>

#include "config.h"

#ifdef ENABLE_NLS
#include <libintl.h>
#define _(String) gettext(String)
#ifdef gettext_noop
#define N_(String) gettext_noop(String)
#else
#define N_(String) (String)
#endif
#else /* NLS = 0 */
#define _(String) (String)
#define N_(String) (String)
#endif

gboolean SaveOnExit;
GString *ConfigFile;

typedef struct _listdata {
	gchar *cmd;
	gboolean defaultwm;
}listdata;

typedef struct _config {
	gfloat delay;
	gboolean restart_selectwm;
	gint default_wm;
	gint goback_btn;
	gint timer;
	GtkWidget *window;
	GtkWidget *list;
}config;

typedef struct _timer_data {
	GtkWidget *timeout;
	GtkWidget *list;
	gint timer;
	gfloat delay;
	config selectwm_config;
}timer_data;

GString *getline(FILE *fichier)
{
	GString *temp;
	gchar buf[128];
	
	temp=g_string_new("");
	do
	{
		fgets(buf,sizeof(buf),fichier);
		g_string_append(temp,buf);
		if(feof(fichier))return temp;
	}while(strstr(buf,"\n")==NULL);
	return temp;
}

gint read_config_file(config *selectwm_config)
{
	FILE *config;
	GString *buf,*buf2,*buf3;
	gchar *c,*text[1];
	gint row;
	gint n=0;
	listdata *ldata;
	
	config=fopen(ConfigFile->str,"rt");
	if(!config)return n;
	buf=g_string_new("");
	while(!feof(config))
	{
		buf=getline(config);
                if(feof(config))
		{
			fclose(config);
			g_string_free(buf,TRUE);
			return n;
		}
		if(buf->str[0]!='#')
		{
			buf2=g_string_new("");
			c=buf->str;
			while(*c!=':' && *c!='\n')
			{
				g_string_append_c(buf2,*c);
				c++;
			}
			c++;
			if(0==strcmp(buf2->str,"delay"))selectwm_config->delay=atof(c);
			if(0==strcmp(buf2->str,"restart_selectwm"))selectwm_config->restart_selectwm=atoi(c);
			if(0==strcmp(buf2->str,"default_wm"))selectwm_config->default_wm=atoi(c);
			if(0==strcmp(buf2->str,"goback_btn"))selectwm_config->goback_btn=atoi(c);
			
			g_string_free(buf2,TRUE);
			buf2=g_string_new("");
			c=buf->str;
			while(*c!='=' && *c!='\n')
			{
				g_string_append_c(buf2,*c);
				c++;
			}
			if(*c!='\n')
			{
				c++;
				buf3=g_string_new("");
				while(*c!='\n')
				{
					g_string_append_c(buf3,*c);
					c++;
				}
				text[0]=buf2->str;
				row=gtk_clist_append(GTK_CLIST(selectwm_config->list),text);
				ldata = (listdata *)g_malloc (sizeof(listdata));
				ldata->cmd = buf3->str;
				ldata->defaultwm = FALSE;
				gtk_clist_set_row_data(GTK_CLIST(selectwm_config->list),row,ldata);
				n++;
			}
			g_string_free(buf2,TRUE);
		}
		g_string_free(buf,TRUE);
	}
	fclose(config);
	return n;
}

void set_defaultwm(GtkWidget *list,gint i)
{
	listdata *ldata;
	GdkColor color = { 0, 0, 0, 0xFFFF };
	
	ldata=(listdata *)gtk_clist_get_row_data(GTK_CLIST(list),i);
	ldata->defaultwm=TRUE;
	gtk_clist_set_foreground(GTK_CLIST(list),i,&color);
	gtk_clist_set_row_data(GTK_CLIST(list),i,ldata);
}

gint read_config(config *selectwm_config)
{
	gint n;
	selectwm_config->delay=10.0;
	selectwm_config->restart_selectwm=FALSE;
	selectwm_config->default_wm=0;
	selectwm_config->goback_btn=2;
	
	n=read_config_file(selectwm_config);

	switch(selectwm_config->goback_btn)
	{
		case 1:
			selectwm_config->restart_selectwm=TRUE;
			break;
		case 2:
			selectwm_config->restart_selectwm=FALSE;
			break;
		case 3:
			break;
	}
	if(n!=0)
	{	
		gtk_clist_select_row(GTK_CLIST(selectwm_config->list),selectwm_config->default_wm,0);
		set_defaultwm(selectwm_config->list,selectwm_config->default_wm);
	}
	return n;
}

void write_config(config *selectwm_config)
{
	FILE *fichier;
	gint i;
	gchar *desc;
	listdata *ldata;
	
	if(SaveOnExit)
	{
		if(NULL!=(fichier=fopen(ConfigFile->str,"wt")))
		{
			fprintf(fichier,"# Automatically generated file.\n# Do not edit (or make a backup copy before doing so)\n");
		
			for(i=0;i<GTK_CLIST(selectwm_config->list)->rows;i++)
			{
				ldata=(listdata *)gtk_clist_get_row_data(GTK_CLIST(selectwm_config->list),i);
				gtk_clist_get_text(GTK_CLIST(selectwm_config->list),i,0,&desc);
				fprintf(fichier,"%s=%s\n",desc,ldata->cmd);
				if(ldata->defaultwm)selectwm_config->default_wm=i;
			}
		
			fprintf(fichier,"delay:%f\n",selectwm_config->delay);
			fprintf(fichier,"restart_selectwm:%d\n",selectwm_config->restart_selectwm);
			if(selectwm_config->default_wm >= i)selectwm_config->default_wm=0;
			fprintf(fichier,"default_wm:%d\n",selectwm_config->default_wm);
			fprintf(fichier,"goback_btn:%d\n",selectwm_config->goback_btn);

			fclose(fichier);
		}
	}
}

typedef struct _ch_options
{
	GtkWidget *window;
	GtkWidget *spin;
	config *selectwm_config;
	
}ch_options;

void change_options(GtkWidget *w, gpointer data)
{
	ch_options *choptions=(ch_options *)data;
	
	choptions->selectwm_config->delay=gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(choptions->spin));
	gtk_widget_destroy(choptions->window);
}

void gob1_callback (GtkWidget *widget, gpointer data)
{
	config *selectwm_config=(config *)data;
	selectwm_config->goback_btn=1;
}

void gob2_callback (GtkWidget *widget, gpointer data)
{
	config *selectwm_config=(config *)data;
	selectwm_config->goback_btn=2;
}

void gob3_callback (GtkWidget *widget, gpointer data)
{
	config *selectwm_config=(config *)data;
	selectwm_config->goback_btn=3;
}

void edit_config(GtkWidget *w, gpointer data)
{
	GtkWidget *window, *hbox,*vbox,*vbox2,*win_vbox,*spin,*label,*radio1,*radio2,*radio3,*frame,*btn,*bbox,*notebook;
	GtkObject *adj;
	config *selectwm_config = (config *)data;
	ch_options choptions;
	
	if(selectwm_config->delay!=0.0)gtk_timeout_remove(selectwm_config->timer);
	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_modal (GTK_WINDOW(window),TRUE);
	gtk_signal_connect (GTK_OBJECT (window), "destroy",
		  GTK_SIGNAL_FUNC(gtk_main_quit),
		  &window);
	gtk_window_set_title (GTK_WINDOW (window), _("Config"));
	gtk_container_set_border_width (GTK_CONTAINER (window), 5);
	gtk_window_set_position(GTK_WINDOW(window),GTK_WIN_POS_MOUSE);
	gtk_window_set_policy (GTK_WINDOW (window), FALSE, FALSE, TRUE);

	win_vbox = gtk_vbox_new (FALSE ,0);
	gtk_container_add (GTK_CONTAINER (window), win_vbox);
	
	notebook=gtk_notebook_new();

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (win_vbox), notebook);


	/*  page de configuration  */

	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
	
	adj = gtk_adjustment_new(selectwm_config->delay,0.0,3600.0,0.1,1.0,0.0);
	spin = gtk_spin_button_new(GTK_ADJUSTMENT(adj),1.0,1);	
	gtk_box_pack_start (GTK_BOX (hbox), spin, FALSE, FALSE, 0);
	
	label = gtk_label_new(_("Timeout in seconds (0 = infinite)"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	frame=gtk_frame_new(_("Go back... button"));
	gtk_container_add (GTK_CONTAINER (vbox), frame);
	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (frame), vbox2);
	
	label = gtk_label_new(_("State of the \"Go back to selectwm when\nthe WM exits\" button when selectwm starts"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);

	
	radio1 = gtk_radio_button_new_with_label(NULL,_("Checked"));
	gtk_signal_connect (GTK_OBJECT (radio1), "clicked",GTK_SIGNAL_FUNC (gob1_callback),selectwm_config);
	radio2 = gtk_radio_button_new_with_label(gtk_radio_button_group (GTK_RADIO_BUTTON (radio1)),_("Not checked"));
	gtk_signal_connect (GTK_OBJECT (radio2), "clicked",GTK_SIGNAL_FUNC (gob2_callback),selectwm_config);
	radio3 = gtk_radio_button_new_with_label(gtk_radio_button_group (GTK_RADIO_BUTTON (radio2)),_("Remember last state"));
	gtk_signal_connect (GTK_OBJECT (radio3), "clicked",GTK_SIGNAL_FUNC (gob3_callback),selectwm_config);

	switch (selectwm_config->goback_btn)
	{
		case 1:
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio1), TRUE);
			break;
		case 2:
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio2), TRUE);
			break;
		case 3:
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio3), TRUE);
			break;
	}
	
	gtk_box_pack_start (GTK_BOX (vbox2), radio1, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), radio2, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), radio3, TRUE, TRUE, 0);
	
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),vbox,gtk_label_new(_("Config")));
	
	
	/*   page 'about'   */


	label = gtk_label_new(""PACKAGE" "VERSION"\nhttp://www.altern.org/ldufresne/selectwm/\n\nCopyright (c) 1999-2000\nLuc Dufresne <ldufresne@altern.org>");
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),label,gtk_label_new(_("About")));

		
	bbox = gtk_hbutton_box_new ();
	gtk_box_pack_start (GTK_BOX (win_vbox), bbox, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (bbox), 10);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), 
		     GTK_BUTTONBOX_SPREAD );
	btn = gtk_button_new_with_label(_("OK"));
	choptions.window=window;
	choptions.spin=spin;
	choptions.selectwm_config=selectwm_config;
	gtk_signal_connect(GTK_OBJECT (btn), "clicked", change_options, &choptions);
	GTK_WIDGET_SET_FLAGS (btn, GTK_CAN_DEFAULT);
	gtk_container_add (GTK_CONTAINER (bbox), btn);
	gtk_widget_grab_default (btn);
	btn = gtk_button_new_with_label(_("Cancel"));
	gtk_signal_connect_object (GTK_OBJECT (btn), "clicked",
		gtk_widget_destroy, GTK_OBJECT(window));
	gtk_container_add (GTK_CONTAINER (bbox), btn);

	GTK_WIDGET_SET_FLAGS (btn, GTK_CAN_DEFAULT);

	

	gtk_widget_show_all (win_vbox);
	gtk_widget_show(window);
	gtk_main();
}

void exec_wm(gpointer a,gpointer b)
{
	gint row;
	listdata *ldata;
	config *selectwm_config = (config *)b;
	
	row = GPOINTER_TO_INT (a);
	ldata=(listdata *)gtk_clist_get_row_data(GTK_CLIST(selectwm_config->list),row);
	if(!selectwm_config->restart_selectwm)
	{
		write_config(selectwm_config);
		execlp(ldata->cmd,ldata->cmd,NULL);
		g_print(_("Error: Cannot start \"%s\"\n"),ldata->cmd);
	}
	else
	{
		write_config(selectwm_config);
		gtk_widget_hide(selectwm_config->window);
		while (gtk_events_pending()) gtk_main_iteration();
		system(ldata->cmd);
		gtk_widget_show(selectwm_config->window);
	}
}

typedef struct _add_data
{
	GtkWidget *window;
	GtkWidget *command;
	GtkWidget *desc;
	GtkWidget *list;
	gint row; /* pour la modification, on sauvegarde la ligne a modifier */
}add_data;

void add_item2(GtkWidget *w, gpointer data)
{
	gchar *text1[1],*buf;
	gint row;
	listdata *ldata;
	
	add_data *add_d = (add_data *)data;
	text1[0]=gtk_entry_get_text(GTK_ENTRY(add_d->desc));
	buf=g_strdup(gtk_entry_get_text(GTK_ENTRY(add_d->command)));
	if(0==strcmp(text1[0],"") || 0==strcmp(buf,""))return;
	row=gtk_clist_append(GTK_CLIST(add_d->list),text1);
	ldata = (listdata *)malloc(sizeof(listdata));
	ldata->cmd=buf;
	ldata->defaultwm=FALSE;
	gtk_clist_set_row_data(GTK_CLIST(add_d->list),row,ldata);
	gtk_widget_destroy(add_d->window);
}

typedef struct _browse_data {
	GtkWidget *window;
	GtkWidget *entry;
}browse_data;

void browse_ok(GtkWidget *w, gpointer data)
{
	browse_data *bdata=(browse_data *)data;
	gtk_entry_set_text(GTK_ENTRY(bdata->entry),gtk_file_selection_get_filename(GTK_FILE_SELECTION(bdata->window)));
	gtk_widget_destroy(bdata->window);
}

void browse(GtkWidget *w, gpointer data)
{
	GtkWidget *filew;
	browse_data bdata;
        
        filew = gtk_file_selection_new (_("Select a window manager"));
        gtk_window_set_modal (GTK_WINDOW(filew),TRUE);
	gtk_signal_connect (GTK_OBJECT (filew), "destroy", GTK_SIGNAL_FUNC(gtk_main_quit), filew);
	gtk_window_set_position(GTK_WINDOW(filew),GTK_WIN_POS_MOUSE);

	bdata.window=filew;
	bdata.entry=(GtkWidget *)data;
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filew)->ok_button), "clicked", (GtkSignalFunc) browse_ok, &bdata);
        
        gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filew)->cancel_button), "clicked", (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT (filew));
        
        gtk_widget_show(filew);
	gtk_main();
}


void add_edit_gui(GtkWidget **window,GtkWidget **okbtn,GtkWidget **entry1,GtkWidget **entry2,GtkWidget **vbox)
{
	GtkWidget *hbox,*label,*bbox,*btn;
	
	*window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_modal (GTK_WINDOW(*window),TRUE);
	gtk_signal_connect (GTK_OBJECT (*window), "destroy",
		  GTK_SIGNAL_FUNC(gtk_main_quit),
		  window);
	gtk_window_set_title (GTK_WINDOW (*window), _("Config"));
	gtk_container_set_border_width (GTK_CONTAINER (*window), 5);
	gtk_window_set_position(GTK_WINDOW(*window),GTK_WIN_POS_MOUSE);
	gtk_window_set_policy (GTK_WINDOW (*window), FALSE, FALSE, TRUE);

	gtk_widget_show(*window);

	*vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (*window), *vbox);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (*vbox), hbox, FALSE, FALSE, 0);

	*entry2 = gtk_entry_new();
	gtk_box_pack_start (GTK_BOX (hbox), *entry2, FALSE, FALSE, 0);
	label = gtk_label_new(_(" Description"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (*vbox), hbox, FALSE, FALSE, 0);
	
	*entry1 = gtk_entry_new();
	gtk_box_pack_start (GTK_BOX (hbox), *entry1, FALSE, FALSE, 0);
	label = gtk_label_new(_(" Command"));
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	
	bbox = gtk_hbutton_box_new ();
	
	gtk_box_pack_start (GTK_BOX (*vbox), bbox, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (bbox), 10);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), 
		     GTK_BUTTONBOX_SPREAD );
	*okbtn = gtk_button_new_with_label(_("OK"));
	GTK_WIDGET_SET_FLAGS (*okbtn, GTK_CAN_DEFAULT);
	gtk_container_add (GTK_CONTAINER (bbox), *okbtn);
	gtk_widget_grab_default (*okbtn);
	btn = gtk_button_new_with_label(_("Cancel"));
	gtk_signal_connect_object (GTK_OBJECT (btn), "clicked",
		gtk_widget_destroy, GTK_OBJECT(*window));
	gtk_container_add (GTK_CONTAINER (bbox), btn);
	GTK_WIDGET_SET_FLAGS (btn, GTK_CAN_DEFAULT);
	
	btn = gtk_button_new_with_label(_("Browse"));
	gtk_signal_connect (GTK_OBJECT (btn), "clicked", browse, GTK_OBJECT(*entry1));
	gtk_container_add (GTK_CONTAINER (bbox), btn);
	GTK_WIDGET_SET_FLAGS (btn, GTK_CAN_DEFAULT);
}

void add_item(GtkWidget *w, gpointer data)
{
	GtkWidget *window,*entry1,*entry2,*okbtn,*vbox;
	add_data add_d;
	GtkWidget *list = w;
	
	add_edit_gui(&window,&okbtn,&entry1,&entry2,&vbox);

	add_d.window=window;
	add_d.list=list;
	add_d.desc=entry2;
	add_d.command=entry1;
	gtk_signal_connect(GTK_OBJECT (okbtn), "clicked",add_item2, &add_d);

	gtk_widget_show_all (vbox);
	gtk_main();
}

void edit_item2(GtkWidget *w, gpointer data)
{
	gchar *text,*buf;
	listdata *ldata;
	
	add_data *add_d = (add_data *)data;
	text=gtk_entry_get_text(GTK_ENTRY(add_d->desc));
	buf=g_strdup(gtk_entry_get_text(GTK_ENTRY(add_d->command)));
	if(0==strcmp(text,"") || 0==strcmp(buf,""))return;

	gtk_clist_set_text(GTK_CLIST(add_d->list),add_d->row,0,text);

	ldata=(listdata *)gtk_clist_get_row_data(GTK_CLIST(add_d->list),add_d->row);
	g_free(ldata->cmd);
	ldata->cmd=buf;
	gtk_clist_set_row_data(GTK_CLIST(add_d->list),add_d->row,ldata);
	gtk_widget_destroy(add_d->window);
}

void edit_item(GtkWidget *w, gpointer data)
{
	GtkWidget *window,*entry1,*entry2,*okbtn,*vbox;
	add_data add_d;
	gint row;
	GtkWidget *list=w;
	gchar *desc;
	listdata *ldata;
	
	add_edit_gui(&window,&okbtn,&entry1,&entry2,&vbox);
	
	row = GPOINTER_TO_INT (GTK_CLIST(list)->selection->data);
	
	add_d.window=window;
	add_d.list=list;
	add_d.desc=entry2;
	add_d.command=entry1;
	add_d.row=row;
	gtk_signal_connect(GTK_OBJECT (okbtn), "clicked",edit_item2, &add_d);

	ldata=(listdata *)gtk_clist_get_row_data(GTK_CLIST(list),row);
	gtk_clist_get_text(GTK_CLIST(list),row,0,&desc);

	gtk_entry_set_text(GTK_ENTRY(entry2),desc);
	gtk_entry_set_text(GTK_ENTRY(entry1),ldata->cmd);

	gtk_widget_show_all (vbox);
	gtk_main();

}
	
void delete_item(GtkWidget *w, gpointer data)
{
	gint row;
	GtkWidget *list=w;
	
	gtk_clist_freeze (GTK_CLIST(list));

	row = GPOINTER_TO_INT (GTK_CLIST(list)->selection->data);
	gtk_clist_remove (GTK_CLIST(list), row);
	gtk_clist_thaw (GTK_CLIST(list));
}

void messagebox(gchar *msg) {
	GtkWidget *dialog_window;
	GtkWidget *button;
	GtkWidget *label;
	GtkWidget *hbox;
	
	dialog_window = gtk_dialog_new ();
	gtk_signal_connect (GTK_OBJECT (dialog_window), "destroy",
			  GTK_SIGNAL_FUNC(gtk_widget_destroy),
			  &dialog_window);

	gtk_window_set_title (GTK_WINDOW (dialog_window), _("Message"));
	gtk_container_set_border_width (GTK_CONTAINER (dialog_window), 0);
	gtk_window_set_policy (GTK_WINDOW (dialog_window), FALSE, FALSE, TRUE);
	gtk_window_set_position(GTK_WINDOW(dialog_window),GTK_WIN_POS_MOUSE);
	
	label = gtk_label_new (msg);
	hbox = gtk_hbox_new(FALSE,0);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog_window)->vbox),
                             hbox, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 10);
	gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
	gtk_widget_show (label);

	button = gtk_button_new_with_label (_("close"));
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog_window)->action_area), 
			  button, TRUE, TRUE, 0);
	
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default (button);
	gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
		GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog_window));

	gtk_widget_show (hbox);
	gtk_widget_show (button);
	gtk_widget_show (dialog_window);
}

void remove_defaultwm(GtkWidget *list)
{
	listdata *ldata;
	gint i;
	GdkColor color = { 0, 0, 0, 0};
	
	for(i=0;i<GTK_CLIST(list)->rows;i++)
	{
		ldata=(listdata *)gtk_clist_get_row_data(GTK_CLIST(list),i);
		if(ldata->defaultwm)ldata->defaultwm=FALSE;
		gtk_clist_set_foreground(GTK_CLIST(list),i,&color);
		gtk_clist_set_row_data(GTK_CLIST(list),i,ldata);
	}

}

void default_item(GtkWidget *w, gpointer data)
{
	gint row;
	GString *buf;
	config *selectwm_config=(config *)data;
	GtkWidget *list=selectwm_config->list;
	gchar *desc;

	row = GPOINTER_TO_INT (GTK_CLIST(list)->selection->data);
	remove_defaultwm(selectwm_config->list);
	set_defaultwm(selectwm_config->list,row);

	buf = g_string_new("");
	gtk_clist_get_text(GTK_CLIST(selectwm_config->list),row,0,&desc);
	g_string_sprintf(buf,_("%s is now the default window manager"),desc);
	messagebox(buf->str);
	g_string_free(buf,TRUE);
}


void list_click(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	GtkWidget *menu,*menu_item;
	config *selectwm_config = (config *)data;
	
	gtk_timeout_remove(selectwm_config->timer);
	if(event->button==1 && event->type==GDK_2BUTTON_PRESS)
		g_list_foreach(GTK_CLIST(widget)->selection,exec_wm,selectwm_config);
	
	if(event->button==3 && event->type==GDK_BUTTON_PRESS)
	{
		menu=gtk_menu_new();

		menu_item=gtk_menu_item_new_with_label(_("add item..."));
		gtk_menu_append(GTK_MENU(menu),menu_item);
		gtk_signal_connect_object(GTK_OBJECT(menu_item),"activate",GTK_SIGNAL_FUNC(add_item),(gpointer)widget);
		gtk_widget_show(menu_item);

		menu_item=gtk_menu_item_new_with_label(_("edit item..."));
		gtk_menu_append(GTK_MENU(menu),menu_item);
		gtk_signal_connect_object(GTK_OBJECT(menu_item),"activate",GTK_SIGNAL_FUNC(edit_item),(gpointer)widget);
		gtk_widget_show(menu_item);

		menu_item=gtk_menu_item_new_with_label(_("set as default"));
		gtk_menu_append(GTK_MENU(menu),menu_item);
		gtk_signal_connect(GTK_OBJECT(menu_item),"activate",GTK_SIGNAL_FUNC(default_item),selectwm_config);
		gtk_widget_show(menu_item);

		menu_item=gtk_menu_item_new_with_label(_("delete item"));
		gtk_menu_append(GTK_MENU(menu),menu_item);
		gtk_signal_connect_object(GTK_OBJECT(menu_item),"activate",GTK_SIGNAL_FUNC(delete_item),(gpointer)widget);
		gtk_widget_show(menu_item);

		
		gtk_menu_popup (GTK_MENU(menu),NULL,NULL,NULL,NULL,event->button,event->time);
	}
}

void list_enter(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	config *selectwm_config = (config *)data;
	
	gtk_timeout_remove(selectwm_config->timer);
	switch (event->keyval)
		{
		case GDK_space:
		case GDK_Return:
			g_list_foreach(GTK_CLIST(widget)->selection,exec_wm,selectwm_config);
			break;
		default:
		}
}

gint timerfnc(gpointer data)
{
	GtkWidget *entry;
	GString *str;
	timer_data *timerdata = (timer_data *)data;
	
	entry=timerdata->timeout;
	timerdata->delay-=0.1;
	if(timerdata->delay<=0.0)
	{
		g_list_foreach(GTK_CLIST(timerdata->list)->selection,exec_wm,&(timerdata->selectwm_config));
		gtk_timeout_remove(timerdata->timer);
	}
	str=g_string_new("");
	g_string_sprintf(str,"%.1f",timerdata->delay);
	gtk_entry_set_text(GTK_ENTRY(entry),str->str);
	g_string_free(str,TRUE);
	return(TRUE);
}

void goback_callback (GtkWidget *widget, gpointer data)
{
	config *selectwm_config=(config *)data;
	gtk_timeout_remove(selectwm_config->timer);
	if (GTK_TOGGLE_BUTTON (widget)->active)
		selectwm_config->restart_selectwm=TRUE;
	else
		selectwm_config->restart_selectwm=FALSE;
}

void quit_selectwm(GtkWidget *w, gpointer data)
{
	config *selectwm_config=(config *)data;
	
	write_config(selectwm_config);
	gtk_main_quit();
}

void version()
{
	g_print(""PACKAGE" "VERSION" http://www.altern.org/ldufresne/selectwm/\n");
	g_print("Copyright (c) 1999-2000 Luc Dufresne <ldufresne@altern.org>\n");
}

void usage()
{
	version();
	g_print("\n");
	g_print(_("usage: selectwm [OPTIONS]\n"));
	g_print(_("  -h, --help\t\t\tprints this message\n"));
	g_print(_("  -v, --version\t\t\tprints version information\n"));
	g_print(_("  -d, --DontSaveOnExit\t\tdon't save the config on exit\n"));
	g_print(_("  -c, --config <filename>\tselect the config file\n"));
}

gint main(gint argc, gchar *argv[])
{
	GtkWidget *window,*scroll, *vbox, *hbox, *timeout, *confbtn, *quitbtn, *restart,*list;
	gint timer,c,opind;
	gfloat delay;
	config selectwm_config;
	timer_data timerdata;
	static struct option long_options[] = {
		{"help", 0, 0, 'h'},
		{"version", 0, 0, 'v'},
		{"DontSaveOnExit", 0, 0, 'd'},
		{"config", 1, 0, 'c'},
		{0, 0, 0, 0}
	};
	
	gtk_init (&argc, &argv);
	#ifdef ENABLE_NLS
	bindtextdomain(PACKAGE,LOCALEDIR);
	textdomain(PACKAGE); /*utile pour choisir le langage*/
	#endif
	
	SaveOnExit=TRUE;
	ConfigFile=g_string_new("");
	g_string_sprintf(ConfigFile,"%s/.selectwmrc",getenv("HOME"));

	while(1)
	{
		opind = 0;
		c = getopt_long (argc, argv, "hvdc:", long_options, &opind);
		if (c == -1)
		break;
		switch(c) 
			{
				case 0:
					break;
				case 'h':
					usage();
					exit(0);
					break;
				case 'v':
					version();
					exit(0);
					break;			
				case 'd':
					g_print(_("don't save the config on exit\n"));
					SaveOnExit=FALSE;
					break;
				case 'c':
					g_print(_("using %s\n"),optarg);
					g_string_sprintf(ConfigFile,"%s",optarg);
					break;
				default:
					usage();
					exit(1);
					break;
			}
	}
	
	gtk_hbutton_box_set_spacing_default(0);
	
	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (GTK_WINDOW (window), ""PACKAGE" "VERSION"");
	gtk_signal_connect (GTK_OBJECT (window), "destroy",GTK_SIGNAL_FUNC(quit_selectwm),&selectwm_config);

	gtk_container_set_border_width (GTK_CONTAINER (window), 2);
	gtk_window_set_position(GTK_WINDOW(window),GTK_WIN_POS_CENTER);
	
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (window), vbox);

	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

	gtk_box_pack_start (GTK_BOX (vbox), scroll, TRUE, TRUE, 0);
	gtk_widget_show (scroll);
	gtk_widget_set_usize (scroll, 150, 150);
	
	list = gtk_clist_new(1);
	gtk_clist_set_selection_mode(GTK_CLIST(list),GTK_SELECTION_BROWSE);
	gtk_clist_set_shadow_type(GTK_CLIST(list),GTK_SHADOW_NONE);
	gtk_clist_set_reorderable (GTK_CLIST(list), TRUE);
	
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroll), list);
	
	selectwm_config.window=window;
	selectwm_config.list=list;
	if(read_config(&selectwm_config)==0)selectwm_config.delay=0;
	
	gtk_signal_connect(GTK_OBJECT(list),"button_press_event",GTK_SIGNAL_FUNC(list_click),&selectwm_config);
	gtk_signal_connect(GTK_OBJECT(list),"key_press_event",GTK_SIGNAL_FUNC(list_enter),&selectwm_config);

	timeout = gtk_entry_new();
	gtk_entry_set_editable(GTK_ENTRY(timeout),FALSE);
	gtk_widget_set_sensitive(timeout,FALSE);
	gtk_box_pack_start (GTK_BOX (vbox), timeout, FALSE, FALSE, 0);

	restart=gtk_check_button_new_with_label(_("Go back to selectwm\nwhen the WM exits"));
	gtk_box_pack_start (GTK_BOX (vbox), restart, FALSE, FALSE, 0);

	if (selectwm_config.restart_selectwm) gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (restart), TRUE);
	gtk_signal_connect (GTK_OBJECT (restart), "toggled", goback_callback, &selectwm_config);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	quitbtn=gtk_button_new_with_label(_("Quit"));
	gtk_signal_connect (GTK_OBJECT (quitbtn), "clicked",GTK_SIGNAL_FUNC (quit_selectwm), &selectwm_config);
	gtk_box_pack_start (GTK_BOX (hbox), quitbtn, TRUE, TRUE, 2);
	confbtn=gtk_button_new_with_label(_("Config"));
	gtk_signal_connect (GTK_OBJECT (confbtn), "clicked",GTK_SIGNAL_FUNC (edit_config), &selectwm_config);
	gtk_box_pack_start (GTK_BOX (hbox), confbtn, TRUE, TRUE, 2);

	gtk_widget_show_all (vbox);
	gtk_widget_show(window);

	delay = selectwm_config.delay;
	if(delay>0.05)
	{
		timer=gtk_timeout_add(100,timerfnc,&timerdata);
		selectwm_config.timer=timer;
		timerdata.list=list;
		timerdata.delay=delay;
		timerdata.timer=timer;
		timerdata.timeout=timeout;
		timerdata.selectwm_config=selectwm_config;
	}
	
	gtk_main();
	return 0;
}
