#include "get_sid_list.h"
#include <pwd.h>
#include <sys/types.h>

#define USERPRIORITY 1
#define SYSTEMPRIORITY 2


/* get_config_priority - given a sid and a username, get the context priority 
                         list for that user and place it in pri_list.  The  
                         maximum number of elements allowed is pri_length.  
                         If which equals USERPRIORITY, the list will come
                         from the user's .default_contexts file.  If which 
                         equals SYSTEMPRIORITY, the list will come from the
                         system configuration file.  The number of sids placed
                         in pri_list is returned.
 */
int get_config_priority (security_id_t from_sid, char *username, 
                         security_id_t *pri_list, int pri_length, int which)
{
    FILE *config_file;    /* The configuration file                    */
    char *fname = 0;      /* The name of the user's configuration file */
    int fname_len;        /* The length of fname                       */
    struct passwd *pwd;   /* The user's passwd structure               */
    int retval;           /* The return value                          */

    if (which == USERPRIORITY)
    {
        /* Get the password structure in order to find the home directory */
        pwd = getpwnam (username);
        fname_len = strlen (pwd->pw_dir) + 19;
        fname = (char *) malloc (fname_len * sizeof (char));
        if (!fname)
        {
            return -1;
        }
        sprintf (fname, "%s%s", pwd->pw_dir, "/.default_contexts");
        config_file = fopen (fname, "r");
    }
    else if (which == SYSTEMPRIORITY)
    {
        config_file = fopen (_DEFCONTEXT_PATH, "r");
    }
    else
    {
        /* Bad which value */
        return -1;
    }

    if (!config_file)
    {
        if (fname)
            free (fname);
        return -1;
    }
    retval = get_context_list (config_file, from_sid, username, pri_list, 
                               pri_length);
    free (fname);
    fclose (config_file);
    return retval;
}


/* get_ordered_sid_list - given a user_name and the current_sid, fill 
                          ordered_list with a prioritized list of sids to 
                          transition to.  If length (the size of the list) is 
                          not large enough, then length is set to the size 
                          needed and 0 is returned.  If successful, the number 
                          of sids actually placed in the list is returned.
 */
int get_ordered_sid_list (char *user_name, security_id_t from_sid, 
                          security_id_t *ordered_list, int *length)
{
    security_id_t *init_list=0; /* The initial sid list - all reachable sids */
    int init_len = 0;           /* The length of the initial sid list        */
    int ret_val;                /* Used for return values                    */
    int *bitmap = 0;            /* An array matching the initial sid list.
                                   Each int corresponds to a sid in the 
                                   initial sid list.  The int will be 1 if 
                                   the corresponding sid has already been 
                                   placed into the ordered list, 0 otherwise */
    int i;                      /* An index into an array                    */
    security_id_t *pri_list;    /* The priority sid list obtained from a 
                                   config file                               */
    security_id_t pri_length;   /* The maximum length of the priority list   */
    int config_length;          /* The actual length of the priority list    */
    int pos = 0;                /* The current position in the ordered list  */


    if (from_sid <= 0)
    {
        /* Get the current_sid and use it for the from_sid */
        from_sid = getsecsid();
    }

    /* call once to get the size for the lists */
    ret_val = security_get_user_sids (from_sid, user_name, strlen(user_name)+1, 
                                      init_list, &init_len);

    if (ret_val != 0)
    {
        if (init_len <= 0)
        {
            /* There are no reachable sids */
            return 0;
        }

        if (*length < init_len)   /* Need more space in ordered list */
        {
            *length = init_len;
            return 0;
        }
        else
        {
            /* malloc more space for init_list */
            init_list = (security_id_t *)malloc(init_len*sizeof(security_id_t));
            if (!init_list)
            {
                return 0;
            }

            /* get all the reachable sids */
            ret_val = security_get_user_sids (from_sid, user_name,  
                                              strlen(user_name)+1, 
                                              init_list, &init_len);
            if (ret_val != 0)
            {
                free (init_list);
                return 0;
            }
        }
    }


    /* Initialize priority list */
    pri_length = MAX_CONFIG_SIDS;
    pri_list = (security_id_t *)malloc (pri_length *sizeof(security_id_t));
    if (!pri_list)
    {
        free (init_list);
        return 0;
    }
 
    /* Initialize bitmap */
    bitmap = (int *)malloc (init_len*sizeof(int));
    if (!bitmap)
    {
        free (init_list);
        free (pri_list);
        return 0;
    }
    for (i = 0; i < init_len; i++)
        bitmap[i] = 0;

    /* get the user's default context list; the sids from here should go
       first in the ordered list */
    config_length = get_config_priority (from_sid, user_name, pri_list,
                                         pri_length, USERPRIORITY);
    add_priority_list (ordered_list, init_list, bitmap, init_len, pri_list,
                       config_length, &pos);

    /* get the sids from the system config file and add to ordered_list */
    config_length = get_config_priority (from_sid, user_name, pri_list,
                                         pri_length, SYSTEMPRIORITY);
    add_priority_list (ordered_list, init_list, bitmap, init_len, pri_list,
                       config_length, &pos);

    /* finish up the list with the rest of the reachable sids */
    ret_val = complete_ordered_list (ordered_list, &pos, init_list, bitmap, 
                                     init_len);
    free (init_list);
    free (pri_list);
    free (bitmap);
    return ret_val;
}


/* insert - given a list, a position pos, and a sid, inserts the sid into the 
            list at pos.  Returns 0 on success, 1 on failure.
 */
int insert (security_id_t *ordered_list, int length, int pos, 
            security_id_t new_item)
{
    int ret_val = 1;

    if ((pos < length) && (pos >= 0))
    {
        ordered_list[pos] = new_item;
        ret_val = 0;
    }

    return ret_val;
}


/* add_priority_list - given an ordered list of sids ordered_list, a current
                       position pos, the total list of sids total_list, and a 
                       priority list pri_list add the sids from pri_list that 
                       are in total_list, but are not yet in ordered list. 
                       Return 0 on success.
 */
int add_priority_list (security_id_t *ordered_list, security_id_t *total_list,
                       int *bitmap, int length, security_id_t *pri_list,
                       int pri_length, int *pos)
{
    int i;
    int location;
    int ret_val = 0;

    for (i = 0; i < pri_length; i++)
    {
        location = locate (total_list, length, pri_list[i]);
        if ((location >= 0) && (location < length) && (!bitmap[location]))
        {
            ret_val = insert (ordered_list, length, *pos, pri_list[i]);
            if (ret_val)
                return ret_val;

            /* Mark that we have already used this sid. */
            bitmap[location] = 1;
            (*pos)++;
        }
    }
    return ret_val;
}


/* locate - given a list, and a sid, return the position of the sid in the list 
 */
int locate (security_id_t *list, int list_len, security_id_t element)
{
    int i;

    for (i = 0; i < list_len; i++)
        if (list[i] == element)
            return i;

    return -1;
}


/* complete_ordered_list - given an ordered_list of sids and a position, insert
                           all the sids in init_list that are not yet in 
                           ordered_list into ordered_list
 */
int complete_ordered_list (security_id_t *ordered_list, int *pos, 
                           security_id_t *init_list, int *bitmap, int length)
{
    int i;
    int ret_val = 0;
    int count = *pos;

    for (i = 0; i < length; i++)
    {
        if (!bitmap[i])
        {
            ret_val = insert (ordered_list, length, *pos, init_list[i]);
            if (!ret_val)
            {
                /* Mark that we have already used this sid */
                bitmap[i] = 1;
                (*pos)++;
                count++;
            }
        }
    }

    return count;
}


/* get_context_list - given current_sid, a security_id_t, and a user_name
                      string, stores in pri_list a list of sids based on 
                      configuration information from infile.  The int 
                      pri_length is the maximum amount of sids that will  
                      fit in pri_list.  Returns the number of sids stored 
                      in pri_list.
*/
int get_context_list (FILE *infile, security_id_t current_sid, char *user_name,
                      security_id_t *pri_list, int pri_length)
{
    int ret_val = 0;        /* Used for return values                    */
    char line[BUF_SIZE];    /* The line from the configuration file that
                               matches the current sid                   */

    /* Find the line in infile that matches current_sid */
    ret_val = find_line (infile, current_sid, line, BUF_SIZE);

    if (ret_val == 0)
    { 
        /* Get the sids from this line */
        ret_val = list_from_string (line, user_name, pri_list, pri_length);
    }
    else
        ret_val = 0;

    return ret_val;
}


/* find_line - given current_sid, a security_id_t, converts current_sid to 
               a security_context_t and searches for an entry in the file
               that matches the security context with the user portion 
               stripped off.  If an entry is found, the remainder of the 
               line is stored in line.  A 0 is returned if an entry is found,
               and a 1 is returned otherwise
*/
int find_line (FILE *infile, security_id_t current_sid, char *line, 
               int length)
{
    int size = length;     /* The length of the current line              */
    char *current_line;    /* The char string containing the current line */
    int pos;               /* The current position in the line            */
    char *blank;           /* A pointer to a blank character              */
    int done = 0;          /* 1 if we have found the appropriate line     */
    int nextline = 0;      /* 1 if it is time to skip to the next line    */
    int ret_val;           /* used for return values                      */
    char *cc_str = 0;      /* contains a string representing the security
                              context we are looking for without the user
                              field                                       */
    int cc_len = 0;        /* the length of cc_str                        */
    char *new_context_str; /* a string representing the security context
                              in the current line without the user field  */
    int part_len;          /* The length before the end of the string we
                              are currently examining                     */

    current_line = (char *) malloc (size * sizeof(char));
    if (current_line == NULL)
    {
        /* Error - Unable to malloc */
        return (1);
    }

    /* Convert current_sid to a context without the user field */
    if (get_current_part_context (current_sid, &cc_str, &cc_len) != 0)
    {
        fprintf (stderr, "Unable to get part_context\n");
        free (current_line);
        if (cc_str != 0)
            free (cc_str);
        return (1);
    }

    while ((!done) && (!feof (infile)))
    {
        nextline = 0;
        fgets (current_line, size, infile);

        if (current_line[strlen(current_line) - 1] != '\n')
        {
            /* Need to log that we may not have gotten all the choices */
        }
        else  /* Strip off the newline */
        {
            current_line[strlen(current_line) - 1] = '\0';
        }

            
        /* Read past leading blanks */
        pos = 0;
        while ((isspace (current_line[pos])) && (pos < size))
        {
            pos++;
        }

        if ((current_line[pos] == '\n') || (pos >= size))
            nextline = 1;


        /* Find next blank -- Everything before this blank should be the
           security context without the user field */
        if (!nextline)
        {
            blank = (char *) index (&current_line[pos], '\t');
            if (blank == 0)
                blank = (char *) index (&current_line[pos], ' ');

            part_len = (int)(blank - &current_line[pos]);
            if (part_len <= -1)
            {
                /* Problem here */
                nextline = 1;
            }
        }
  
        if (!nextline)
        {
            new_context_str = &current_line[pos];

            if (strncmp (cc_str, new_context_str, (cc_len-1)) != 0)
                nextline = 1;
            else 
            {
                done = 1;
                pos += part_len;
            }
        }

    }

    if (done)
    {
        /* Read past leading blanks */
        while (((current_line[pos] == ' ') || 
                (current_line[pos] == '\t'))&& (pos < size))
            pos++;

        if (strlen (&current_line[pos]) < length)
            strcpy (line, &current_line[pos]);
        else
        {
            strncpy (line, &current_line[pos], length-1);
            line[length-1] = '\0';
        }
        ret_val = 0;
    }
    else
        ret_val = 1;
 
    free (current_line);
    if (cc_str != 0)
        free (cc_str);
    return ret_val;
}

         
/* list_from_string - given a string, and a user name, pull out the partial
                      security contexts from the string and combine with the 
                      username to make a full security context.  Each security 
                      context will be converted to a security_id_t, and if 
                      successful stored in pri_list.  The number of elements 
                      stored in pri_list is returned.
*/
int list_from_string (char *instr, char *user, security_id_t *pri_list, 
                      int pri_length)
{
    int pos = 0;                /* The current position in instr         */
    int separator = 0;          /* points to a space in instr            */
    int length;                 /* the distance from the current position
                                   to the next blank space               */
    security_context_t current_context;  /* The current security context */
    int current_context_len;    /* length of current_context             */
    security_id_t current_sid;  /* The sid that matches current_context  */
    int done = 0;               /* 1 if we are finished making the list  */
    int count = 0;              /* The number of sids added to pri_list  */

    while ((!done) && (count < pri_length)) 
    { 
        /* Read past blank spaces */
        while ((pos < strlen(instr)) && (isblank (instr[pos])))
            pos++;

        /* Find out if there is another context after this one */
        separator = pos;
        while ((separator < strlen(instr)) && (!isspace (instr[separator])))
            separator++;
        length = separator - pos;

        if (separator == strlen (instr))
        {
            /* This must be the last context */
            done = 1;
            if (length <= -1)
            {
                /* This is not a valid context */
            }
        }

        if (length >= 0)
        {
            current_context_len = length + strlen (user) + 2;
            current_context = (security_context_t)malloc(current_context_len
                                                         *sizeof(char));
            if (current_context == NULL)
            {
                /* Problem with the malloc */
                exit (1);
            }

            strcpy (current_context, user);
            strcat (current_context, ":");
            strncat (current_context, &instr[pos], length);
            current_context[current_context_len-1] = '\0';
 
            if (security_context_to_sid (current_context,current_context_len,
                                         &current_sid) == 0)
            {
                pri_list[count] = current_sid;
                count++;
            }

            pos = pos + length;
            free (current_context);
        }
    }

    return count;
}


/* get_current_part_context - given a sid, converts it to a security context and
                              strips off the user part.  Space is malloced for
                              the partial context which is stored in 
                              *part_context.  The length is stored in 
                              part_length.  Upon success a 0 is returned.
 */
int get_current_part_context (security_id_t current_sid, char **part_context, 
                              int *part_len)
{
    security_context_t cc_str;  /* The security context that matches the 
                                   current sid                              */
    int cc_len = 0;             /* The length of cc_str                     */
    char *part_con_ptr;         /* a pointer to a string containg cc_str 
                                   without the user field                   */
    char *ptr;                  /* points to just after the user field in 
                                   cc_str                                   */

    if (security_sid_to_context (current_sid, cc_str, &cc_len) != 0)
    {
        cc_str = (char *) malloc (cc_len * sizeof (char));
        if (!cc_str)
        {
            return 1;
        }
        if (security_sid_to_context (current_sid, cc_str, &cc_len) != 0)
        {
            free (cc_str);
            return 1;
        }
    }

    ptr = index (cc_str, ':');
    ptr++;
    *part_len = (int)(cc_len - (ptr - cc_str));
    (part_con_ptr) = (char *) malloc (*part_len * sizeof(char));
    if (!part_con_ptr)
    {
        return 1;
    }
    strcpy (part_con_ptr, ptr);
    *part_context = part_con_ptr;
   
    free (cc_str);
    return 0;
}
