/*
 * Copyright (C) 2006 Sven Peter <svpe@gmx.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <X11/Xlib.h>
#include <NVCtrl/NVCtrl.h>
#include <NVCtrl/NVCtrlLib.h>

#include "sensors-applet.h"

#define GPU_CORE_TEMP "GPUCoreTemp"
#define AMBIENT_TEMP "AmbientTemp"

/* for error handling */
#define NVIDIA_ERROR (nvidia_sensors_interface_error_quark())

/* for error handling */
static GQuark nvidia_sensors_interface_error_quark(void) {
	static GQuark quark = 0;
	gchar *string;

	if (quark == 0) {
		string = g_strdup_printf("%s-error", SENSORS_INTERFACE_STRING(NVIDIA));
		quark = g_quark_from_string(string);
		g_free(string);
	}

	return quark;
}

/* global variables */
Display *nvidia_sensors_dpy; /* the connection to the X server */

/* returns the value of the sensor */
gdouble nvidia_sensors_interface_get_sensor_value(const gchar *path, 
						  const gchar *id, 
						  SensorType type,
						  GError **error)
{
	Bool res;
	int temp;


        if (g_ascii_strcasecmp(path, GPU_CORE_TEMP) == 0) {
                res = XNVCTRLQueryAttribute (nvidia_sensors_dpy, 0, 0, 
                                             NV_CTRL_GPU_CORE_TEMPERATURE, 
                                             &temp);
        } else if (g_ascii_strcasecmp(path, AMBIENT_TEMP) == 0) {
                res = XNVCTRLQueryAttribute (nvidia_sensors_dpy, 0, 0, 
                                             NV_CTRL_AMBIENT_TEMPERATURE, 
                                             &temp);
        } else {
                g_set_error(error, NVIDIA_ERROR, 0, "Invalid path string passed to nvidia_sensors_interface_get_sensor_value");
                return 0;
        }
                

        if (res != True) {
                /* when res isn't true something went wrong */
                g_set_error(error, NVIDIA_ERROR, 0, "XNVCTRLQueryAttribute returned false");
                return 0;
        }
        
        /* convert the int to gdouble and return it */
        return (gdouble)temp;
}

/* creates the connection to the X server and checks whether the
 * NV-CONTROL extension is loaded */
void nvidia_sensors_interface_init(SensorsApplet *sensors_applet)
{
	int dummy;
	int event_base, error_base;


	/* register this interface regardless */
	sensors_applet_register_sensors_interface(sensors_applet,
						  NVIDIA,
						  nvidia_sensors_interface_get_sensor_value);
        
	/* create the connection to the X server */
	if (!(nvidia_sensors_dpy = XOpenDisplay(NULL))) {
		/* no connection to the X server avaible */
		return;
	}

	/* check if the NV-CONTROL extension is available on this X
         * server - if so add the two sensors if they exist */
	if (XNVCTRLQueryExtension(nvidia_sensors_dpy, &event_base, &error_base)) {
                if (XNVCTRLQueryAttribute(nvidia_sensors_dpy, 0, 0, NV_CTRL_GPU_CORE_TEMPERATURE, &dummy)) {
                        
                        sensors_applet_add_sensor(sensors_applet,
                                                  GPU_CORE_TEMP,
                                                  GPU_CORE_TEMP,
                                                  _("GPU"),
                                                  NVIDIA,
                                                  TRUE,
                                                  TEMP_SENSOR,
                                                  GPU_ICON);
                }

                if (XNVCTRLQueryAttribute(nvidia_sensors_dpy, 0, 0, NV_CTRL_AMBIENT_TEMPERATURE, &dummy)) {
                        
                        sensors_applet_add_sensor(sensors_applet,
                                                  AMBIENT_TEMP,
                                                  AMBIENT_TEMP,
                                                  _("Ambient"),
                                                  NVIDIA,
                                                  FALSE,
                                                  TEMP_SENSOR,
                                                  GENERIC_ICON);
                }
        }
}

