#define _XOPEN_SOURCE 500

#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <sys/time.h>
#include <linux/major.h>
typedef unsigned char u_char;   /* horrible, for scsi.h */
#include "sg_include.h"
#include "sg_err.h"
#include "llseek.h"

/* A utility program for the Linux OS SCSI generic ("sg") device driver.
*  Copyright (C) 1999 - 2002 D. Gilbert and P. Allworth
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2, or (at your option)
*  any later version.

   This program is a specialization of the Unix "dd" command in which
   one or both of the given files is a scsi generic device or a raw
   device. A block size ('bs') is assumed to be 512 if not given. This
   program complains if 'ibs' or 'obs' are given with some other value.
   If 'if' is not given or 'if=-' then stdin is assumed. If 'of' is
   not given or 'of=-' then stdout assumed.  Multipliers:
     'c','C'  *1       'b','B' *512      'k' *1024      'K' *1000
     'm' *(1024^2)     'M' *(1000^2)     'g' *(1024^3)  'G' *(1000^3)

   A non-standard argument "bpt" (blocks per transfer) is added to control
   the maximum number of blocks in each transfer. The default value is 128.
   For example if "bs=512" and "bpt=32" then a maximum of 32 blocks (16KB
   in this case) are transferred to or from the sg device in a single SCSI
   command.

   Version 4.15 20020728
*/

#define DEF_BLOCK_SIZE 512
#define DEF_BLOCKS_PER_TRANSFER 128

#define ME "sg_dd: "

// #define SG_DEBUG

#define DEF_TIMEOUT 40000       /* 40,000 millisecs == 40 seconds */
#define SG_HEAD_SZ sizeof(struct sg_header)
#define SCSI_CMD10_LEN 10
#define READ_CAP_DATA_LEN 8

#ifndef SG_MAX_SENSE
#define SG_MAX_SENSE 16
#endif

#ifndef RAW_MAJOR
#define RAW_MAJOR 255   /*unlikey value */
#endif

#define FT_OTHER 0              /* filetype other than sg or raw device */
#define FT_SG 1                 /* filetype is sg char device */
#define FT_RAW 2                /* filetype is raw char device */

#define STR_SZ 1024
#define INOUTF_SZ 512
#define EBUFF_SZ 512

static unsigned char rdCmdBlk[SCSI_CMD10_LEN] =
                    {0x28, 0, 0, 0, 0, 0, 0, 0, 0, 0};
static unsigned char wrCmdBlk[SCSI_CMD10_LEN] =
                    {0x2a, 0, 0, 0, 0, 0, 0, 0, 0, 0};

static int dd_count = -1;
static int in_full = 0;
static int in_partial = 0;
static int out_full = 0;
static int out_partial = 0;
static int do_coe = 0;

static void install_handler (int sig_num, void (*sig_handler) (int sig))
{
    struct sigaction sigact;
    sigaction (sig_num, NULL, &sigact);
    if (sigact.sa_handler != SIG_IGN)
    {
	sigact.sa_handler = sig_handler;
	sigemptyset (&sigact.sa_mask);
	sigact.sa_flags = 0;
	sigaction (sig_num, &sigact, NULL);
    }
}

void print_stats()
{
    if (0 != dd_count)
        fprintf(stderr, "  remaining block count=%d\n", dd_count);
    fprintf(stderr, "%d+%d records in\n", in_full - in_partial, in_partial);
    fprintf(stderr, "%d+%d records out\n", out_full - out_partial,
	    out_partial);
}

static void interrupt_handler(int sig)
{
    struct sigaction sigact;

    sigact.sa_handler = SIG_DFL;
    sigemptyset (&sigact.sa_mask);
    sigact.sa_flags = 0;
    sigaction (sig, &sigact, NULL);
    fprintf(stderr, "Interrupted by signal,");
    print_stats ();
    kill (getpid (), sig);
}

static void siginfo_handler(int sig)
{
    fprintf(stderr, "Progress report, continuing ...\n");
    print_stats ();
}

int dd_filetype(const char * filename)
{
    struct stat st;

    if (stat(filename, &st) < 0)
        return FT_OTHER;
    if (S_ISCHR(st.st_mode)) {
        if (RAW_MAJOR == major(st.st_rdev))
            return FT_RAW;
        else if (SCSI_GENERIC_MAJOR == major(st.st_rdev))
            return FT_SG;
    }
    return FT_OTHER;
}

void usage()
{
    fprintf(stderr, "Usage: "
            "sg_dd  [if=<infile>] [skip=<n>] [of=<ofile>] [seek=<n>]\n"
            "              [bs=<num>] [bpt=<num>] [count=<n>] [time=0|1]\n"
            " where\n"
            " 'bpt' is blocks_per_transfer (default is 128)\n"
	    " 'time' 0->no timing(def), 1->time plus calculate throughput\n"
	    " 'coe' 1->continue on sg error, 0->exit on error (def)\n");
}

/* Return of 0 -> success, -1 -> failure, 2 -> try again */
int read_capacity(int sg_fd, int * num_sect, int * sect_sz)
{
    int res;
    unsigned char rcCmdBlk [10] = {0x25, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    unsigned char rcBuff[SG_HEAD_SZ + sizeof(rcCmdBlk) + 512];
    int rcInLen = SG_HEAD_SZ + sizeof(rcCmdBlk);
    int rcOutLen;
    unsigned char * buffp = rcBuff + SG_HEAD_SZ;
    struct sg_header * rsghp = (struct sg_header *)rcBuff;

    rcOutLen = SG_HEAD_SZ + READ_CAP_DATA_LEN;
    rsghp->pack_len = 0;                /* don't care */
    rsghp->pack_id = 0;
    rsghp->reply_len = rcOutLen;
    rsghp->twelve_byte = 0;
    rsghp->result = 0;
#ifndef SG_GET_RESERVED_SIZE
    rsghp->sense_buffer[0] = 0;
#endif
    memcpy(rcBuff + SG_HEAD_SZ, rcCmdBlk, sizeof(rcCmdBlk));

    while (((res = write(sg_fd, rcBuff, rcInLen)) < 0) && (EINTR == errno))
        ;
    if (res < 0) {
        perror("read_capacity (wr) error");
        return -1;
    }
    if (res < rcInLen) {
        fprintf(stderr, "read_capacity (wr) problems\n");
        return -1;
    }
    
    memset(buffp, 0, 8);
    while (((res = read(sg_fd, rcBuff, rcOutLen)) < 0) && (EINTR == errno))
        ;
    if (res < 0) {
        perror("read_capacity (rd) error");
        return -1;
    }
    if (res < rcOutLen) {
        fprintf(stderr, "read_capacity (rd) problems\n");
        return -1;
    }
#ifdef SG_GET_RESERVED_SIZE
    res = sg_err_category(rsghp->target_status, rsghp->host_status,
                          rsghp->driver_status, rsghp->sense_buffer, 
                          SG_MAX_SENSE);
    if (SG_ERR_CAT_MEDIA_CHANGED == res)
        return 2; /* probably have another go ... */
    else if (SG_ERR_CAT_CLEAN != res) {
        sg_chk_n_print("read capacity", rsghp->target_status, 
                       rsghp->host_status, rsghp->driver_status, 
                       rsghp->sense_buffer, SG_MAX_SENSE);
        return -1;
    }
#else
    if ((rsghp->result != 0) || (0 != rsghp->sense_buffer[0])) {
        fprintf(stderr, "read_capacity result=%d\n", rsghp->result);
        if (0 != rsghp->sense_buffer[0])
            sg_print_sense("read_capacity", rsghp->sense_buffer, SG_MAX_SENSE);
        return -1;
    }
#endif

    *num_sect = 1 + ((buffp[0] << 24) | (buffp[1] << 16) |
                (buffp[2] << 8) | buffp[3]);
    *sect_sz = (buffp[4] << 24) | (buffp[5] << 16) | 
               (buffp[6] << 8) | buffp[7];
/* printf("number of sectors=%d, sector size=%d\n", *num_sect, *sect_sz); */
    return 0;
}

/* -1 -> unrecoverable error, 0 -> successful, 1 -> recoverable (ENOMEM),
   2 -> try again */
int sg_read(int sg_fd, unsigned char * buff, int blocks, int from_block,
            int bs)
{
    int inLen = SG_HEAD_SZ + SCSI_CMD10_LEN;
    int outLen, res;
    unsigned char * rdCmd = buff + SG_HEAD_SZ;
    struct sg_header * isghp = (struct sg_header *)buff;
    struct sg_header * osghp = (struct sg_header *)(buff + SCSI_CMD10_LEN);
    
    outLen = SG_HEAD_SZ + (bs * blocks);
    isghp->pack_len = 0;                /* don't care */
    isghp->pack_id = from_block;
    isghp->reply_len = outLen;
    isghp->twelve_byte = 0;
    isghp->result = 0;
#ifndef SG_GET_RESERVED_SIZE
    isghp->sense_buffer[0] = 0;
#endif
    memcpy(rdCmd, rdCmdBlk, SCSI_CMD10_LEN);
    rdCmd[2] = (unsigned char)((from_block >> 24) & 0xFF);
    rdCmd[3] = (unsigned char)((from_block >> 16) & 0xFF);
    rdCmd[4] = (unsigned char)((from_block >> 8) & 0xFF);
    rdCmd[5] = (unsigned char)(from_block & 0xFF);
    rdCmd[7] = (unsigned char)((blocks >> 8) & 0xff);
    rdCmd[8] = (unsigned char)(blocks & 0xff);

    while (((res = write(sg_fd, buff, inLen)) < 0) && (EINTR == errno))
        ;
    if (res < 0) {
        if (ENOMEM == errno)
            return 1;
        perror("reading (wr) on sg device, error");
        return -1;
    }
    if (res < inLen) {
        fprintf(stderr, "reading (wr) on sg device, problems, "
		"ask=%d, got=%d\n", inLen, res);
        return -1;
    }

    while (((res = read(sg_fd, buff + SCSI_CMD10_LEN, outLen)) < 0) &&
           (EINTR == errno))
        ;
    if (res < 0) {
        perror("reading (rd) on sg device, error");
        return -1;
    }
    if (res < outLen)
    {
        fprintf(stderr, "reading (rd) on sg device, problems, "
		"ask=%d, got=%d\n", outLen, res);
        return -1;
    }
#ifdef SG_GET_RESERVED_SIZE
    res = sg_err_category(osghp->target_status, osghp->host_status,
                          osghp->driver_status, osghp->sense_buffer, 
                          SG_MAX_SENSE);
    switch (res) {
    case SG_ERR_CAT_CLEAN:
        break;
    case SG_ERR_CAT_RECOVERED:
        fprintf(stderr, "Recovered error while reading block=%d, num=%d\n",
               from_block, blocks);
        break;
    case SG_ERR_CAT_MEDIA_CHANGED:
        return 2;
    default:
        sg_chk_n_print("reading", osghp->target_status, 
                       osghp->host_status, osghp->driver_status, 
                       osghp->sense_buffer, SG_MAX_SENSE);
        if (do_coe) {
            memset(buff + SG_HEAD_SZ + SCSI_CMD10_LEN, 0, bs * blocks);
            fprintf(stderr, ">> unable to read at blk=%d for "
                        "%d bytes, use zeros\n", from_block, bs * blocks);
            return 0; /* fudge success */
        }
        else
            return -1;
    }
#else
    if ((osghp->result != 0) || (0 != osghp->sense_buffer[0])) {
        fprintf(stderr, "reading result=%d\n", osghp->result);
        if (0 != osghp->sense_buffer[0])
            sg_print_sense("after read(rd)", osghp->sense_buffer, SG_MAX_SENSE);
        return -1;
    }
#endif
    return 0;
}

/* -1 -> unrecoverable error, 0 -> successful, 1 -> recoverable (ENOMEM),
   2 -> try again */
int sg_write(int sg_fd, unsigned char * buff, int blocks, int to_block,
             int bs)
{
    int outLen = SG_HEAD_SZ;
    int inLen, res;
    unsigned char * wrCmd = buff + SG_HEAD_SZ;
    struct sg_header * isghp = (struct sg_header *)buff;
    struct sg_header * osghp = (struct sg_header *)(buff + SCSI_CMD10_LEN);
    
    inLen = SG_HEAD_SZ + SCSI_CMD10_LEN + (bs * blocks);
    isghp->pack_len = 0;                /* don't care */
    isghp->pack_id = to_block;
    isghp->reply_len = outLen;
    isghp->twelve_byte = 0;
    isghp->result = 0;
#ifndef SG_GET_RESERVED_SIZE
    isghp->sense_buffer[0] = 0;
#endif
    memcpy(wrCmd, wrCmdBlk, SCSI_CMD10_LEN);
    wrCmd[2] = (unsigned char)((to_block >> 24) & 0xFF);
    wrCmd[3] = (unsigned char)((to_block >> 16) & 0xFF);
    wrCmd[4] = (unsigned char)((to_block >> 8) & 0xFF);
    wrCmd[5] = (unsigned char)(to_block & 0xFF);
    wrCmd[7] = (unsigned char)((blocks >> 8) & 0xff);
    wrCmd[8] = (unsigned char)(blocks & 0xff);

    while (((res = write(sg_fd, buff, inLen)) < 0) && (EINTR == errno))
        ;
    if (res < 0) {
        if (ENOMEM == errno)
            return 1;
        perror("writing (wr) on sg device, error");
        return -1;
    }
    if (res < inLen) {
        fprintf(stderr, "writing (wr) on sg device, problems,"
		" ask=%d, got=%d\n", inLen, res);
        return -1;
    }

    while (((res = read(sg_fd, buff + SCSI_CMD10_LEN, outLen)) < 0) &&
           (EINTR == errno))
        ;
    if (res < 0) {
        perror("writing (rd) on sg device, error");
        return -1;
    }
    if (res < outLen)
    {
        fprintf(stderr, "writing (rd) on sg device, problems, ask=%d, got=%d\n",
                outLen, res);
        return -1;
    }
#ifdef SG_GET_RESERVED_SIZE
    res = sg_err_category(osghp->target_status, osghp->host_status,
                          osghp->driver_status, osghp->sense_buffer,
                          SG_MAX_SENSE);
    switch (res) {
    case SG_ERR_CAT_CLEAN:
        break;
    case SG_ERR_CAT_RECOVERED:
        fprintf(stderr, "Recovered error while writing block=%d, num=%d\n",
               to_block, blocks);
        break;
    case SG_ERR_CAT_MEDIA_CHANGED:
        return 2;
    default:
        sg_chk_n_print("writing", osghp->target_status, 
                       osghp->host_status, osghp->driver_status, 
                       osghp->sense_buffer, SG_MAX_SENSE);
        if (do_coe) {
            fprintf(stderr, ">> ignored errors for out blk=%d for "
                    "%d bytes\n", to_block, bs * blocks);
            return 0; /* fudge success */
        }
        else
            return -1;
    }
#else
    if ((osghp->result != 0) || (0 != osghp->sense_buffer[0])) {
        fprintf(stderr, "writing result=%d\n", osghp->result);
        if (0 != osghp->sense_buffer[0])
            sg_print_sense("after write(rd)", osghp->sense_buffer, 
                           SG_MAX_SENSE);
        return -1;
    }
#endif
    return 0;
}

int get_num(char * buf)
{
    int res, num;
    char c;

    res = sscanf(buf, "%d%c", &num, &c);
    if (0 == res)
        return -1;
    else if (1 == res)
        return num;
    else {
        switch (c) {
        case 'c':
        case 'C':
            return num;
        case 'b':
        case 'B':
            return num * 512;
        case 'k':
            return num * 1024;
        case 'K':
            return num * 1000;
        case 'm':
            return num * 1024 * 1024;
        case 'M':
            return num * 1000000;
        case 'g':
            return num * 1024 * 1024 * 1024;
        case 'G':
            return num * 1000000000;
        default:
            fprintf(stderr, "unrecognized multiplier\n");
            return -1;
        }
    }
}


int main(int argc, char * argv[])
{
    int skip = 0;
    int seek = 0;
    int bs = 0;
    int ibs = 0;
    int obs = 0;
    int bpt = DEF_BLOCKS_PER_TRANSFER;
    char str[STR_SZ];
    char * key;
    char * buf;
    char inf[INOUTF_SZ];
    int in_type = FT_OTHER;
    char outf[INOUTF_SZ];
    int out_type = FT_OTHER;
    int do_time = 0;
    int res, k, t, buf_sz;
    int infd, outfd, blocks;
    unsigned char * wrkBuffRaw;
    unsigned char * wrkBuff;
    unsigned char * wrkPos;
    int in_num_sect = 0;
    int out_num_sect = 0;
    int in_sect_sz, out_sect_sz;
    char ebuff[EBUFF_SZ];
    int blocks_per;
    int req_count;
    struct timeval start_tm, end_tm;

    inf[0] = '\0';
    outf[0] = '\0';
    if (argc < 2) {
        usage();
        return 1;
    }

    for(k = 1; k < argc; k++) {
        if (argv[k])
            strncpy(str, argv[k], STR_SZ);
        else
            continue;
        for(key = str, buf = key; *buf && *buf != '=';)
            buf++;
        if (*buf)
            *buf++ = '\0';
        if (strcmp(key,"if") == 0)
            strncpy(inf, buf, INOUTF_SZ);
        else if (strcmp(key,"of") == 0)
            strncpy(outf, buf, INOUTF_SZ);
        else if (0 == strcmp(key,"ibs"))
            ibs = get_num(buf);
        else if (0 == strcmp(key,"obs"))
            obs = get_num(buf);
        else if (0 == strcmp(key,"bs"))
            bs = get_num(buf);
        else if (0 == strcmp(key,"bpt"))
            bpt = get_num(buf);
        else if (0 == strcmp(key,"skip"))
            skip = get_num(buf);
        else if (0 == strcmp(key,"seek"))
            seek = get_num(buf);
        else if (0 == strcmp(key,"count"))
            dd_count = get_num(buf);
        else if (0 == strcmp(key,"coe"))
            do_coe = get_num(buf);
        else if (0 == strcmp(key,"time"))
            do_time = get_num(buf);
        else {
            fprintf(stderr, "Unrecognized argument '%s'\n", key);
            usage();
            return 1;
        }
    }
    if (bs <= 0) {
        bs = DEF_BLOCK_SIZE;
        fprintf(stderr, "Assume default 'bs' (block size) of %d bytes\n", bs);
    }
    if ((ibs && (ibs != bs)) || (obs && (obs != bs))) {
        fprintf(stderr, "If 'ibs' or 'obs' given must be same as 'bs'\n");
        usage();
        return 1;
    }
    if ((skip < 0) || (seek < 0)) {
        fprintf(stderr, "skip and seek cannot be negative\n");
        return 1;
    }
#ifdef SG_DEBUG
    fprintf(stderr, ME "if=%s skip=%d of=%s seek=%d count=%d\n",
           inf, skip, outf, seek, dd_count);
#endif
    install_handler (SIGINT, interrupt_handler);
    install_handler (SIGQUIT, interrupt_handler);
    install_handler (SIGPIPE, interrupt_handler);
    install_handler (SIGUSR1, siginfo_handler);

    infd = STDIN_FILENO;
    outfd = STDOUT_FILENO;
    if (inf[0] && ('-' != inf[0])) {
	in_type = dd_filetype(inf);

	if (FT_SG == in_type) {
            if ((infd = open(inf, O_RDWR)) < 0) {
                snprintf(ebuff, EBUFF_SZ, ME "could not open %s for sg "
			 "reading", inf);
                perror(ebuff);
                return 1;
            }
            t = bs * bpt;	
#ifdef SG_SET_RESERVED_SIZE
	    res = ioctl(infd, SG_SET_RESERVED_SIZE, &t);
	    if (res < 0) {
		perror(ME "SG_SET_RESERVED_SIZE error");
		return 1;
	    }
#endif
        }
	if (FT_SG != in_type) {
            if ((infd = open(inf, O_RDONLY)) < 0) {
                snprintf(ebuff, EBUFF_SZ, ME "could not open %s for reading",
			 inf);
                perror(ebuff);
                return 1;
            }
            else if (skip > 0) { 
                llse_loff_t offset = skip;

                offset *= bs;       /* could exceed 32 bits here! */
                if (llse_llseek(infd, offset, SEEK_SET) < 0) {
                    snprintf(ebuff, EBUFF_SZ, ME "couldn't skip to "
		    	     "required position on %s", inf);
                    perror(ebuff);
                    return 1;
                }
            }
        }
    }
    if (outf[0] && ('-' != outf[0])) {
	out_type = dd_filetype(outf);

	if (FT_SG == out_type) {
            if ((outfd = open(outf, O_RDWR)) < 0) {
                snprintf(ebuff, EBUFF_SZ, ME "could not open %s for "
			 "sg writing", outf);
                perror(ebuff);
                return 1;
            }
            t = bs * bpt;
#ifdef SG_SET_RESERVED_SIZE
	    res = ioctl(outfd, SG_SET_RESERVED_SIZE, &t);
	    if (res < 0) {
		perror(ME "SG_SET_RESERVED_SIZE error");
		return 1;
	    }
#endif
        }
	else {
	    if (FT_OTHER == out_type) {
                if ((outfd = open(outf, O_WRONLY | O_CREAT, 0666)) < 0) {
                    snprintf(ebuff, EBUFF_SZ, ME "could not open %s "
		    	     "for writing", outf);
                    perror(ebuff);
                    return 1;
                }
            }
            else {
                if ((outfd = open(outf, O_WRONLY)) < 0) {
                    snprintf(ebuff, EBUFF_SZ, ME "could not open %s "
		    	     "for raw writing", outf);
                    perror(ebuff);
                    return 1;
                }
            }
            if (seek > 0) {
                llse_loff_t offset = seek;

                offset *= bs;       /* could overflow here! */
                if (llse_llseek(outfd, offset, SEEK_SET) < 0) {
                    snprintf(ebuff, EBUFF_SZ, ME "couldn't seek to "
		    	     "required position on %s", outf);
                    perror(ebuff);
                    return 1;
                }
            }
        }
    }
    if ((STDIN_FILENO == infd) && (STDOUT_FILENO == outfd)) {
        fprintf(stderr, "Can't have both 'if' as stdin _and_ 'of' as stdout\n");
        return 1;
    }
    if (0 == dd_count)
        return 0;
    else if (dd_count < 0) {
        if (FT_SG == in_type) {
            res = read_capacity(infd, &in_num_sect, &in_sect_sz);
            if (2 == res) {
                fprintf(stderr, 
			"Unit attention, media changed(in), continuing\n");
                res = read_capacity(infd, &in_num_sect, &in_sect_sz);
            }
            if (0 != res) {
                fprintf(stderr, "Unable to read capacity on %s\n", inf);
                in_num_sect = -1;
            }
            else {
#if 0
                if (0 == in_sect_sz)
                    in_sect_sz = bs;
                else if (in_sect_sz > bs)
                    in_num_sect *=  (in_sect_sz / bs);
                else if (in_sect_sz < bs)
                    in_num_sect /=  (bs / in_sect_sz);
#endif
                if (in_num_sect > skip)
                    in_num_sect -= skip;
            }
        }
        if (FT_SG == out_type) {
            res = read_capacity(outfd, &out_num_sect, &out_sect_sz);
            if (2 == res) {
                fprintf(stderr, "Unit attention, media changed(out),"
			" continuing\n");
                res = read_capacity(outfd, &out_num_sect, &out_sect_sz);
            }
            if (0 != res) {
                fprintf(stderr, "Unable to read capacity on %s\n", outf);
                out_num_sect = -1;
            }
            else {
                if (out_num_sect > seek)
                    out_num_sect -= seek;
            }
        }
#ifdef SG_DEBUG
    fprintf(stderr, "Start of loop, count=%d, in_num_sect=%d,"
	    " out_num_sect=%d\n", dd_count, in_num_sect, out_num_sect);
#endif
        if (in_num_sect > 0) {
            if (out_num_sect > 0)
                dd_count = (in_num_sect > out_num_sect) ? out_num_sect :
                                                       in_num_sect;
            else
                dd_count = in_num_sect;
        }
        else
            dd_count = out_num_sect;
    }
    if (dd_count <= 0) {
        fprintf(stderr, "Couldn't calculate count, please give one\n");
        return 1;
    }

    if ((FT_RAW == in_type) || (FT_RAW == out_type)) {
        size_t psz = getpagesize();
        wrkBuffRaw = malloc((bs * bpt) + (2 * psz));
        if (0 == wrkBuffRaw) {
            fprintf(stderr, "Not enough user memory for raw\n");
            return 1;
        }
        wrkPos = (unsigned char *)
		 (((unsigned long)wrkBuffRaw + (2 * psz) - 1) & (~(psz - 1)));
	wrkBuff = wrkPos - (SG_HEAD_SZ + SCSI_CMD10_LEN);
	/* logic assumes  getpagesize() > ( SG_HEAD_SZ + SCSI_CMD10_LEN) */
    }
    else {
	wrkBuffRaw= malloc(bs * bpt + SG_HEAD_SZ + SCSI_CMD10_LEN);
	if (0 == wrkBuffRaw) {
	    fprintf(stderr, "Not enough user memory\n");
	    return 1;
	}
	wrkBuff = wrkBuffRaw;
	wrkPos = wrkBuff + SG_HEAD_SZ + SCSI_CMD10_LEN;
    }

    blocks_per = bpt;
#ifdef SG_DEBUG
    fprintf(stderr, "Start of loop, count=%d, blocks_per=%d\n", 
	    dd_count, blocks_per);
#endif
    if (do_time) {
        start_tm.tv_sec = 0;
        start_tm.tv_usec = 0;
        gettimeofday(&start_tm, NULL);
    }
    req_count = dd_count;

    while (dd_count > 0) {
        blocks = (dd_count > blocks_per) ? blocks_per : dd_count;
        if (FT_SG == in_type) {
            res = sg_read(infd, wrkBuff, blocks, skip, bs);
            if (1 == res) {     /* ENOMEM, find what's available+try that */
#ifdef SG_GET_RESERVED_SIZE
                if (ioctl(infd, SG_GET_RESERVED_SIZE, &buf_sz) < 0) {
                    perror("RESERVED_SIZE ioctls failed");
                    break;
                }
#else
                buf_sz = SG_BIG_BUFF;
#endif
                blocks_per = (buf_sz + bs - 1) / bs;
                blocks = blocks_per;
                fprintf(stderr, "Reducing read to %d blocks per loop\n", 
			blocks_per);
                res = sg_read(infd, wrkBuff, blocks, skip, bs);
            }
            else if (2 == res) {
                fprintf(stderr, "Unit attention, media changed,"
			" continuing (r)\n");
                res = sg_read(infd, wrkBuff, blocks, skip, bs);
            }
            if (0 != res) {
                fprintf(stderr, "sg_read failed, skip=%d\n", skip);
                break;
            }
            else
                in_full += blocks;
        }
        else {
            while (((res = read(infd, wrkPos, blocks * bs)) < 0) &&
                   (EINTR == errno))
                ;
            if (res < 0) {
                snprintf(ebuff, EBUFF_SZ, ME "reading, skip=%d ", skip);
                perror(ebuff);
                break;
            }
            else if (res < blocks * bs) {
                dd_count = 0;
                blocks = res / bs;
                if ((res % bs) > 0) {
                    blocks++;
                    in_partial++;
                }
            }
            in_full += blocks;
        }

        if (FT_SG == out_type) {
            res = sg_write(outfd, wrkBuff, blocks, seek, bs);
            if (1 == res) {     /* ENOMEM, find what's available+try that */
#ifdef SG_GET_RESERVED_SIZE
                if (ioctl(outfd, SG_GET_RESERVED_SIZE, &buf_sz) < 0) {
                    perror("RESERVED_SIZE ioctls failed");
                    break;
                }
#else
                buf_sz = SG_BIG_BUFF;
#endif
                blocks_per = (buf_sz + bs - 1) / bs;
                blocks = blocks_per;
                fprintf(stderr, "Reducing write to %d blocks per loop\n", 
			blocks);
                res = sg_write(outfd, wrkBuff, blocks, seek, bs);
            }
            else if (2 == res) {
                fprintf(stderr, "Unit attention, media changed,"
			" continuing (w)\n");
                res = sg_write(outfd, wrkBuff, blocks, seek, bs);
            }
            else if (0 != res) {
                fprintf(stderr, "sg_write failed, seek=%d\n", seek);
                break;
            }
            else
                out_full += blocks;
        }
        else {
            while (((res = write(outfd, wrkPos, blocks * bs)) < 0)
                   && (EINTR == errno))
                ;
            if (res < 0) {
                snprintf(ebuff, EBUFF_SZ, ME "writing, seek=%d ", seek);
                perror(ebuff);
                break;
            }
            else if (res < blocks * bs) {
                fprintf(stderr, "output file probably full, seek=%d ", seek);
                blocks = res / bs;
                out_full += blocks;
                if ((res % bs) > 0)
                    out_partial++;
                break;
            }
            else
                out_full += blocks;
        }
        if (dd_count > 0)
            dd_count -= blocks;
        skip += blocks;
        seek += blocks;
    }
    if ((do_time) && (start_tm.tv_sec || start_tm.tv_usec)) {
        struct timeval res_tm;
        double a, b;

        gettimeofday(&end_tm, NULL);
        res_tm.tv_sec = end_tm.tv_sec - start_tm.tv_sec;
        res_tm.tv_usec = end_tm.tv_usec - start_tm.tv_usec;
        if (res_tm.tv_usec < 0) {
            --res_tm.tv_sec;
            res_tm.tv_usec += 1000000;
        }
        a = res_tm.tv_sec;
        a += (0.000001 * res_tm.tv_usec);
        b = (double)bs * (req_count - dd_count);
        printf("time to transfer data was %d.%06d secs",
               (int)res_tm.tv_sec, (int)res_tm.tv_usec);
        if ((a > 0.00001) && (b > 511))
            printf(", %.2f MB/sec\n", b / (a * 1000000.0));
        else
            printf("\n");
    }

    free(wrkBuffRaw);
    if (STDIN_FILENO != infd)
        close(infd);
    if (STDOUT_FILENO != outfd)
        close(outfd);
    res = 0;
    if (0 != dd_count) {
        fprintf(stderr, "Some error occurred,");
	res = 1;
    }
    print_stats();
    return res;
}
