/*
 * Smart Gnome Control - hamlib interface file
 * Copyright (C) 2004 by Mark J. Fine <fineware@users.sourceforge.net>
 *
 * This program provides a graphical user interface to multiple
 * communications receivers using the Hamlib library.
 *
 * $Id: extra.c,v 1.38 2005/01/16 03:23:55 fineware Exp $
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 *      Free Software Foundation, Inc.
 *      59 Temple Place - Suite 330
 *      Boston, MA  02111-1307
 *      USA
 *
 */

#include "support.h"
#include "extra.h"
#include "database.h"
#include "logfile.h"

const guint		dWidth = 17;
const guint		dHeight = 24;
const guint		mWidth = 51;
const guint		mHeight = 31;
static GSList*	rig_list = NULL;

typedef struct {
	gchar*	label;
	double	start;
	double	end;
	gchar*	type;
} band_plan_t;

band_plan_t		plan_text[90] = {
  {" Nav",  100,  150,"N"},
  {" LW ",  150,  280,"B"},
  {" Nav",  280,  530,"N"},
  {" BCB",  530, 1710,"B"},
  {" Nav", 1710, 1800,"N"},
  {"160m", 1800, 2000,"A"},
  {" Mar", 2065, 2096,"M"},
  {" Mar", 2174, 2191,"M"},
  {"120m", 2300, 2495,"B"},
  {"Time", 2495, 2505,"T"},
  {"Fixd", 2505, 2850,"F"},
  {"Aero", 2850, 3155,"E"},
  {"Fixd", 3155, 3400,"F"},
  {" 90m", 3200, 3400,"B"},
  {"Aero", 3400, 3500,"E"},
  {" 80m", 3500, 4000,"A"},
  {" 75m", 3900, 4000,"B"},
  {"Fixd", 4000, 4065,"F"},
  {" Mar", 4065, 4438,"M"},
  {"Aero", 4438, 4750,"E"},
  {" 60m", 4750, 5060,"B"},
  {"Time", 4995, 5005,"T"},
  {"Fixd", 5060, 5450,"F"},
  {"Aero", 5450, 5730,"E"},
  {" 49m", 5730, 6295,"B"},
  {" Mar", 6200, 6525,"M"},
  {"Aero", 6525, 6765,"E"},
  {"Fixd", 6765, 7000,"F"},
  {" 40m", 7000, 7300,"A"},
  {" 41m", 6890, 7600,"B"},
  {"Fixd", 7600, 8195,"F"},
  {" Mar", 8195, 8815,"M"},
  {"Aero", 8815, 9040,"E"},
  {"Fixd", 9040, 9500,"F"},
  {" 31m", 9250, 9900,"B"},
  {"Fixd", 9900, 9995,"F"},
  {"Time", 9995,10005,"T"},
  {"Aero",10005,10100,"E"},
  {" 30m",10100,10150,"A"},
  {"Fixd",10150,11175,"F"},
  {"Aero",11175,11400,"E"},
  {"Fixd",11400,11700,"F"},
  {" 25m",11500,12160,"B"},
  {"Fixd",12160,12230,"F"},
  {" Mar",12230,13200,"M"},
  {"Aero",13200,13360,"E"},
  {"Fixd",13360,13600,"F"},
  {" 22m",13570,13870,"B"},
  {"Fixd",13870,14000,"F"},
  {" 20m",14000,14350,"A"},
  {"Fixd",14350,14990,"F"},
  {"Time",14990,15010,"T"},
  {"Aero",15010,15030,"E"},
  {" 19m",15030,15800,"B"},
  {"Fixd",15800,16360,"F"},
  {" Mar",16360,17410,"M"},
  {"Fixd",17410,17550,"F"},
  {" 16m",17480,17900,"B"},
  {"Aero",17900,18030,"E"},
  {"Fixd",18030,18060,"F"},
  {" 16m",18068,18168,"A"},
  {"Fixd",18168,18780,"F"},
  {" Mar",18780,18900,"M"},
  {" 15m",18900,19020,"B"},
  {"Fixd",19020,19680,"F"},
  {" Mar",19680,19800,"M"},
  {"Fixd",19800,19990,"F"},
  {"Time",19990,20010,"T"},
  {"Fixd",20010,21000,"F"},
  {" 15m",21000,21450,"A"},
  {" 13m",21450,21850,"B"},
  {"Astr",21850,21870,"R"},
  {"Aero",21870,22000,"E"},
  {" Mar",22000,22855,"M"},
  {"Fixd",22855,23200,"F"},
  {"Aero",23200,23350,"E"},
  {"Fixd",23350,24890,"F"},
  {" 12m",24890,24990,"A"},
  {"Time",24990,25010,"T"},
  {"Fixd",25010,25070,"F"},
  {" Mar",25070,25210,"M"},
  {"Fixd",25210,25550,"F"},
  {"Astr",25550,25600,"R"},
  {" 11m",25600,26100,"B"},
  {" Mar",26100,26200,"M"},
  {"Fixd",26200,26965,"F"},
  {" CB ",26965,27405,"C"},
  {"Fixd",27405,28000,"F"},
  {" 10m",28000,29700,"A"},
  {"Fixd",29700,30000,"F"}};

gint rig_list_make (const struct rig_caps *caps, gpointer data);
gint hamlib_riglist_widget_compare_func (GtkCList *c_list, gconstpointer s1,
                                         gconstpointer s2);
gint bwlist_compare_func                (gconstpointer s1, gconstpointer s2);
gint modelist_compare_func              (gconstpointer s1, gconstpointer s2);


void
show_message                           (gchar* in_title,
                                        gchar* in_message,
										gchar* in_type)
{
  GtkWidget *messagebox1;
  GtkWidget *dialog_vbox6;
  GtkWidget *button32;
  GtkWidget *dialog_action_area6;

  messagebox1 = gnome_message_box_new (in_message, in_type,
                              GNOME_STOCK_BUTTON_OK, NULL);
  gtk_container_remove (GTK_CONTAINER (GNOME_DIALOG (messagebox1)->action_area), GNOME_DIALOG (messagebox1)->buttons->data);
  GNOME_DIALOG (messagebox1)->buttons = NULL;
  gtk_widget_set_name (messagebox1, "messagebox1");
  gtk_object_set_data (GTK_OBJECT (messagebox1), "messagebox1", messagebox1);
  gtk_window_set_title (GTK_WINDOW (messagebox1), in_title);
  gtk_window_set_position (GTK_WINDOW (messagebox1), GTK_WIN_POS_CENTER);
  gtk_window_set_modal (GTK_WINDOW (messagebox1), TRUE);

  dialog_vbox6 = GNOME_DIALOG (messagebox1)->vbox;
  gtk_widget_set_name (dialog_vbox6, "dialog_vbox6");
  gtk_object_set_data (GTK_OBJECT (messagebox1), "dialog_vbox6", dialog_vbox6);
  gtk_widget_show (dialog_vbox6);

  gnome_dialog_append_button (GNOME_DIALOG (messagebox1), GNOME_STOCK_BUTTON_OK);
  button32 = GTK_WIDGET (g_list_last (GNOME_DIALOG (messagebox1)->buttons)->data);
  gtk_widget_set_name (button32, "button32");
  gtk_widget_ref (button32);
  gtk_object_set_data_full (GTK_OBJECT (messagebox1), "button32", button32,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (button32);
  GTK_WIDGET_SET_FLAGS (button32, GTK_CAN_DEFAULT);

  dialog_action_area6 = GNOME_DIALOG (messagebox1)->action_area;
  if (dialog_action_area6 != NULL)
    gtk_widget_set_name (dialog_action_area6, "dialog_action_area6");
  gtk_widget_ref (dialog_action_area6);
  gtk_object_set_data_full (GTK_OBJECT (messagebox1), "dialog_action_area6", dialog_action_area6,
                            (GtkDestroyNotify) gtk_widget_unref);
  
  gtk_widget_show(messagebox1);
}


void
show_yes_no_message                    (gchar* in_title,
                                        gchar* in_message,
										gchar* in_type,
										GtkSignalFunc	ok_function,
										GtkSignalFunc	cancel_function)
{
  GtkWidget *YesNoDlg;
  GtkWidget *y_nvbox;
  GtkWidget *y_n_OKBtn;
  GtkWidget *y_n_CancelBtn;
  GtkWidget *y_n_action_area;

  YesNoDlg = gnome_message_box_new (in_message, in_type,
                              GNOME_STOCK_BUTTON_OK, NULL);
  gtk_container_remove (GTK_CONTAINER (GNOME_DIALOG (YesNoDlg)->action_area), GNOME_DIALOG (YesNoDlg)->buttons->data);
  GNOME_DIALOG (YesNoDlg)->buttons = NULL;
  gtk_widget_set_name (YesNoDlg, "YesNoDlg");
  gtk_object_set_data (GTK_OBJECT (YesNoDlg), "YesNoDlg", YesNoDlg);
  gtk_window_set_title (GTK_WINDOW (YesNoDlg), in_title);
  gtk_window_set_position (GTK_WINDOW (YesNoDlg), GTK_WIN_POS_CENTER);
  gtk_window_set_modal (GTK_WINDOW (YesNoDlg), TRUE);

  y_nvbox = GNOME_DIALOG (YesNoDlg)->vbox;
  gtk_widget_set_name (y_nvbox, "y_nvbox");
  gtk_object_set_data (GTK_OBJECT (YesNoDlg), "y_nvbox", y_nvbox);
  gtk_widget_show (y_nvbox);

  gnome_dialog_append_button (GNOME_DIALOG (YesNoDlg), GNOME_STOCK_BUTTON_YES);
  y_n_OKBtn = GTK_WIDGET (g_list_last (GNOME_DIALOG (YesNoDlg)->buttons)->data);
  gtk_widget_set_name (y_n_OKBtn, "y_n_OKBtn");
  gtk_widget_ref (y_n_OKBtn);
  gtk_object_set_data_full (GTK_OBJECT (YesNoDlg), "y_n_OKBtn", y_n_OKBtn,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (y_n_OKBtn);
  GTK_WIDGET_SET_FLAGS (y_n_OKBtn, GTK_CAN_DEFAULT);
  if (ok_function != NULL)
  	gtk_signal_connect (GTK_OBJECT (y_n_OKBtn), "clicked", ok_function, NULL);

  gnome_dialog_append_button (GNOME_DIALOG (YesNoDlg), GNOME_STOCK_BUTTON_NO);
  y_n_CancelBtn = GTK_WIDGET (g_list_last (GNOME_DIALOG (YesNoDlg)->buttons)->data);
  gtk_widget_set_name (y_n_CancelBtn, "y_n_CancelBtn");
  gtk_widget_ref (y_n_CancelBtn);
  gtk_object_set_data_full (GTK_OBJECT (YesNoDlg), "y_n_CancelBtn", y_n_CancelBtn,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (y_n_CancelBtn);
  GTK_WIDGET_SET_FLAGS (y_n_CancelBtn, GTK_CAN_DEFAULT);
  if (cancel_function != NULL)
  	gtk_signal_connect (GTK_OBJECT (y_n_CancelBtn), "clicked", cancel_function, NULL);

  y_n_action_area = GNOME_DIALOG (YesNoDlg)->action_area;
  if (y_n_action_area != NULL)
    gtk_widget_set_name (y_n_action_area, "y_n_action_area");
  gtk_widget_ref (y_n_action_area);
  gtk_object_set_data_full (GTK_OBJECT (YesNoDlg), "y_n_action_area", y_n_action_area,
                            (GtkDestroyNotify) gtk_widget_unref);
  
  gtk_widget_show(YesNoDlg);
}

void
process_keys                           (GtkWidget*   in_widget,
                                        GdkEventKey* in_event)
{
	switch (in_event->keyval) {
		case GDK_BackSpace :
			set_focus(app1,"kpBS");
			set_tuner_display(11);
			break;
		case GDK_Left : ;
		case GDK_KP_Left :
			if ((in_event->state & GDK_CONTROL_MASK) == GDK_CONTROL_MASK) {
				set_focus(app1,"LLeftBtn");
				tune_up_down(-sml_step);
			}
			else {
				set_focus(app1,"LeftBtn");
				tune_up_down(-lrg_step);
			}
			break;
		case GDK_Right : ;
		case GDK_KP_Right :
			if ((in_event->state & GDK_CONTROL_MASK) == GDK_CONTROL_MASK) {
				set_focus(app1,"RRightBtn");
				tune_up_down(sml_step);
			}
			else {
				set_focus(app1,"RightBtn");
				tune_up_down(lrg_step);
			}
			break;
	}
}


gchar*
get_pathname_from_full_path            (gchar* in_path)
{
	gint	len;
	gchar*	ptr;
	gchar*	pathname = NULL;
	
	ptr = strrchr(in_path,'/');
	if (ptr) {
		len = ptr - in_path + 1;
		pathname = g_strndup(in_path,len);
	}
	else
		pathname = g_strdup("<blank>");
	return pathname;
}


gchar*
get_filename_from_full_path            (gchar* in_path)
{
	gchar*	ptr;
	gchar*	filename = NULL;
	
	ptr = strrchr(in_path,'/');
	if (ptr)
		filename = g_strdup(ptr+1);
	else
		filename = g_strdup("<blank>");
	return filename;
}


gchar*
get_extension_from_full_path           (gchar* in_path)
{
	gchar*	ptr;
	gchar*	extension = NULL;
	
	ptr = strrchr(in_path,'.');
	if (ptr)
		extension = g_strdup(ptr+1);
	else
		extension = g_strdup("<blank>");
	return extension;
}


/** widget control primitives **/
void
set_app_title()
{
	gchar* title_str = g_strdup_printf("Smart Gnome Control %s - [%d] %s %s - [%s]",VERSION,cur_rig_model,cur_rig_make,cur_rig_name,db_ext_name);
	gtk_window_set_title(GTK_WINDOW(app1),title_str);
}


void
update_immediate()
{
	while (gtk_events_pending())
		gtk_main_iteration();
}


void
set_widget_text_color                  (GtkWidget*      inWidget,
                                        gchar           *widget_item,
                                        gchar           *color_str)
{
  GtkRcStyle* rc_style;
  GdkColor text_color;
  GtkWidget* outWidget = lookup_widget(inWidget,widget_item);
  if (outWidget) {
	gdk_color_parse(color_str,&text_color);
	rc_style = gtk_rc_style_new();
	rc_style->fg[GTK_STATE_NORMAL] = text_color;
	rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_FG;
	gtk_widget_modify_style(outWidget,rc_style);
	gtk_rc_style_unref(rc_style);
  }
}


void
set_clist_row_color                    (GtkWidget*      inWidget,
                                        gchar           *widget_item,
                                        gchar           *color_str,
                                        int             in_row)
{
  GdkColor text_color;
  GtkWidget* outWidget = lookup_widget(inWidget,widget_item);
  if (outWidget) {
	gdk_color_parse(color_str,&text_color);
	gtk_clist_set_foreground(GTK_CLIST(outWidget),in_row,&text_color);
  }
}


void
set_clist_row_background               (GtkWidget*      inWidget,
                                        gchar           *widget_item,
                                        gchar           *color_str,
                                        int             in_row)
{
  GdkColor text_color;
  GtkWidget* outWidget = lookup_widget(inWidget,widget_item);
  if (outWidget) {
	gdk_color_parse(color_str,&text_color);
	gtk_clist_set_background(GTK_CLIST(outWidget),in_row,&text_color);
  }
}


void
set_widget_color                       (GtkWidget*      inWidget,
                                        gchar           *widget_item,
                                        gchar           *color_str)
{
  GtkRcStyle* rc_style;
  GdkColor bg_color;
  GtkWidget* outWidget = lookup_widget(inWidget,widget_item);
  if (outWidget) {
	gdk_color_parse(color_str,&bg_color);
	rc_style = gtk_rc_style_new();
	rc_style->bg[GTK_STATE_NORMAL] = bg_color;
	rc_style->bg_pixmap_name[GTK_STATE_NORMAL] = "<none>";
	rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_BG;
	gtk_widget_modify_style(outWidget,rc_style);
	gtk_rc_style_unref(rc_style);
  }
}


void
set_visible                            (GtkWidget*      inWidget,
                                        gchar           *widget_item,
                                        gboolean        status)
{
  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item) {
    if (status)
	  gtk_widget_show(gen_item);
    else
	  gtk_widget_hide(gen_item);
  }
}


gboolean
get_visible                            (GtkWidget*      inWidget,
                                        gchar           *widget_item)
{
  gboolean status=FALSE;
  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item)
    status = GTK_WIDGET_VISIBLE(gen_item);
  return status;
}


void
toggle_visible                         (GtkWidget       *item,
                                        gchar           *widget_item)
{
  GtkWidget* gen_item = lookup_widget(item,widget_item);
  if (gen_item) {
    if GTK_WIDGET_VISIBLE(gen_item)
	    gtk_widget_hide(gen_item);
    else
	    gtk_widget_show(gen_item);
  }
}


gboolean
get_focus                              (GtkWidget*      inWidget,
                                        gchar           *widget_item)
{
  GtkWidget*	gen_item;
  GtkArg		arg;
  gboolean		has_focus;

  has_focus = FALSE;	
  gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item) {
	arg.name = "has_focus";
	gtk_widget_get(gen_item, &arg);
	has_focus = GTK_VALUE_BOOL(arg);
  }
  return has_focus;
}


void
set_focus                              (GtkWidget*      inWidget,
                                        gchar           *widget_item)
{
  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item) {
	if (GTK_WIDGET_CAN_FOCUS(gen_item))
		gtk_widget_grab_focus(gen_item);
  }
}


void
set_enabled                            (GtkWidget*      inWidget,
                                        gchar           *widget_item,
                                        gboolean        status)
{
  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item)
    gtk_widget_set_sensitive(gen_item,status);
}


gboolean
get_enabled                            (GtkWidget*      inWidget,
                                        gchar           *widget_item)
{
  int status = FALSE;

  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item)
    status = GTK_WIDGET_SENSITIVE(gen_item);
  return status;
}


void
toggle_enabled                         (GtkMenuItem     *menuitem,
                                        gchar           *widget_item)
{
  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(menuitem),widget_item);
  if (gen_item) {
    if GTK_WIDGET_SENSITIVE(gen_item)
	    gtk_widget_set_sensitive(gen_item,FALSE);
    else
	    gtk_widget_set_sensitive(gen_item,TRUE);
  }
}


void
set_down                               (GtkWidget*      inWidget,
                                        gchar           *widget_item,
                                        gboolean        status)
{
  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gen_item),status);
}


gboolean
get_down                               (GtkWidget*      inWidget,
                                        gchar           *widget_item)
{
  int status = FALSE;

  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item)
    status = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gen_item));
  return status;
}


void
set_checked                            (GtkWidget*      inWidget,
                                        gchar           *widget_item,
                                        gboolean        status)
{
  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(gen_item),status);
}


void
set_scale_value                        (GtkWidget*      inWidget,
                                        gchar           *widget_item,
                                        gdouble         value)
{
  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item) {
	GtkAdjustment* adj = gtk_range_get_adjustment(GTK_RANGE(gen_item));
    gtk_adjustment_set_value(adj,(gfloat)value);
  }
}


void
set_scale_range                        (GtkWidget*      inWidget,
                                        gchar           *widget_item,
                                        gran_t          range,
                                        int             is_float)
{
  GtkAdjustment* adj;
  gfloat     min, max, step;
  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item) {
	if (is_float) {
		min = -range.max.f;
		max = -range.min.f;
		step = range.step.f;
	}
	else {
		min = -range.max.i;
		max = -range.min.i;
		step = range.step.i;
	}
	adj = gtk_range_get_adjustment(GTK_RANGE(gen_item));
	adj->value = 0;
	adj->lower = min;
	adj->upper = max;
	adj->step_increment = step;
	adj->page_increment = step*3;
	adj->page_size = 0;
  }
}


gdouble
get_scale_value                        (GtkWidget*      inWidget,
                                        gchar           *widget_item)
{
  gdouble value = 0.0;
  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item) {
	GtkAdjustment* adj = gtk_range_get_adjustment(GTK_RANGE(gen_item));
    value = adj->value;
  }
  return value;
}


void
set_toolbar_style                      (GtkWidget*      inWidget,
                                        gchar           *widget_item,
                                        GtkToolbarStyle style)
{
  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item)
	gtk_toolbar_set_style(GTK_TOOLBAR(gen_item),style);
}


void
add_menu_item                          (GtkWidget* opt_menu,
                                        gchar*     menu_str)
{
	GtkWidget* glade_menuitem = gtk_menu_item_new_with_label(menu_str);
	gtk_widget_show(glade_menuitem);
	gtk_menu_append(GTK_MENU (opt_menu), glade_menuitem);
}


void
set_option                             (GtkWidget*      inWidget,
                                        gchar           *widget_item,
                                        guint           item)
{
  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item)
    gtk_option_menu_set_history(GTK_OPTION_MENU(gen_item),item);
}


void
set_text                               (GtkWidget*      inWidget,
                                        gchar           *widget_item,
                                        gchar           *text)
{
  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item) {
	if (GTK_IS_LABEL(gen_item))
		gtk_label_set_text(GTK_LABEL(gen_item),text);
	else if (GTK_IS_ENTRY(gen_item))
		gtk_entry_set_text(GTK_ENTRY(gen_item),text);
	else if (GTK_IS_COMBO(gen_item))
		gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(gen_item)->entry),text);
  }
}


void
get_text                               (GtkWidget*      inWidget,
                                        gchar           *widget_item,
                                        gchar           **text)
{
  GtkWidget* gen_item = lookup_widget(GTK_WIDGET(inWidget),widget_item);
  if (gen_item) {
	if (GTK_IS_LABEL(gen_item))
		gtk_label_get(GTK_LABEL(gen_item),text);
	if (GTK_IS_ENTRY(gen_item))
		text[0] = gtk_entry_get_text(GTK_ENTRY(gen_item));
	if (GTK_IS_COMBO(gen_item))
		text[0] = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(gen_item)->entry));
  }
}


/** special widget control primitives **/
void
set_display_cursor                     (GtkWidget* inWidget,
                                        GdkCursorType inCursor)
{
  gchar*	w_name;
  GdkCursor *cursor;
  GdkWindow* digits_win = gtk_widget_get_parent_window(inWidget);
  GtkWidget* gen_item = lookup_widget(app1,"appbar1");

  if (ispower) {
	cursor = gdk_cursor_new (inCursor);
	gdk_window_set_cursor (digits_win, cursor);
	gdk_cursor_destroy (cursor);
	if (inCursor == GDK_HAND2) {
		w_name = strdup(inWidget->name);
		if (strcmp(w_name,"eventbox1") == 0)
			gnome_appbar_set_status(GNOME_APPBAR(gen_item),"Left mouse btn - tune down, right btn - tune up, middle btn - zero digit");
		else if (strcmp(w_name,"eventbox2") == 0)
			gnome_appbar_set_status(GNOME_APPBAR(gen_item),"Click to select memory channel");
		else if (strcmp(w_name,"eventbox3") == 0)
			gnome_appbar_set_status(GNOME_APPBAR(gen_item),"Click to display Band Plan");
		g_free(w_name);
	}
	else
		gnome_appbar_set_status(GNOME_APPBAR(gen_item),"");	
  }
}


void
set_display_mouse                      (GtkWidget* inWidget,
                                        GdkEventButton  *inEvent)
{
  guint power;
  freq_t freq, dfreq;
  gdouble inX;
  GtkArg arg;
  gchar* fStr;

  GdkWindow* digits_win = gtk_widget_get_parent_window(inWidget);
  gdk_input_window_get_pointer(digits_win, GDK_CORE_POINTER, &inX, NULL, NULL, NULL, NULL, NULL);
  arg.name = "x";
  gtk_widget_get(inWidget, &arg);
  gint inX2 = (gint) (inX - GTK_VALUE_INT(arg)) / dWidth;
  if (inX2 != 7) {
	if (inX2 > 7)
	  inX2--;
	freq = (frequency * 1000.0);
  	power = 9 - inX2;
  	dfreq = (guint64) pow (10, power);
  	switch (inEvent->button) {
		case 3:
		case 4: /* RIGHT button or wheel up: increase digit */
			freq += dfreq;
			if (freq <= maxf) {
				frequency = freq / 1000.0;
				set_freq(FALSE);
			}
			break;
		case 2: /*MIDDLE button: zero digit */
  			fStr = g_strdup_printf("%10d",(int)freq);
			fStr[inX2] = '0';
			freq = (freq_t)atol(fStr);
			if ((freq >= minf) && (freq <= maxf)) {
				frequency = freq / 1000.0;
				set_freq(FALSE);
			}
			g_free(fStr);
			break;
		case 1:
		case 5: /* LEFT button or wheel down: decrease digit */
			freq -= dfreq;
			if (freq >= minf) {
				frequency = freq / 1000.0;
				set_freq(FALSE);
			}
			break;
		default:
			break;
	}
  }
}


gint
get_list_num_rows                      (GtkWidget* inWidget,
                                        gchar*     list_name)
{
	GtkWidget*	gen_item;
	gint		row_nums;
	
	row_nums = -1;
	gen_item = lookup_widget(inWidget,list_name);
	if (gen_item)
		row_nums = GTK_CLIST(gen_item)->rows;
	return row_nums;
}


gint
get_list_selected_row                  (GtkWidget* inWidget,
                                        gchar*     list_name)
{
	GtkWidget*	gen_item;
	GList*		rec_list;
	gint		row_num;
	
	row_num = -1;
	gen_item = lookup_widget(inWidget,list_name);
	if (gen_item) {
		rec_list = GTK_CLIST(gen_item)->selection;
		if (rec_list) {
			row_num = GPOINTER_TO_INT(g_list_nth_data(rec_list,0));
		}
	}
	return row_num;
}


void
set_list_selected_row                  (GtkWidget* inWidget,
                                        gchar*     list_name,
                                        gint       in_row)
{
	GtkWidget*	gen_item;
	gboolean	has_focus;
	
	gen_item = lookup_widget(inWidget,list_name);
	if (gen_item) {
		has_focus = get_focus(inWidget,list_name);
		if (has_focus) {
			gtk_clist_freeze(GTK_CLIST(gen_item));
			gtk_clist_unselect_all(GTK_CLIST(gen_item));
		}
		gtk_clist_select_row(GTK_CLIST(gen_item),in_row,0);
		gtk_clist_moveto(GTK_CLIST(gen_item),in_row,0,1,1);
		if (has_focus) {
			GTK_CLIST(gen_item)->focus_row = in_row;
			gtk_clist_thaw(GTK_CLIST(gen_item));
		}
	}
}


void
get_list_clicked_row_col               (GtkWidget* inWidget,
                                        GdkEventButton  *inEvent,
                                        gint* in_row,
                                        gint* in_col)
{
  gdouble	inX, inY;
  GtkArg	args[2];

  if (inEvent->button == 1) {
	gdk_input_window_get_pointer(inEvent->window, inEvent->deviceid, &inX, &inY, NULL, NULL, NULL, NULL);
    args[0].name = "x";
	args[1].name = "y";
	gtk_widget_get(inWidget, &args[0]);
	gtk_widget_get(inWidget, &args[1]);
	inX -= GTK_VALUE_INT(args[0]);
	inY -= GTK_VALUE_INT(args[1]);
	gtk_clist_get_selection_info(GTK_CLIST(inWidget), (gint)inX, (gint)inY, in_row, in_col);
  }
}


gint
get_list_clicked_row                   (GtkWidget* inWidget,
                                        GdkEventButton  *inEvent)
{
  gdouble	inX, inY;
  gint		result = -1;
  gint		in_row;
  GtkArg	args[2];

  if (inEvent->button == 1) {
	gdk_input_window_get_pointer(inEvent->window, inEvent->deviceid, &inX, &inY, NULL, NULL, NULL, NULL);
    args[0].name = "x";
	args[1].name = "y";
	gtk_widget_get(inWidget, &args[0]);
	gtk_widget_get(inWidget, &args[1]);
	inX -= GTK_VALUE_INT(args[0]);
	inY -= GTK_VALUE_INT(args[1]);
	if (gtk_clist_get_selection_info(GTK_CLIST(inWidget), (gint)inX, (gint)inY, &in_row, NULL))
		result = in_row;
  }
  return result;
}


gpointer
get_list_row_data                      (GtkWidget* inWidget,
                                        gchar*     list_name,
                                        gint       in_row)
{
	GtkWidget*	gen_item;
	gpointer	result;
	
	gen_item = lookup_widget(inWidget,list_name);
	if (gen_item)
		result = gtk_clist_get_row_data(GTK_CLIST(gen_item),in_row);
	else
		result = NULL;
	return result;
}


gint
get_notebook_page                      (GtkWidget* inWidget,
                                        gchar*     notebook)
{
	GtkWidget*	gen_item;
	gint		page_num;
	
	page_num = -1;
	gen_item = lookup_widget(inWidget, notebook);
	if (gen_item)
		page_num = gtk_notebook_get_current_page(GTK_NOTEBOOK(gen_item));
	return page_num;
}


void
updateUTC                              (GtkWidget* inWidget)
{
  time_t    s_time;
  struct tm g_time;

  time(&s_time);
  gmtime_r(&s_time,&g_time);
  gchar* gmt_time = g_strdup_printf("%02d:%02d",g_time.tm_hour,g_time.tm_min);
  set_text(inWidget,"UTC2Lbl",gmt_time);
  g_free(gmt_time);
}


/* lookup list functions */
void
do_Lookup_clicked                      (GtkWidget       *in_widget,
                                        GdkEventButton  *in_event)
{
  gint	in_row = -1;
  gint	in_col = -1;
  gchar	*text;

  get_list_clicked_row_col(in_widget,in_event,&in_row,&in_col);
  if ((in_widget) && (in_row > -1)) {
	switch (in_col) {
		case 0 : ;
		case 1 : gtk_clist_get_text(GTK_CLIST(in_widget), in_row, 2, &text);
			 	 show_curr_freq(text);
			 	 break;
		case 2 : ;
		case 3 : if ((log_name) && (ispower))
				     show_log_entry(1,TRUE,in_row);
			     break;
	}
  }
}


/* band plan functions */
void
fill_band_display()
{
  gchar *buf, **text;
  guint i;

  GtkWidget* bc_list = lookup_widget(app1,"BCList");
  GtkWidget* ute_list = lookup_widget(app1,"UteList");

  if (bc_list && ute_list) {
	GtkWidget* c_list = bc_list;
	for (i=0; i<90; i++) {
		buf = g_strdup_printf("%s;%5.0f;%5.0f;%s", plan_text[i].label,
	    	                                       plan_text[i].start,
	        	                                   plan_text[i].end,
	            	                               plan_text[i].type);
		text = g_strsplit(buf, ";", 4);
		switch(plan_text[i].type[0]) {
			case 'A' :
			case 'B' :
			case 'C' : c_list = bc_list; break;
			case 'E' :
			case 'F' :
			case 'M' :
			case 'N' :
			case 'T' :
			case 'R' : c_list = ute_list; break;
			default : break;
		}
		gtk_clist_append(GTK_CLIST(c_list),text);
		gtk_clist_set_row_data(GTK_CLIST(c_list), i, GUINT_TO_POINTER(i));
		g_free(buf);
		g_strfreev(text);
  	}
	gtk_clist_select_row(GTK_CLIST(bc_list), 0, 0);
	gtk_clist_select_row(GTK_CLIST(ute_list), 0, 0);
  }
}


void
do_BandPlan_clicked                    (GtkWidget       *widget,
                                        GdkEventButton  *event)
{
  gint in_row = -1;
  gint in_col = -1;

  get_list_clicked_row_col(widget,event,&in_row,&in_col);
  tune_from_grid(widget,in_row,in_col);
}


double
get_grid_freq                          (GtkWidget* in_widget,
                                        gint  in_row,
                                        gint  in_col)
{
  gchar		*text;
  double	result = 0;

  if ((in_row > -1) && (in_col > 0) && (in_col < 3)) {
	if (in_widget) {
		gtk_clist_get_text(GTK_CLIST(in_widget), in_row, in_col, &text);
		result = strtod(text,NULL);
	}
  }
  return result;
}


void
tune_from_grid                         (GtkWidget* in_widget,
                                        gint  in_row,
                                        gint  in_col)
{
  double	r_freq;

  r_freq = get_grid_freq(in_widget,in_row,in_col);
  if ((r_freq >= minf/1000)&&(r_freq <= maxf/1000)) {
	frequency = r_freq;
	set_freq(FALSE);
  }
}


/* frequency history functions */
double
get_history_freq                       (gint  in_row)
{
  gchar		*text;
  double	result = 0;

  if (in_row > -1) {
	GtkWidget* gen_item = lookup_widget(GTK_WIDGET(app1),"HistList");
	if (gen_item) {
		gtk_clist_get_text(GTK_CLIST(gen_item), in_row, 1, &text);
		result = strtod(text,NULL);
	}
  }
  return result;
}


void
tune_from_history                      (gint  in_row,
                                        gint  set_row)
{
  double	r_freq;

  r_freq = get_history_freq(in_row);
  if ((r_freq >= minf/1000)&&(r_freq <= maxf/1000)) {
	if (set_row)
		set_list_selected_row(app1,"HistList",in_row);
	else
		hist_index = in_row;
	set_history_buttons();
	frequency = r_freq;
	set_freq(FALSE);
  }
}


void
add_to_history()
{
	int		i;
	struct	tm g_time;
	time_t	s_time;
	gchar	*buf, **text;

	if (get_history_freq(hist_index)!=frequency) {
		GtkWidget*	hist_list = lookup_widget(app1,"HistList");
		if (hist_list) {
			time(&s_time);
			gmtime_r(&s_time,&g_time);
			buf = g_strdup_printf("%02d:%02d:%02d;%11.3f",g_time.tm_hour,
														g_time.tm_min,
														g_time.tm_sec,
														frequency);
			text = g_strsplit(buf, ";", 2);
			hist_index++;
			if (hist_index == hist_total) {
				hist_total++;
				gtk_clist_append(GTK_CLIST(hist_list), text);
			}
			else {
				gtk_clist_insert(GTK_CLIST(hist_list),hist_index, text);
				for (i = hist_total;i > hist_index; --i)
					gtk_clist_remove(GTK_CLIST(hist_list),i);
				hist_total = hist_index+1;
			}
			gtk_clist_set_row_data(GTK_CLIST(hist_list), hist_index, GUINT_TO_POINTER(hist_index));
			g_free(buf);
			g_strfreev(text);
			if (hist_total > 200) {
				gtk_clist_remove(GTK_CLIST(hist_list),0);
				hist_index -= 1;
				hist_total -= 1;
			}
			set_list_selected_row(app1,"HistList",hist_index);
		}
		set_history_buttons();
	}
}


void
set_history_buttons()
{
  set_enabled(app1,"HistBackBtn",ispower && (hist_index > 0));
  set_enabled(app1,"HistForwBtn",ispower && (hist_index < (hist_total-1)));
}


void
tune_from_hist_entry()
{
  gchar*	text;
  gdouble	r_freq;

  get_text(app1,"HistEntry",&text);
  g_strstrip(text);
  if (strcmp(text,"") != 0) {
	r_freq = g_strtod(text,NULL);
	if ((r_freq >= (minf/1000000.0)) && (r_freq <= (maxf/1000000.0)))
		r_freq = r_freq * 1000.0;
	if ((r_freq >= minf/1000.0) && (r_freq <= maxf/1000.0))
		frequency = r_freq;
	set_text(app1,"HistEntry","");
	set_freq(TRUE);
  }
}


void
get_all                                (int do_history)
{
  get_power();
  get_freq();
  if (do_history && ispower)
	add_to_history();
  get_vfo();
  get_ant();
  get_agc();
  get_mode_bw();
  get_blanker();
  get_dnr();
  get_atten();
  get_preamp();
  get_notch();
  get_anf();
  get_af();
  get_rf();
  get_if();
  get_pi();
  get_po();
  get_cp();
  get_sq();
  get_no();
  get_nr();
  get_ap();
  /*get_to();*/
  get_bwc();
  get_mem();
  set_scan_display();
  set_timer_display();
}


/** power control interface **/
void
check_power()
{
	has_power = (hrig->caps->get_powerstat != NULL);
	set_enabled(app1,"PwrBtn",has_power);
	if (!has_power)
		ispower = TRUE;
}


void
set_power()
{
  powerstat_t	status;

  ispower = !ispower;
  if (ispower)
	status = 1;
  else
	status = 0;
  rig_set_powerstat(hrig, status);
  get_all(TRUE);
}


void
get_power()
{
  gint			retcode;
  powerstat_t	pwr1;

  if (has_power) {
	retcode = rig_get_powerstat(hrig, &pwr1);
	usleep(1000);
	retcode = rig_get_powerstat(hrig, &pwr1);
	if (retcode == RIG_OK) {
		switch (pwr1) {
			case 0 : ispower = FALSE; break;
			case 1 : ispower = TRUE; break;
			default : break;
		}
	}
	else {
		ispower = FALSE;
		g_warning(_("Hamlib error: Could not get power status."));
	}
  }
  else {
	ispower = TRUE;
  }
  set_power_display();
}


void
set_power_display()
{
  set_history_buttons();
  set_enabled(app1,"KeyPad",ispower);
  set_enabled(app1,"UpDownPad",ispower);
}


/** tuning controls interface **/
void
set_tuner_display                      (guint key_code)
{
  double	r_freq;
  int		len;
  gchar*	o_text;

  if (ispower) {
	if (tune_ok) {
		switch (key_code) {
			case 0 :
			case 1 :
			case 2 :
			case 3 :
			case 4 :
			case 5 :
			case 6 :
			case 7 :
			case 8 :
			case 9 :	e_text = g_strdup_printf("%d",key_code);
						tune_ok = FALSE;
						break;
			case 10 : 	e_text = g_strdup("0.");
						tune_ok = FALSE;
						break;
			default : 	break;
		}
	}
	else {
		if (e_text) {
			int len = strlen(e_text);
    	    int dot_pos = strcspn(e_text,".");
			switch (key_code) {
				case 0 :
				case 1 :
				case 2 :
				case 3 :
				case 4 :
				case 5 :
				case 6 :
				case 7 :
				case 8 :
				case 9 :	e_text = g_strdup_printf("%s%d",e_text,key_code);
							break;
            	case 10 :	if (dot_pos == len)
								e_text = g_strdup_printf("%s%c",e_text,'.');
							break;
				case 11 :	if (len > 0)
								e_text = g_strndup(e_text,len-1);
							if (len <= 1) {
								tune_ok = TRUE;
							}
							break;
				case 13 :	if (len > 0) {
								r_freq = strtod(e_text,NULL);
								if ((r_freq >= (minf/1000000.0)) && (r_freq <= (maxf/1000000.0)))
									r_freq = r_freq * 1000.0;
								if ((r_freq >= minf/1000.0) && (r_freq <= maxf/1000.0)) {
									frequency = r_freq;
									tune_ok = TRUE;
								}
							}
							break;
				default :	break;
			}
			if (tune_ok) {
				g_free(e_text);
				set_freq(TRUE);
			}
		}
	}
	if (!tune_ok) {
		len = strlen(e_text);
		o_text = g_strndup("           ",(11-len));
		o_text = g_strdup_printf("%s%s",o_text,e_text);
		set_freq_display_text(app1,o_text);
	}
  }
}


void
set_tuning_help                        (double amount)
{
  GtkWidget*	gen_item;
  gchar*		gen_dir;
  gchar*		gen_amt;

  gen_item = lookup_widget(app1,"appbar1");
  if (amount < 0.00)
	gen_dir = g_strdup("down");
  else
	gen_dir = g_strdup("up");
  gen_amt = g_strdup_printf("Left mouse btn - tune %s %7.2f kHz, right btn - set stepsize",
  	gen_dir,fabs(amount));
  if (amount != 0.00)
  	gnome_appbar_set_status(GNOME_APPBAR(gen_item),gen_amt);
  else
	gnome_appbar_set_status(GNOME_APPBAR(gen_item),"");
  g_free(gen_dir);
  g_free(gen_amt);
}


void
handle_tuning                          (GtkWidget       *in_widget,
                                        GdkEventButton  *in_event,
                                        double          amount)
{
  GtkWidget*		gen_menu;
	
  switch (in_event->button) {
	case 1: /* LEFT button - step up or down */
			tune_up_down(amount);
			break;
	case 3: /* RIGHT button - show menu */
			if (lrg_step == fabs(amount))
				gen_menu = b_popup;
			else
				gen_menu = l_popup;
			if (gen_menu)
				gtk_menu_popup(GTK_MENU(gen_menu),NULL,NULL,NULL,NULL,
				in_event->button,in_event->time);
			break;
	default:
			break;
  }
}


void
tune_up_down                           (double amount)
{
	gint	retcode;

	if (!ismem) {
		frequency += amount;
		set_freq(fabs(amount) != sml_step);
	}
	else {
		if (amount > 0) {
			memch++;
			if ((min_mem_edge != -1) && (max_mem_edge != -1)) {
				if (memch == (max_mem_edge+1))
					memch = min_mem_ch;
			}
			if (memch == (max_mem_ch+1)) {
				if ((min_mem_edge != -1) && (max_mem_edge != -1))
					memch = min_mem_edge;
				else
					memch = min_mem_ch;
			}
		}
		else {
			memch--;
			if (memch == (min_mem_ch-1)) {
				if ((min_mem_edge != -1) && (max_mem_edge != -1))
					memch = max_mem_edge;
				else
					memch = max_mem_ch;
			}
			if ((min_mem_edge != -1) && (max_mem_edge != -1)) {
				if (memch == (min_mem_edge-1))
					memch = max_mem_ch;
			}
		}

		retcode = rig_set_mem(hrig, RIG_VFO_CURR, memch);
		if (retcode != RIG_OK)
			g_warning(_("Hamlib error: Could not set memory channel."));
		get_all(FALSE);

		set_list_selected_row(mem1,"MemList",memch-min_mem_ch); 
	}
}


void
mem_up_down                            (double amount)
{
	if (!ismem)
		set_mem();
	tune_up_down(amount);
}


/** frequency control interface **/
void
check_freq()
{
	if (hrig->state.rx_range_list[0].end != 0) {
		minf = hrig->state.rx_range_list[0].start;
		maxf = hrig->state.rx_range_list[0].end;
	}
	else {
		minf=0;
		maxf=30000000;
	}
	has_get_freq = (hrig->caps->get_freq != NULL);
}


void
set_freq                               (int	do_history)
{
  if (frequency < (minf/1000.0))
	frequency = (maxf/1000.0);
  if (frequency > (maxf/1000.0))
	frequency = (minf/1000.0);
  freq_t freq1 = frequency * 1000;
  rig_set_freq(hrig, RIG_VFO_CURR, freq1);
  if (do_history)
	add_to_history();
  get_freq();
  get_mode_bw();
  if (get_down(tun1,"FreqLockBtn"))
	  refresh_who_list(NULL);
}


void
get_freq()
{
  gint			retcode;
  freq_t		freq1;

  if (has_get_freq) {
	if (ispower) {
		retcode = rig_get_freq(hrig, RIG_VFO_CURR, &freq1);
		if (retcode != RIG_OK) {
			freq1 = 100000;
			g_warning(_("Hamlib error: Could not get frequency."));
		}
		else
			frequency = freq1/1000.0;
	}
	else
		frequency = 0.0;
  }
  set_freq_display();
}


gchar*
get_meterband()
{
	int		i;
	gchar*	result;

	if (!ispower)
		result = g_strdup("    ");
	else {
		if (frequency > 30000.0)
			result = g_strdup(" VHF");
		else {
			result = g_strdup(" SW ");
			for (i=89;i>=0;i--) {
				if ((frequency >= plan_text[i].start)&&(frequency <= plan_text[i].end)) {
					g_free(result);
					result = g_strdup(plan_text[i].label);
					break;
				}
			}
		}
	}
	return result;
}


void
set_freq_display_value                 (GtkWidget* inWidget,
                                        double freq_val)
{
  gchar* freqstr=NULL;
  gchar* mb_str;

  if (ispower) {
	freqstr = g_strdup_printf("%11.3f",freq_val);
	mb_str = get_meterband();
  }
  else {
	freqstr = g_strdup("           ");
	mb_str = g_strdup("    ");
  }
  set_text(inWidget,"MeterLbl",mb_str);
  g_free(mb_str);
  set_freq_display_text(inWidget,freqstr);
}

void
set_freq_display_text                  (GtkWidget* inWidget,
                                        gchar* freq_str)
{
  guint			i;
  gint			src_x;
  gint			dest_x;
  GtkPixmap*	outWidget;
  GtkPixmap*	inDigits;
  GdkGC*		digit_gc;
  GdkWindow*	digits_win;
  GdkPixmap*	digit_pix;
  static GdkPixmap*	digits_pix;
	
  src_x = 0;
  dest_x = 0;
  outWidget = (GtkPixmap*) lookup_widget(GTK_WIDGET(inWidget),"freq_display");
  inDigits = (GtkPixmap*) lookup_widget(GTK_WIDGET(inWidget),"digits_pix");

  if (outWidget && inDigits) {
    digits_win = gtk_widget_get_parent_window(GTK_WIDGET(outWidget));
    digit_gc = gdk_gc_new(digits_win);
    digit_pix = gdk_pixmap_new(digits_win,(11*dWidth),dHeight,-1);
	if (!digits_pix) {
		digits_pix = gdk_pixmap_new(digits_win,(12*dWidth),dHeight,-1);
		gtk_pixmap_get(inDigits,&digits_pix,NULL);
	}

    for(i=0;i<11;i++) {
      switch(freq_str[i]){
	    case '0' : src_x = 0*dWidth; break;
	    case '1' : src_x = 1*dWidth; break;
  	    case '2' : src_x = 2*dWidth; break;
  	    case '3' : src_x = 3*dWidth; break;
	    case '4' : src_x = 4*dWidth; break;
	    case '5' : src_x = 5*dWidth; break;
	    case '6' : src_x = 6*dWidth; break;
	    case '7' : src_x = 7*dWidth; break;
	    case '8' : src_x = 8*dWidth; break;
	    case '9' : src_x = 9*dWidth; break;
	    case ' ' : src_x = 10*dWidth; break;
	    case '.' : src_x = 11*dWidth; break;
        default : break;
	  }
	  dest_x = i*dWidth;
      gdk_draw_pixmap(digit_pix, digit_gc, digits_pix, src_x, 0, dest_x, 0, dWidth, (dHeight+1));
    }
	g_free(freq_str);
    gtk_pixmap_set(outWidget,digit_pix,NULL);
	gdk_pixmap_unref(digit_pix);
	gdk_gc_unref(digit_gc);
  }
  update_immediate();
}


void
set_freq_display()
{
  set_visible(app1,"MeterLbl",ispower);
  set_visible(app1,"HzLbl",ispower);
  set_freq_display_value(app1,frequency);
  update_freq_db_display();
}


void
update_freq_db_display()
{
  GtkWidget* gen_item;
	
  gen_item = lookup_widget(GTK_WIDGET(app1),"clist1");
  if ((ispower) && (num_freqs > 0)) {
	if (gen_item)
		get_freq_records(GTK_CLIST(gen_item), frequency);
  }
  else {
	if (gen_item)
		gtk_clist_clear(GTK_CLIST(gen_item));
  }
}


/** meter control interface **/
void
check_meter()
{
	has_meter = (rig_has_get_level(hrig,RIG_LEVEL_STRENGTH));
	set_visible(app1,"meter",has_meter);
}

gboolean
get_meter()
{
  gint			retcode;
  value_t		val;

  if (ispower) {
	if (has_meter) {
		retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_STRENGTH, &val);
		if (retcode != RIG_OK) {
			val.i = 0;
			meter_level = 0;
			g_warning(_("Hamlib error: Could not get meter."));
		}
    	else {
			if (val.i <= 0) { //convert to S level
				meter_level = 9 + (val.i / 6);
				if (meter_level <= 0)
					meter_level = 1;
			}
			else { //already at S level use nearest mult of 10
				meter_level = (val.i / 10) * 10;
				if (meter_level == 0)
					meter_level = 9;
			}
		}
	}
	else
		meter_level = 0;
  }
  else
	  meter_level = 0;
  set_meter_display();
  
  return TRUE;
}


void
set_meter_display_value                (GtkWidget* inWidget)
{
  gint			src_x;
  GtkPixmap*	outWidget;
  GtkPixmap*	inMeters;
  GdkGC*		meter_gc;
  GdkWindow*	meters_win;
  GdkPixmap*	meter_pix;
  static GdkPixmap*	meters_pix;
	
  src_x = 0;
  outWidget = (GtkPixmap*) lookup_widget(GTK_WIDGET(inWidget),"meter");
  inMeters = (GtkPixmap*) lookup_widget(GTK_WIDGET(inWidget),"meters_pix");

  if (outWidget && inMeters) {
    meters_win = gtk_widget_get_parent_window(GTK_WIDGET(outWidget));
    meter_gc = gdk_gc_new(meters_win);
    meter_pix = gdk_pixmap_new(meters_win,mWidth,mHeight,-1);
	if (!meters_pix) {
	    meters_pix = gdk_pixmap_new(meters_win,(16*mWidth),mHeight,-1);
    	gtk_pixmap_get(inMeters,&meters_pix,NULL);
	}

    switch(meter_level){
		case 1 :
	    case 2 :
  	    case 3 :
  	    case 4 :
	    case 5 :
	    case 6 :
	    case 7 :
	    case 8 :
	    case 9 :
	    case 10 : src_x = meter_level*mWidth; break;
	    case 20 : src_x = 11*mWidth; break;
	    case 30 : src_x = 12*mWidth; break;
	    case 40 : src_x = 13*mWidth; break;
	    case 50 : src_x = 14*mWidth; break;
	    case 60 : src_x = 15*mWidth; break;
        default : break;
	}
    gdk_draw_pixmap(meter_pix, meter_gc, meters_pix, src_x, 0, 0, 0, mWidth, (mHeight+1));
    gtk_pixmap_set(outWidget,meter_pix,NULL);
	gdk_pixmap_unref(meter_pix);
	gdk_gc_unref(meter_gc);
  }
  update_immediate();
}


void
set_meter_display()
{
  set_visible(app1,"meter",has_meter);
  set_meter_display_value(app1);
}


/** vfo control interface **/
void
check_vfo()
{
	GtkWidget*	opt_menu;
	GtkWidget*	opt_menu_menu;
	gchar*		m_item;
	int			i;

	has_get_vfo = (hrig->caps->get_vfo != NULL);
	has_set_vfo = (hrig->caps->set_vfo != NULL);
	int num_vfos = hrig->state.vfo_list;
	num_vfos &= 0xf;
	has_vfo = (num_vfos > 1);
	set_enabled(app1,"VFOBox",(has_vfo && has_set_vfo));
	opt_menu = lookup_widget(app1,"VFOBox");
	if (opt_menu) {
		gtk_option_menu_remove_menu(GTK_OPTION_MENU(opt_menu));
		opt_menu_menu = gtk_menu_new ();
		add_menu_item(opt_menu_menu," ");
		for (i=0; i<8; i++) {
			int vfo_num = num_vfos;
			vfo_num &= (1<<i);
			if (vfo_num > 0) {
				m_item = g_strdup_printf("%c",(0x41+i));
				add_menu_item(opt_menu_menu,m_item);
				g_free(m_item);
			}
		}
		gtk_option_menu_set_menu(GTK_OPTION_MENU(opt_menu),opt_menu_menu);
  		gtk_signal_connect(GTK_OBJECT(GTK_OPTION_MENU(opt_menu)->menu),
  						   "deactivate", GTK_SIGNAL_FUNC(on_vfo_selected),
  						   NULL);
	}
}


void
on_vfo_selected                        (GtkMenuShell    *menu_shell,
                                        gpointer        data)
{
  GtkWidget* active_item = gtk_menu_get_active(GTK_MENU(menu_shell));
  gint item_index = g_list_index(menu_shell->children, active_item);
  if (item_index != 0)
	set_vfo(item_index);
}


void
set_vfo                                (gint	item_index)
{
  vfo_t		vfo1 = 0;

  if (has_set_vfo) {
	vfo = item_index;
	switch (vfo) {
		case 1 : vfo1 = RIG_VFO_A; break;
		case 2 : vfo1 = RIG_VFO_B; break;
		case 3 : vfo1 = RIG_VFO_C; break;
		default : break;
	}
	rig_set_vfo(hrig, vfo1);
	get_all(FALSE);
  }
}


void
get_vfo()
{
  gint			retcode;
  vfo_t			vfo1;

  if (has_get_vfo) {
	if (ispower) {
		if (has_vfo) {
			retcode = rig_get_vfo(hrig, &vfo1);
			if (retcode != RIG_OK) {
				vfo1 = RIG_VFO_A;
				g_warning(_("Hamlib error: Could not get vfo."));
			}
			else {
				switch (vfo1 & 0x03) {
					case RIG_VFO_NONE : vfo = 0; break;
					case RIG_VFO_A    : vfo = 1; break;
					case RIG_VFO_B    : vfo = 2; break;
					case RIG_VFO_C    : vfo = 3; break;
					default : break;
				}
			}
			ismem = ((vfo1 & RIG_VFO_MEM)==RIG_VFO_MEM);
		}
		else
			vfo = 1;
	}
	else
		vfo = 0;
  }
  else
	  vfo = RIG_VFO_CURR;
  set_vfo_display();
  set_mem_display();
}


void
set_vfo_display()
{
  set_visible(app1,"VFOLbl",ispower);
  set_option(app1,"VFOBox", ispower ? vfo : 0);
  set_enabled(app1,"VFOBox",(ispower&&has_vfo&&has_set_vfo));
}

/** antenna control interface **/
void
check_ant()
{
	GtkWidget*	opt_menu;

	has_get_ant = (hrig->caps->get_ant != NULL);
	has_set_ant = (hrig->caps->set_ant != NULL);
	opt_menu = lookup_widget(app1,"AntBox");
	if (opt_menu) {
		gtk_signal_connect(GTK_OBJECT(GTK_OPTION_MENU(opt_menu)->menu),
        	               "deactivate", GTK_SIGNAL_FUNC(on_ant_selected),
            	           NULL);
	}
}


void
on_ant_selected                        (GtkMenuShell    *menu_shell,
                                        gpointer        data)
{
  GtkWidget* active_item = gtk_menu_get_active(GTK_MENU(menu_shell));
  gint item_index = g_list_index(menu_shell->children, active_item);
  if (item_index != 0)
	set_ant(item_index);
}


void
set_ant                                (gint	item_index)
{
  ant_t		ant1 = 0;

  if (has_set_ant) {
	ant = item_index;
	switch (ant) {
		case 1 : ant1 = RIG_ANT_1; break;
		case 2 : ant1 = RIG_ANT_2; break;
		case 3 : ant1 = RIG_ANT_3; break;
		default : break;
	}
	rig_set_ant(hrig, RIG_VFO_CURR, ant1);
	get_ant();
  }
}


void
get_ant()
{
  gint			retcode;
  ant_t			ant1;

  if (ispower) {
	if (has_get_ant) {
		retcode = rig_get_ant(hrig, RIG_VFO_CURR, &ant1);
		if (retcode != RIG_OK) {
			ant1 = RIG_ANT_1;
			g_warning(_("Hamlib error: Could not get antenna."));
		}
		else {
			switch (ant1) {
				case RIG_ANT_NONE : ant = 0; break;
				case RIG_ANT_1    : ant = 1; break;
				case RIG_ANT_2    : ant = 2; break;
				case RIG_ANT_3    : ant = 3; break;
				default : break;
			}
		}
	}
	else
		ant = 1;
  }
  else
	  ant = 0;
  set_ant_display();
}


void
set_ant_display()
{
  set_visible(app1,"AntLbl",ispower);
  set_option(app1,"AntBox", ispower ? ant : 0);
  set_enabled(app1,"AntBox",ispower&&has_set_ant);
}


/** agc control interface **/
void
check_agc()
{
	GtkWidget*	opt_menu;

	has_get_agc = rig_has_get_level(hrig, RIG_LEVEL_AGC);
	has_set_agc = rig_has_set_level(hrig, RIG_LEVEL_AGC);
	opt_menu = lookup_widget(app1,"AGCBox");
	if (opt_menu) {
		gtk_signal_connect(GTK_OBJECT(GTK_OPTION_MENU(opt_menu)->menu),
        	               "deactivate", GTK_SIGNAL_FUNC(on_agc_selected),
            	           NULL);
	}
}


void
on_agc_selected                        (GtkMenuShell    *menu_shell,
                                        gpointer        data)
{
  GtkWidget* active_item = gtk_menu_get_active(GTK_MENU(menu_shell));
  gint item_index = g_list_index(menu_shell->children, active_item);
  if (item_index != 0)
	set_agc(item_index);
}


void
set_agc                                (gint	item_index)
{
  value_t	agc1;

  if (has_set_agc) {
	agc = item_index;
	switch (agc) {
		case 1 : agc1.i = RIG_AGC_OFF; break;
		case 2 : agc1.i = RIG_AGC_SLOW; break;
		case 3 : agc1.i = RIG_AGC_MEDIUM; break;
		case 4 : agc1.i = RIG_AGC_FAST; break;
		case 5 : agc1.i = RIG_AGC_SUPERFAST; break;
		default : break;
	}
	rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_AGC, agc1);
	get_agc();
  }
}


void
get_agc()
{
  gint		retcode;
  value_t	val;

  if (ispower && has_get_agc) {
	retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_AGC, &val);
	if (retcode != RIG_OK) {
		val.i = RIG_AGC_OFF;
		g_warning(_("Hamlib error: Could not get AGC."));
	}
   	else {
		switch (val.i) {
			case RIG_AGC_OFF       : agc = 1; break;
			case RIG_AGC_SLOW      : agc = 2; break;
			case RIG_AGC_MEDIUM    : agc = 3; break;
			case RIG_AGC_FAST      : agc = 4; break;
			case RIG_AGC_SUPERFAST : agc = 5; break;
			default : break;
		}
	}
  }
  else
	agc = 0;
  set_agc_display();
}


void
set_agc_display()
{
  set_visible(app1,"AGCLbl",ispower);
  set_option(app1,"AGCBox", ispower ? agc : 0);
  set_enabled(app1,"AGCBox",ispower&&has_set_agc);
}


/** mode and bandwidth controls interface **/
gint
bwlist_compare_func                    (gconstpointer s1,
                                        gconstpointer s2)
{
  double num1 = GPOINTER_TO_INT(s1)/1000.0;
  double num2 = GPOINTER_TO_INT(s2)/1000.0;
  gchar *d1;
  gchar *d2;
  gint result;

  d1 = g_strdup_printf("%4.1f",num1);
  d2 = g_strdup_printf("%4.1f",num2);
  result = g_strcasecmp(d2, d1);
  g_free(d1);
  g_free(d2);

  return result;
}


void
check_bw()
{
	GtkWidget*	opt_menu;
	GtkWidget*	opt_menu_menu;
	gchar*		m_item;
	int			i=0;

	has_get_mode = (hrig->caps->get_mode != NULL);
	has_set_mode = (hrig->caps->set_mode != NULL);
	if (bw_list != NULL) {
		g_list_free(bw_list);
		bw_list = NULL;
	}
	while (hrig->state.filters[i].width != 0) {
		gint pbw = hrig->state.filters[i].width;
		if ((pbw > 0) && (g_list_find(bw_list,GINT_TO_POINTER(pbw))==NULL))
			bw_list = g_list_insert_sorted(bw_list,GINT_TO_POINTER(pbw),
										  (GCompareFunc)bwlist_compare_func);
		i++;
	}
	opt_menu = lookup_widget(app1,"BWBox");
	if (opt_menu) {
		gtk_option_menu_remove_menu(GTK_OPTION_MENU(opt_menu));
		opt_menu_menu = gtk_menu_new ();
		add_menu_item(opt_menu_menu," ");
		for (i=0; i < g_list_length(bw_list); i++){
			double pbwd = GPOINTER_TO_INT(g_list_nth_data(bw_list,i))/1000.0;
			m_item = g_strdup_printf("%4.1f",pbwd);
			add_menu_item(opt_menu_menu,m_item);
			g_free(m_item);
		}
		gtk_option_menu_set_menu(GTK_OPTION_MENU(opt_menu),opt_menu_menu);
  		gtk_signal_connect(GTK_OBJECT(GTK_OPTION_MENU(opt_menu)->menu),
  						   "deactivate", GTK_SIGNAL_FUNC(on_bw_selected),
  						   NULL);
	}
}


void
on_bw_selected                        (GtkMenuShell    *menu_shell,
                                       gpointer        data)
{
  GtkWidget* active_item = gtk_menu_get_active(GTK_MENU(menu_shell));
  gint item_index = g_list_index(menu_shell->children, active_item);
  if (item_index != 0)
	set_bw(item_index);
}


void
set_bw                                 (gint	item_index)
{
  pbwidth_t		width1 = 0;
  rmode_t		mode1;

  if (has_set_mode) {
	bw = item_index;

	mode1 = GPOINTER_TO_INT(g_list_nth_data(mode_list,mode-1));
	width1 = GPOINTER_TO_INT(g_list_nth_data(bw_list,bw-1));

	rig_set_mode(hrig, RIG_VFO_CURR, mode1, width1);
	get_mode_bw();
	get_atten();
	get_preamp();
  }
}


void
set_bw_display()
{
  set_visible(app1,"BWLbl",ispower);
  set_option(app1,"BWBox", ispower ? bw : 0);
  set_enabled(app1,"BWBox",ispower&&has_set_mode);
}


gint
modelist_compare_func                  (gconstpointer s1,
                                        gconstpointer s2)
{
  double num1 = GPOINTER_TO_INT(s1);
  double num2 = GPOINTER_TO_INT(s2);
  gchar *d1;
  gchar *d2;
  gint result;

  d1 = g_strdup_printf("%16.0f",num1);
  d2 = g_strdup_printf("%16.0f",num2);
  result = g_strcasecmp(d1, d2);
  g_free(d1);
  g_free(d2);

  return result;
}


gchar*
get_mode_str                           (gint	in_mode)
{
	rmode_t		mode1;
	gchar*		m_item;
	
	mode1 = GPOINTER_TO_INT(g_list_nth_data(mode_list,in_mode-1));
	switch(mode1){
		case RIG_MODE_AM : m_item = strdup("AM"); break;
		case RIG_MODE_CW : m_item = strdup("CW"); break;
		case RIG_MODE_USB : m_item = strdup("USB"); break;
		case RIG_MODE_LSB : m_item = strdup("LSB"); break;
		case RIG_MODE_RTTY : m_item = strdup("RTTY"); break;
		case RIG_MODE_FM : m_item = strdup("FM"); break;
		case RIG_MODE_WFM : m_item = strdup("FM-Wide"); break;
		case RIG_MODE_CWR : m_item = strdup("CW-R"); break;
		case RIG_MODE_RTTYR : m_item = strdup("RTTY-R"); break;
		case RIG_MODE_AMS : m_item = strdup("AMS"); break;
		case RIG_MODE_PKTLSB : m_item = strdup("Packet-LSB"); break;
		case RIG_MODE_PKTUSB : m_item = strdup("Packet-USB"); break;
		case RIG_MODE_PKTFM : m_item = strdup("PKT-FM"); break;
		case RIG_MODE_FAX : m_item = strdup("FAX"); break;
		case RIG_MODE_ECSSLSB : m_item = strdup("ECSS-LSB"); break;
		case RIG_MODE_ECSSUSB : m_item = strdup("ECSS-USB"); break;
		default : m_item = strdup(" "); break;
	}
	return m_item;
}


void
check_mode()
{
	GtkWidget*	opt_menu;
	GtkWidget*	opt_menu_menu;
	gchar*		m_item;
	int			i=0;
	int			j=0;

	if (mode_list != NULL) {
		g_list_free(mode_list);
		mode_list = NULL;
	}
	while (hrig->state.filters[i].modes != 0) {
		gint mw = hrig->state.filters[i].modes;
		for (j=0; j<16; j++) {
			gint mode_num = mw;
		    mode_num &= (1<<j);
			if ((mode_num > 0) && (g_list_find(mode_list,GINT_TO_POINTER(mode_num))==NULL))
				mode_list = g_list_insert_sorted(mode_list,GINT_TO_POINTER(mode_num),
											  (GCompareFunc)modelist_compare_func);
		}
		i++;
	}
	opt_menu = lookup_widget(app1,"ModeBox");
	if (opt_menu) {
		gtk_option_menu_remove_menu(GTK_OPTION_MENU(opt_menu));
		opt_menu_menu = gtk_menu_new ();
		add_menu_item(opt_menu_menu," ");
		for (i=0; i < g_list_length(mode_list); i++){
			int mwd = GPOINTER_TO_INT(g_list_nth_data(mode_list,i));
			switch(mwd){
				case RIG_MODE_AM : m_item = strdup("AM"); break;
				case RIG_MODE_CW : m_item = strdup("CW"); break;
				case RIG_MODE_USB : m_item = strdup("USB"); break;
				case RIG_MODE_LSB : m_item = strdup("LSB"); break;
				case RIG_MODE_RTTY : m_item = strdup("RTTY"); break;
				case RIG_MODE_FM : m_item = strdup("FM"); break;
				case RIG_MODE_WFM : m_item = strdup("FM-Wide"); break;
				case RIG_MODE_CWR : m_item = strdup("CW-R"); break;
				case RIG_MODE_RTTYR : m_item = strdup("RTTY-R"); break;
				case RIG_MODE_AMS : m_item = strdup("AMS"); break;
				case RIG_MODE_PKTLSB : m_item = strdup("Packet-LSB"); break;
				case RIG_MODE_PKTUSB : m_item = strdup("Packet-USB"); break;
				case RIG_MODE_PKTFM : m_item = strdup("PKT-FM"); break;
				case RIG_MODE_FAX : m_item = strdup("FAX"); break;
				case RIG_MODE_ECSSLSB : m_item = strdup("ECSS-LSB"); break;
				case RIG_MODE_ECSSUSB : m_item = strdup("ECSS-USB"); break;
				default : m_item = strdup(" "); break;
			}
			add_menu_item(opt_menu_menu,m_item);
			g_free(m_item);
		}
		gtk_option_menu_set_menu(GTK_OPTION_MENU(opt_menu),opt_menu_menu);
  		gtk_signal_connect(GTK_OBJECT(GTK_OPTION_MENU(opt_menu)->menu),
  						   "deactivate", GTK_SIGNAL_FUNC(on_mode_selected),
  						   NULL);
	}
}


void
on_mode_selected                        (GtkMenuShell    *menu_shell,
                                         gpointer        data)
{
  GtkWidget* active_item = gtk_menu_get_active(GTK_MENU(menu_shell));
  gint item_index = g_list_index(menu_shell->children, active_item);
  if (item_index != 0)
	set_mode(item_index);
}


void
set_mode                               (gint	item_index)
{
  if (has_set_mode) {
	mode = item_index;
	set_bw((gint)bw);
  }
}


void
set_mode_display()
{
  set_visible(app1,"ModeLbl",ispower);
  set_option(app1,"ModeBox", ispower ? mode : 0);
  set_enabled(app1,"ModeBox",ispower&&has_set_mode);
}


void
get_mode_bw()
{
  gint			retcode;
  rmode_t		mode1;
  pbwidth_t		width1;
  int 			i;

  if (ispower && has_get_mode) {
    retcode = rig_get_mode(hrig, RIG_VFO_CURR, &mode1, &width1);
    if (retcode != RIG_OK) {
	  mode1 = 1;
	  width1 = rig_passband_normal(hrig,mode1);
	  g_warning(_("Hamlib error: Could not get mode and bandwidth."));
    }
    else {
	  mode = 0;
	  for (i=0; i<g_list_length(mode_list); i++) {
		if (mode1 == GPOINTER_TO_INT(g_list_nth_data(mode_list,i)))
			mode = i+1;
	  }
	  bw = 0;
	  for (i=0; i<g_list_length(bw_list); i++) {
		if (width1 == GPOINTER_TO_INT(g_list_nth_data(bw_list,i)))
			bw = i+1;
	  }
    }
  }
  else {
	  mode = 0;
	  bw = 0;
  }
  set_bw_display();
  set_mode_display();
}


/** noise blanker control interface **/
void
check_blanker()
{
	has_noise = (rig_has_get_func(hrig,RIG_FUNC_NB));
	set_visible(app1,"NoiseBtn",has_noise);
	set_visible(app1,"noise_blanker1",has_noise);
}


void
set_blanker()
{
  int	status;

  if (has_noise) {
	isnoise = !isnoise;
	if (isnoise)
		status = 1;
	else
		status = 0;
	rig_set_func(hrig, RIG_VFO_CURR, RIG_FUNC_NB, status);
	get_blanker();
  }
}


void
get_blanker()
{
  gint			retcode;
  int			status;

  if (ispower && has_noise) {
   	retcode = rig_get_func(hrig, RIG_VFO_CURR, RIG_FUNC_NB, &status);
	if (retcode != RIG_OK) {
		isnoise = FALSE;
		g_warning(_("Hamlib error: Could not get noise blanker."));
	}
    else
		isnoise = (status == 1);
  }
  else
	isnoise = FALSE;
  set_blanker_display();
}


void
set_blanker_display()
{
  set_visible(app1,"NoiseLbl",ispower&&isnoise&&has_noise);
  if (has_noise) {
	set_enabled(app1,"NoiseBtn",ispower);
	set_down(app1,"NoiseBtn",ispower&&isnoise);
	set_enabled(app1,"noise_blanker1",ispower);
  }
}


/** noise reduction control interface **/
void
check_dnr()
{
	has_dnr = (rig_has_get_func(hrig,RIG_FUNC_NR));
	set_visible(app1,"DNRBtn",has_dnr);
	set_visible(app1,"noise_reduction1",has_dnr);
}


void
set_dnr()
{
  int	status;

  if (has_dnr) {
	isdnr = !isdnr;
	if (isdnr)
		status = 1;
	else
		status = 0;
	rig_set_func(hrig, RIG_VFO_CURR, RIG_FUNC_NR, status);
	get_dnr();
  }
}


void
get_dnr()
{
  gint			retcode;
  int			status;

  if (ispower && has_dnr) {
   	retcode = rig_get_func(hrig, RIG_VFO_CURR, RIG_FUNC_NR, &status);
	if (retcode != RIG_OK) {
		isdnr = FALSE;
		g_warning(_("Hamlib error: Could not get noise reduction."));
	}
	else
		isdnr = (status == 1);
  }
  else
	isdnr = FALSE;
  set_dnr_display();
}


void
set_dnr_display()
{
  set_visible(app1,"DNRLbl",ispower&&isdnr&&has_dnr);
  if (has_dnr) {
	set_enabled(app1,"DNRBtn",ispower);
	set_down(app1,"DNRBtn",ispower&&isdnr);
	set_enabled(app1,"noise_reduction1",ispower);
  }
}


/** attenuator control interface **/
void
check_atten()
{
	int	i=0;

	num_attens = 0;
	has_atten = (rig_has_get_level(hrig,RIG_LEVEL_ATT));
	set_visible(app1,"AttBtn",has_atten);
	set_visible(app1,"attenuator1",has_atten);
	if (has_atten) {
		i = 0;
		atten_val[0] = 0;
		while (hrig->state.attenuator[i] != 0) {
			atten_val[i+1] = hrig->state.attenuator[i];
			i++;
		}
		num_attens = i;
	}
	if (num_attens <= 1)
		set_text(app1,"AttLbl","ATT");
}


void
set_atten()
{
  value_t	status;

  if (has_atten) {
	cur_atten++;
	if (cur_atten > num_attens)
		cur_atten = 0;
	status.i = atten_val[cur_atten];
	rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_ATT, status);
	get_atten();
	get_preamp();
  }
}


void
get_atten()
{
  int		i;
  gint		retcode;
  value_t	val;

  if (ispower && has_atten) {
	retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_ATT, &val);
	if (retcode != RIG_OK) {
		cur_atten = 0;
		g_warning(_("Hamlib error: Could not get attenuator."));
	}
    else
		for (i = 0; i <= num_attens; i++) {
			if (val.i == atten_val[i])
				cur_atten = i;
		}
  }
  else
	cur_atten = 0;
  set_atten_display();
}


void
set_atten_display()
{
  set_visible(app1,"AttLbl",ispower&&(cur_atten > 0)&&has_atten);
  if (has_atten) {
	if (num_attens > 1) {
		gchar* atext = g_strdup_printf("AT%d",cur_atten);
		set_text(app1,"AttLbl",atext);
		g_free(atext);
	}
	set_enabled(app1,"AttBtn",ispower);
	set_down(app1,"AttBtn",ispower&&(cur_atten > 0));
	set_enabled(app1,"attenuator1",ispower);
  }
}


/** preamp control interface **/
void
check_preamp()
{
	int	i;

	num_preamps = 0;
	has_preamp = (rig_has_get_level(hrig,RIG_LEVEL_PREAMP));
	set_visible(app1,"PreampBtn",has_preamp);
	set_visible(app1,"preamp1",has_preamp);
	if (has_preamp) {
		i = 0;
		preamp_val[0] = 0;
		while (hrig->state.preamp[i] != 0) {
			preamp_val[i+1] = hrig->state.preamp[i];
			i++;
		}
		num_preamps = i;
	}
	if (num_preamps <= 1)
		set_text(app1,"PreampLbl","PRE");
}


void
set_preamp()
{
  value_t	status;

  if (has_preamp) {
	cur_preamp++;
	if (cur_preamp > num_preamps)
		cur_preamp = 0;
	status.i = preamp_val[cur_preamp];
	rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_PREAMP, status);
	get_preamp();
	get_atten();
  }
}


void
get_preamp()
{
  int		i;
  gint		retcode;
  value_t	val;

  if (ispower && has_preamp) {
	retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_PREAMP, &val);
	if (retcode != RIG_OK) {
		cur_preamp = 0;
		g_warning(_("Hamlib error: Could not get preamp."));
	}
   	else
		for (i = 0; i <= num_preamps; i++) {
			if (val.i == preamp_val[i])
				cur_preamp = i;
		}
  }
  else
	cur_preamp = 0;
  set_preamp_display();
}


void
set_preamp_display()
{
  set_visible(app1,"PreampLbl",ispower&&(cur_preamp > 0)&&has_preamp);
  if (has_preamp) {
	if (num_preamps > 1) {
		gchar* ptext = g_strdup_printf("PR%d",cur_preamp);
		set_text(app1,"PreampLbl",ptext);
		g_free(ptext);
	}
	set_enabled(app1,"PreampBtn",ispower);
	set_down(app1,"PreampBtn",ispower&&(cur_preamp > 0));
	set_enabled(app1,"preamp1",ispower);
  }
}


/** notch control interface **/
void
check_notch()
{
	has_notch = (rig_has_get_func(hrig,RIG_FUNC_MN));
	set_visible(app1,"NotchBtn",has_notch);
	set_visible(app1,"notch1",has_notch);
}


void
set_notch()
{
  int	status;

  if (has_notch) {
	isnotch = !isnotch;
	if (isnotch)
		status = 1;
	else
		status = 0;
	rig_set_func(hrig, RIG_VFO_CURR, RIG_FUNC_MN, status);
	get_notch();
  }
}


void
get_notch()
{
  gint			retcode;
  int			status;

  if (ispower && has_notch) {
	retcode = rig_get_func(hrig, RIG_VFO_CURR, RIG_FUNC_MN, &status);
	if (retcode != RIG_OK) {
		isnotch = FALSE;
		g_warning(_("Hamlib error: Could not get notch."));
	}
    else
		isnotch = (status == 1);
  }
  else
	isnotch = FALSE;
  set_notch_display();
}


void
set_notch_display()
{
  set_visible(app1,"NotchLbl",ispower&&isnotch&&has_notch);
  if (has_notch) {
	set_enabled(app1,"NotchBtn",ispower);
	set_down(app1,"NotchBtn",ispower&&isnotch);
	set_enabled(app1,"notch1",ispower);
  }
}


/** automatic notch filter control interface **/
void
check_anf()
{
	has_anf = (rig_has_get_func(hrig,RIG_FUNC_ANF));
	set_visible(app1,"ANFBtn",has_anf);
	set_visible(app1,"auto_notch1",has_anf);
}


void
set_anf()
{
  int	status;

  if (has_anf) {
	isanf = !isanf;
	if (isanf)
		status = 1;
	else
		status = 0;
	rig_set_func(hrig, RIG_VFO_CURR, RIG_FUNC_ANF, status);
	get_anf();
  }
}


void
get_anf()
{
  gint			retcode;
  int			status;

  if (ispower && has_anf) {
	retcode = rig_get_func(hrig, RIG_VFO_CURR, RIG_FUNC_ANF, &status);
	if (retcode != RIG_OK) {
		isanf = FALSE;
		g_warning(_("Hamlib error: Could not get automatic notch filter."));
	}
    else
		isanf = (status == 1);
  }
  else
	isanf = FALSE;
  set_anf_display();
}


void
set_anf_display()
{
  set_visible(app1,"ANFLbl",ispower&&isanf&&has_anf);
  if (has_anf) {
	set_enabled(app1,"ANFBtn",ispower);
	set_down(app1,"ANFBtn",ispower&&isanf);
	set_enabled(app1,"auto_notch1",ispower);
  }
}


/** AF level control interface **/
void
check_af()
{
	af_range.min.f = 0.0;
	af_range.max.f = 1.0;
	af_range.step.f = 0.01;
	has_af = (rig_has_get_level(hrig,RIG_LEVEL_AF));
	set_visible(app1,"AFBox",has_af);
	if (has_af) {
		if ((hrig->state.level_gran[3].min.f != 0.0) &&
		    (hrig->state.level_gran[3].max.f != 0.0)) {
			af_range.min.f = hrig->state.level_gran[3].min.f;
			af_range.max.f = hrig->state.level_gran[3].max.f;
			af_range.step.f = hrig->state.level_gran[3].step.f;
		}
		if (af_range.step.f == 0.0)
			af_range.step.f = (af_range.max.f - af_range.min.f)/100;
		set_scale_range(app1,"AFScale",af_range,TRUE);
	}
}


void
set_af                                 (GtkAdjustment *adj,
                                        gpointer       gptr)
{
  value_t	val;

  if (has_af) {
	val.f = -get_scale_value(app1,"AFScale");
	if (val.f != afval) {
		rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_AF, val);
		get_af();
	}
  }
}


void
get_af()
{
  gint			retcode;
  value_t		val;

  if (ispower && has_af) {
	retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_AF, &val);
	if (retcode != RIG_OK) {
		afval = 0.0;
		g_warning(_("Hamlib error: Could not get af level."));
	}
    else
		afval = val.f;
  }
  else
	afval = 0.0;
  set_af_display();
}


void
set_af_display()
{
  if (has_af) {
	gchar* afstr = g_strdup_printf("%03d",(int)(afval*100.0));
	set_text(app1,"AFAmt",afstr);
	set_scale_value(app1,"AFScale",-afval);
	g_free(afstr);
  }
}


/** RF level control interface **/
void
check_rf()
{
	rf_range.min.f = 0.0;
	rf_range.max.f = 1.0;
	rf_range.step.f = 0.01;
	has_rf = (rig_has_get_level(hrig,RIG_LEVEL_RF));
	set_visible(app1,"RFBox",has_rf);
	if (has_rf) {
		if ((hrig->state.level_gran[4].min.f != 0.0) &&
		    (hrig->state.level_gran[4].max.f != 0.0)) {
			rf_range.min.f = hrig->state.level_gran[4].min.f;
			rf_range.max.f = hrig->state.level_gran[4].max.f;
			rf_range.step.f = hrig->state.level_gran[4].step.f;
		}
		if (rf_range.step.f == 0.0)
			rf_range.step.f = (rf_range.max.f - rf_range.min.f)/100;
		set_scale_range(app1,"RFScale",rf_range,TRUE);
	}
}


void
set_rf                                 (GtkAdjustment *adj,
                                        gpointer       gptr)
{
  value_t	val;

  if (has_rf) {
	val.f = -get_scale_value(app1,"RFScale");
	if (val.f != rfval) {
		rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_RF, val);
		get_rf();
	}
  }
}


void
get_rf()
{
  gint			retcode;
  value_t		val;

  if (ispower && has_rf) {
	retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_RF, &val);
	if (retcode != RIG_OK) {
		rfval = 0.0;
		g_warning(_("Hamlib error: Could not get rf level."));
	}
    else
		rfval = val.f;
  }
  else
	rfval = 0.0;
  set_rf_display();
}


void
set_rf_display()
{
  if (has_rf) {
	gchar* rfstr = g_strdup_printf("%03d",(int)(rfval*100.0));
	set_text(app1,"RFAmt",rfstr);
	set_scale_value(app1,"RFScale",-rfval);
	g_free(rfstr);
  }
}


/** IF level control interface **/
void
check_if()
{
	if_range.min.i = -2000;
	if_range.max.i = +2000;
	if_range.step.i = 10;
	has_if = (rig_has_get_level(hrig,RIG_LEVEL_IF));
	set_visible(app1,"IFBox",has_if);
	if (has_if) {
		if ((hrig->state.level_gran[6].min.i != 0) &&
		    (hrig->state.level_gran[6].max.i != 0)) {
			if_range.min.i = hrig->state.level_gran[6].min.i;
			if_range.max.i = hrig->state.level_gran[6].max.i;
			if_range.step.i = hrig->state.level_gran[6].step.i;
		}
		if (if_range.step.i == 0.0)
			if_range.step.i = (if_range.max.i - if_range.min.i)/100;
		set_scale_range(app1,"IFScale",if_range,FALSE);
	}
}


void
set_if                                 (GtkAdjustment *adj,
                                        gpointer       gptr)
{
  value_t	val;

  if (has_if) {
	val.i = -get_scale_value(app1,"IFScale");
	if (val.i != ifval) {
		rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_IF, val);
		get_if();
	}
  }
}


void
get_if()
{
  gint			retcode;
  value_t		val;

  if (ispower && has_if) {
	retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_IF, &val);
	if (retcode != RIG_OK) {
		ifval = 0.0;
		g_warning(_("Hamlib error: Could not get if level."));
	}
    else
		ifval = val.i;
  }
  else
	ifval = 0.0;
  set_if_display();
}


void
set_if_display()
{
  if (has_if) {
	gchar* ifstr = g_strdup_printf("%+05d",(int)ifval);
	set_text(app1,"IFAmt",ifstr);
	set_scale_value(app1,"IFScale",-ifval);
	g_free(ifstr);
  }
}


/** PBT IN level control interface **/
void
check_pi()
{
	pi_range.min.f = 0.0;
	pi_range.max.f = 1.0;
	pi_range.step.f = 0.01;
	has_pi = (rig_has_get_level(hrig,RIG_LEVEL_PBT_IN));
	set_visible(app1,"PIBox",has_pi);
	if (has_pi) {
		if ((hrig->state.level_gran[9].min.f != 0.0) &&
		    (hrig->state.level_gran[9].max.f != 0.0)) {
			pi_range.min.f = hrig->state.level_gran[9].min.f;
			pi_range.max.f = hrig->state.level_gran[9].max.f;
			pi_range.step.f = hrig->state.level_gran[9].step.f;
		}
		if (pi_range.step.f == 0.0)
			pi_range.step.f = (pi_range.max.f - pi_range.min.f)/100;
		set_scale_range(app1,"PIScale",pi_range,TRUE);
	}
}


void
set_pi                                 (GtkAdjustment *adj,
                                        gpointer       gptr)
{
  value_t	val;

  if (has_pi) {
	val.f = -get_scale_value(app1,"PIScale");
	if (val.f != pival) {
		rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_PBT_IN, val);
		get_pi();
	}
  }
}


void
get_pi()
{
  gint			retcode;
  value_t		val;

  if (ispower && has_pi) {
	retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_PBT_IN, &val);
	if (retcode != RIG_OK) {
		pival = 0.0;
		g_warning(_("Hamlib error: Could not get pbt in level."));
	}
    else
		pival = val.f;
  }
  else
	pival = 0.0;
  set_pi_display();
}


void
set_pi_display()
{
  if (has_pi) {
	gchar* pistr = g_strdup_printf("%+05d",(int)pival);
	set_text(app1,"PIAmt",pistr);
	set_scale_value(app1,"PIScale",-pival);
	g_free(pistr);
  }
}


/** PBT OUT level control interface **/
void
check_po()
{
	po_range.min.f = 0.0;
	po_range.max.f = 1.0;
	po_range.step.f = 0.01;
	has_po = (rig_has_get_level(hrig,RIG_LEVEL_PBT_OUT));
	set_visible(app1,"POBox",has_po);
	if (has_po) {
		if ((hrig->state.level_gran[10].min.f != 0.0) &&
		    (hrig->state.level_gran[10].max.f != 0.0)) {
			po_range.min.f = hrig->state.level_gran[10].min.f;
			po_range.max.f = hrig->state.level_gran[10].max.f;
			po_range.step.f = hrig->state.level_gran[10].step.f;
		}
		if (po_range.step.f == 0.0)
			po_range.step.f = (po_range.max.f - po_range.min.f)/100;
		set_scale_range(app1,"POScale",po_range,TRUE);
	}
}


void
set_po                                 (GtkAdjustment *adj,
                                        gpointer       gptr)
{
  value_t	val;

  if (has_po) {
	val.f = -get_scale_value(app1,"POScale");
	if (val.f != poval) {
		rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_PBT_OUT, val);
		get_po();
	}
  }
}


void
get_po()
{
  gint			retcode;
  value_t		val;

  if (ispower && has_po) {
	retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_PBT_OUT, &val);
	if (retcode != RIG_OK) {
		poval = 0.0;
		g_warning(_("Hamlib error: Could not get pbt out level."));
	}
    else
		poval = val.f;
  }
  else
	poval = 0.0;
  set_po_display();
}


void
set_po_display()
{
  if (has_po) {
	gchar* postr = g_strdup_printf("%+05d",(int)poval);
	set_text(app1,"POAmt",postr);
	set_scale_value(app1,"POScale",-poval);
	g_free(postr);
  }
}


/** CW PITCH level control interface **/
void
check_cp()
{
	cp_range.min.i = -5000;
	cp_range.max.i = +5000;
	cp_range.step.i = 50;
	has_cp = (rig_has_get_level(hrig,RIG_LEVEL_CWPITCH));
	set_visible(app1,"CPBox",has_cp);
	if (has_cp) {
		if ((hrig->state.level_gran[11].min.i != 0) &&
		    (hrig->state.level_gran[11].max.i != 0)) {
			cp_range.min.i = hrig->state.level_gran[11].min.i;
			cp_range.max.i = hrig->state.level_gran[11].max.i;
			cp_range.step.i = hrig->state.level_gran[11].step.i;
		}
		if (cp_range.step.i == 0.0)
			cp_range.step.i = (cp_range.max.i - cp_range.min.i)/100;
		set_scale_range(app1,"CPScale",cp_range,FALSE);
	}
}


void
set_cp                                 (GtkAdjustment *adj,
                                        gpointer       gptr)
{
  value_t	val;

  if (has_cp) {
	val.i = -get_scale_value(app1,"CPScale");
	if (val.i != cpval) {
		rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_CWPITCH, val);
		get_cp();
	}
  }
}


void
get_cp()
{
  gint			retcode;
  value_t		val;

  if (ispower && has_cp) {
	retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_CWPITCH, &val);
	if (retcode != RIG_OK) {
		cpval = 0.0;
		g_warning(_("Hamlib error: Could not get cw pitch level."));
	}
    else
		cpval = val.i;
  }
  else
	cpval = 0.0;
  set_cp_display();
}


void
set_cp_display()
{
  if (has_cp) {
	gchar* cpstr = g_strdup_printf("%+05d",(int)cpval);
	set_text(app1,"CPAmt",cpstr);
	set_scale_value(app1,"CPScale",-cpval);
	g_free(cpstr);
  }
}


/** SQUELCH level control interface **/
void
check_sq()
{
	sq_range.min.f = 0.0;
	sq_range.max.f = 1.0;
	sq_range.step.f = 0.01;
	has_sq = (rig_has_get_level(hrig,RIG_LEVEL_SQL));
	set_visible(app1,"SQBox",has_sq);
	if (has_sq) {
		if ((hrig->state.level_gran[5].min.f != 0.0) &&
		    (hrig->state.level_gran[5].max.f != 0.0)) {
			sq_range.min.f = hrig->state.level_gran[5].min.f;
			sq_range.max.f = hrig->state.level_gran[5].max.f;
			sq_range.step.f = hrig->state.level_gran[5].step.f;
		}
		if (sq_range.step.f == 0.0)
			sq_range.step.f = (sq_range.max.f - sq_range.min.f)/100;
		set_scale_range(app1,"SQScale",sq_range,TRUE);
	}
}


void
set_sq                                 (GtkAdjustment *adj,
                                        gpointer       gptr)
{
  value_t	val;

  if (has_sq) {
	val.f = -get_scale_value(app1,"SQScale");
	if (val.f != sqval) {
		rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_SQL, val);
		get_sq();
	}
  }
}


void
get_sq()
{
  gint			retcode;
  value_t		val;

  if (ispower && has_sq) {
	retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_SQL, &val);
	if (retcode != RIG_OK) {
		sqval = 0.0;
		g_warning(_("Hamlib error: Could not get squelch level."));
	}
    else
		sqval = val.f;
  }
  else
	sqval = 0.0;
  set_sq_display();
}


void
set_sq_display()
{
  if (has_sq) {
	gchar* sqstr = g_strdup_printf("%03d",(int)(sqval*100.0));
	set_text(app1,"SQAmt",sqstr);
	set_scale_value(app1,"SQScale",-sqval);
	g_free(sqstr);
  }
}


/** NOTCH level control interface **/
void
check_no()
{
	no_range.min.i = 0;
	no_range.max.i = +5000;
	no_range.step.i = 10;
	has_no = (rig_has_get_level(hrig,RIG_LEVEL_NOTCHF));
	set_visible(app1,"NOBox",has_no);
	if (has_no) {
		if ((hrig->state.level_gran[15].min.i != 0) &&
		    (hrig->state.level_gran[15].max.i != 0)) {
			no_range.min.i = hrig->state.level_gran[15].min.i;
			no_range.max.i = hrig->state.level_gran[15].max.i;
			no_range.step.i = hrig->state.level_gran[15].step.i;
		}
		if (no_range.step.i == 0.0)
			no_range.step.i = (no_range.max.i - no_range.min.i)/100;
		set_scale_range(app1,"NOScale",no_range,FALSE);
	}
}


void
set_no                                 (GtkAdjustment *adj,
                                        gpointer       gptr)
{
  value_t	val;

  if (has_no) {
	val.i = -get_scale_value(app1,"NOScale");
	if (val.i != noval) {
		rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_NOTCHF, val);
		get_no();
	}
  }
}


void
get_no()
{
  gint			retcode;
  value_t		val;

  if (ispower && has_no) {
	retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_NOTCHF, &val);
	if (retcode != RIG_OK) {
		noval = 0.0;
		g_warning(_("Hamlib error: Could not get notch level."));
	}
    else
		noval = val.i;
  }
  else
	noval = 0.0;
  set_no_display();
}


void
set_no_display()
{
  if (has_no) {
	gchar* nostr = g_strdup_printf("%04d",(int)noval);
	set_text(app1,"NOAmt",nostr);
	set_scale_value(app1,"NOScale",-noval);
	g_free(nostr);
  }
}


/** NOISE REDUCTION level control interface **/
void
check_nr()
{
	nr_range.min.f = 0.0;
	nr_range.max.f = 1.0;
	nr_range.step.f = 0.1;
	has_nr = (rig_has_get_level(hrig,RIG_LEVEL_NR));
	set_visible(app1,"NRBox",has_nr);
	if (has_nr) {
		if ((hrig->state.level_gran[8].min.f != 0.0) &&
		    (hrig->state.level_gran[8].max.f != 0.0)) {
			nr_range.min.f = hrig->state.level_gran[8].min.f;
			nr_range.max.f = hrig->state.level_gran[8].max.f;
			nr_range.step.f = hrig->state.level_gran[8].step.f;
		}
		if (nr_range.step.f == 0.0)
			nr_range.step.f = (nr_range.max.f - nr_range.min.f)/100;
		set_scale_range(app1,"NRScale",nr_range,TRUE);
	}
}


void
set_nr                                 (GtkAdjustment *adj,
                                        gpointer       gptr)
{
  value_t	val;

  if (has_nr) {
	val.f = -get_scale_value(app1,"NRScale");
	if (val.f != nrval) {
		rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_NR, val);
		get_nr();
	}
  }
}


void
get_nr()
{
  gint			retcode;
  value_t		val;

  if (ispower && has_nr) {
	retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_NR, &val);
	if (retcode != RIG_OK) {
		nrval = 0.0;
		g_warning(_("Hamlib error: Could not get noise reduction level."));
	}
    else
		nrval = val.f;
  }
  else
	nrval = 0.0;
  set_nr_display();
}


void
set_nr_display()
{
  if (has_nr) {
	gchar* nrstr = g_strdup_printf("%03d",(int)(nrval*100.0));
	set_text(app1,"NRAmt",nrstr);
	set_scale_value(app1,"NRScale",-nrval);
	g_free(nrstr);
  }
}


/** AUTO PASSBAND FILTER level control interface **/
void
check_ap()
{
	ap_range.min.f = 0.0;
	ap_range.max.f = 1.0;
	ap_range.step.f = 0.1;
	has_ap = (rig_has_get_level(hrig,RIG_LEVEL_APF));
	set_visible(app1,"APBox",has_ap);
	if (has_ap) {
		if ((hrig->state.level_gran[7].min.f != 0.0) &&
		    (hrig->state.level_gran[7].max.f != 0.0)) {
			ap_range.min.f = hrig->state.level_gran[7].min.f;
			ap_range.max.f = hrig->state.level_gran[7].max.f;
			ap_range.step.f = hrig->state.level_gran[7].step.f;
		}
		if (ap_range.step.f == 0.0)
			ap_range.step.f = (ap_range.max.f - ap_range.min.f)/100;
		set_scale_range(app1,"APScale",ap_range,TRUE);
	}
}


void
set_ap                                 (GtkAdjustment *adj,
                                        gpointer       gptr)
{
  value_t	val;

  if (has_ap) {
	val.f = -get_scale_value(app1,"APScale");
	if (val.f != apval) {
		rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_APF, val);
		get_ap();
	}
  }
}


void
get_ap()
{
  gint			retcode;
  value_t		val;

  if (ispower && has_ap) {
	retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_APF, &val);
	if (retcode != RIG_OK) {
		apval = 0.0;
		g_warning(_("Hamlib error: Could not get apf level."));
	}
    else
		apval = val.f;
  }
  else
	apval = 0.0;
  set_ap_display();
}


void
set_ap_display()
{
  if (has_ap) {
	gchar* apstr = g_strdup_printf("%03d",(int)(apval*100.0));
	set_text(app1,"APAmt",apstr);
	set_scale_value(app1,"APScale",-apval);
	g_free(apstr);
  }
}


/** Tone level control interface **/
/*void
check_to()
{
	to_range.min.f = 0.0;
	to_range.max.f = 1.0;
	to_range.step.f = 0.1;
	has_to = (rig_has_get_level(hrig,RIG_LEVEL_TONE));
	set_visible(app1,"TOBox",has_to);
	if (has_to) {
		if ((hrig->state.level_gran[32].min.f != 0.0) &&
		    (hrig->state.level_gran[32].max.f != 0.0)) {
			to_range.min.f = hrig->state.level_gran[32].min.f;
			to_range.max.f = hrig->state.level_gran[32].max.f;
			to_range.step.f = hrig->state.level_gran[32].step.f;
		}
		if (to_range.step.f == 0.0)
			to_range.step.f = (to_range.max.f - to_range.min.f)/100;
		set_scale_range(app1,"TOScale",to_range,TRUE);
	}
}


void
set_to                                 (GtkAdjustment *adj,
                                        gpointer       gptr)
{
  value_t	val;

  val.f = -get_scale_value(app1,"TOScale");
  if (val.f != toval) {
	rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_TONE, val);
	get_to();
  }
}


void
get_to()
{
  gint			retcode;
  value_t		val;

  if (ispower) {
	if (has_to) {
		retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_TONE, &val);
		if (retcode != RIG_OK) {
			toval = 0.0;
			g_warning(_("Hamlib error: Could not get tone level."));
		}
    	else
			toval = val.f;
	}
	else
		toval = 0.0;
  }
  else
	  toval = 0.0;
  set_to_display();
}


void
set_to_display()
{
  if (has_to) {
	gchar* tostr = g_strdup_printf("%03d",(int)(toval*100.0));
	set_text(app1,"TOAmt",tostr);
	set_scale_value(app1,"TOScale",-toval);
	g_free(tostr);
  }
}*/


/** BWC level control interface **/
void
check_bwc()
{
	bwc_range.min.i = 10;
	bwc_range.max.i = 9990;
	bwc_range.step.i = 10;
	/*has_bwc = (rig_has_get_level(hrig,RIG_LEVEL_BWC));*/
	has_bwc = FALSE;
	set_visible(app1,"BWCBox",has_bwc);
	if (has_bwc) {
		if ((hrig->state.level_gran[31].min.i != 0) &&
		    (hrig->state.level_gran[31].max.i != 0)) {
			bwc_range.min.i = hrig->state.level_gran[31].min.i;
			bwc_range.max.i = hrig->state.level_gran[31].max.i;
			bwc_range.step.i = hrig->state.level_gran[31].step.i;
		}
		if (bwc_range.step.i == 0.0)
			bwc_range.step.i = (bwc_range.max.i - bwc_range.min.i)/100;
		set_scale_range(app1,"BWCScale",bwc_range,FALSE);
	}
}


void
set_bwc                                (GtkAdjustment *adj,
                                        gpointer       gptr)
{
  value_t	val;

  if (has_bwc) {
	val.i = -get_scale_value(app1,"BWCScale");
	if (val.i != bwcval) {
	    /*rig_set_level(hrig, RIG_VFO_CURR, RIG_LEVEL_BWC, val);*/
		get_bwc();
	}
  }
}


void
get_bwc()
{
  gint			retcode;
  value_t		val;

  if (ispower && has_bwc) {
	/*retcode = rig_get_level(hrig, RIG_VFO_CURR, RIG_LEVEL_BWC, &val);*/
	retcode = RIG_OK;
	val.i = 0.0;
	if (retcode != RIG_OK) {
		bwcval = 0.0;
		g_warning(_("Hamlib error: Could not get bwc level."));
	}
    else
		bwcval = val.i;
  }
  else
	  bwcval = 0.0;
  set_bwc_display();
}


void
set_bwc_display()
{
  if (has_bwc) {
	gchar* bwcstr = g_strdup_printf("%04d",(int)bwcval);
	set_text(app1,"BWCAmt",bwcstr);
	set_scale_value(app1,"BWCScale",-bwcval);
	g_free(bwcstr);
  }
}


/** external tape control interface **/
void
set_tape()
{
  istape = !istape;
  set_tape_display();
}


void
set_tape_display()
{
  set_visible(app1,"TapeLbl",ispower&&istape);
  set_enabled(app1,"TapeBtn",ispower);
  set_down(app1,"TapeBtn",ispower&&istape);
  set_enabled(app1,"tape_vox1",ispower);
}


/** memory mode control interface **/
void
check_mem()
{
	int		i, delta;
	gchar 	*buf, *m_num, **text;

	has_get_mem = (hrig->caps->get_mem != NULL);
	has_set_mem = (hrig->caps->set_mem != NULL);
	i = 0;
	min_mem_ch = -1;
	max_mem_ch = -1;
	min_mem_edge = -1;
	max_mem_edge = -1;
	while (hrig->state.chan_list[i].type != 0) {
		switch (hrig->state.chan_list[i].type) {
		case RIG_MTYPE_MEM :
			min_mem_ch = hrig->state.chan_list[i].start;
			max_mem_ch = hrig->state.chan_list[i].end;
			break;
		case RIG_MTYPE_EDGE :
			min_mem_edge = hrig->state.chan_list[i].start;
			max_mem_edge = hrig->state.chan_list[i].end;
			break;
		default : break;
		}
		i++;
	}
	if (!has_get_mem)
		memch = min_mem_ch;
	ismem = FALSE;

	GtkWidget* mem_list = lookup_widget(mem1, "MemList");
	if (mem_list) {
		gtk_clist_clear(GTK_CLIST(mem_list));
		delta = max_mem_ch - min_mem_ch + 1;
		for (i = min_mem_ch; i < (max_mem_ch+1); i++) {
			m_num = g_strdup_printf("/apps/sgcontrol/%d/mem%d",
									hrig->caps->rig_model,
									i);
  			buf = gconf_client_get_string (confclient, m_num, NULL);
			if (buf) {
				text = g_strsplit(buf, ";", 3);
				gtk_clist_append(GTK_CLIST(mem_list),text);
				gtk_clist_set_row_data(GTK_CLIST(mem_list), i-min_mem_ch, 
										GUINT_TO_POINTER(i));
				g_free(buf);
				g_strfreev(text);
			}
			g_free(m_num);
		}
		if ((min_mem_edge != -1) && (max_mem_edge != -1)) {
			delta = max_mem_edge - min_mem_edge + 1;
			for (i = min_mem_edge; i < (max_mem_edge+1); i++) {
				m_num = g_strdup_printf("/apps/sgcontrol/%d/mem%d",
										hrig->caps->rig_model,
										i);
	  			buf = gconf_client_get_string (confclient, m_num, NULL);
				if (buf) {
					text = g_strsplit(buf, ";", 3);
					gtk_clist_append(GTK_CLIST(mem_list),text);
					gtk_clist_set_row_data(GTK_CLIST(mem_list), i-min_mem_edge, 
											GUINT_TO_POINTER(i));
					g_free(buf);
					g_strfreev(text);
				}
				g_free(m_num);
			}
		}
	}
}


void
set_mem()
{
  gint	retcode;
  vfo_t	new_vfo;

  if (has_set_mem) {
	ismem = !ismem;
	if (ispower) {
		if (ismem)
			new_vfo = RIG_VFO_MEM;
		else
			new_vfo = RIG_VFO_VFO;

		retcode = rig_set_vfo(hrig, new_vfo);
		if (retcode != RIG_OK)
			g_warning(_("Hamlib error: Could not set vfo/memory."));

		if (ismem) {
			retcode = rig_set_mem(hrig, RIG_VFO_CURR, memch);
			if (retcode != RIG_OK)
				g_warning(_("Hamlib error: Could not set memory channel."));
		}
	}
	get_all(FALSE);
  }
}


void
get_mem()
{
  gint			retcode;
  int			ch1;

  if (ispower && has_get_mem) {
	retcode = rig_get_mem(hrig, RIG_VFO_CURR, &ch1);
	if (retcode != RIG_OK) {
		memch = min_mem_ch;
		g_warning(_("Hamlib error: Could not get memory channel."));
	}
	else
		memch = ch1;
  }
  set_mem_display();
}


void
set_mem_display()
{
  set_enabled(app1,"MemBtn",ispower);
  set_down(app1,"MemBtn",ispower&&ismem);
  set_visible(app1,"MemLbl",ispower&&ismem);
  set_visible(app1,"Mem2Lbl",ispower);
  gchar* chstr = g_strdup_printf("%03d",memch);
  set_text(app1,"Mem2Lbl",chstr);
  g_free(chstr);
}


void
get_memory_list()
{
	guint 		i, delta, resval;
	gchar 		*buf, *status, *m_num, **text;
	channel_t	chan;
	gfloat		pcent;

	GtkWidget* mem_list = lookup_widget(mem1, "MemList");
	if (mem_list) {
		gtk_clist_clear(GTK_CLIST(mem_list));
		delta = max_mem_ch - min_mem_ch + 1;
		GtkWidget* app_bar = lookup_widget(app1, "appbar1");
		for (i = min_mem_ch; i < (max_mem_ch+1); i++) {
			chan.channel_num = i;
			resval = rig_get_channel(hrig, &chan);
			if ((resval == RIG_OK) && (chan.freq != 0))
				buf = g_strdup_printf("%d;%11.3f;%s", chan.channel_num,
													  chan.freq/1000.0,
													  chan.channel_desc);
			else
				buf = g_strdup_printf("%d;%s;%s", chan.channel_num,
												  "<blank>",
												  "");

			m_num = g_strdup_printf("/apps/sgcontrol/%d/mem%d",
									hrig->caps->rig_model,
									chan.channel_num);
			gconf_client_set_string(confclient, m_num, buf, NULL);
			text = g_strsplit(buf, ";", 3);
			gtk_clist_append(GTK_CLIST(mem_list),text);
			gtk_clist_set_row_data(GTK_CLIST(mem_list), i-min_mem_ch,
									GUINT_TO_POINTER(i));
			g_free(buf);
			g_free(m_num);
			g_strfreev(text);
			if (app_bar) {
				pcent = (gfloat)(i - min_mem_ch + 1) / (gfloat)delta;
				status = g_strdup_printf("Collecting memory #%d",i);
				gnome_appbar_set_status(GNOME_APPBAR(app_bar), status);
				gnome_appbar_set_progress(GNOME_APPBAR(app_bar), pcent);
				update_immediate();
				g_free(status);
			}

		}
		if ((min_mem_edge != -1) && (max_mem_edge != -1)) {
			delta = max_mem_edge - min_mem_edge + 1;
			for (i = min_mem_edge; i < (max_mem_edge+1); i++) {
				chan.channel_num = i;
				resval = rig_get_channel(hrig, &chan);
				if ((resval == RIG_OK) && (chan.freq != 0))
					buf = g_strdup_printf("%d;%11.3f;%s", chan.channel_num,
														  chan.freq/1000.0,
														  chan.channel_desc);
				else
					buf = g_strdup_printf("%d;%s;%s", chan.channel_num,
													  "<blank>",
													  "");
				m_num = g_strdup_printf("/apps/sgcontrol/%d/mem%d",
										hrig->caps->rig_model,
										chan.channel_num);
				gconf_client_set_string(confclient, m_num, buf, NULL);
				text = g_strsplit(buf, ";", 3);
				gtk_clist_append(GTK_CLIST(mem_list),text);
				gtk_clist_set_row_data(GTK_CLIST(mem_list), i-min_mem_edge,
										GUINT_TO_POINTER(i));
				g_free(buf);
				g_free(m_num);
				g_strfreev(text);
				if (app_bar) {
					pcent = (i - min_mem_edge + 1) / delta;
					status = g_strdup_printf("Collecting memory edge #%d",i);
					gnome_appbar_set_status(GNOME_APPBAR(app_bar), status);
					gnome_appbar_set_progress(GNOME_APPBAR(app_bar), pcent);
					gnome_appbar_refresh(GNOME_APPBAR(app_bar));
					g_free(status);
				}

			}
		}
		if (app_bar) {
			status = g_strdup_printf(" ");
			gnome_appbar_set_status(GNOME_APPBAR(app_bar), status);
			gnome_appbar_set_progress(GNOME_APPBAR(app_bar), 0);
			gnome_appbar_refresh(GNOME_APPBAR(app_bar));
			g_free(status);
		}
	}
	gtk_widget_show(mem1);
}


void
select_from_memory_list                (GtkWidget* inWidget,
                                        GdkEventButton  *inEvent)
{
  gint	in_row;
  gint	ch;
  gchar	*text;
  gchar	*text1;

  in_row = get_list_clicked_row(inWidget, inEvent);
  if (in_row != -1) {
	GtkWidget* gen_item = lookup_widget(GTK_WIDGET(mem1),"MemList");
	if (gen_item) {
		gtk_clist_get_text(GTK_CLIST(gen_item), in_row, 0, &text);
		gtk_clist_get_text(GTK_CLIST(gen_item), in_row, 1, &text1);
		if (strcmp(text1,"<blank>") != 0) {
			ch = strtol(text,NULL,10);
			if (rig_set_mem(hrig, RIG_VFO_CURR, ch) == RIG_OK) {
				ismem = TRUE;
				memch = ch;
			}
			get_all(FALSE);
		}
	}
  }
}


/** scanning mode control interface **/
void
set_scan()
{
  isscan = !isscan;
  set_scan_display();
}


void
set_scan_display()
{
  set_enabled(app1,"ScanBtn",ispower);
  set_visible(app1,"ScanLbl",ispower&&isscan);
  set_visible(app1,"Scan2Lbl",ispower);
}


/** timer control interface **/
void
set_timer()
{
  istimer = !istimer;
  set_timer_display();
}


void
set_timer_display()
{
  set_enabled(app1,"TimerBtn",ispower);
  set_visible(app1,"TimerLbl",ispower&&istimer);
}


/** hamlib general control interface **/
void
hamlib_new                             (guint   rigid,
                                        gchar*  port,
                                        gchar*  port_text,
                                        gchar*  rig_make,
                                        gchar*  rig_name)
{
  int	has_controls;

  hrig = rig_init(rigid);
  if (hrig) {
	if (g_strcasecmp(port,"Other...") == 0)
	  strncpy(hrig->state.rigport.pathname,port_text,80);
	else
	  strncpy(hrig->state.rigport.pathname,port,80);
	gint retcode = rig_open(hrig);
	if (retcode != RIG_OK) {
	  rig_cleanup(hrig);
	  return;
	}

	cur_rig_model = rigid;
	if (cur_rig_make)
		g_free(cur_rig_make);
	cur_rig_make = g_strdup(rig_make);
	if (cur_rig_name)
		g_free(cur_rig_name);
	cur_rig_name = g_strdup(rig_name);

	set_app_title();

	check_power();
	check_meter();
	check_blanker();
	check_dnr();
	check_atten();
	check_preamp();
	check_notch();
	check_anf();
	check_vfo();
	check_bw();
	check_ant();
	check_agc();
	check_mode();
	check_freq();
	check_mem();
	check_af();
	check_rf();
	check_if();
	check_pi();
	check_po();
	check_cp();
	check_sq();
	check_no();
	check_nr();
	check_ap();
	/*check_to();*/
	check_bwc();
	has_controls = (has_af || has_rf || has_if || has_pi || has_po ||
		            has_cp || has_sq || has_no || has_nr || has_ap ||
					/*has_to ||*/ has_bwc);
	if (get_visible(app1,"CPanel") && !has_controls)
		set_checked(app1,"control_panel1",FALSE);
	set_visible(app1,"control_panel1",has_controls);

	get_all(TRUE);
  }
}


void
hamlib_destroy()
{
  if (hrig) {
	rig_close(hrig);
    rig_cleanup(hrig);
  }
}


/** hamlib radio list control interface **/
GSList*
get_hamlib_radio_list()
{
  gint status;
  if (!rig_list) {
	  rig_load_all_backends();
	  status = rig_list_foreach(rig_list_make, NULL);
  }
  return rig_list;
}


gint
rig_list_make                          (const struct rig_caps *caps,
                                        gpointer data)
{
  rig_list = g_slist_append(rig_list, (gpointer) caps);
  return 1;
}


void
hamlib_fill_radio_list                 (GSList* rig_list,
                                        GtkCList* c_list)
{
  struct rig_caps* rcaps;
  gchar *buf, **text;
  guint i;

  guint num_rigs = g_slist_length(rig_list);
  for (i=0; i < num_rigs; i++) {
	rcaps = (struct rig_caps*) g_slist_nth_data(rig_list,i);
	buf = g_strdup_printf("%d;%s;%s;%s", rcaps->rig_model,
	                                     rcaps->mfg_name,
	                                     rcaps->model_name,
	                                     rcaps->version);
	text = g_strsplit(buf, ";", 4);
	gtk_clist_append(c_list,text);
	gtk_clist_set_row_data(c_list, i, GUINT_TO_POINTER(rcaps->rig_model));
	g_free(buf);
	g_strfreev(text);
  }

  gtk_clist_set_compare_func(c_list, hamlib_riglist_widget_compare_func );
  gtk_clist_set_sort_column(c_list, 1);
  gtk_clist_set_sort_type(c_list, GTK_SORT_ASCENDING);
  gtk_clist_sort(c_list);
  gtk_clist_select_row(c_list, 0, 0);
}


gint
hamlib_riglist_widget_compare_func     (GtkCList *c_list,
                                        gconstpointer s1,
                                        gconstpointer s2)
{
  GtkCListRow *row1 = (GtkCListRow *) s1;
  GtkCListRow *row2 = (GtkCListRow *) s2;
  gchar *d1;
  gchar *d2;
  gint result;

  d1 = g_strdup_printf("%s %s",GTK_CELL_TEXT (row1->cell[1])->text,GTK_CELL_TEXT (row1->cell[2])->text);
  d2 = g_strdup_printf("%s %s",GTK_CELL_TEXT (row2->cell[1])->text,GTK_CELL_TEXT (row2->cell[2])->text);
  result = g_strcasecmp(d1, d2);
  g_free(d1);
  g_free(d2);

  return result;
}


void
hamlib_riglist_set_defaults()
{
  GtkWidget* port_widget;
  gint sel_row;
  guint rig_id = -1;
  gchar* text[10];
  gchar* rig_make[25];
  gchar* rig_name[25];
  gchar* port_str = NULL;
  gchar* port_text = NULL;

  port_widget = lookup_widget(opt1,"RadioList");
  if (port_widget) {
	if (GTK_CLIST(port_widget)->selection) {
		sel_row = GPOINTER_TO_INT(GTK_CLIST(port_widget)->selection->data);
		gtk_clist_get_text(GTK_CLIST(port_widget),sel_row, 0, text);
		rig_id = atoi(text[0]);
		gconf_client_set_int(confclient,"/apps/sgcontrol/rig_model",rig_id,NULL);

		gtk_clist_get_text(GTK_CLIST(port_widget),sel_row, 1, rig_make);
		gconf_client_set_string(confclient,"/apps/sgcontrol/rig_make",rig_make[0],NULL);

		gtk_clist_get_text(GTK_CLIST(port_widget),sel_row, 2, rig_name);
		gconf_client_set_string(confclient,"/apps/sgcontrol/rig_name",rig_name[0],NULL);
	}
  }
  port_widget = lookup_widget(opt1,"PortBoxEdit");
  if (port_widget) {
	port_str = gtk_editable_get_chars(GTK_EDITABLE(port_widget),0,-1);
	gconf_client_set_string(confclient,"/apps/sgcontrol/rig_port",port_str,NULL);
  }
  port_widget = lookup_widget(opt1,"PortEdit");
  if (port_widget) {
	port_text = gtk_editable_get_chars(GTK_EDITABLE(port_widget),0,-1);
	gconf_client_set_string(confclient,"/apps/sgcontrol/port_text",port_text,NULL);
  }

  hamlib_destroy();
  hamlib_new(rig_id,port_str,port_text,rig_make[0],rig_name[0]);
}


void
set_visual_defaults()
{
  hamlib_destroy();

  if (get_visible(app1,"KeyPad"))
	gconf_client_set_int(confclient,"/apps/sgcontrol/keypad",1,NULL);
  else
	gconf_client_set_int(confclient,"/apps/sgcontrol/keypad",0,NULL);

  if (get_visible(app1,"CPanelBox"))
	gconf_client_set_int(confclient,"/apps/sgcontrol/controlpanel",1,NULL);
  else
	gconf_client_set_int(confclient,"/apps/sgcontrol/controlpanel",0,NULL);

  if (get_visible(app1,"HistoryBox"))
	gconf_client_set_int(confclient,"/apps/sgcontrol/history",1,NULL);
  else
	gconf_client_set_int(confclient,"/apps/sgcontrol/history",0,NULL);

  if (get_visible(app1,"BandPlanBox"))
	gconf_client_set_int(confclient,"/apps/sgcontrol/bandplan",1,NULL);
  else
	gconf_client_set_int(confclient,"/apps/sgcontrol/bandplan",0,NULL);

  if (get_down(tun1,"FreqLockBtn"))
	gconf_client_set_int(confclient,"/apps/sgcontrol/lock_freq",1,NULL);
  else
	gconf_client_set_int(confclient,"/apps/sgcontrol/lock_freq",0,NULL);
}
