/*
 * Copyright (C) 2000 Richard Groult <rgroult@jalix.org>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*directory.cpp*/

#include "directory.h"
#include "directoryview.h"

#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qapplication.h>

#include <kio/job.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <klineeditdlg.h>


Directory::Directory( DirectoryView* parent,
			 ImageViewer *iv,  ImageListeView *imageList, MainWindow *mw )
	: QListViewItem( parent, i18n("Directory /") ), f("/")
{
	this->dirView = parent;
	this->iv = iv;
	this->imageList = imageList;
	this->mw  = mw;

	list.setAutoDelete( TRUE );

	p = NULL;
	readable = TRUE;

	full.append( "/" );

	p_open = new QPixmap(BarIcon("folder_open",16));
	p_close = new QPixmap(BarIcon("folder",16));
	p_none = new QPixmap();
}

Directory::Directory(Directory* parent, const char * filename ,
			DirectoryView* dirView, ImageViewer *iv, ImageListeView  *imageList, MainWindow *mw )
    : QListViewItem(parent), f(filename)
{
	this->dirView = dirView;
	this->iv = iv;
	this->imageList = imageList;
	this->mw  = mw;

	list.setAutoDelete( TRUE );

	p = parent;
	readable = TRUE;

	full.append(p->fullName()+f.name()+"/");

	p_open = new QPixmap(BarIcon("folder_open",16));
	p_close = new QPixmap(BarIcon("folder",16));
	p_none = new QPixmap();
}


Directory::~Directory()
{
}


Directory*
Directory::find(const char* dir)
{
	QListViewItem* dirt;
	dirt=firstChild();
	while(dirt != 0)
	{
		if(QString(dir) == dirt->text(0))
			break;
		dirt = dirt->nextSibling ();
	}
	if(dirt)
		if(dirt->text(1)==i18n("Directory"))
			return (Directory*)dirt;
	
	return NULL;
}

void 
Directory::setOpen( bool o )
{
	if ( o && !childCount() )
	{
		QApplication::setOverrideCursor( waitCursor ); // this might take time
		listView()->setUpdatesEnabled( FALSE );
		listView()-> setCurrentItem(this);
	
		QString s( fullName() );
		QDir thisDir( s );
		if(dirView->showHiddenDir())
			thisDir.setFilter(QDir::All|QDir::Hidden);
		else
			thisDir.setFilter(QDir::All);
		if ( !thisDir.isReadable() )
		{
			readable = FALSE;
			p_open=p_close=new QPixmap(BarIcon("folder_locked",16));
			QApplication::restoreOverrideCursor();  // restore original cursor
			return;
		 }
		 const QFileInfoList * files = thisDir.entryInfoList();
		 if ( files )
		 {
		 	 QFileInfoListIterator it( *files );
		 	 QFileInfo * f;
	
		 	 while( (f=it.current()) != 0 )
		 	 {
		 	  	  ++it;
		 	  	  if ( f->fileName() == "." || f->fileName() == ".." )
		 	  	      ; // nothing
		 	  	  else
		 	  	  {
		 	  	  	  if ( f->isDir() || f->isSymLink() )
		 	  	  	    	  (void)new Directory( this, f->fileName(), dirView, iv, imageList, mw);
		 	  	  	  else
		 	  	  	  {
		 	  	  	   	   QString ext =  QFileInfo(f->fileName()).extension();
		 	  	  	   	   if ( (ext=="zip"))
		 	  	  	   	   	(void)new CompressedFileItem(this, f->fileName(), fullName(),  mw, dirView, iv, imageList);		 	  	  	   	   
		 	  	  	  }
		 	  	  }
		 	 }
			 dirView->sort();
		}
		if(!childCount())
		     setExpandable(false);
		QApplication::restoreOverrideCursor();      // restore original cursor
		QListViewItem::setOpen( o );	
		listView()->setUpdatesEnabled( TRUE );
	}
	else
	if(!o && childCount() )
	{
		QListViewItem
			*dirt=firstChild(),
			*oldDir;			
		while(dirt != 0)
		{	
			if(dirt->text(1)!=i18n("Directory"))
			;
			else
			{
			 	 if(dirt->isSelected())
				 	((Directory*)dirt)->setSelected(false);
				 ((Directory*)dirt)->setOpen(false);
			}
			oldDir = dirt;
			dirt = dirt->nextSibling ();

			delete(oldDir);
		}		
		QListViewItem::setOpen( false );
	}
}

void
Directory::setSelected(bool select)
{

	if(isSelected() && select)
		return;

	QListViewItem::setSelected(select);
	if(!dirView->isDropping())
	{
		listView()->triggerUpdate();
		qApp->processEvents();
		
       		if(select)
        	{
			loadDir(true);
			dirView->startWatchDir(fullName());
		}
		else
		{
			unLoadDir();
			dirView->stopWatchDir(fullName());
		}
	}
}


void 
Directory::refresh()
{
	QString s( fullName() );
	QDir thisDir( s );
	
	//QApplication::setOverrideCursor( waitCursor ); // this might take time
	const QFileInfoList * files = thisDir.entryInfoList();
	QStringList after;		
	QStringList news;		
	QStringList deleted;		
		
	if ( files ) 
	{
		QFileInfoListIterator it( *files );
		QFileInfo * f;
		QString ext;
		
		while( (f=it.current()) != 0 )
		{
			++it;
			ext = f->extension( FALSE ).lower();

			if( (ext == "jpg") ||  (ext == "jpeg") ||
				(ext == "gif") ||
				(ext == "png") ||
				(ext == "bmp") ||
				(ext == "tif") || (ext == "tiff") || 
				(ext == "mng") ||
				(ext == "xpm") ||
				(ext == "pbm") ||
				(ext == "pnm") ||
				(ext == "pgm") ||
				(ext == "ppm") ||
				(ext == "xbm") )
			{
				QString filename=f->fileName();
				FileItem *item=imageList->findItem(filename);
				if(!item)
				{
					news.append(filename);
				}
				after.append(filename);
			}
		}
	}
	FileItem *item;
	for ( item=list.first();  item!= 0; item=list.next() )
	{
		QString itemname=item->text(0);
		if(!after.contains(itemname))
		{
			deleted.append(itemname);
		}		
	}
	for ( QStringList::Iterator it = news.begin(); it != news.end(); ++it )
	{
		list.append(new ImageFileItem( imageList, this, *it, (const char*)fullName(), iv));
		mw->slotAddImage();
	}
	int toDel=0;
	for ( QStringList::Iterator it = deleted.begin(); it != deleted.end(); ++it )
	{
		ImageFileItem *item=(ImageFileItem *)imageList->findItem(*it);
		list.remove(item);
		toDel++;
	}
	mw->slotRemoveImage(toDel);
	if(!news.isEmpty() || deleted.isEmpty())
	{
		imageList->sort();
		if(!news.isEmpty())
			imageList->slotLoadFirst();
	}
	//QApplication::restoreOverrideCursor();	// restore original cursor	
}

void
Directory::goTo(const char* dest)
{

}

void
Directory::setup()
{
	setExpandable( TRUE );
	QListViewItem::setup();
}

QString
Directory::fullName()
{
	return full;
}

QString
Directory::name()
{
	return f.name();
}

QString
Directory::path()
{
	QDir dir=QFileInfo(fullName()).dir();
	if (dir.cdUp())
		return dir.absPath();
	else
		return QString();
}

const QPixmap*
Directory::pixmap ( int column) const
{
	if ( column == 0 )
	{
		if (isOpen())
			return p_open;
		else
			return p_close;
	}
	else
		return p_none;
}

QString
Directory::text( int column ) const
{
	if ( column == 0 )
		return f.name();
	else
	if(column == 2 )
		return "";
	else
	if ( readable )
		return i18n("Directory");
	else
		return i18n("Locked");
}

void
Directory::loadFirst()
{

}

void
Directory::removeImage(ImageFileItem *imf)
{
	if(!imf)
	{
		return;
	}
	
	if(list.findRef(imf)!=-1)
	{
		list.removeRef(imf);
		mw->slotRemoveImage(1);
	}
	else
	{ 
	}    
}

void
Directory::unLoadDir()
{
	if(mw->preview())	
		imageList->stopLoading();
	
	mw->slotRemoveImage(list.count());
	list.clear();
	iv->updateStatus();

	if(mw->preview())	
		imageList->slotLoadFirst();
}

void
Directory::loadDir(bool refresh)
{
	mw->setMessage(QString().sprintf(i18n("Loading directory %s..."),(const char*)text(0)));
	listView()->setSelected(this, TRUE);
	if(!imageList->hasImages()) iv->loadImage(NULL);
	if(mw->preview())	
		imageList->stopLoading();

	QString s( fullName() );
	QDir thisDir( s );
	if(dirView->showHiddenFile())
		thisDir.setFilter(QDir::All|QDir::Hidden|QDir::Files);
	else
		thisDir.setFilter(QDir::All|QDir::Files);
	QApplication::setOverrideCursor( waitCursor ); // this might take time
	const QFileInfoList * files = thisDir.entryInfoList();

	if ( files )
	{
		QFileInfoListIterator it( *files );
		QFileInfo * f;
		QString ext;
		int num=0;
		while( (f=it.current()) != 0 )
		{
			++it;
			ext = f->extension( FALSE ).lower();
			if( (ext == "jpg") ||  (ext == "jpeg") ||
				(ext == "gif") ||
				(ext == "png") ||
				(ext == "bmp") ||
				(ext == "tif") || (ext == "tiff")  ||
				(ext == "mng") ||
				(ext == "xpm") ||
				(ext == "pbm") ||
				(ext == "pnm") ||
				(ext == "pgm") ||
				(ext == "ppm") ||
				(ext == "xbm") )
				{
					list.append(new ImageFileItem( imageList, this, f->fileName(), (const char*)fullName(), iv));
					num++;
				}
		}
		mw->slotAddImage(num);
		if(refresh)
		{
			imageList->slotLoadFirst();
			iv->updateStatus();
		}
	}
	imageList->sort();
	if(mw->preview())	
		imageList->slotLoadFirst();
	imageList->arrangeItemsInGrid () ;
	QApplication::restoreOverrideCursor();	// restore original cursor
	mw->setMessage(i18n("Ready"));
}

QString
Directory::key (int column, bool ascending) const
{
	if(column==0)
		return (text(0)+"|"+text(1)+"|"+text(2)).lower();
	else
	if(column==1)
		return (text(1)+"|"+text(0)+"|"+text(2)).lower();
	else
		return (text(2)+"|"+text(0)+"|"+text(1)).lower();
}

void
Directory::createDir(QString dirName)
{
	KURL url = KURL("file://"+fullName()+dirName);
	KIO::mkdir( url );
	setExpandable(true);
	if(isOpen())
		new Directory(this, dirName, dirView, iv, imageList, mw);
}

void
Directory::rename()
{
	if(newDirName!="")
	{	
		f.setName(newDirName);
		full=path()+"/"+newDirName+"/";
		mw->slotRefresh();
		repaint();
	}	
}

void
Directory::rename(QString newDirName)
{
	this->newDirName=newDirName;
	
	QString oldN=fullName();
	QString newN=path()+"/"+newDirName;	
	
	KURL orgName = KURL("file://"+oldN);
	KURL newName = KURL("file://"+newN);
	
	KIO::SimpleJob *renameJob=KIO::rename(orgName, newName, true );
	QObject::connect(renameJob, SIGNAL(result( KIO::Job *)),
		dirView, SLOT(renameDone( KIO::Job *)));

}





