/**
 * SIEGE socket library
 *
 * Copyright (C) 2001 Jeffrey Fulmer <jdfulmer@armstrong.com>
 * This file is part of Siege
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifdef  HAVE_CONFIG_H
# include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <stdio.h>
#include <signal.h>
#include <setjmp.h>

#include <joedog/joedog.h>
#include "sock.h"

#ifdef  HAVE_ARPA_INET_H
# include <arpa/inet.h>
#endif/*HAVE_ARPA_INET_H*/

#ifdef  HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#endif/*HAVE_SYS_SOCKET_H*/

#ifdef  HAVE_SYS_SELECT_H
# include <sys/select.h>
#  if  TIME_WITH_SYS_TIME
#   include <sys/time.h>
#   include <time.h>
#  else
#   if HAVE_SYS_TIME_H
#    include <sys/time.h>
#   else
#    include <time.h>
#   endif
#  endif/*TIME_WITH_SYS_TIME*/
#endif/*HAVE_SYS_SELECT_H*/

#ifdef  HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif/*HAVE_NETINET_IN_H*/

#ifdef  HAVE_NETDB_H
# include <netdb.h>
#endif/*HAVE_NETDB_H*/

#ifdef HAVE_SSL
# include <openssl/ssl.h>
# include <openssl/err.h>
# include <openssl/rsa.h>
# include <openssl/crypto.h>
# include <openssl/x509.h>
# include <openssl/pem.h>
  SSL      *ssl  =  NULL;
  SSL_CTX  *ctx  =  NULL;
#endif /* HAVE_SSL */

int conn;
int sock;
int USE_SSL;
jmp_buf env;

void handler( int );

/**
 * returns int socket
 * _SSL: 1=use encryption,
 * _SSL: 0=not encrypted
 * 
 * Uses RAND_seed for randomness on all platforms.
 */
int 
SIEGEsocket( char *hn, int port, int _SSL )
{
  /* ssl connection */
  int  serr;

  /* socket timeout */
  int timo = 8;
  struct timeval     to  = { timo, 0 };
  fd_set readset, writeset;

  /* buffer for RAND_seed */
  char buf[1024];

  /* socket structs */
  struct hostent     *hp;
  struct protoent    *pe;
  struct sockaddr_in srv;
  
  USE_SSL = _SSL;
  signal( SIGALRM, handler );

  if( USE_SSL ){ 
    #ifdef HAVE_SSL
      SSL_load_error_strings();
      SSLeay_add_ssl_algorithms();
      ctx = SSL_CTX_new( SSLv3_client_method());
      if( ctx == NULL ){ joe_warning( "ctx is NULL" ); }
      /* http://www.openssl.org/support/faq.html#USER1 
       * Perhaps someday I'll learn to read the FAQ 
       * first and then code second, but probably not. 
       * Not all OSes have /dev/urandom, we must seed 
       * the PRNG 
       */
      memset( buf, 0, sizeof( buf ));
      RAND_seed( buf, sizeof( buf ));
      ssl = SSL_new( ctx );
      SSL_set_connect_state( ssl );
    #else
      return -1;
    #endif /* HAVE_SSL */
  }
 
  /* get the protocol, error on failure  */ 
  if(( pe = getprotobyname( "tcp" )) == NULL ){
    joe_error( "protocol failure" );
  }
  
  /* get the host, else error on failure */ 
  if(( hp = gethostbyname( hn )) != NULL ){ 
    memset( &srv, 0, sizeof(srv));
    memcpy( &srv.sin_addr, hp->h_addr, hp->h_length );
    srv.sin_family = AF_INET;
    srv.sin_port = htons( port );
  }
  else{ joe_error( "hostname failure" ); }

  /* create a socket, return -1 on failure */
  if(( sock = socket( AF_INET, SOCK_STREAM, pe->p_proto )) < 0 ){
    return -1;
  }
  
  /* set sock options  */
  setsockopt( sock, SOL_SOCKET, SO_KEEPALIVE, 0, 0 );

  /* connect to server, return -1 on failure  */
  if( setjmp( env ) == 0 ){
    alarm( timo );
    if(( conn = connect( sock, (struct sockaddr *) &srv, sizeof(srv))) < 0 ){
      SIEGEclose(); 
      return -1; 
    }
    alarm( 0 );
  } /* if setjmp */
  else{   
    printf( "CONNECT: timed out.\n" );
    close( sock );
    alarm( 0 );
    return -1;
  }

  /* if requested, encrypt the transaction  */
  if( USE_SSL ){
    #ifdef HAVE_SSL
      /* currently a fatal error, should it be? */
      if(( SSL_set_fd( ssl, sock )) < 0 ){
        joe_fatal( "unable to create secure socket!" );
      }
      if(( serr = SSL_connect( ssl )) < 0 ){
        int problem = SSL_get_error( ssl, serr ); 
        fprintf( stderr, "SSL_connect: want to %s more...\n",
               ( problem == SSL_ERROR_WANT_READ) ? "read" : "write");
        return -1; 
      }
      SSL_get_cipher( ssl ); 
    #else 
      return -1;
    #endif /* HAVE_SSL */
  }

  return sock;
}

/**
 * returns int 
 * conditional, writes to encrypted 
 * or unencrypted socket.
 */
int 
SIEGEsocket_write( char *buf, unsigned int len )
{
  /* assign from write */
  int t = 0;

  if( USE_SSL ){
    #ifdef HAVE_SSL
      /* write to the encrypted socket */
      return t = SSL_write( ssl, buf, len );
    #endif /* HAVE_SSL */
  }
  else{
    return t = write( sock, buf, len );   
  }
}

/**
 * returns int 
 * conditional, reads from encrypted 
 * or unencrypted socket.
 */ 
int 
SIEGEsocket_read( char *buf, unsigned int len )
{
  int t = 0;
#ifdef  HAVE_SYS_SELECT_H
  int    nf;
  int    timo  =  4;  /* HARDCODED, timo second timeout */
  struct timeval to;
  fd_set readset, writeset;

  to.tv_sec = timo;
  to.tv_usec = 0;

  FD_ZERO( &readset );
  FD_ZERO( &writeset );
  FD_SET( sock, &readset );
#endif/*HAVE_SYS_SELECT_H*/

  if( USE_SSL ){
    #ifdef HAVE_SSL
      #ifdef  HAVE_SYS_SELECT_H
      if(( nf = select( sock + 1, &readset, &writeset, NULL, &to )) < 0 ){
        write( 1, "socket timeout\n", 15 );
        return -1;
      }
      else
      #endif/*HAVE_SYS_SELECT_H*/
        return t = SSL_read( ssl, buf, len );
    #endif /* HAVE_SSL */
  }
  else{
    #ifdef  HAVE_SYS_SELECT_H
    if(( nf = select( sock + 1, &readset, &writeset, NULL, &to )) < 0 ){ 
      write( 1, "socket timeout\n", 15 );
      return -1;
    }
    else
    #endif/*HAVE_SYS_SELECT_H*/
      return t = read( sock, buf, len );
  }
}

/**
 * returns void
 * frees ssl resources if using ssl and
 * closes the connection and the socket.
 */
void
SIEGEclose( )
{
  #ifdef HAVE_SSL
    if( USE_SSL ){
      SSL_shutdown( ssl ); 
    }
    SSL_free( ssl );
    SSL_CTX_free( ctx ); 
  #endif 
  close( conn );
  close( sock );

  return;
}

/**
 * returns void
 * signal handler implemented for connection
 * time outs, yech! non-blocking connections
 * were choking on some platforms.
 */
void
handler( int sig )
{
  longjmp( env, 1 );
}
