#include "sjog.h"
#include "sjog_list.h"
#include "sjog_list_callbacks.h"
#include "sjog_enlightenment.h"

/* -------------------------------------------------------------------------- */

/*
 * Call the configuration file parsing and creates the list window
 * (only the window, not the list).
 */
void
sjog_list_init()
{
   /* call the parsing of the config file */
   if (eMenu)
   sjog_init_tree_entries();
   else
      sjog_init_list_entries();

   listTimer = 0;
   /* create the list window and register callbacks */
   listWindow =
      windowsAreTransient ? gtk_window_new(GTK_WINDOW_DIALOG) :
      gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(GTK_WINDOW(listWindow), "S-Jog");

   gtk_signal_connect(GTK_OBJECT(listWindow), "delete-event",
                      GTK_SIGNAL_FUNC(sjog_list_signal_delete_event_callback),
                      NULL);

   /* TT - This callback is used to determine where to move the window
    * just before it's shown, and just after the widget has been allocated.
    */
   gtk_signal_connect(GTK_OBJECT(listWindow), "realize",
                      GTK_SIGNAL_FUNC(sjog_list_signal_realize_callback),
                      NULL);
   
   /* creates a 1 column width list */
   cmdListUI = gtk_clist_new(1);
   gtk_clist_set_column_auto_resize(GTK_CLIST(cmdListUI), 0, TRUE);

   /* be warned when a row is selected */
   gtk_signal_connect(GTK_OBJECT(cmdListUI), "select-row",
                      GTK_SIGNAL_FUNC(sjog_list_row_selected_callback), NULL);

   gtk_container_add(GTK_CONTAINER(listWindow), cmdListUI);

}

/* -------------------------------------------------------------------------- */

/*
 * Builds the widgets of the actions list.
 * @arg list the list model.
 */
void
sjog_list_set_list()
{
   guint length, i = 0;      /* commands list length and loop index */
   sjog_cmd_t *cmd;          /* temporary pointer for commands insertion loop */
   gchar *item[1];           /* GtkCList needs an array of pointers to gchar */


   /* default is to select 1st item */
   selectedIndex = 0;

   /* get the number of items to put in the list */
   length = g_list_length(list);

   if (length == 0)
   {
      fprintf(stderr, "Error: the list contains no items, exiting\n");
      exit(2);
   }

   D("list contains %d items\n", length);

   /* loops over the items and add them to the list */
   for (i = 0; i <= length - 1; i++)
   {
      cmd = (sjog_cmd_t *) g_list_nth_data(list, i);
      item[0] = cmd->name;
      gtk_clist_append(GTK_CLIST(cmdListUI), item);
   }

   /* select the 1st item */
   gtk_clist_select_row(GTK_CLIST(cmdListUI), selectedIndex, 0);

}

/* -------------------------------------------------------------------------- */

/*
 * Shows the list window.
 */
void
sjog_list_show()
{

   D_ENTER;

   if (listWindow == NULL)
   {
      sjog_list_init();
      if (eMenu)
         sjog_tree_set_list();
      else
         sjog_list_set_list();
   }

   if (resetMenu == TRUE)
   {
      /* select the 1st item */
      gtk_clist_select_row(GTK_CLIST(cmdListUI), 0, 0);
   }

   if (!GTK_WIDGET_VISIBLE(listWindow))
   {
      /* change the mode */
      sjog_set_mode(SJOG_MODE_LIST);

      /* show the window */
      /* sjog_window_set_position(GTK_WINDOW(listWindow)); */
      /* TT - The above sjog_window_set_position() function is now depracated,
       * as the functionaliy has been moved into the "realize" event callback
       */
      gtk_widget_show_all(listWindow);

      /* reset the window hidding timer */
      sjog_list_reset_timer();
   }
}

/* -------------------------------------------------------------------------- */

/*
 * Hides the list window.
 */
void
sjog_list_hide()
{

   D_ENTER;

   gtk_timeout_remove(listTimer);
   gtk_widget_hide_all(listWindow);
   /* jump to scroll mode when no window is shown */
   sjog_set_mode(SJOG_MODE_SCROLL);
}

/* -------------------------------------------------------------------------- */

/*
 * Remove the old hidding timer and put a new one.
 */
void
sjog_list_reset_timer()
{

   D_ENTER;

   if (listTimer != 0)
      gtk_timeout_remove(listTimer);
   listTimer =
      gtk_timeout_add(SJOG_TIMEOUT, (GtkFunction) sjog_list_timer_callback,
                      NULL);
}

/* -------------------------------------------------------------------------- */

/*
 * Do whatever needed depending on the jog position.
 */
void
sjog_list_check_jog_position(int jog)
{
   switch (jog)
   {
     case JOG_BOTTOM:
        sjog_list_select_next();
        break;
     case JOG_PUSHED:
        sjog_list_exec_cmd();
        break;
     case JOG_TOP:
        sjog_list_select_previous();
        break;
     case JOG_BACK:
   sjog_list_go_back();
   break;
   }
}

/* -------------------------------------------------------------------------- */

/*
 * Called when the jog is pushed while we are in LIST_MODE. Either executes
 * a command or jump to one of the other modes (VOLUME, BRIGHTNESS..)
 */
void
sjog_list_exec_cmd()
{
   sjog_cmd_t *sjog_cmd;               /* sjog_cmd to execute */
   static sjog_cmd_t *prev_cmd = NULL; /* previously executed command */
   GList *list_element;                /* glib element that contains the cmd */
   GNode *tree_element;                /* glib element that contains the cmd */
   gchar *cmd;                         /* cmd string to execute */

   /* remove the timer and hide the window */
   sjog_list_hide();

   D_ENTER;

   if (eMenu)
   {
      tree_element = g_node_nth_child(currentTreeNode, selectedIndex);
      sjog_cmd = (sjog_cmd_t *) tree_element->data;
   }
   else
   {
      list_element = g_list_nth(list, selectedIndex);
      sjog_cmd = (sjog_cmd_t *) list_element->data;
   }

   D("sjog_cmd->path = %s\n", sjog_cmd->path);

   /* redo allows execution of the last executed command  */
   if (strcmp(sjog_cmd->path, "redo") == 0) {

      /* no previous command */
      if (prev_cmd == NULL) return;

      sjog_cmd = prev_cmd;
   }
   else {
      prev_cmd = sjog_cmd;
   }


   /* enter brightness mode */
   if (strcmp(sjog_cmd->path, "brightness") == 0)
   {
      sjog_set_mode(SJOG_MODE_BRIGHTNESS);
      sjog_brightness_show();
   }

   /* enter volume mode */
   else if (strcmp(sjog_cmd->path, "volume") == 0)
   {
      sjog_set_mode(SJOG_MODE_VOLUME);
      sjog_volume_show();
   }
   
   else if (eMenu && sjog_cmd->type == SJOG_CMD_TYPE_MENU)
   {
      sjog_list_reset_timer();

      /* return to parent menu if special "parent" item is selected */
      if (strcmp(sjog_cmd->path, "parent") == 0)
      {
         sjog_list_go_back();

      }

      /* enter a new menu only if there is something in it */
      else if (g_node_n_children(tree_element))
      {
         gtk_clist_clear(GTK_CLIST(cmdListUI));
         currentTreeNode = tree_element;
         sjog_tree_set_list();
      }

      /* force re-display */
      sjog_list_show();
   }

   /* hide does nothing and makes the menu disappear */
   else if (strcmp(sjog_cmd->path, "hide") == 0) {
      return;
   }

   else
   {
      cmd = g_strdup_printf("%s&", sjog_cmd->path);

      D("cmd = %s\n", cmd);

      /* launch the command */
      system(cmd);

      g_free(cmd);
   }
}

/* -------------------------------------------------------------------------- */


void
sjog_list_go_back()
{
   guint index;
   
        if (eMenu)
        {

      D_ENTER;

                if (currentTreeNode->parent)
                {
         /* force selected index */
         index = g_node_child_position(currentTreeNode->parent, currentTreeNode);
                        sjog_list_reset_timer();
                        gtk_clist_clear(GTK_CLIST(cmdListUI));
                        currentTreeNode = currentTreeNode->parent;
                        sjog_tree_set_list();

                        /* force re-display */
                        sjog_list_hide();
                        sjog_list_show();
         
         gtk_clist_select_row(GTK_CLIST(cmdListUI), index, 0);
                }
      else
         sjog_list_hide();
        }
}


/*
 * Selects the next item in the application list.
 */
void
sjog_list_select_next()
{
   guint length;

   if (eMenu)
      length = g_node_n_children(currentTreeNode);
   else
      length = g_list_length(list);

   /* if we hit bottom, go to top */
   if (selectedIndex == (length - 1))
      selectedIndex = 0;
   else
      selectedIndex++;

   D("selectedIndex=%d\n", selectedIndex);

   gtk_clist_select_row(GTK_CLIST(cmdListUI), selectedIndex, 0);
   sjog_list_reset_timer();

}

/* -------------------------------------------------------------------------- */

/*
 * Selects the previous item in the application list.
 */
void
sjog_list_select_previous()
{
   guint length;

   if (eMenu)
      length = g_node_n_children(currentTreeNode);
   else
      length = g_list_length(list);

   /* if we hit top, go to bottom */
   if (selectedIndex == 0)
      selectedIndex = length - 1;
   else
      selectedIndex--;

   D("selectedIndex=%d\n", selectedIndex);

   gtk_clist_select_row(GTK_CLIST(cmdListUI), selectedIndex, 0);
   sjog_list_reset_timer();
}

/*
 * modelines
 * ex:set ai ts=3 sw=3:
 * ei:set itab=spaces noat:
 * vim:set et:
 */
