/* 
 * dbskkd.c
 * inetd-style SKK Server for Berkeley-DBnized dictionary file
 * Version dbskkd-0.2beta
 * based on SKK Version 3.9.3
 * Copyright (c) 1996 by Kenji Rikitake <kenji@reseau.toyonaka.osaka.jp>
 * Original copyright notice at the end of this source file
 * $Id: dbskkd.c,v 1.9 1996/11/09 12:16:39 kenji Exp $
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <netdb.h>
#include <signal.h>
#include <errno.h>
#include <limits.h>
#include <syslog.h>
#include <db.h>

/* dbskkd working directory */
#ifndef SERVER_DIR
#define	SERVER_DIR	"/usr/local/etc/dbskkd/"
#endif /* SERVER_DIR */

/* dbskkd dictionary file (path relative to SERVER_DIR) */
#ifndef JISHO_FILE
#define JISHO_FILE	"./SKK-JISYO.L.db"	
#endif /* JISHO_FILE */

#define	BUFSIZE		1024	/* max size of a request */

#define CLIENT_END	'0'
#define CLIENT_REQUEST	'1'
#define CLIENT_VERSION	'2'
#define CLIENT_HOST	'3'

#define SERVER_ERROR	"0"
#define SERVER_FOUND	"1"
#define SERVER_NOT_FOUND "4"

#define STDIN	(fileno(stdin))
#define STDOUT	(fileno(stdout))

void err __P((const char *, ...));
void setproctitle __P((const char *, ...));

char	pgmver[] = "dbskkd-0.2 ";	/* version number */

char	hname[BUFSIZ];	/* hostname and IP address */
DB	*jisho_dbp;	/* DB structure for the shared dictionary */
HASHINFO jisho_hash;	/* HASHINFO for the shared dictionary */

/*
 * main server loop:
 * DB-style hashed database search 
 */

int search()
{	
  unsigned char	combuf[BUFSIZE];
  register unsigned char *pbuf;
  int length, errcode, get_status;
  DBT key, data;

  if ((length = read(STDIN, &combuf[0], BUFSIZE)) <= 0) {
    err("read error from stdin: %s", strerror(errno));
    /* NOTREACHED */
  }

  switch (combuf[0]) {

  case CLIENT_END:
    syslog(LOG_NOTICE, "end of conversion requested");
    return -1;
    /* NOTREACHED */
    break; /* foolproof */

  case CLIENT_VERSION:
    if (write(STDOUT, pgmver, strlen(pgmver)) < 0) {
      err("write error (version string): %s", 
	  strerror(errno));
      /* NOTREACHED */
    }
    return 0;
    /* NOTREACHED */
    break; /* foolproof */

  case CLIENT_HOST:
    if (write(STDOUT, hname, strlen(hname)) < 0) {
      err("write error (host info): %s", strerror(errno));
      /* NOTREACHED */
    }
    return 0;
    /* NOTREACHED */
    break; /* foolproof */

  case CLIENT_REQUEST:
    /* get size of key */
    key.data = &combuf[1];
    for (pbuf = &combuf[1];
         *pbuf != ' ' && pbuf != &combuf[length - 1];
         pbuf++) {}
    key.size = pbuf - &combuf[1];
	
    if (key.size <= 0) {
      err("invalid keysize = %d", key.size);
      /* NOTREACHED */
    }
    else {
      get_status = jisho_dbp->get(jisho_dbp, &key, &data, 0);
    }
    switch (get_status) {
    case -1:
      err("fatal error on DB get routine: %s", strerror(errno));
      /* NOTREACHED */
      break; /* foolproof */
    case 0: /* found */
      /* sending found code and the result data string with LF */
      if ((errcode = write(STDOUT, SERVER_FOUND, 1)) >= 0) {
        if ((errcode = write(STDOUT, data.data, data.size)) >= 0) {
	  errcode = write(STDOUT, "\n", 1);
          if (errcode < 0) {
	    err("write error (converted data): %s",
		 strerror(errno));
	    /* NOTREACHED */
          }
          return 0;
          /* NOTREACHED */
        }
      }
      break;
    case 1: /* NOT found */
      /* sending error code and the key string with LF */
      /* this action is required by skkinput */
      if ((errcode = write(STDOUT, SERVER_NOT_FOUND, 1)) >= 0) {
        if ((errcode = write(STDOUT, key.data, key.size)) >= 0) {
	  errcode = write(STDOUT, "\n", 1);
          if (errcode < 0) {
	    err("write error (NOT_FOUND message): %s",
		 strerror(errno));
	    /* NOTREACHED */
          }
          return 0;
          /* NOTREACHED */
        }
      }
      return 0;
      /* NOTREACHED */
      break; /* foolproof */
    default: /* unknown value */ 
      err("unknown return value %d from DB get routine", get_status);
      /* NOTREACHED */
      break; /* foolproof */
    }
    break;
  default:
    err("unknown client request code %d", combuf[0]);
    /* NOTREACHED */
    break; /* foolproof */
  }
  /* NOTREACHED */
}

/*
 * main program
 */

int main(argc, argv)
int argc;
char *argv[];
{
  struct hostent *hp;
  char **p, *lp;
  struct in_addr addr;
  struct sockaddr_in sin;
  int sval;
  
  /* open syslog */
  openlog("dbskkd", LOG_PID | LOG_CONS, LOG_DAEMON);

  /* chdir to server directory */
  if (chdir(SERVER_DIR) != 0) {
    err("cannot chdir() to %s: %s", SERVER_DIR, strerror(errno));
    }
  /* set umask to g-w, o-rwx */
  (void)umask(027);

  /* open dictionary db file */
  if ((jisho_dbp = dbopen(JISHO_FILE, O_RDONLY, S_IRUSR, DB_HASH, &jisho_hash)) == NULL) {
    err("cannot dbopen() the dictionary file %s: %s", JISHO_FILE,
	 strerror(errno));
    /* NOTREACHED */
  }

  /* get host info */
  if (gethostname(hname, BUFSIZE) < 0) {
    err("gethostname: %s", strerror(errno));
    /* NOTREACHED */
  }
  hp = gethostbyname(hname);
  p = hp->h_addr_list;
  while (*p != NULL) {
    strcat(hname, ":");
    memcpy(&addr.s_addr, *p, hp->h_length);
    strcat(hname, inet_ntoa(addr));
    p++;
  }
  strcat(hname, ": ");

  /* get peer info */
  sval = sizeof(sin);
  if (getpeername(0, (struct sockaddr *)&sin, &sval) < 0) {
    err("getpeername: %s", strerror(errno));
    /* NOTREACHED */
    }
  if (hp = gethostbyaddr((char *)&sin.sin_addr.s_addr,
      sizeof(sin.sin_addr.s_addr), AF_INET))
    lp = hp->h_name;
  else
    lp = inet_ntoa(sin.sin_addr);

#ifdef LOG_PEERINFO
  /* log peer info */
  syslog(LOG_NOTICE, "connected from %s", lp);
#endif /* LOG_PEERINFO */
  /* set process status string */
  setproctitle("serving %s", lp);

  /* command loop */
  while(search() == 0) {}

  /* close dictionary db file */
  if (jisho_dbp->close(jisho_dbp)) 
    err("cannot dbp->close() the dictionary file: %s", strerror(errno));
    /* NOTREACHED */

  exit(0);
}

/* 
 * the function err() for syslog retains 
 * the following copyright conditions
 */
/*
 * Copyright (c) 1983, 1993
 *      The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#if __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

void
#if __STDC__
err(const char *fmt, ...)
#else
err(fmt, va_alist)
        char *fmt;
        va_dcl
#endif
{
        va_list ap;
#if __STDC__
        va_start(ap, fmt);
#else
        va_start(ap);
#endif
        (void)vsyslog(LOG_ERR, fmt, ap);
        va_end(ap);
        exit(1);
        /* NOTREACHED */
}

/* end of program */

/* original skkserv comment follows */

/* skkserv.c
 * SKK Server Version 3.9.3 of Jun 17, 1994
 * Copyright (C) 1994 Yukiyoshi Kameyama (kam@riec.tohoku.ac.jp)
 *
 * Programmed by
 * Yukiyoshi Kameyama (kam@riec.tohoku.ac.jp)
 * Research Institute of Electrical Communication
 * Tohoku University
 *
 * Contributed by 
 * Nobu Takahashi (nt@hpycla.yhp.co.jp) for System V patch
 * Chikahiro Kimura (kimura@2oa.kb.nec.co.jp) for "ntohs" bug
 * Koji Kawaguchi (guchi@pfu.fujitsu.co.jp) for various improvement
 * Hironobu Takahashi (takahasi@tiny.or.jp) for patch for char/int typing
 * Kazushi Marukawa (kazushi@kubota.co.jp) for cleaning up behavior in PRIVATE mode
 * TSUMURA Kazumasa (tsumura@fml.ec.tmit.ac.jp) for pointing out a bug in okuri-ari/nasi test
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either versions 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with SKK, see the file COPYING.  If not, write to the Free
 * Software Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * Description
 *	"skkserv" is a server daemon for an SKK/Nemacs client, which
 *	provides a facility of fast looking-up of an SKK-dictionary.
 *	skkserv and an SKK/Nemacs client can run on different hosts 
 *	which are connected by a TCP/IP network.
 *
 *	An old-style SKK-dictionary is a sorted sequence of lines in 
 *	the form: 
 *		YOMI /TANGO1/TANGO2/.../TANGOn/
 *	A new-style SKK-dictionary consists of two parts:
 *	The first part begins with the line
 *		;; okuri-ari entries.
 *	followed by the reverse-sorted sequence of lines in the same
 *	form as old-style.  In the first part, all the YOMI entries
 *	have "okurigana".
 *	The second part begins with the line
 *		;; okuri-nasi entries.
 *	followed by the sorted sequence of lines in the same form as
 *	old-style.  In the second part, all the YOMI entries do NOT
 *	have "okurigana".
 *	Other lines whose first two characters are ";;" will be ignored.
 *
 *	NOTE: the "sort" and "reverse-sort" should be emacs-sort, not
 *	UNIX sort (unsigned char, shorter is earlier).
 *
 *	Once invoked with a dictionary, the server runs permanently, 
 *	and waits for client's requests.
 *	A client, usually invoked by SKK/Nemacs, requests the server
 *	to transform a string (YOMI) to another string(/TANGO1/.../).
 *	Communication between the server and a client is done through
 *	a TCP-port "skkserv" by a UNIX socket.
 *	The server looks up the dictionary with the input string
 *	as the key, and replies its value to the client.
 *	If the input string is not found in the dictionary it replies
 *	'Not Found'.
 */

/*
 * Client Request Form
 *   "0"	end of connection
 *   "1eee "	eee is keyword in EUC code with ' ' at the end
 *   "2"	skkserv version number
 *   "3"	hostname and its IP addresses
 *
 * Server Reply Form for "1eee"
 *   "0"	Error
 *   "1eee"	eee is the associated line separated by '/'
 *   "4"	Not Found
 *
 * Server Reply Form for "2"
 *   "A.B "	A for major version number, B for minor version number 
 *		followed by a space
 *
 * Server Reply Form for "3"
 *   "string:addr1:...: "  string for hostname, addr1 for an IP address
 *	           	followed by a space
 *			addrs should be in dotted quad form
 */

/* end of source file */
