/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to track operations.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgpayeepluginwidget.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgobjectmodel.h"
#include "skgsortfilterproxymodel.h"
#include "skgdocumentbank.h"
#include "skgpayeeobject.h"
#include "skgtransactionmng.h"

#include <QDomDocument>
#include <QKeyEvent>

SKGPayeePluginWidget::SKGPayeePluginWidget ( SKGDocument* iDocument )
        :SKGTabPage ( iDocument )
{
    SKGTRACEIN ( 1, "SKGPayeePluginWidget::SKGPayeePluginWidget" );

    ui.setupUi ( this );

    ui.kAddButton->setIcon ( KIcon ( "list-add" ) );
    ui.kModifyButton->setIcon ( KIcon ( "dialog-ok-apply" ) );
    ui.kDeleteUnusedButton->setIcon ( KIcon ( "edit-delete" ) );

    objectModel = new SKGObjectModel ( static_cast<SKGDocumentBank*>(getDocument()), "v_payee_display", "1=0", this, "", false );
    SKGSortFilterProxyModel* modelproxy = new SKGSortFilterProxyModel ( this );
    modelproxy->setSourceModel ( objectModel );
    modelproxy->setSortRole ( Qt::UserRole );
    modelproxy->setDynamicSortFilter ( true );
    ui.kView->setModel ( modelproxy );

    ui.kFilterEdit->setProxy ( modelproxy );

    //Add registered global action in contextual menu
    if ( SKGMainPanel::getMainPanel() )
    {
        ui.kView->insertGlobalAction ( "edit_delete" );
        ui.kView->insertGlobalAction ();
        ui.kView->insertGlobalAction ( "edit_find" );
        ui.kView->insertGlobalAction ( "open_report" );

        ui.kView->resizeColumnToContents ( 0 );
    }

    ui.kTitle->setPixmap ( KIcon ( "dialog-information" ).pixmap ( 22, 22 ), KTitleWidget::ImageLeft );

    connect ( ( const QObject* ) getDocument(), SIGNAL ( tableModified ( QString, int ) ), this, SLOT ( dataModified ( QString, int ) ), Qt::QueuedConnection );
    dataModified ( "", 0 );

    connect ( ui.kView, SIGNAL ( selectionChangedDelayed() ), this, SLOT ( onSelectionChanged() ) );

    connect ( objectModel, SIGNAL ( beforeReset() ), ui.kView, SLOT ( saveSelection() ) );
    connect ( objectModel, SIGNAL ( afterReset() ), ui.kView, SLOT ( resetSelection() ) );

    //Set Event filters to catch CTRL+ENTER or SHIFT+ENTER
    this->installEventFilter ( this );
}

SKGPayeePluginWidget::~SKGPayeePluginWidget()
{
    SKGTRACEIN ( 1, "SKGPayeePluginWidget::~SKGPayeePluginWidget" );
    objectModel=NULL;
}

bool SKGPayeePluginWidget::eventFilter ( QObject *object, QEvent *event )
{
    if ( event && event->type() == QEvent::KeyPress )
    {
        QKeyEvent *keyEvent = ( QKeyEvent * ) event;
        if ( ( keyEvent->key() == Qt::Key_Return || keyEvent->key() == Qt::Key_Enter ) && object==this )
        {
            if ( QApplication::keyboardModifiers() &Qt::ControlModifier && ui.kAddButton->isEnabled() )
            {
                ui.kAddButton->click();
            }
            else if ( QApplication::keyboardModifiers() &Qt::ShiftModifier && ui.kModifyButton->isEnabled() )
            {
                ui.kModifyButton->click();
            }
        }
    }

    return false;
}

void SKGPayeePluginWidget::dataModified ( const QString& iTableName, int iIdTransaction )
{
    SKGTRACEIN ( 1, "SKGPayeePluginWidget::dataModified" );
    Q_UNUSED ( iIdTransaction );

    if ( iTableName=="refund" || iTableName.isEmpty() )
    {
        //Correction bug 2299394 vvv
        if ( ui.kView->isAutoResized() ) ui.kView->resizeColumnsToContentsDelayed();
        //Correction bug 2299394 ^^^
        onEditorModified();
    }
}

void SKGPayeePluginWidget::onSelectionChanged()
{
    SKGTRACEIN ( 10, "SKGPayeePluginWidget::onSelectionChanged" );

    SKGObjectBase::SKGListSKGObjectBase objs=getSelectedObjects();
    if ( objs.count() )
    {
        SKGPayeeObject obj=objs.at ( 0 );
        ui.kNameInput->setText ( obj.getName() );
        ui.kAddressEdit->setText ( obj.getAddress() );
    }

    onEditorModified();
    emit selectionChanged();
}

SKGObjectBase::SKGListSKGObjectBase SKGPayeePluginWidget::getSelectedObjects()
{
    return ui.kView->getSelectedObjects();
}

int SKGPayeePluginWidget::getNbSelectedObjects()
{
    return ui.kView->getNbSelectedObjects();
}

QString SKGPayeePluginWidget::getState()
{
    SKGTRACEIN ( 10, "SKGPayeePluginWidget::getState" );
    QDomDocument doc ( "SKGML" );
    QDomElement root = doc.createElement ( "parameters" );
    doc.appendChild ( root );

    root.setAttribute ( "filter", ui.kFilterEdit->lineEdit()->text() );

    //Memorize table settings
    root.setAttribute ( "view", ui.kView->getState() );
    return doc.toString();
}

void SKGPayeePluginWidget::setState ( const QString& iState )
{
    SKGTRACEIN ( 10, "SKGPayeePluginWidget::setState" );
    QDomDocument doc ( "SKGML" );
    doc.setContent ( iState );
    QDomElement root = doc.documentElement();

    QString hideUseless=root.attribute ( "hideUseless" );
    QString filter=root.attribute ( "filter" );

    if ( !filter.isEmpty() ) ui.kFilterEdit->setText ( filter );

    if ( objectModel )
    {
        bool previous=objectModel->blockRefresh ( true );
        onFilterChanged();
        objectModel->blockRefresh ( previous );
    }

    //!!! Must be done here after onFilterChanged
    ui.kView->setState ( root.attribute ( "view" ) );
}

QString SKGPayeePluginWidget::getDefaultStateAttribute()
{
    return "SKGPAYEE_DEFAULT_PARAMETERS";
}

QWidget* SKGPayeePluginWidget::getWidgetForPrint()
{
    return ui.kView;
}

void SKGPayeePluginWidget::onEditorModified()
{
    _SKGTRACEIN ( 10, "SKGPayeePluginWidget::onEditorModified" );
    int nb=getNbSelectedObjects();
    ui.kModifyButton->setEnabled ( !ui.kNameInput->text().isEmpty() && nb==1 );
    ui.kAddButton->setEnabled ( !ui.kNameInput->text().isEmpty() && nb<=1 );
}

void SKGPayeePluginWidget::onFilterChanged()
{
    SKGTRACEIN ( 10, "SKGPayeePluginWidget::onFilterChanged" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    //Update model
    if ( objectModel )
    {
        QString filter;
        objectModel->setFilter ( filter );
        objectModel->refresh();

        //Correction bug 2299394 vvv
        if ( ui.kView->isAutoResized() ) ui.kView->resizeColumnsToContentsDelayed();
        //Correction bug 2299394 ^^^
    }

    QApplication::restoreOverrideCursor();
}

void SKGPayeePluginWidget::onAddPayee()
{
    SKGError err;
    _SKGTRACEINRC ( 10, "SKGPayeePluginWidget::onAddPayee", err );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    QString name=ui.kNameInput->text();
    SKGPayeeObject payee;
    {
        SKGBEGINTRANSACTION ( *getDocument(), i18nc ( "Noun, name of the user action", "Payee creation '%1'", name )  , err );

        err=SKGPayeeObject::createPayee ( static_cast<SKGDocumentBank*>(getDocument()), name, payee );
        if ( err.isSucceeded() ) err=payee.setAddress ( ui.kAddressEdit->text() );
        if ( err.isSucceeded() ) err=payee.save();
    }
    //status bar
    if ( err.isSucceeded() )
    {
        err=SKGError ( 0, i18nc ( "Successful message after an user action", "Payee '%1' created", name ) );
        ui.kView->selectObject ( payee.getUniqueID() );
    }
    else err.addError ( ERR_FAIL, i18nc ( "Error message", "Payee creation failed" ) );
    QApplication::restoreOverrideCursor();

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGPayeePluginWidget::onModifyPayee()
{
    SKGError err;
    _SKGTRACEINRC ( 10, "SKGPayeePluginWidget::onModifyPayee", err );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    //Get Selection
    SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();

    int nb=selection.count();
    if ( nb==1 )
    {
        SKGBEGINTRANSACTION ( *getDocument(), i18nc ( "Noun, name of the user action", "Payee '%1' update", ui.kNameInput->text() )  , err );

        //Modification of category object
        SKGPayeeObject payee=selection[0];
        err=payee.setName ( ui.kNameInput->text() );
        if ( err.isSucceeded() ) err=payee.setAddress ( ui.kAddressEdit->text() );
        if ( err.isSucceeded() ) err=payee.save();

        //status bar
        if ( err.isSucceeded() ) err=SKGError ( 0, i18nc ( "Successful message after an user action", "Payee '%1' updated", ui.kNameInput->text() ) );
        else err.addError ( ERR_FAIL, i18nc ( "Error message", "Payee update failed" ) );
    }
    QApplication::restoreOverrideCursor();

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGPayeePluginWidget::onDoubleClick()
{
    _SKGTRACEIN ( 10, "SKGPayeePluginWidget::onDoubleClick" );

    //Get selection
    SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
    if ( selection.count() ==1 )
    {
        SKGPayeeObject payee ( selection.at ( 0 ) );
        QString name=payee.getName();

        //Build where clause and title
        QString wc="t_PAYEE = '"+SKGServices::stringToSqlString ( name ) +'\'';
        QString title=i18nc ( "Noun, a list of items", "Operations assigned to payee '%1'",  name );

        //Open
        if ( QApplication::keyboardModifiers() &Qt::ControlModifier && QApplication::keyboardModifiers() &Qt::ShiftModifier )
        {
            //Call debug plugin
            QDomDocument doc ( "SKGML" );
            QDomElement root = doc.createElement ( "parameters" );
            doc.appendChild ( root );
            root.setAttribute ( "sqlOrder", "SELECT * from v_operation_display WHERE "+wc );

            SKGMainPanel::getMainPanel()->openPage ( SKGMainPanel::getMainPanel()->getPluginByName ( "Debug plugin" ), -1, doc.toString() );
        }
        else
        {
            //Call operation plugin
            QDomDocument doc ( "SKGML" );
            doc.setContent ( getDocument()->getParameter ( "SKGOPERATION_DEFAULT_PARAMETERS" ) );
            QDomElement root = doc.documentElement();
            if ( root.isNull() )
            {
                root=doc.createElement ( "parameters" );
                doc.appendChild ( root );
            }

            root.setAttribute ( "operationTable", "v_operation_display" );
            root.setAttribute ( "operationWhereClause", wc );
            root.setAttribute ( "title", title );
            root.setAttribute ( "title_icon", "skrooge_payee" );
            root.setAttribute ( "currentPage", "-1" );

            SKGMainPanel::getMainPanel()->openPage ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
        }
    }
}

void SKGPayeePluginWidget::cleanEditor()
{
    if ( getNbSelectedObjects() ==0 )
    {
        ui.kNameInput->setText ( "" );
        ui.kAddressEdit->setText ( "" );
    }
}

void SKGPayeePluginWidget::activateEditor()
{
    ui.kNameInput->setFocus();
}

bool SKGPayeePluginWidget::isEditor()
{
    return true;
}

void SKGPayeePluginWidget::onDeleteUnused()
{
    SKGError err;
    _SKGTRACEINRC ( 10, "SKGPayeePluginWidget::onDeleteUnused", err );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    {
        SKGBEGINTRANSACTION ( *getDocument(), i18nc ( "Noun, name of the user action", "Delete unused payees" )  , err );

        //Modification of payee object
        QString sql="DELETE FROM payee WHERE NOT EXISTS (SELECT 1 FROM operation WHERE operation.r_payee_id=payee.id)";
        err=SKGServices::executeSqliteOrder ( getDocument(), sql );

        //status bar
        if ( err.isSucceeded() ) err=SKGError ( 0, i18nc ( "Successful message after an user action", "Unused payees deleted" ) );
        else err.addError ( ERR_FAIL, i18nc ( "Error message", "Unused payees deletion failed" ) );
    }
    QApplication::restoreOverrideCursor();

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

QWidget* SKGPayeePluginWidget::zoomableWidget() const
{
    return ui.kView;
}
#include "skgpayeepluginwidget.moc"


