#ifeval SC.have_fortran

private variable EntryPoint = struct {  % {{{
   name,		% function/subroutine name
   mangled_name,	% name as it must be called from C scope
   retval,
   args,
   untyped_args		% arguments whose types still need to be discerned
};
private variable Func_Wrapper_Suffix = "sfwrap";
private variable Return_Value_Name = "OUTPUT";
private variable Fortran_String_Type = "character";
private variable Implicits = String_Type[26];
Implicits[[0:7]] = "real";
Implicits[[8:13]] = "integer";
Implicits[[14:25]] = "real";
% }}}

private define is_comment(ch) % {{{
{
   if (orelse {ch == 'C'} {ch == 'c'} {ch == '*'})
      return 1;
   return 0;
} % }}}

private define eat_white_lines(fp, eof_ok) % {{{
{
   variable ch, l, buf, nlines = 0, nchars = 0;

   forever {

	if ( fread (&ch, Char_Type, 1, fp) == -1) {
	   !if (eof_ok) eof_error(_function_name);
	   return nlines;
	}

	if (ch == NEWLINE) {
	   nlines++;
	   nchars = 0;
	}
	else if (andelse {nchars == 0} {is_comment(ch)}) {
	   % Comment line are considered white, and thus skipped
	   if (andelse {fgets(&buf,fp) == -1} {not(eof_ok)})
		eof_error(_function_name);
	   nlines++;
	   nchars = 0; 
	}
	else {
	   nchars++;
	   !if (is_white(ch)) break;
	}
   }

   if (nchars) backup(fp, nchars);
   return nlines;
} % }}}

private define get_line(fp, eof_ok) % {{{
{
   variable l;

   do {
	() = eat_white_lines(fp, eof_ok);
	if (fgets(&l,fp) == -1) {
	   if (eof_ok) return NULL;
	   eof_error(_function_name);
	}

	if (string_match(l, "\\(.*\\)!.*$",1))	% remove embedded comments
	   l = get_match(l, 1);

	forever {

	   if (eat_white_lines(fp, 1)) continue;

	   % look ahead for possible line continuation
	   variable buf, n = fread(&buf, Char_Type, 6, fp);
	   if (orelse {n < 6} {buf[0]==TAB} {buf[5]==SPACE} {buf[5]==TAB}) {
#ifexists LLong_Type
		n = typecast(n, LLong_Type);
#endif
		() = fseek(fp, -n, SEEK_CUR);
		break;
	   }
	   else {			   % append continuation to prev line
		if (fgets(&buf,fp) == -1)
		   eof_error(_function_name);

	   if (string_match(buf, "\\(.*\\)!.*$",1))  % remove embedded comments
		buf = get_match(buf, 1);

		l += " " + buf;
	   }
	}

	% Strip line labels, if present
	if (string_match(l, "^[ \t]*[0-9]+\\(.*\\)", 1))
	   l = get_match(l, 1);

	% Compress each continguous bit of whitespace into single space
	l = strcompress(l," \t\r\n\f");

   } while (l == EMPTY);		% return only non-blank lines

   return strlow(l);			% and in lower case
} % }}}

private define symbol_new(name, type) % {{{
{
   variable s = struct { name, type, dim};
   s.name = name;
   s.type = type;
   s.dim = 0;
   return s;
}
% }}}

private define emit_line() % {{{
{
   variable args = __pop_args(_NARGS);
   emit("      ");
   emit( __push_args(args) );
   emit("\n");
} % }}}

private define emit_comment() % {{{
{
   variable args = __pop_args(_NARGS);
   emit("c      ");
   emit( __push_args(args) );
   emit("\n");
} % }}}

private define emit_function_wrappers(in, function_entry_points) % {{{
{
   !if (length(function_entry_points)) return 1;

   variable out = Func_Wrapper_Suffix + "_" + path_basename(in);
   SC.outfp = fopen(out, "w+");
   if (SC.outfp == NULL) return 0;

   % Ensure this appears as a dependency in generated makefile content
   SC.obj_code = [ SC.obj_code, path_sans_extname(out) + ".o" ];
   SC.ldflags = [ SC.ldflags, "$(FCLIBS)" ];

   emit_comment("This file was generated by SLIRP (version %s)", SC.version);
   emit_comment("(c) 2005 Massachusetts Institute of Technology %s",
						"(mnoble@space.mit.edu)");
   emit_comment("It contains FORTRAN subroutine wrappers for the functions");
   emit_comment("contained within %s, to maximize portability.\n",in);

   foreach(function_entry_points) {

	variable f = (), args = String_Type[0], decls = String_Type[0], arg;
	if (f.retval.type == VOID) continue;

	foreach([f.args])  {
	   arg = ();
	   args = [args, arg.name];
	   decls = [sprintf("%s %s", arg.type, arg.name), decls ];
	}

	% First the subroutine declaration
	variable d = sprintf("      subroutine %s%s (", f.name,
	      						Func_Wrapper_Suffix);
	emit(d);
	variable pos = strlen(d) + 1, i = 0, nargs = length(args);
	while (i < nargs) {
	   if (i) { emit(","); pos++; }
	   arg = args[i];
	   if (pos >= 60) {			% precaution to avoid lines
		emit("\n     $ ");		% that are too long
		pos = 8;
	   }
	   emit("%s",arg);
	   pos += strlen(arg);
	   i++;
	}
	emit(")\n");

	emit_line("external %s",f.name);	% declare wrapped func as
	decls[-1] = decls[-1] + "," + f.name;	% external, typed accordingly

	% Now the variable declarations
	array_map(Void_Type, &emit_line, decls);

	% And finally the actual function call
	emit_line("%s = %s(%s)",f.retval.name,f.name,strjoin( args[[1:]], ","));
	emit_line("end\n");
   }

   () = fclose(SC.outfp);
   return 1;
} % }}}

private define make_c_prototype(m) % {{{
{
   variable p = sprintf("extern void %s ( ", m.name);

   variable sig = EMPTY, ref, extra;
   foreach(m.args) {
	variable a = (), name = EMPTY;
	variable tmap = get_typemap(a.type, 0);
	if (a.type == Fortran_String_Type) {
	   ref = EMPTY;
	   % Insert a string len param, so that the FORTRAN
	   % entry point may be called correctly from C scope
	   extra = sprintf (", FTN_STR_LEN");
	   % Note that an annotation will be transparently applied
	   % to elide this length param from S-Lang scope calls
	}
	else {
	   % If this is a function return val, ensure that it's named
	   % in the signature, so that the OUTPUT annotation is applied
	   % All other args are left unnamed in the prototype, both for
	   % brevity and to avoid clashes with reserved C identifiers
	   if (a.name == Return_Value_Name)
		name = Return_Value_Name;
	   ref = "*";
	   extra = EMPTY;
	}
	p = sprintf("%s%s%s %s%s, ", p, tmap.ltype, ref, name, extra);
   }
   return sprintf("%s );", strtrim_end(p, ", "));
} % }}}

private define make_line_descriptor(line) % {{{
{
   variable ld = struct { line, index, len };
   ld.line = line;
   ld.index = 0;
   ld.len = strlen(line);
   return ld;
} % }}}

private define get_token(ld) % {{{
{
   % Whitespace-delimited line tokenizer; EOL is indicated
   % by EMPTY return token (and line index is not advanced)

   variable line = ld.line, len = ld.len, i = ld.index, token = EMPTY;
   if (i >= len) return token;

   while (i < len) {				% skip leading whitespace
	variable ch = line[i];
	if (ch == NEWLINE) return token;
	!if (is_white(ch)) break;
	i++;
   }

   while (i < len) {

	ch = char(line[i]);

	if (Single_Char_Tokens[ch]) {
	   if (token == EMPTY) {
		i++;
		token = ch;
	   }
	   break;
	}

	token += ch;
	i++;
	if (is_white(line[i]))
	   break;
   }

   ld.index = i;
   return token;
} % }}}

private define get_type_token(token_ref, type_ref, ld) % {{{
{
   variable token = @token_ref, type, size = EMPTY;

   switch(token)
   { case "double" :
	token = get_token(ld);
	if (token == "precision")
	   type = "double precision";
	else
	   type = "double complex";
   }
   { case "complex" or case "real" or case "logical" or
     case "integer" or case Fortran_String_Type :

	type = token;
   }
   { return 0; }

   token = get_token(ld);
   if (token == "*") {
	size = get_token(ld);
	if (size == "(") {		% character string size: ignored
	   () = get_token(ld);		%	actual # of chars
	   () = get_token(ld);		% 	close paren
	}
	else
	   size = "*" + size;

	token = get_token(ld);
	if (type != Fortran_String_Type)
	   type = sprintf("%s%s",type,size);
   }

   @token_ref = token;			% NB: token advanced when type found
   @type_ref = type;
   return 1;
} % }}}

private define parse_implicit_type_spec(tok, ld) % {{{
{
   variable type, size;
   while (tok != EMPTY) {

	() = get_type_token(&tok, &type, ld);		% "(" ignored here
	tok = get_token(ld);				% get first letter
	do {
	   variable lower = tok, upper;
	   tok = get_token(ld);
	   if (tok == "-") {
		upper = get_token(ld);
		tok = get_token(ld);
	   }
	   else
		upper = lower;

	   lower = int(lower) - 97;
	   upper = int(upper) - 97;
	   Implicits[[lower:upper]] = type;

	} while (tok != ")");

	tok = get_token(ld);
   }
} % }}}

private define resolve_arg_types(fp, m) % {{{
{
   variable close_block = 0, allow_implicit = 1;

   forever {

	variable type = EMPTY, size = EMPTY;
	variable line = strtrans(get_line(fp, 0), ",", " ");  % commas removed
	variable ld =  make_line_descriptor(line);
	variable tok = get_token(ld);

	while (tok != EMPTY) {

	   if (get_type_token(&tok, &type, ld)) {

		% Got a type, so resolve any named arguments of this type
		while (tok != EMPTY) {
		   variable symbol = m.untyped_args[tok];
		   if (symbol != NULL) {
			assoc_delete_key (m.untyped_args, tok);
			symbol.type = type;
			tok = get_token(ld);
			if (tok == "(") {	% record array dimensionality
			   forever {
				tok = get_token(ld);
				if (tok == ")") break;
				if (tok == ":")
				   tok = get_token(ld);
				else
				   symbol.dim++;
			   }

			   % Passing character string arrays not supported yet
			   % The C array must be morphed into a 1D sequence of
			   % FORTRAN strings, all of equal length, b4 passing
			   if (symbol.type == Fortran_String_Type)
				return NULL, warn_ignore( SC.infname +
					":" + m.name,
					"String arrays unsupported: " +
					symbol.name);

			   tok = get_token(ld);
			}
		   }
		   else
			tok = get_token(ld);
		}

	   }
	   else if (tok == "implicit") {

		tok = get_token(ld);

		if (tok == "none")
		   allow_implicit = 0;
		else 
		   parse_implicit_type_spec(tok, ld);

		break;
	   }
	   else {

		if (andelse {strncmp(tok, "common", 6)}
			    {strncmp(tok, "parameter",9)}
			    {strncmp(tok, "dimension",9)}
			    {strncmp(tok, "include",7)})
		   close_block = 1;

		break;
	   }

	}

	if (close_block) {
	   while (tok != "end")
		tok = get_line(fp, 0);
	   break;
	}
   }

   variable untyped = assoc_get_keys(m.untyped_args);
   if (length(m.untyped_args)) {

	!if (allow_implicit)
	    return NULL, warn_ignore(SC.infname + ":" + m.name,
				sprintf("Implicitly typed arguments: %s",
				strjoin( assoc_get_keys(m.untyped_args)," ")));
	else {

	    foreach(m.untyped_args) {

		variable typechar, a = ();
		if (a.name == Return_Value_Name)
		   typechar = m.name[0];
		else
		   typechar = a.name[0];

		if (typechar < 97 or typechar > 122)
		   return NULL, warn_ignore(SC.infname + ":" + m.name,
			sprintf("argument of unknown type: %c", typechar));

		a.type = Implicits[ typechar - 97 ];
	    }
	}
   }

   return m;
} % }}}

private define entry_point_new(retval, signature) % {{{
{
   variable m = @EntryPoint;
   m.args = Struct_Type[0];
   m.untyped_args = Assoc_Type[Any_Type, NULL];

   () = string_match(signature,"^\\([^(]+\\)\\(.*\\)",1);
   m.name = strtrim(get_match(signature, 1));
   m.mangled_name = m.name;
   variable arglist = str_delete_chars(get_match(signature, 2),"() \t");
   foreach (strtok(arglist, ",")) {
	variable arg = ();
	arg = symbol_new(arg, EMPTY);
	m.untyped_args[arg.name] = arg;
	m.args = [m.args, arg];
   }

   m.retval = symbol_new(Return_Value_Name, retval);
   if (retval != VOID) {
	m.args = [m.retval, m.args ];
	m.mangled_name += Func_Wrapper_Suffix;	% C scope calls functions via
   }						% subroutine wrappers

   m.mangled_name += SC.fortran_mangle_suffix;
   if (andelse	{SC.fortran_mangle_extra_uscore} {is_substr(m.name, "_")})
	m.mangled_name += "_";
	
   if (SC.fortran_mangle_upcase)
	m.mangled_name = strup(m.mangled_name);

   if (retval == EMPTY)
	m.untyped_args[m.name] = m.retval;

   return m;
} % }}}

private define entry_point_referer(fmap, ignored) % {{{
{
   return fmap.data.mangled_name;
} % }}}

private define emit_c_prototype(fmap) % {{{
{
   if (fmap.nargs) {
	variable signature = struct_map(String_Type, fmap.args, "type");
	signature = strjoin(signature, ",");
   }
   else
	signature = VOID;
   emit("extern LINKAGE void %s (%s);\n", fmap.data.mangled_name, signature);
} % }}}

private define map_to_c(entryp) % {{{
{
   if (entryp == NULL) return 0;

   % Vectorizing FORTRAN funcs with args > 2D is unsupported, b/c
   % transpose(3D) != [transpose(2D), transpose(2D) ]; supporting
   % this requires that each stride be transposed independently
   variable i, j, maxdim = max([0, struct_map(Int_Type, entryp.args, "dim") ]);
   if (andelse {maxdim > 1} {SC.vectorize or try_vectorize[entryp.name]}) {
	warn_novec(entryp.name, "%dD FORTRAN arrays not vectorizable", maxdim);
	dont_vectorize[entryp.name] = 1;
	assoc_delete_key(SC.interface.functions, entryp.name);
   }

   % This function parse call prohibits overloading and applies annotations
   variable fmap = parse_func_decl( make_c_prototype(entryp), 1, 1);
   if (fmap == NULL) return 0;
   fmap.referer = &entry_point_referer;
   fmap.prototype_hook = &emit_c_prototype;
   fmap.data = entryp;		% store entry point (for name mangling, etc)
   fmap.language = FORTRAN;

   % Prevent FORTRAN use of call-by-ref semantics from potentially
   % being misconstrued in usage statements as C array arguments
   for (i=0, j=0; i < fmap.nargs; i++, j++) {
	variable c_arg = fmap.args[i], fortran_arg = entryp.args[j];
	variable mnemonic = strtrans(c_arg.mnemonic, "[]", "");
	if (fortran_arg.dim) {
	   loop(fortran_arg.dim) mnemonic += "[]";
	   c_arg.layout = COLUMN_MAJOR;
	}
	c_arg.mnemonic = mnemonic;
	if (i == 0 and entryp.retval.type != VOID) {	% apply to function
	   variable retmap = fmap.argmaps[AM_Out][-1];	% return values, too
	   retmap.args[0].mnemonic = mnemonic;
	}
	if (fortran_arg.type == Fortran_String_Type)
	   i++;			% Skip hidden C string length arguments
   }
   return 1;
} % }}}

private define function_decl(fp, functions_array_ref) % {{{
{
   SC.decl = get_line(fp, 1);
   if (SC.decl == NULL) return 0;
   !if (string_match(SC.decl,
		"\\C^\\([A-Za-z \t*12468()]*\\)[  \t]*function\\(.*\\)", 1))
	return 0;

   variable retval = strtrim_end(get_match(SC.decl,1));
   variable entryp = entry_point_new(retval, get_match(SC.decl,2));
   !if (map_to_c(resolve_arg_types(fp, entryp)))
	 return 0;

   @functions_array_ref = [ @functions_array_ref, entryp ];
   return 1;
} % }}}

private define subroutine_decl(fp) % {{{
{
   if (SC.decl == NULL) return 0;
   !if (string_match(SC.decl, "\\Csubroutine\\(.*\\)", 1)) return 0;
   variable entryp = entry_point_new(VOID, get_match(SC.decl,1));
   return map_to_c( resolve_arg_types(fp, entryp) );
} % }}}

private define parse_fortran(fp) % {{{
{
   % FIXME: option needed to turn off subroutine(func) wrapping

   add_tokens(Single_Char_Tokens, "* -");
   variable function_entry_points = Struct_Type[0];

   while( not(feof(fp)) )
	!if (function_decl(fp, &function_entry_points))
	    !if (subroutine_decl(fp))
		continue;

   () = fclose(fp);

   !if (emit_function_wrappers(SC.infname, function_entry_points))
	abort("Could not create function wrappers for: %s", SC.infname);

   remove_tokens(Single_Char_Tokens, "* -");
} % }}}

define f2cprotos(file) % {{{
{
   variable ffp = fopen(file, "rb"), symbols = Assoc_Type[Any_Type, NULL];
   if (ffp == NULL) return NULL;

   SC.infname = file;
   SC.parse_file = &parse_fortran;
   SC.wrapped_some_fortran = 1;

   % Convenience:  automatically add .o file to link line
   SC.obj_code = [ SC.obj_code, path_sans_extname(file) + ".o" ];

   % Ensure SLang-scoped function names do not reflect subroutine wrapping
   SC.renames["\\(.+\\)"+Func_Wrapper_Suffix+"$"] = EMPTY;

   return ffp;
} % }}}

#else

define f2cprotos(file) % {{{
{
   return NULL;
} % }}}

#endif

provide("slirpf2c");
