/* GdkPixbuf library - FITS image loader	{{{
 *
 * The FITS file format is defined by the NASA Office of Standards and
 * Technology (NOST).  See http://fits.gsfc.nasa.gov for more information.
 *
 * Copyright (C) Massachusetts Institute of Technology
 * Authors: Michael S. Noble <mnoble@space.mit.edu>
 *
 * NOTE: The gdk-pixbuf library presently supports only 2D images, but
 *       the FITS format allows for images of up to 999 dimensions.  This
 *	 module will return valid a GdkPixbuf* for:
 *
 *	 2D FITS images, in which case the entire image is returned
 *	 3D images, in which case ONLY the 2D slice along Z=1 is returned
 *	 N > 3D images ONLY if each additional dimension is of size 1,
 *		in which case, again, only a 2D slice is returned 
 *
 * Note that even for incremental loading the entire image will be read
 * before a GdkPixbuf is created.  This supports scaling the pixel values
 * from the native file type to 8-bit, and makes incremental loading
 * (and the overall implementation) much cleaner, but probably violates
 * the GdkPixbufLoader "philosophy".
 * 
 * This code was drawn from the XV file viewer and heavily adapted.
 * The original XV FITS module was written by David Robinson:
 *
 * Copyright 1992, 1993, 1994 by David Robinson.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  This
 * software is provided "as is" without express or implied warranty.
 *
 * -----------------------------------------------------------------------
 *
 * This software was partially developed by the MIT Center for Space
 * Research under contract SV1-61010 from the Smithsonian Institution.
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * the supporting documentation, and that the name of the Massachusetts
 * Institute of Technology not be used in advertising or publicity
 * pertaining to distribution of the software without specific, written
 * prior permission.  The Massachusetts Institute of Technology makes
 * no representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 * 
 * THE MASSACHUSETTS INSTITUTE OF TECHNOLOGY DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE MASSACHUSETTS
 * INSTITUTE OF TECHNOLOGY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 }}} */



/* Header inclusions and type defintions {{{ */
#include <stdio.h>
#include <stdlib.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <string.h>
#include "gdk-pixbuf/gdk-pixbuf.h"
#include "gdk-pixbuf/gdk-pixbuf-io.h"

#ifndef SIZEOF_CHAR				/* just a mnemonic */
#define SIZEOF_CHAR 1
#endif

#define GDK_FITS_NOMEM	"Insufficient memory to load FITS file"
#define GDK_FITS_WRERR  "Error writing FITS file"
#define GDK_FITS_UNSUP  "Cannot output image with requested # bits per channel"
#define GDK_FITS_NULLP  "Attempted to dereference a NULL pointer"
#define MODULE_NAME	"fits"
#define maxmin(x, max, min) {\
  maxmin_t=(x);  if(maxmin_t > max) max=maxmin_t; \
  if (maxmin_t<min) min=maxmin_t;}

/* MONO returns total intensity of r,g,b triple (i = .33R + .5G + .17B) */
#define MONO(rd,gn,bl) ( ((int)(rd)*11 + (int)(gn)*16 + (int)(bl)*5) >> 5)

static const char* _version_string = _VERSION_STRING;
typedef enum _FIOState	{ READING_FITS_HEADER, READING_FITS_PIXELS } FIOState;
typedef enum  datatype	{ T_INT, T_LOG, T_STR, T_NOVAL } KEY_TYPE;

typedef struct {
  
  #define BLOCKSIZE 2880	/* FITS record size, in bytes */
  #define CARDSIZE  80

   /* FITS fields */
  FILE*	   fp;		/* file pointer */
  char*	   buffer;	/* buffer of size BLOCKSIZE bytes */
  int	   bufpos;	/* current pos (in bytes) of read ptr w/in buffer */
  int	   bitpix;	/* FITS BITPIX convention: indicates pixel datatype */
  int	   psize;	/* size of each pixel, in bytes */
  int	   naxis;	/* number of axes */
  int	   curr_axis;	/* last axis value parsed from header */
  int	   nx;		/* width of image */
  int	   ny;		/* height of image */
  gulong   npixels;	/* number of pixels in the image */
  FIOState iostate;
  char	   scratch[81];	/* workspace for odd jobs */

  void*  fits_pixels;
  gulong image_nbytes_needed;
  gulong header_bytes;

  /* gdk-pixbuf fields */
  GdkPixbufModulePreparedFunc	prepared_func;
  GdkPixbufModuleUpdatedFunc	updated_func;
  gpointer			user_data;
  GdkPixbuf			*pixbuf;
  GError			**error;

} FITS;
/* }}} */

/* Miscellaneous: error handling, allocation {{{ */
static int set_g_error(GError **error, const char *cause)
{
   g_set_error (error, GDK_PIXBUF_ERROR, GDK_PIXBUF_ERROR_CORRUPT_IMAGE, cause);
   return -1;
}

static int
error_cleanup(FITS *fs, char *errstr)
{
   g_free(fs->buffer);
   g_free(fs->fits_pixels);
   return set_g_error(fs->error,errstr);
}

static GdkPixbuf*
allocate_pixbuf(FITS *fs)
{
   fs->pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, fs->nx, fs->ny);
   return fs->pixbuf;
}

static FITS* allocate_fits(FILE *fp, GError **error)
{
   FITS *fs = (FITS*) calloc (1, sizeof(FITS) );

   if (fs != NULL) {
	fs->buffer = (char *) malloc(BLOCKSIZE);
	if (fs->buffer == NULL) {
	   g_free(fs);
	   fs = NULL;
	}
	fs->fp = fp;
	fs->error = error;
   }

   return fs;
}
/* }}} */

static int getkey(FITS *fs, char *name, KEY_TYPE type, long int* value) /*{{{*/
{

  /* Read the named keyword from the FITS header, and parse its value
   * according to it keytype.  This routine is concerned only with:
   *
   *    keytype = T_LOG
   *		value is logical, either 'T' or 'F' in column 30.
   *    keytype = T_INT
   *		value is an integer, right justified in columns 11-30.
   *
   * Returns zero on success, or non-zero (with a filled-in GError**)
   *
   * The buffer position will be advanced 80 bytes for each key extracted.
   * (see the FITS standard for definition of keys as 80-byte 'cards')
   */

   int  i, ptr;
   char namestr[9];
   char *card = fs->buffer + fs->bufpos;

   memcpy(namestr, card, (size_t) 8);

   for (i=7; i>=0 && namestr[i] == ' '; i--);
   namestr[i+1] = '\0';

   if (strcmp(namestr, name) != 0) {
	sprintf(fs->scratch,"FITS keyword %s not found in file", name);
	return error_cleanup(fs,fs->scratch);
   }
  
   /* get start of value */
   ptr = 10;
   while (ptr < CARDSIZE && card[ptr] == ' ') ptr++;
   if (ptr == CARDSIZE) {
	sprintf(fs->scratch,"FITS keyword %s missing a value",name);
	return error_cleanup(fs,fs->scratch);
   }
  
   if (type == T_LOG) {
	/* LOGICAL */
	if (ptr != 29 || (card[29] != 'T' && card[29] != 'F')) {
	   sprintf(fs->scratch,"FITS keyword %s has bad 'LOGICAL' value", name);
	   return error_cleanup(fs,fs->scratch);
	}
    *value = (card[29] == 'T');
   } 
   else {
   	/* INTEGER */
	int j;
	long int ival;
	char num[21];
    
	if (ptr > 29) {
	   sprintf(fs->scratch,"FITS keyword %s has bad INTEGER value", name);
	   return error_cleanup(fs,fs->scratch);
	}

	memcpy(num, &card[ptr], (size_t) (30-ptr));
	num[30-ptr] = '\0';
	j = sscanf(num, "%ld", &ival);
	if (j != 1) {
	   sprintf(fs->scratch,"FITS keyword %s has bad INTEGER value", name);
	   return error_cleanup(fs,fs->scratch);
	}
	*value = ival;
   }
 
   fs->bufpos += CARDSIZE;
   return 0;
}	/* }}} */

static void putkey(FITS *fs,char *name,KEY_TYPE type,int ival,char *sval)/*{{{*/
{
  /* write a header record into the 80 byte buffer card.
   * The keyword name is passed in name. The value type is in dtype; this
   * can have the following values:
   *    dtype = T_NOVAL
   *         no keyword value is written
   *    dtype = T_LOG
   *         a logical value, either 'T' or 'F' in column 30 is written
   *    dtype = T_INT
   *         an integer is written, right justified in columns 11-30
   *    dtype = T_STR
   *         a string 'sval' of length ival is written, in columns 9-80.
   */

   char *card = fs->buffer + fs->bufpos;

   fs->bufpos += CARDSIZE;
   memcpy(card, name, (size_t) strlen(name));
  
   if (type == T_NOVAL) return;
  
   if (type == T_STR) {
	if (ival <= 0) return;
	if (ival > 72) ival = 72;
	memcpy(card+8, sval, (size_t) ival);
	return;
   }
  
   card[8] = '=';
  
   if (type == T_LOG)
	card[29] = ival ? 'T' : 'F';
   else {					/* T_INT */
	sprintf(card+10, "%20d", ival);
	card[30] = ' ';
   }
}	/* }}} */

static int read_header(FITS *fs)	/* {{{ */
{
   /* Fill out the given FITS *fs with the results of scanning
    * the prevously opened file referred to by FITS *fs.
    *
    * Returns 0 on success, or non-zero (with a filled-in GError **).
    *
    * Will return an error message if the primary data unit is not a
    * 2 or 3-dimensional array.
    */
 
   int j, key_error;
   char name[9];
   long int val;         /* keyword value */

   fs->bufpos = 0;
   if (fs->bitpix == 0) {

	/* read SIMPLE key */
	key_error = getkey(fs, "SIMPLE", T_LOG, &val);
  	if (key_error) return key_error;
	if (val == 0)
	   return error_cleanup(fs,"Not a SIMPLE FITS file");

	/* read BITPIX key */
	key_error = getkey(fs, "BITPIX", T_INT, &val);
	if (key_error) return key_error;

	switch(val) {
	   case 8:	/* ASCII characters or 8-bit unsigned integers */
	   case 16:	/* 16-bit, twos complement signed integers  */
	   case 32:	/* 32-bit, twos complement signed integers */
	   case -32:	/* IEEE 32-bit floating point values */
	   case -64:	/* IEEE 64-bit floating point values */
		break;
	   default:
		return error_cleanup(fs,"Bad BITPIX value in FITS file");
	}

	j = fs->bitpix = val;
	if (j<0) j = -j;
	fs->psize = j/8;
   }

   if (fs->naxis == 0) {
	/* read NAXIS key */
	key_error = getkey(fs, "NAXIS", T_INT, &val);
	if (key_error) return key_error;
	if (val < 0 || val > 999)
	return error_cleanup(fs,"Bad NAXIS value in FITS file");
	if (val == 0)
	return error_cleanup(fs,"FITS file does not contain a primary image");
	if (val < 2)
	return error_cleanup(fs,"FITS file has fewer than two dimensions");

	fs->naxis = val;
	fs->curr_axis = 1;
   }

   /* Read NAXISnnn keys: see comment regarding ND images at top of module */
   while(fs->curr_axis <= fs->naxis && fs->bufpos < BLOCKSIZE) {

    sprintf(name, "NAXIS%d", fs->curr_axis);
    key_error = getkey(fs, name, T_INT, &val);
    if (key_error) return key_error;
    if (val < 0)
	return error_cleanup(fs,"Bad NAXISn value in FITS file");
    if (val == 0)
	return error_cleanup(fs,"FITS file does not contain a primary image");
    
    if (fs->curr_axis == 1)
	fs->nx = val;
    else if (fs->curr_axis == 2) {
	fs->ny = val;
	fs->npixels = fs->nx * fs->ny;
    }
    else if (fs->curr_axis > 2 && val != 1) {
	sprintf(fs->scratch,
	      "Unable to read non-trivial %d-dimensional FITS image",fs->naxis);
	return error_cleanup(fs,fs->scratch);
    }

    fs->curr_axis++;
  }

  /* Ignore remainder of header */
  while (fs->bufpos < BLOCKSIZE && fs->iostate == READING_FITS_HEADER) {

	if (strncmp(fs->buffer + fs->bufpos, "END     ", (size_t) 8) == 0) {
	   fs->iostate = READING_FITS_PIXELS;
	}
	else
	   fs->bufpos += CARDSIZE;
  }

  fs->bufpos = 0;
  return 0;
}	/* }}} */

static int write_header(FITS *fs)	/* {{{ */
{
  memset(fs->buffer, ' ', BLOCKSIZE);	/* Write minimalist FITS header */
  putkey(fs, "SIMPLE", T_LOG, 1, NULL);
  putkey(fs, "BITPIX", T_INT, fs->bitpix, NULL);
  putkey(fs, "NAXIS",  T_INT, 2, NULL);
  putkey(fs, "NAXIS1", T_INT, fs->nx, NULL);
  putkey(fs, "NAXIS2", T_INT, fs->ny, NULL);

  sprintf(fs->scratch, "Created by FITS GdkPixbuf Module (SLgtk), version %s",
							_version_string);
  putkey(fs, "HISTORY",T_STR,strlen(fs->scratch),fs->scratch); 

  strcpy(fs->scratch, "Author: Michael S. Noble");
  strcat(fs->scratch," (MIT Kavli Institute for Astrophysics)");
  putkey(fs, "HISTORY",T_STR,strlen(fs->scratch),fs->scratch); 

  putkey(fs, "END", T_NOVAL, 0, NULL);

  if (fwrite(fs->buffer, sizeof(char), BLOCKSIZE, fs->fp) != BLOCKSIZE)
	return set_g_error(fs->error, GDK_FITS_WRERR);

  return 0;
}				/* }}} */

static int scale_to_8bit(FITS *fs) /* {{{ */
{
   guchar *cbuff;
   int i, numread;

  if (fs->bitpix == 8)
     return fs->npixels;

   numread = fs->npixels;
   cbuff = (guchar*) malloc(numread);
   if (cbuff == NULL) {
	set_g_error(fs->error, GDK_FITS_NOMEM);
	return 0;
   }

  if (fs->bitpix == 16) {		/* convert short int to byte */
    short int *buffer = fs->fits_pixels;
    int max, min, maxmin_t;
    float scale;
    
    min = max = buffer[0];
    for (i=1; i < numread; i++, buffer++) maxmin(*buffer, max, min);
    scale = (max == min) ? 0. : 255./(float)(max-min);
    
    /* rescale and convert */
    for (i=0, buffer = fs->fits_pixels; i < numread; i++)
      cbuff[i] = (guchar)(scale*(float)((int)buffer[i]-min));
    
  } 

  else if (fs->bitpix == 32) {		/* convert long int to byte */
    int *buffer = fs->fits_pixels;
    int max, min, maxmin_t;
    float scale, fmin;
    
    min = max = buffer[0];
    for (i=1; i < numread; i++, buffer++) maxmin(*buffer, max, min);
    scale = (max == min) ? 1. : 255./((double)max-(double)min);
    fmin = (float)min;
    
    /* rescale and convert */
    if (scale < 255./2.1e9) /* is max-min too big for an int ? */
      for (i=0, buffer = fs->fits_pixels; i < numread; i++)
	cbuff[i] = (guchar)(scale*((float)buffer[i]-fmin));
    else /* use integer subtraction */
      for (i=0, buffer = fs->fits_pixels; i < numread; i++)
	cbuff[i] = (guchar)(scale*(float)(buffer[i]-min));
    
   
  } 
  else if (fs->bitpix == -32) {		/* convert float to byte */
    float *buffer = fs->fits_pixels;
    float max, min, maxmin_t, scale;
    
    min = max = buffer[0];
    for (i=1; i < numread; i++, buffer++) maxmin(*buffer, max, min);
    scale = (max == min) ? 0. : 255./(max-min);
    
    /* rescale and convert */
    for (i=0, buffer = fs->fits_pixels; i < numread; i++)
      cbuff[i] = (guchar)(scale*(buffer[i]-min));
  } 

  else if (fs->bitpix == -64) {		/* convert double to byte */
    double *buffer = fs->fits_pixels;
    double max, min, maxmin_t, scale;
    
    min = max = buffer[0];
    for (i=1; i < numread; i++, buffer++) maxmin(*buffer, max, min);
    scale = (max == min) ? 0. : 255./(max-min);
    
    /* rescale and convert */
    for (i=0, buffer = fs->fits_pixels; i < numread; i++)
      cbuff[i] = (guchar)(scale*(buffer[i]-min));
  }

  g_free(fs->fits_pixels);
  fs->fits_pixels = cbuff;
  return numread;
} /* }}} */

static int make_native(FITS *fs, int nelem)	/* {{{ */
{
  /* convert the raw data, as stored in the FITS file, to the format
   * appropiate for the data representation of the host computer.
   * Assumes that
   *  short int = 2 or more byte integer
   *  int       = 4 or more byte integer
   *  float     = 4 byte floating point, not necessarily IEEE.
   *  double    = 8 byte floating point.
   * 
   * Returns number of elements "fixed,"  or zero on error 
   */

  int   i, n = nelem;
  guchar* ptr = fs->fits_pixels;

  if (nelem <= 0)
     return 0;

  /*
   * conversions. Although the data may be signed, reverse using unsigned 
   * variables.
   * Because the native int types may be larger than the types in the file,
   * we start from the end and work backwards to avoid overwriting data
   * prematurely.
   */

  /* convert from big-endian two-byte signed integer to native form */
  if (fs->bitpix == 16) {
    unsigned short int *iptr=(unsigned short int *)ptr;
    iptr += n-1;    /* last short int */
    ptr += (n-1)*2; /* last pair of bytes */
    for (i=0; i < n; i++, ptr-=2, iptr--)
      *iptr = (((int)*ptr) << 8) | (int)(ptr[1]);
    }

  /* convert from big-endian four-byte signed integer to native form */
  else if (fs->bitpix == 32) {
    unsigned int *iptr = (unsigned int *)ptr;
    iptr += n-1;     /* last integer */
    ptr  += (n-1)*4; /* last 4 bytes */
    for (i=0; i < n; i++, ptr-=4, iptr--)
      *iptr = ((unsigned int)ptr[0] << 24) |
	      ((unsigned int)ptr[1] << 16) |
	      ((unsigned int)ptr[2] << 8)  |
	      ((unsigned int)ptr[3]);
  }
  
  /* convert from IEE 754 single precision to native form */
  else if (fs->bitpix == -32) {
    int j, k, expo;
    float *exps = (float *)malloc(256 * sizeof(float));

    if (exps == NULL)
	return 0;

    exps[150] = 1.;
    for (i=151; i < 256; i++) exps[i] = 2.*exps[i-1];
    for (i=149; i >= 0; i--) exps[i] = 0.5*exps[i+1];
	      
    for (i=0; i < n; i++, ptr+=4) {
      k = (int)*ptr;
      j = ((int)ptr[1] << 16) | ((int)ptr[2] << 8) | (int)ptr[3];
      expo = ((k & 127) << 1) | (j >> 23);
      if ((expo | j) == 0) *(float *)ptr = 0.;
      else *(float *)ptr = exps[expo]*(float)(j | 0x800000);
      if (k & 128) *(float *)ptr = - *(float *)ptr;
    }

    free(exps);

  }
  
  /* convert from IEE 754 double precision to native form */
  else if (fs->bitpix == -64) {
    int expo, k, l;
    unsigned int j;
    double *exps = (double *)malloc(2048 * sizeof(double));

    if (exps == NULL)
       return 0;

    exps[1075] = 1.;
    for (i=1076; i < 2048; i++) exps[i] = 2.*exps[i-1];
    for (i=1074; i >= 0; i--) exps[i] = 0.5*exps[i+1];
	      
    for (i=0; i < n; i++, ptr+=8) {
      k = (int)*ptr;
      j = ((unsigned int)ptr[1] << 24) | ((unsigned int)ptr[2] << 16) |
	((unsigned int)ptr[3] << 8) | (unsigned int)ptr[4];
      l = ((int)ptr[5] << 16) | ((int)ptr[6] << 8) | (int)ptr[7];
      expo = ((k & 127) << 4) | (j >> 28);
      if ((expo | j | l) == 0) *(double *)ptr = 0.;
      else *(double *)ptr = exps[expo] * (16777216. *
		         (double)((j&0x0FFFFFFF)|0x10000000) + (double)l);
      if (k & 128) *(double *)ptr = - *(double *)ptr;
    }

    free(exps);
  }

  return nelem;
}	/* }}} */

static int get_pixels(FITS *fs)	/* {{{ */
{
  /* reads nelem values into the buffer.
   * returns NULL for success or an error message.
   * Copes with the fact that the last 2880 byte record of the FITS file
   * may be truncated, and should be padded out with zeros.
   *  bitpix   type of data
   *    8        byte
   *   16        short int       (NOT 2-byte integer)
   *   32        int             (NOT 4-byte integer)
   *  -32        float
   *  -64        double
   *
   * Returns the number of elements actually read.
   */

  int nelem = fs->npixels;
  int res = fread(fs->fits_pixels, (size_t) fs->psize, (size_t) nelem, fs->fp);

  /* if failed to read all the data because at end of file */
  if (res != nelem && feof(fs->fp)) {
    /* nblock is the number of elements in a record. 
       size is always a factor of BLOCKSIZE */

    int loffs, nblock = BLOCKSIZE/fs->psize;

    /*
     * the last record might be short; check this.
     * loffs is the offset of the start of the last record from the current
     * position.
     */

    loffs = ((fs->npixels + nblock - 1) / nblock - 1) * nblock;

    /* if we read to the end of the penultimate record */
    if (res >= loffs) {
      /* pad with zeros */
      memset((char *)fs->fits_pixels + res*fs->psize, 0,
	    			(size_t) ((nelem-res)*fs->psize));
      res = nelem;
    }
  }

  return make_native(fs, res);
}	/* }}} */

static void reflect_image(FITS *fs) /* {{{ */
{
   /* By convention FITS uses a Cartesian coordinate system, with the
    * first pixel (0,0) in the lower left corner.  This routine reflects
    * the image through the line y=ny/2, to match the "usual" compsci
    * graphics convention of having (0,0) in the upper left corner.
   */
  register int i, j, v;
  register guchar *chunk1, *chunk2;
  register int ny = fs->ny;
  register int reflection_point = ny / 2;
  register int nx = fs->nx;
  register guchar *pixels = fs->fits_pixels;
  
  for (i=0; i < reflection_point; i++) {
    chunk1 = &pixels[i*nx];
    chunk2 = &pixels[(ny-1-i) * nx];
    for (j=0; j < nx; j++) {
      v = *chunk1;
      *(chunk1++) = *chunk2;
      *(chunk2++) = v;
    }
  }
} /* }}} */

static int read_image(FITS *fs) /* {{{ */
{
  /* Reads a byte image from the FITS file fs. The image contains nelem pixels.
   * If bitpix = 8, then the image is loaded as stored in the file.
   * Otherwise, it is rescaled so that the minimum value is stored as 0, and
   * the maximum is stored as 255.  Returns the number of image pixels read.
   */

  int numread;

  fs->fits_pixels = malloc(fs->npixels * fs->psize);
  if (fs->fits_pixels == NULL) {
	set_g_error(fs->error, GDK_FITS_NOMEM);
	return 0;
  }

  numread = get_pixels(fs);
  if (numread == 0) return 0;	   /* it's FITS-legal to have no pixel data */

  return scale_to_8bit(fs);
} /* }}} */

int write_image(FITS *fs)	/* {{{ */
{
   register unsigned char *rgb_pixels, *p;
   register FILE *fp = fs->fp;
   register int x, y, w = fs->nx, h = fs->ny, num_channels, row_stride;

   if (write_header(fs) == -1)
	return FALSE;

   /* FITS images are greyscale, with no transparency. Here we output only
    * 2D images, and implicitly skip any alpha channel w/in the pixbuf by
    * incrementing the pixel index according to the total # of channels */

   rgb_pixels = gdk_pixbuf_get_pixels (fs->pixbuf);
   num_channels = gdk_pixbuf_get_n_channels(fs->pixbuf);
   row_stride = gdk_pixbuf_get_rowstride(fs->pixbuf);

   for (y=h-1; y >= 0; y--) {		/* flip lines; see reflect_image() */
	p = rgb_pixels + y*row_stride;
	for (x=0; x < w; x++, p += num_channels)
	   putc(MONO(p[0], p[1], p[2]), fp);
   }

   /* Pad with zeros until last block is completely filled */
   fs->npixels = w * h;
   x = ((fs->npixels + BLOCKSIZE - 1) / BLOCKSIZE) * BLOCKSIZE - fs->npixels;
   while (x--) putc('\0', fp);

   return TRUE;
}		/* }}} */

static void fits_to_rgb(FITS *fs) /* {{{ */
{
   /* Expand single-byte FITS image pixels into 3-byte gdk-pixbuf
   * RGB pixels, taking into account the alignment padding 
   * introduced by gdk-pixbuf to optimize memory access */

   register int x;
   register int nx = fs->nx;
   register int padding = gdk_pixbuf_get_rowstride(fs->pixbuf) - nx*3;
   register guchar *rgb_pixel = gdk_pixbuf_get_pixels(fs->pixbuf);
   register guchar *fits_pixel = fs->fits_pixels;
   register guchar *last_fits_pixel = fits_pixel + fs->npixels;

   while (fits_pixel < last_fits_pixel) {
	for (x = 0; x < nx; x++, fits_pixel++, rgb_pixel += 3)
		rgb_pixel[0] = rgb_pixel[1] = rgb_pixel[2] = *fits_pixel;
	rgb_pixel += padding;
   }

   /* at this point the fits pixels are no longer needed */
   g_free(fs->fits_pixels);
   fs->fits_pixels = NULL;
} /* }}} */

/* GdkPixbuf interface {{{ */
static gpointer
gdk_pixbuf__fits_image_begin_load(GdkPixbufModuleSizeFunc size_func,
                                  GdkPixbufModulePreparedFunc prepared_func,
                                  GdkPixbufModuleUpdatedFunc updated_func,
                                  gpointer user_data,
                                  GError **error)
{
   FITS *context = allocate_fits(NULL, NULL);

   if ( context == NULL) {
	if (error != NULL)
	   set_g_error(error, GDK_FITS_NOMEM);
   }
   else {
	context->prepared_func = prepared_func;
	context->updated_func = updated_func;
	context	->user_data = user_data;
   }

   return (gpointer) context;
}

static gboolean
gdk_pixbuf__fits_image_stop_load(gpointer context, GError **error)
{
   FITS *fs = (FITS *) context;
   fs->error = error;

   if (fs == NULL || error == NULL) {
	if (error != NULL)
	   set_g_error(error, GDK_FITS_NULLP);
	return FALSE;
   }

   if (allocate_pixbuf(fs) == NULL)
	return FALSE;

   if (fs->prepared_func)
	fs->prepared_func(fs->pixbuf, NULL, fs->user_data);

   if (make_native(fs, fs->npixels) != fs->npixels)
	return FALSE;

   scale_to_8bit(fs);
   reflect_image(fs);
   fits_to_rgb(fs);

   if (fs->pixbuf)
	g_object_unref(fs->pixbuf);
   g_free(fs->buffer);
   g_free(fs);

   return TRUE;
}

static gboolean
gdk_pixbuf__fits_image_load_increment(gpointer context,
				const guchar* buf,
                                guint size,
				GError **error)
{
   /* Read a FITS file, incrementally from the given buf, which MUST be  */
   /* at least BLOCKSIZE bytes in size for this module to work properly. */
   FITS *fs = (FITS *) context;
   int bufpos = 0, hdr_nbytes_to_copy;

   if (fs == NULL || buf == NULL || error == NULL) {
	if (error != NULL)
	   set_g_error(error, GDK_FITS_NULLP);
	return FALSE;
   }

   fs->error = error;

   while (fs->iostate == READING_FITS_HEADER) {

	   hdr_nbytes_to_copy = MIN( BLOCKSIZE - fs->bufpos, size - bufpos);

	   if (hdr_nbytes_to_copy) {
		memcpy(fs->buffer + fs->bufpos,buf + bufpos,hdr_nbytes_to_copy);
		bufpos += hdr_nbytes_to_copy;
		fs->bufpos += hdr_nbytes_to_copy;
	   }

	   if (fs->bufpos == BLOCKSIZE) {
		/* Have enough header bytes to continue looking for cards */
		if (read_header(fs) !=0)
		   return FALSE;
	   }
	   else {
		/* Not done with header, but we have no more bytes left  */
		/* in this buffer.  So, remember # of bytes read thus    */
		/* far into the current header block, then rtrn for more */
		return TRUE;
	   }
   }

   /* Done retrieving header, ready to grab image pixels */
   /* For accurate count of pixel data remaining in buffer (if any),
   * reduce buffer size by number of bytes in the keywords read */
   if (fs->fits_pixels == NULL)  {
	size -= bufpos;
	buf += bufpos;
	fs->image_nbytes_needed = fs->npixels * fs->psize;
	fs->fits_pixels = malloc(fs->image_nbytes_needed);
	if (fs->fits_pixels == NULL) {
	   set_g_error(fs->error,GDK_FITS_NOMEM);
	   return FALSE;
	}
	fs->bufpos = 0;
   }

   /* We have to keep track of the exact number of bytes */
   /* read for the image, since some of the bytes in the */
   /* final FITS block might just be padding, not pixels */
   size = MIN(fs->image_nbytes_needed,size);
   if (size > 0) {
	memcpy( ((char*)fs->fits_pixels)+fs->bufpos, buf, size);
	fs->bufpos += size;
	fs->image_nbytes_needed -= size;
   }
   return TRUE;
}

static GdkPixbuf *
gdk_pixbuf__fits_image_load (FILE *fp, GError **error)
{
   FITS *fs;
   int  numread;
   GdkPixbuf *pixbuf;
   register guchar* fits_pixel;
   register guchar* last_fits_pixel;

   if (fp == NULL || error == NULL) {
	if (error != NULL)
	   set_g_error(error, GDK_FITS_NULLP);
	return FALSE;
   }

   fs = allocate_fits(fp, error);
   if (fs == NULL) {
	set_g_error(error,GDK_FITS_NOMEM);
	return NULL;
   }
      
   while (fs->iostate == READING_FITS_HEADER) {

	numread = fread(fs->buffer, SIZEOF_CHAR, (size_t) BLOCKSIZE, fs->fp);
	if (numread != BLOCKSIZE) {
	   (void)error_cleanup(fs,"Error reading FITS file header");
	   return NULL;
	}
	if (read_header(fs) != 0)
	   return NULL;
   }

   numread = read_image(fs);
   if (numread == 0)
	return NULL;				/* error: nothing read */

   if (numread < fs->npixels) {			/* read only partial image */
	fits_pixel = (guchar*)fs->fits_pixels + (fs->psize * numread);
	last_fits_pixel = fits_pixel + fs->npixels;
	while (fits_pixel < last_fits_pixel)
	   *fits_pixel++ = 0x80;		/* pad with grey */
   }

   if ( (pixbuf = allocate_pixbuf(fs)) != NULL) {
	reflect_image(fs);
	fits_to_rgb(fs);
   }

   g_free(fs->buffer);
   g_free(fs);

   return pixbuf;
}

static gboolean gdk_pixbuf__fits_image_save (FILE *fp,
					GdkPixbuf *pixbuf,
					gchar  **param_keys,
					gchar  **param_values,
					GError **error)
{
   FITS *fs;
   int bps;
   gboolean retval;
  
   (void) param_keys; (void) param_values;

   if (fp == NULL || pixbuf == NULL || error == NULL) {
	if (error != NULL)
	   set_g_error(error, GDK_FITS_NULLP);
	return FALSE;
   }

   bps = gdk_pixbuf_get_bits_per_sample(pixbuf);
   if (bps != 8) {
	set_g_error(error, GDK_FITS_UNSUP);
	return FALSE;
   }

   if ( (fs = allocate_fits(fp, error)) == NULL) {
	set_g_error(error, GDK_FITS_NOMEM);
	return FALSE;
   }

   fs->pixbuf = pixbuf;
   fs->bitpix = bps;
   fs->nx = gdk_pixbuf_get_width (fs->pixbuf);
   fs->ny = gdk_pixbuf_get_height (fs->pixbuf);

   retval = write_image(fs);

   g_free(fs->buffer);
   g_free(fs);

   return retval;
}

#ifdef SLGTK_STATIC_MODULE
static
#endif

void
fill_vtable (GdkPixbufModule *module)
{
   (void) _version_string;			/* silence compile warning */

   module->begin_load = gdk_pixbuf__fits_image_begin_load;
   module->stop_load = gdk_pixbuf__fits_image_stop_load;
   module->load_increment = gdk_pixbuf__fits_image_load_increment;
   module->load = gdk_pixbuf__fits_image_load;	/* faster for entire image */
   module->save = gdk_pixbuf__fits_image_save;	
}

#ifdef SLGTK_STATIC_MODULE
static
#endif

void
fill_info (GdkPixbufFormat *info)
{
   static GdkPixbufModulePattern signature[] = {
	{ (unsigned char*)"SIMPLE  =                    T", NULL, 100 }, 
	{ NULL, NULL, 0 }
   };
   static gchar * mime_types[] = {
	"image/fits",
	NULL
   };
   static gchar * extensions[] = {
	MODULE_NAME,
	NULL
   };

   info->name = MODULE_NAME;
   info->signature = signature;
   info->description = "The FITS file format";
   info->mime_types = mime_types;
   info->extensions = extensions;
   info->flags = GDK_PIXBUF_FORMAT_WRITABLE;
}
#ifdef SLGTK_STATIC_MODULE
extern gboolean
gdk_pixbuf_add_module(
		gchar *name,
      		GdkPixbufModuleFillInfoFunc fill_info,
		GdkPixbufModuleFillVtableFunc fill_vtable);
#endif

gboolean load_fits_pixbuf_module()
{
#ifdef SLGTK_STATIC_MODULE
   /* The add_module function is a gdk-pixbuf extension bundled   */
   /* with SLgtk, in order to include the FITS pixbuf reader w/in */
   /* an entirely-statically-linked SLgtk module.		  */
   return gdk_pixbuf_add_module(MODULE_NAME,fill_info,fill_vtable);
#else
   return FALSE;
#endif
}
/* }}} */
