/*
#ident	"@(#)smail/src:RELEASE-3_2_0_115:log.c,v 1.57 2003/04/02 22:15:06 woods Exp"
 */

/*
 *    Copyright (C) 1987, 1988 Ronald S. Karr and Landon Curt Noll
 *    Copyright (C) 1992  Ronald S. Karr
 * 
 * See the file COPYING, distributed with smail, for restriction
 * and warranty information.
 */

/*
 * log.c:
 *	system and per-message logging functions
 *
 *	These functions send information to a per-system log file and to
 *	a per-message log file, manage the creation use and removal of
 *	per-message logs and handle panic and fatal messages.
 *
 *	external functions:  open_system_logs, close_system_logs,
 *			     open_msg_log, close_msg_log, unlink_msg_log,
 *			     panic, write_log, send_log, scan_msg_log
 */

#include "defs.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

#ifdef STDC_HEADERS
# include <stdlib.h>
# include <stddef.h>
#else
# ifdef HAVE_STDLIB_H
#  include <stdlib.h>
# endif
#endif

#ifdef HAVE_STRING_H
# if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#  include <memory.h>
# endif
# include <string.h>
#endif
#ifdef HAVE_STRINGS_H
# include <strings.h>
#endif

#ifdef __STDC__
# include <stdarg.h>
#else
# include <varargs.h>
#endif

#if defined(HAVE_UNISTD_H)
# include <unistd.h>
#endif

#if defined(UNIX_SYS5) || defined(POSIX_OS) || defined(USE_FCNTL)
# include <fcntl.h>
#else
# if defined(UNIX_BSD)
#  include <sys/file.h>
# endif
#endif
#ifdef __GLIBC__
# include <sys/file.h>
#endif

#include "smail.h"
#include "main.h"
#include "addr.h"
#include "alloc.h"
#include "smailstring.h"
#include "dys.h"
#include "log.h"
#include "spool.h"
#include "exitcodes.h"
#include "debug.h"
#include "extern.h"
#include "smailport.h"

#ifdef STANDALONE
# define xmalloc malloc
# define xrealloc realloc
# define xfree free
#endif	/* STANDALONE */

/* exported variables */
FILE *msg_logfile = NULL;	/* open stream to per-message log */
FILE *panicfile = NULL;		/* open stream to panic log file */
FILE *logfile = NULL;		/* open stream to system log file */

/* variables local to this file */
static char *msg_log_fn;	/* name of per-message log file */

/* functions local to this file */
static void build_msg_log_fn __P((void));
static FILE *open_a_log __P((char *, char *, char *, unsigned int));
static void try_mkdir __P((char *));
static void write_log_va __P((int, char *, va_list));


/*
 * open_system_logs - open the panic and the general information log files.
 *
 * Access to the system log should not require that much be functional
 * or that resources be allocated to send to the system log.  For that
 * reason, we allocate all resources in advance, while resources are
 * available and the system is assumed to be somewhat sane.
 *
 * If reasonable, the system logs should be opened after a message has
 * been read to a spool file so that if a panic occurs because a log
 * could not be opened we can recover mail later when the problem is
 * solved.
 */
void
open_system_logs()
{
    static char panicbuf[BUFSIZ];	/* stdio buffer to avoid unchecked mallocs */
    static char logbuf[BUFSIZ];		/* stdio buffer to avoid unchecked mallocs */

    if (!panicfile) {
	panicfile = open_a_log(panic_fn, "panic log", panicbuf, log_mode);
	if (!panicfile) {
	    if (only_testing) {
		fprintf(stderr, "%s: [%lu] cannot open panic log %s: %s\n", program, (long unsigned int) getpid(), panic_fn, strerror(errno));
	    } else {
		fprintf(stderr, "%s: cannot open panic log %s: %s\n", program, panic_fn, strerror(errno));
	    }
	    panicfile = stderr;
	}
    }
    if (!logfile) {
	logfile = open_a_log(log_fn, "system log", logbuf, log_mode);
	if (!logfile) {
	    if (only_testing) {
		fprintf(stderr, "%s: [%lu] cannot open system log %s: %s\n", program, (long unsigned int) getpid(), log_fn, strerror(errno));
	    } else {
		fprintf(stderr, "%s: cannot open system log %s: %s\n", program, log_fn, strerror(errno));
	    }
	    logfile = stderr;
	}
    }
    return;
}

/*
 * open_a_log() - open a log file for append, creating it if necessary, and if
 *                auto_mkdir is set then perhaps also creating its directory.
 *
 * If 'buf' is set then call setbuf() with it.
 *
 * Returns a pointer to a STDIO FILE structure.
 */
/*
 * XXX this should probably be rewritten to use fopen(fqfn, "a")
 * [with a fix to use umask((~mode) & 0777)) too of course]
 */
static FILE *
open_a_log(fqfn, fn, buf, mode)
    char *fqfn;				/* fully qualified filename */
    char *fn;				/* internal filename (for debug/error msgs) */
    char *buf;				/* stdio buffer for fp */
    unsigned int mode;			/* access modes to create with */
{
    FILE *fp = NULL;
    int fd;
    unsigned int oumask = umask(0);	/* we supply an explicit mode */

#ifdef O_APPEND
    fd = open(fqfn, O_CREAT | O_APPEND | O_WRONLY, mode);
#else
    /* limited to V7 open semantics  XXX this old junk should go away.... */
    fd = open(fqfn, 1);
    if (fd < 0) {
	DEBUG3(DBG_LOG_HI, "open_a_log(%s): %s: open() failed, will try creat(): %s\n", fqfn, fn, strerror(errno));
	fd = creat(fqfn, mode);
    } else {
	(void) lseek(fd, (off_t) 0L, SEEK_END);	/* XXX could this ever fail? */
    }
#endif
    if (fd < 0) {
#ifdef O_APPEND
	DEBUG3(DBG_LOG_MID, "open_a_log(%s): %s: initial open() failed: %s\n", fqfn, fn, strerror(errno));
#else
	DEBUG3(DBG_LOG_MID, "open_a_log(%s): %s: initial creat() failed: %s\n", fqfn, fn, strerror(errno));
#endif

	/* perhaps the directory just needs to be created */
	if (auto_mkdir && errno == ENOENT) {
	    try_mkdir(fqfn);
#ifdef O_APPEND
	    fd = open(fqfn, O_CREAT | O_APPEND | O_WRONLY, mode);
#else
	    /* limited to V7 open semantics  XXX this old junk should go away.... */
	    /* this time we know it's not there so just create it */
	    fd = creat(fqfn, mode);
#endif
	    if (fd < 0) {
#ifdef O_APPEND
		DEBUG3(DBG_LOG_LO, "open_a_log(%s): %s: second try open() failed: %s\n", fqfn, fn, strerror(errno));
#else
		DEBUG3(DBG_LOG_LO, "open_a_log(%s): %s: second try creat() failed: %s\n", fqfn, fn, strerror(errno));
#endif
	    }
	}
    }
    if (fd >= 0) {
	if (!(fp = fdopen(fd, "a"))) {
	    DEBUG3(DBG_LOG_LO, "open_a_log(%s): %s: fdopen() failed: %s\n", fqfn, fn, strerror(errno));
	}
	if (fp && buf) {
	    setbuf(fp, buf);
	}
    }
    (void) umask(oumask);

    return fp;
}

/*
 * try_mkdir - try to build the directory for the given filename
 */
static void
try_mkdir(fn)
    char *fn;
{
    char *slash = strrchr(fn, '/');
    char *dr;
    struct stat st;

    if (slash == NULL) {
	return;				/* ignore bad filename */
    }

    /* figure directory name */
    while (slash > fn && *(slash - 1) == '/') {
	--slash;
    }
    if (slash == fn) {
	return;			/* root directory */
    }
    dr = xmalloc((size_t) (slash - fn + 1));
    (void) memcpy(dr, fn, (size_t) (slash - fn));
    dr[slash - fn] = '\0';

    DEBUG1(DBG_LOG_LO, "make directory %s: ", dr);
    errno = 0;
    (void) mkdir(dr, auto_mkdir_mode);
    DEBUG1(DBG_LOG_LO, "%s\n", strerror(errno));

    if (stat(dr, &st) == -1 && errno == ENOENT) {
	char *slash2 = strrchr(dr, '/');

	if (slash2) {
	    while (slash2 > dr && *(slash2 - 1) == '/') {
		--slash2;
	    }
	    if (slash2 != dr) {
		*slash2 = '\0';
		DEBUG1(DBG_LOG_LO, "    make parent directory %s: ", dr);
		errno = 0;
		(void) mkdir(dr, auto_mkdir_mode);
		DEBUG1(DBG_LOG_LO, "%s\n", strerror(errno));
		*slash2 = '/';
		DEBUG1(DBG_LOG_LO, "    re-make directory %s: ", dr);
		errno = 0;
		(void) mkdir(dr, auto_mkdir_mode);
		DEBUG1(DBG_LOG_LO, "%s\n", strerror(errno));
	    }
	}
    }

    xfree(dr);
}

/*
 * close_system_logs - close the panic and general info log files.
 */
void
close_system_logs()
{
    if (logfile) {
	(void) fclose(logfile);
	logfile = NULL;
    }
    if (panicfile) {
	(void) fclose(panicfile);
	panicfile = NULL;
    }
}


/*
 * open_msg_log - open message log file, one per message
 *
 * a per-message log should be opened once for each message and closed
 * when done processing a message.  It is intended to be information
 * that a sender might be interested in, and will be sent back to
 * the sender if the return_to_sender flag is set when processing of
 * the message is completed.
 */
void
open_msg_log()
{
    static char msgbuf[BUFSIZ];		/* stdio buffer to avoid unchecked mallocs */

    /*
     * if msg_fn not yet set up create a suitably unique value
     */
    if (msg_logfile) {
	(void) fclose(msg_logfile);
	msg_logfile = NULL;
    }
    build_msg_log_fn();
    msg_logfile = open_a_log(msg_log_fn, "message log", msgbuf, message_log_mode);
    if (!msg_logfile) {
	/*
	 * otherwise, panic.  We are assuming that the mail queue entry can be
	 * scanned at a later date (note: errno may not be useful from open_a_log)
	 */
	panic(EX_OSFILE, "cannot open %s/%s: %s", spool_dir, msg_log_fn, strerror(errno));
	/*NOTREACHED*/
    }
    return;
}

/*
 * build_msg_log_fn - build the name for the per-message log file
 */
static void
build_msg_log_fn()
{
    static char buf[sizeof("msglog/") + SPOOL_FN_LEN + 1];

    (void) sprintf(buf, "msglog/%s", spool_fn);
    msg_log_fn = buf;
}

/*
 * close_msg_log - close the per-message log file
 *
 * This should be called when further processing of a message is
 * being postponed to some point in the future.
 */
void
close_msg_log()
{
    if (msg_logfile) {
	(void) fclose(msg_logfile);
	msg_logfile = NULL;
    }
}

/*
 * unlink_msg_log - close and unlink the per-message log file
 *
 * use this when a message has been processed completely.
 */
void
unlink_msg_log()
{
    close_msg_log();
    if (msg_log_fn) {
	(void) unlink(msg_log_fn);
	msg_log_fn = NULL;
    }
}


/*
 * panic - panic and die!
 *
 * Attempt to write to the panic and system log files. and no matter what we
 * also try to write to stderr.
 *
 * If this is called after spooling a message, then the message will be put in
 * the error queue so that the postmaster can reprocess it after correcting
 * whatever error caused the panic in the first place.
 *
 */
/*VARARGS2*/
#ifdef __STDC__
void
panic(int exitcode, char *fmt, ...)
#else
void
panic(exitcode, fmt, va_alist)
    int exitcode;			/* we will call exit(exitcode) */
    char *fmt;				/* printf(3) format */
    va_dcl                              /* arguments for printf */
#endif
{
    va_list ap;

    if (errfile) {
	VA_START(ap, fmt);
	write_log_va(WRITE_LOG_TTY, fmt, ap);
	va_end(ap);
	if (only_testing) {
	    exit(force_zero_exitvalue ? 0 : exitcode);
	}
    }
    VA_START(ap, fmt);
    write_log_va(WRITE_LOG_PANIC, fmt, ap);
    va_end(ap);

    VA_START(ap, fmt);
    write_log_va(WRITE_LOG_SYS, fmt, ap);
    va_end(ap);

    if (daemon_pid != 0 && daemon_pid == getpid()) {
	write_log(WRITE_LOG_SYS, "smail daemon exiting on panic");
	if (daemon_pidfile && *daemon_pidfile == '/') {
	    (void) unlink(daemon_pidfile);
	}
    }
    if (spool_fn) {
	freeze_message();		/* put message in error/ directory */
    }

    /* XXX we _REALLY_ should have an option to (try to) trigger a core dump here.... */

    exit(force_zero_exitvalue ? 0 : exitcode);
    /* NOTREACHED */
}

#ifndef NODEBUG
# define WRITE_LOG_DEBUG	(WRITE_LOG_LAST << 2) /* magic for write_log_va() */
#endif

/*
 * write_log - write to the per-message log, and perhaps the system log
 *
 * write a log message to the various log files, where `who' is the
 * bitwise OR of WRITE_LOG_SYS, WRITE_LOG_MLOG and/or WRITE_LOG_PANIC.
 *
 * Note that messages with a leading 'X' are for per-message logs.
 */
/*VARARGS2*/
#ifdef __STDC__
void
write_log(int who, char *fmt, ...)
#else
void
write_log(who, fmt, va_alist)
    int who;				/* mask of log files to be written */
    char *fmt;				/* printf(3) format */
    va_dcl                              /* arguments for printf */
#endif
{
    va_list ap;
#ifndef NODEBUG
    int wct = 0;

    if (debug && errfile && (debug > 1 || fmt[0] != 'X')) {
	/* if we are debugging at all, print all logged messages */
	(void) fputs("write_log(", errfile);
	if (who & WRITE_LOG_SYS) {
	    (void) fputs("SYS", errfile);
	    wct++;
	}
	if (who & WRITE_LOG_PANIC) {
	    (void) fprintf(errfile, "%sPANIC", wct++ ? "|" : "");
	}
	if (who & WRITE_LOG_MLOG) {
	    (void) fprintf(errfile, "%sMLOG", wct++ ? "|" : "");
	}
	if (who & WRITE_LOG_TTY) {
	    (void) fprintf(errfile, "%sTTY", wct++ ? "|" : "");
	}
	if (who & WRITE_LOG_DEBUG) {
	    (void) fprintf(errfile, "%sDEBUG", wct++ ? "|" : "");
	}
	(void) fputs("): ", errfile);
	VA_START(ap, fmt);
	write_log_va(WRITE_LOG_DEBUG, fmt, ap);
	va_end(ap);
	if (only_testing || who == WRITE_LOG_TTY) {
	    return;
	}
    }
#endif /* NODEBUG */
    if (errfile && ((who & WRITE_LOG_TTY) ||
		   ((who & (WRITE_LOG_MLOG|WRITE_LOG_PANIC)) &&
		    (error_processing == TERMINAL ||
		     error_processing == ERROR_DEFAULT) && /* XXX ??? */
		    fmt[0] != 'X'))) {
	VA_START(ap, fmt);
	write_log_va(WRITE_LOG_TTY, fmt, ap);
	va_end(ap);
    }
    if (only_testing) {
	return;
    }
    if (who & WRITE_LOG_SYS) {
	VA_START(ap, fmt);
	write_log_va(WRITE_LOG_SYS, fmt, ap);
	va_end(ap);
    }
    if (who & WRITE_LOG_PANIC) {
	VA_START(ap, fmt);
	write_log_va(WRITE_LOG_PANIC, fmt, ap);
	va_end(ap);
    }
    if (who & WRITE_LOG_MLOG) {
	VA_START(ap, fmt);
	write_log_va(WRITE_LOG_MLOG, fmt, ap);
	va_end(ap);
    }
    return;
}

static void
write_log_va(who, fmt, ap)
    int who;				/* mask of log files to be written */
    char *fmt;				/* printf(3) format */
    va_list ap;                         /* arguments for printf */
{
    switch (who) {
#ifndef NODEBUG
    case WRITE_LOG_DEBUG:
	(void) vfprintf(errfile, fmt, ap);
	/* log messages don't come with their own \n */
	(void) fprintf(errfile, "\n");
	(void) fflush(errfile);
	break;

#endif	/* NODEBUG */

    case WRITE_LOG_TTY:
	if (only_testing) {
	    (void) fprintf(errfile, "%s: ", program);
	} else {
	    (void) fprintf(errfile, "%s: %s: [%lu] ", program, time_stamp(), (long unsigned int) getpid());
	}
	if (message_id) {
	    (void) fprintf(errfile, "[%s] ", message_id);
	}
	(void) vfprintf(errfile, fmt, ap);
	/* log messages don't come with their own \n */
	(void) fprintf(errfile, "\n");
	(void) fflush(errfile);
	if (ferror(errfile)) {
	    int oerrno = errno;
	    int fd;

	    fd = fileno(errfile);
	    errfile = NULL;
	    panic(EX_IOERR, "error writing %s[%d]: %s", "errfile", fd, strerror(oerrno));
	    /*NOTREACHED*/
	}
	break;

    case WRITE_LOG_SYS:
	/* write out permanent log to the system log file */
	if (logfile == NULL || panicfile == NULL) {
	    open_system_logs();		/* if system log not open, open it */
	}
#ifdef LOG_EXTRA_NEWLINE
	(void) fputc('\n', logfile);
#endif
	if (only_testing) {
	    (void) fprintf(logfile, "%s: %s: [%lu] log: ", program, time_stamp(), (long unsigned int) getpid());
	} else {
	    (void) fprintf(logfile, "%s: [%lu] ", time_stamp(), (long unsigned int) getpid());
	}
	if (message_id) {
	    (void) fprintf(logfile, "[%s] ", message_id);
	}
	(void) vfprintf(logfile, fmt, ap);
	/* log messages don't come with their own \n */
	(void) fprintf(logfile, "\n");
	(void) fflush(logfile);
	if (ferror(logfile)) {
	    int oerrno = errno;
	    int fd;
	    struct str str;
		
	    STR_INIT(&str);
	    str_printf_va(&str, fmt, ap);
	    fd = fileno(logfile);
	    logfile = NULL;
	    panic(EX_IOERR, "error writing the following message to %s[%d]: %s\n%s",
		  log_fn, fd, strerror(oerrno), STR(&str));
	    /*NOTREACHED*/
	}
	break;

    case WRITE_LOG_PANIC:
	if (panicfile == NULL) {
	    open_system_logs();		/* if system log not open, open it */
	}
#ifdef LOG_EXTRA_NEWLINE
	(void) fputc('\n', logfile);
#endif
	if (only_testing) {
	    (void) fprintf(panicfile, "%s: %s: [%lu] panic: ", program, time_stamp(), (long unsigned int) getpid());
	} else {
	    (void) fprintf(panicfile, "%s: [%lu] ", time_stamp(), (long unsigned int) getpid());
	}
	if (message_id) {
	    (void) fprintf(panicfile, "[%s] ", message_id);
	}
	(void) vfprintf(panicfile, fmt, ap);
	/* log messages don't come with their own \n */
	(void) fprintf(panicfile, "\n");
	(void) fflush(panicfile);
	if (ferror(panicfile)) {
	    int oerrno = errno;
	    int fd;

	    /*
	     * Note: no sense potentially going in a loop here by calling
	     * panic() again, so instead just clear the errant panicfile so
	     * that nothing tries again, and so in case panic() itself calls
	     * itself then mabye the open_system_logs() above will work.  In
	     * the mean time just try writing the message to errfile and
	     * logfile if possible.
	     */
	    fd = fileno(panicfile);
	    panicfile = NULL;
	    if (errfile) {
		(void) fprintf(errfile, "error writing following message to %s[%d]: %s\n", panic_fn, fd, strerror(oerrno));
		(void) vfprintf(errfile, fmt, ap);
		/* log messages don't come with their own \n */
		(void) fprintf(errfile, "\n");
		(void) fflush(errfile);
	    }
	    if (logfile) {
		(void) fprintf(logfile, "error writing following message to %s[%d]: %s", panic_fn, fd, strerror(oerrno));
		(void) vfprintf(logfile, fmt, ap);
		/* log messages don't come with their own \n */
		(void) fprintf(logfile, "\n");
		(void) fflush(logfile);
	    }
	}
	break;

    /*
     * NOTE:  if there is no spool_dir set, then a per-message logfile
     *	      cannot be created, so don't bother writing per-message
     *	      log entries.
     */
    case WRITE_LOG_MLOG:
	if (spool_dir) {
	    if (msg_logfile == NULL) {
		open_msg_log();		/* if message log not open, open it */
	    }

	    /* write out the message to the per-message log file */
	    (void) vfprintf(msg_logfile, fmt, ap);
	    /* log messages don't come with their own \n */
	    (void) fprintf(msg_logfile, "\n");
	    (void) fflush(msg_logfile);
	    if (ferror(msg_logfile)) {
		int oerrno = errno;
		int fd;
		struct str str;
		
		STR_INIT(&str);
		str_printf_va(&str, fmt, ap);
		fd = fileno(msg_logfile);
		msg_logfile = NULL;
		panic(EX_IOERR, "error writing the following message to %s/%s[%d]: %s\n%s",
		      spool_dir, msg_log_fn, fileno(msg_logfile), strerror(oerrno), STR(&str));
		/*NOTREACHED*/
	    }
	}
	break;

    }
}

/*
 * send_log - write out the per-message log file.
 *
 * Send the per-message log to a file with or without lines that begin
 * with the magic X character.  Also, an optional banner string will
 * be output before any log data.  If no log data is output then the
 * banner string is also not output.
 *
 * Note: we reopen to keep from changing the write position while
 *       reading.
 */
void
send_log(f, all, banner)
    FILE *f;				/* send to this file */
    int all;				/* if TRUE, also send X lines */
    char *banner;			/* if non-NULL precedes log output */
{
    FILE *mlog;				/* message log file */
    char *mln;				/* line from log file */
    int lastx = FALSE;			/* last line started with 'X' */

    build_msg_log_fn();
    /* reopen log file to avoid changing the write position */
    if (! (mlog = fopen(msg_log_fn, "r"))) {
	DEBUG3(DBG_LOG_MID, "send_log(): no message log file: %s/%s: %s.\n", spool_dir, msg_log_fn, strerror(errno));
	return;
    }
    /* copy from message-log to the specified file */
    while ((mln = read_line(mlog))) {
	int len;

	/* trim the trailing newline, if any */
	len = strlen(mln);
	if (mln[len - 1] == '\n') {
	    mln[len - 1] = '\0';
	}
	if (!all && *mln == 'X') {
	    lastx = TRUE;
	    continue;
	}
	if (lastx) {
	    /* ignore any SMTP replies following the 'X' line too */
	    if (strlen(mln) >= 4 && isdigit((int) mln[0]) && isdigit((int) mln[1]) && isdigit((int) mln[2])) {
		if (mln[3] == ' ') {
		    lastx = FALSE;
		}
		continue;
	    } else {
		lastx = FALSE;
	    }
	}
	if (banner) {
	    (void) fputs(banner, f);
	    banner = NULL;		/* output banner at most once */
	}
	(void) fprintf(f, " %s\n", mln);
    }
    (void) fclose(mlog);		/* don't need this anymore */
}

/*
 * scan_msg_log - scan for X entries in the per-message logfile
 *
 * if first is TRUE, open the message log and scan for the first line
 * marked with an X, return that line.  On successive calls, where
 * first is FALSE, return successive lines marked with an X.  Return
 * NULL if no more such lines are found.
 *
 * returned value points to a string area which may be reused on subsequent
 * calls to scan_msg_log.
 */
char *
scan_msg_log(first)
    int first;				/* TRUE to re-open the message log */
{
    static FILE *mlog = NULL;		/* opened message log */
    static struct str line;		/* line marked with X */
    static int inited_line = FALSE;	/* TRUE if STR_INIT called for line */
    register int c, last_c;

    build_msg_log_fn();
    if (!mlog) {
	DEBUG2(DBG_LOG_HI, "scan_msg_log(): %s scanning message log file: %s.\n", first ? "starting" : "re-starting", msg_log_fn);
	if (!(mlog = fopen(msg_log_fn, "r"))) {	/* reopen log file */
	    DEBUG2(DBG_LOG_MID, "scan_msg_log(): no message log file: %s: %s.\n", msg_log_fn, strerror(errno));
	    return NULL;
	}
    }
    if (first) {
	rewind(mlog);
    }
    last_c = '\n';

    /* scan for line beginning with X */
    while ((c = getc(mlog)) != EOF) {
	if (c == 'X' && last_c == '\n') {
	    break;
	}
	last_c = '\n';
    }
    if (c == EOF) {
	/* reached end of file without finding another X line */
	DEBUG1(DBG_LOG_HI, "scan_msg_log(): finished scanning message log file: %s.\n", msg_log_fn);
	(void) fclose(mlog);
	mlog = NULL;
	return NULL;
    }
    /* found a line marked with X, read it in */
    if (! inited_line) {
	STR_INIT(&line);
	inited_line = TRUE;
    } else {
	STR_CHECK(&line);
	STR_CLEAR(&line);
    }
    STR_NEXT(&line, 'X');
    last_c = 'X';
    while ((c = getc(mlog)) != EOF) {
	if (last_c == '\n' && c == 'X') {
	    break;
	}
	STR_NEXT(&line, c);
	last_c = c;
    }
    if (c == 'X') {
	ungetc(c, mlog);		/* back for next time */
    }
    if (last_c == '\n') {
	STR_PREV(&line);		/* trim off the last newline */
    }
    STR_NEXT(&line, '\0');

    /* return that line */
    DEBUG1(DBG_LOG_HI, "scan_msg_log() returns:\n'%s'\n", STR(&line));

    return STR(&line);
}


#ifdef STANDALONE

/*
cc -g -DSTANDALONE -o log.stand log.c spool.o alloc.o string.o sysdep.o config.o ascii.o
*/

int debug = 0;				/* adjust as desired */

static char *panic_fn = "paniclog";
static char *log_fn = "logfile";
static char *cons_fn = "/dev/console";
static char *spool_fn = "spoolfile";
char *msg_log_dir = "/tmp";
char *program = "log";
int exitvalue = 0;
enum er_proc error_processing = MAIL_BACK;
int return_to_sender = FALSE;
int islocal = TRUE;
char *sender = "nsc!tron";
char *sender_name = "MasterControlProgram";
char *local_sender;
int force_zero_exitvalue = FALSE;
FILE *errfile = stderr;
long unsigned int daemon_pid = 4000000000;	/* 0xEE6B2800 */
int real_uid = 0;

char *
bind_compute_domain()
{
    return NULL;
}

/*
 * excersize the logging code by performing the various operations
 * in a sequence given at run time on the standard input.
 */
/* ARGSUSED */
int
main(argc, argv)
    int argc;
    char *argv[];
{
    register int c;			/* hold key for current operation */
    char *line;				/* a line of input */
    char *p;

    while ((c = getchar()) != EOF) {
	switch (c) {
	case 'O':
	    open_system_logs();
	    break;
	case 'C':
	    close_system_logs();
	    break;
	case 'o':
	    open_msg_log();
	    break;
	case 'c':
	    close_msg_log();
	    break;
	case 's':
	    send_log(stdout, FALSE, "|----- log without X lines -----|\n");
	    break;
	case 'S':
	    send_log(stdout, TRUE, "|----- log with X lines -----|\n");
	    break;
	case 'X':
	    for (p = scan_msg_log(TRUE); p; p = scan_msg_log(FALSE)) {
		printf("%s\n", p);
	    }
	    break;
	case 'p':
	    if ((line = read_line(stdin)))
		panic(EX_OSERR, line);
	    break;
	case 'l':
	    if ((line = read_line(stdin)))
		write_log(WRITE_LOG_SYS, line);
	    break;
	case 'm':
	    if ((line = read_line(stdin)))
		write_log(WRITE_LOG_MLOG, line);
	    break;
	case 'L':
	    if ((line = read_line(stdin)))
		write_log(WRITE_LOG_MLOG|WRITE_LOG_SYS|WRITE_LOG_PANIC, line);
	    break;
	case ' ':
	case '\t':
	case '\n':
	    break;
	default:
	    (void)fprintf(stderr, "%c -- huh?\n", c);
	    break;
	}
    }
    exit(EX_OK);
}
#endif	/* STANDALONE */
