/***************************************************************************
                          smb4kprint  -  description
                             -------------------
    begin                : Tue Mar 30 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qtimer.h>
#include <qfile.h>

// KDE includes
#include <klocale.h>
#include <kapplication.h>
#include <kmessagebox.h>
#include <kurl.h>

// system includes
#include <stdlib.h>

// application specific includes
#include "smb4kprint.h"
#include "smb4kcore.h"
#include "smb4kdefs.h"


Smb4KPrint::Smb4KPrint( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_proc = new KProcess( this, "PrintProcess" );
  m_proc->setUseShell( true );
  
  m_config = kapp->config();
  
  m_item = NULL;
  
  connect( m_proc, SIGNAL( receivedStdout( KProcess *, char *, int ) ), this, SLOT( slotReceivedStdout( KProcess *, char *, int ) ) );
  connect( m_proc, SIGNAL( receivedStderr( KProcess *, char *, int ) ), this, SLOT( slotReceivedStderr( KProcess *, char *, int ) ) );
  connect( m_proc, SIGNAL( processExited( KProcess * ) ),               this, SLOT( slotProcessExited( KProcess * ) ) );
}


Smb4KPrint::~Smb4KPrint()
{
  if ( m_item )
  {
    delete m_item;
  }
  
  abort();
}


/****************************************************************************
   Initializes the printing of a file. This function looks for the mime 
   type and takes care of the further processing.   
****************************************************************************/


bool Smb4KPrint::init( Smb4KPrintInfo *info )
{
  m_config->setGroup( "Programs" );
  QString enscript = m_config->readPathEntry( "enscript" );
  QString dvips = m_config->readPathEntry( "dvips" );  

  m_info = info;  
  bool ok = true;
  
  if ( QFile::exists( m_info->path() ) )
  {  
    // We want to determine the mimetype of the incoming 
    // file. Let's use KFileItem for this, because KFileMetaInfo
    // does not work very reliable.
    KURL u;
    u.setPath( m_info->path() );
    
    if ( m_item )
    {
      delete m_item;
    }
    
    m_item = new KFileItem( KFileItem::Unknown, KFileItem::Unknown, u, false );
    
    QString type = m_item->mimetype();
    
//    qDebug( type );
    
    if ( type == "application/postscript" || 
         type == "application/pdf" ||
         type.startsWith( "image" ) )
    {
      startPrinting();
    } 
    else if ( type == "application/x-dvi" && !dvips.isEmpty() )
    {
      convertDVIToPS();
    }
    else if ( ( type.startsWith( "text" ) || type.startsWith( "message" ) ) && !enscript.isEmpty() )
    {
      if ( KMessageBox::warningContinueCancel( (QWidget *)this, QString( i18n( "The mimetype (%1) indicates that this is a text file. Printing it, however, might fail. Do you want to continue?" ) ).arg( type ) ) == KMessageBox::Continue )
      {
        convertTextToPS();
      }
    }
    else
      emit error( ERROR_MIMETYPE_NOT_SUPPORTED, type );
  }
  else
  {
    emit error( ERROR_FILE_NOT_FOUND, m_info->path() );
    ok = false; 
  }
  
  return ok;
}


/****************************************************************************
   Aborts the current process.
****************************************************************************/

void Smb4KPrint::abort()
{
  if ( m_proc->isRunning() )
  {
    m_proc->kill();
  }
}


/****************************************************************************
   Starts the actual printing.
****************************************************************************/

void Smb4KPrint::startPrinting()
{
  Smb4KAuthInfo *auth = ((Smb4KCore *)parent())->passwdReader()->getAuth( m_info->workgroup(), m_info->host(), m_info->printer() );
  
  QString uri;
  
  // It seems that we must not quote the entries for the DEVICE_URI
  // environment variable. Printing will fail if do it.
  
  if ( !m_info->workgroup().isEmpty() )
  {
    if ( !auth->user().isEmpty() )
    {
      uri = QString( "smb://%1:%2@%3/%4/%5" ).arg( auth->user() ).arg( auth->password() ).arg( m_info->workgroup() ).arg( m_info->host() ).arg( m_info->printer() );
    }
    else
    {
      uri = QString( "smb://%1/%2/%3" ).arg( m_info->workgroup() ).arg( m_info->host() ).arg( m_info->printer() );
    }
  }
  else
  {
    if ( !auth->user().isEmpty() )
    {
      uri = QString( "smb://%1:%2@%3/%4" ).arg( auth->user() ).arg( auth->password() ).arg( m_info->host() ).arg( m_info->printer() );
    }
    else
    {
      uri = QString( "smb://%1/%2" ).arg( m_info->host() ).arg( m_info->printer() );
    }
  }
  
  m_proc->setEnvironment( "DEVICE_URI", uri );
  
  // FIXME: If smbspool starts to use argv[1] and argv[5] 
  // we have to implement them!
  QString command = QString( "smbspool 111 %1 \"Smb4K print job\" %2 \"\" %3" ).arg( getenv( "USER" ) ).arg( m_info->copies() ).arg( KProcess::quote( m_info->path() ) );
  
  *m_proc << command;
  
  startProcess( Print );
}


/****************************************************************************
   Converts DVI files to PS using dvips.
****************************************************************************/

void Smb4KPrint::convertDVIToPS()
{
  QString path = m_info->path().section( "/", 0, -2 );
  QString file = m_info->path().section( "/", -1, -1 );
  
  *m_proc << "cd "+KProcess::quote( path )+" && dvips -P pdf -o /tmp/smb4k_print_$USER.ps "+KProcess::quote( file );
  startProcess( Convert );
}


/****************************************************************************
   Converts text files to PS using enscript.
****************************************************************************/

void Smb4KPrint::convertTextToPS()
{
  *m_proc << "enscript -1 -B --ps-level=2 -o /tmp/smb4k_print_$USER.ps "+KProcess::quote( m_info->path() );
  startProcess( Convert );
}


/****************************************************************************
   Starts any process of this class.
****************************************************************************/

void Smb4KPrint::startProcess( int state )
{
  m_state = state;
  m_buffer = QString::null;
  
  if ( m_state == Print )
  {
    emit running( PRINT_SEND_FILE, true );
  }
  else
  {
    emit running( PRINT_CONVERT_FILE, true );
  }
    
  m_proc->start( KProcess::NotifyOnExit, KProcess::AllOutput );
}


/****************************************************************************
   If the KProcess ended, this function takes care of the further
   processing.
****************************************************************************/

void Smb4KPrint::endProcess()
{
  switch ( m_state )
  {
    case Print:
      endPrintProcess();
      emit running( PRINT_STOP, false );
      break;
    case Convert:
      endConversionProcess();
      break;
    default:
      break;
  }
  
  m_state = Idle;
  m_proc->clearArguments();
}


/****************************************************************************
   Post-processing of any print job.
****************************************************************************/

void Smb4KPrint::endPrintProcess()
{
  if ( m_buffer.contains( "NT_STATUS", true ) != 0 )
  {
    if ( m_buffer.contains( "NT_STATUS_ACCESS_DENIED" ) != 0 || m_buffer.contains( "NT_STATUS_LOGON_FAILURE" ) != 0 )
    {
      int state = Smb4KPasswdReader::None;
       
      if ( m_buffer.contains( "NT_STATUS_ACCESS_DENIED" ) != 0 )
      {
        state = Smb4KPasswdReader::AccessDenied;
      }
      else if (m_buffer.contains( "NT_STATUS_LOGON_FAILURE" ) != 0  )
      {
        state = Smb4KPasswdReader::LogonFailure;
      }
      
      if ( ((Smb4KCore *)parent())->passwdReader()->askpass( m_info->workgroup(), m_info->host(), m_info->printer(), state ) )
      {
        QTimer::singleShot( 50, this, SLOT( slotRetry() ) );
      }
    }
    else
    {
      emit error( ERROR_PRINTING, m_buffer );
    }
  }
  else
  {
    QStringList list = QStringList::split( '\n', m_buffer, false );
    
    if ( list.last().startsWith( "putting file" ) )
    {
      QFile::remove( "/tmp/smb4k_print_"+QString( "%1" ).arg( getenv( "USER" ) )+".ps" );
      emit running( PRINT_STOP, false );
    }
  }
}


/****************************************************************************
   Ends the conversion process.
****************************************************************************/

void Smb4KPrint::endConversionProcess()
{
  if ( m_buffer.contains( "command not found", true ) != 0 )
  {
    QString missing = m_buffer.section( ":", -2, -2 ).section( ":", -1, -1 ).stripWhiteSpace();
        
    emit error( ERROR_COMMAND_NOT_FOUND, missing );
    emit running( PRINT_STOP, false );
    m_state = Idle;
    m_proc->clearArguments();
  }
  else
  {  
    // Set the URL to the newly created PS file:
    m_info->setPath( "/tmp/smb4k_print_"+QString( "%1" ).arg( getenv( "USER" ) )+".ps" );
  
    QTimer::singleShot( 50, this, SLOT( slotRetry() ) );
  }
}


/****************************************************************************
   Reads the advanced options for smbclient
****************************************************************************/

const QString Smb4KPrint::getSmbclientOptions()
{
  QString options;
  
  m_config->setGroup( "Samba" );

  if ( !m_config->readEntry( "Client Resolve Order", QString::null ).isEmpty() )
    options.append( QString( " -R \"%1\"" ).arg( m_config->readEntry( "Client Resolve Order", QString::null ) ) );

  if ( !m_config->readEntry( "Port", QString::null ).isEmpty() )
    options.append( QString( " -p %1" ).arg( m_config->readNumEntry( "Port", 139 ) ) );

  if ( !m_config->readEntry( "Client Buffer Size", QString::null ).isEmpty() )
    options.append( QString( " -b %1" ).arg( m_config->readNumEntry( "Client Buffer Size", 65520 ) ) );

  if ( m_config->readBoolEntry( "Use Kerberos", false ) )
  {
    if ( ((Smb4KCore *)parent())->getSambaVersion().startsWith( "3" ) )
      options.append( " -k" );
  }

  if ( !m_config->readEntry( "NetBIOS Name", QString::null ).isEmpty() )
    options.append( QString( " -n \"%1\"" ).arg( m_config->readEntry( "NetBIOS Name", QString::null ) ) );

  if ( !m_config->readEntry( "NetBIOS Scope", QString::null ).isEmpty() )
    options.append( QString( " -i \"%1\"" ).arg( m_config->readEntry( "NetBIOS Scope", QString::null ) ) );

  if ( !m_config->readEntry( "Socket Options", QString::null ).isEmpty() )
    options.append( QString( " -O \"%1\"" ).arg( m_config->readEntry( "Socket Options", QString::null ) ) );
  
  return options;
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KPrint::slotReceivedStdout( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KPrint::slotReceivedStderr( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
  
  if ( m_buffer.contains( "NT_STATUS" ) != 0 )
  {
    abort();
  }
}


void Smb4KPrint::slotProcessExited( KProcess * )
{
  endProcess();
}


void Smb4KPrint::slotRetry()
{
  startPrinting();
}

#include "smb4kprint.moc"
