/***************************************************************************
    smb4kglobal  -  This is the global namespace for Smb4K.
                             -------------------
    begin                : Sa Apr 2 2005
    copyright            : (C) 2005 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qfile.h>
#include <qdir.h>
#include <qtextstream.h>
#include <qobject.h>
#include <qlayout.h>
#include <qlabel.h>

// KDE includes
#include <kconfig.h>
#include <kprocess.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kdebug.h>
#include <kdialogbase.h>
#include <kiconloader.h>
#include <kcombobox.h>

// other includes
#include <sys/utsname.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>

// application specific includes
#include "smb4kglobal.h"
#include "smb4kdefs.h"

#define TIME 50

class Smb4KGlobalPrivate : public QObject
{
  Q_OBJECT

  public:
    Smb4KGlobalPrivate();
    ~Smb4KGlobalPrivate();
    QTimer *timer();
    KConfig *config();
    const QString specifyUser( const QString &host );

  protected slots:
    void slotSpecifyUserTextChanged( const QString &text );

  private:
    QTimer *m_timer;
    KConfig *m_config;
    KDialogBase *m_user_dialog;
};


Smb4KGlobalPrivate::Smb4KGlobalPrivate()
{
  m_timer = new QTimer();
  m_timer->start( TIME, false );

  m_config = NULL;
}


Smb4KGlobalPrivate::~Smb4KGlobalPrivate()
{
  delete m_timer;
  delete m_config;
}


QTimer *Smb4KGlobalPrivate::timer()
{
  return m_timer;
}


KConfig *Smb4KGlobalPrivate::config()
{
  return m_config ? m_config : (m_config = new KConfig( "smb4krc", false, false, "config" ));
}


const QString Smb4KGlobalPrivate::specifyUser( const QString &host )
{
  QString username = QString::null;

  m_user_dialog = new KDialogBase( KDialogBase::Plain, i18n( "Specify User" ), KDialogBase::Ok|KDialogBase::Cancel, KDialogBase::Ok, 0, 0, false, true );
  m_user_dialog->enableButtonOK( false );

  // Set up the ask pass dialog.
  QFrame *frame = m_user_dialog->plainPage();
  QGridLayout *layout = new QGridLayout( frame );
  layout->setSpacing( 5 );

  QLabel *pic = new QLabel( frame );
  pic->setPixmap( DesktopIcon( "personal" ) );
  pic->setMargin( 10 );

  QLabel *text = new QLabel( i18n( "Please specify a user name." ), frame );

  QLabel *userLabel = new QLabel( i18n( "User:" ), frame );
  KComboBox *userCombo = new KComboBox( true, frame, 0 );
  userCombo->setDuplicatesEnabled( false );

  connect ( userCombo, SIGNAL( textChanged( const QString &) ),
            this,     SLOT( slotSpecifyUserTextChanged( const QString & ) ) );

  QSpacerItem *spacer1 = new QSpacerItem( 10, 10, QSizePolicy::Expanding, QSizePolicy::Preferred );

  layout->addWidget( pic, 0, 0, 0 );
  layout->addMultiCellWidget( text, 0, 0, 1, 3, 0 );
  layout->addWidget( userLabel, 1, 0, 0 );
  layout->addMultiCellWidget( userCombo, 1, 1, 1, 4, 0 );
  layout->addItem( spacer1, 0, 2 );

  // Read the list of logins, that were already defined
  // for this 'homes' share.
  QStringList list;

  if ( config()->hasGroup( "Homes Shares" ) )
  {
    config()->setGroup( "Homes Shares" );

    if ( config()->hasKey( host ) )
    {
      list = config()->readListEntry( host, ',' );
    }
  }

  if ( !list.isEmpty() )
  {
    userCombo->insertStringList( list, -1 );
  }

  userCombo->setCurrentText( QString::null );

  // Do the last things before showing.
  userCombo->setFocus();
  m_user_dialog->setFixedSize( m_user_dialog->sizeHint() );

  if ( m_user_dialog->exec() == KDialogBase::Accepted )
  {
    // Write the new list of logins to the config file.
    if ( list.contains( userCombo->currentText() ) == 0 )
    {
      list.append( userCombo->currentText() );
    }

    int index = 0;
    while ( index < userCombo->count() )
    {
      if ( list.contains( userCombo->text( index ) ) == 0 )
      {
        list.append( userCombo->text( index ) );
      }

      index++;
    }

    list.sort();

    config()->setGroup( "Homes Shares" );
    config()->writeEntry( host, list, ',' );

    username = userCombo->currentText();
  }

  delete m_user_dialog;

  return username;
}


void Smb4KGlobalPrivate::slotSpecifyUserTextChanged( const QString &text )
{
  if ( !text.isEmpty() )
  {
    m_user_dialog->enableButtonOK( true );
  }
  else
  {
    m_user_dialog->enableButtonOK( false );
  }
}


Smb4KGlobalPrivate p;




/****************************************************************************
   Returns the global options for smbclient
****************************************************************************/

const QString Smb4KGlobal::smbclientOptions()
{
  QString options;

  p.config()->setGroup( "Samba" );

  if ( !p.config()->readEntry( "Client Resolve Order", QString::null ).isEmpty() )
  {
    options.append( QString( " -R %1" ).arg( KProcess::quote( p.config()->readEntry( "Client Resolve Order", QString::null ) ) ) );
  }

  options.append( QString( " -p %1" ).arg( p.config()->readNumEntry( "Port", 139 ) ) );

  if ( !p.config()->readEntry( "Client Buffer Size", QString::null ).isEmpty() )
  {
    options.append( QString( " -b %1" ).arg( p.config()->readNumEntry( "Client Buffer Size", 65520 ) ) );
  }

  if ( p.config()->readBoolEntry( "Use Kerberos", false ) )
  {
    options.append( " -k" );
  }

  if ( !p.config()->readEntry( "NetBIOS Name", QString::null ).isEmpty() )
  {
    options.append( QString( " -n %1" ).arg( KProcess::quote( p.config()->readEntry( "NetBIOS Name", QString::null ) ) ) );
  }

  if ( !p.config()->readEntry( "NetBIOS Scope", QString::null ).isEmpty() )
  {
    options.append( QString( " -i %1" ).arg( KProcess::quote( p.config()->readEntry( "NetBIOS Scope", QString::null ) ) ) );
  }

  if ( !p.config()->readEntry( "Socket Options", QString::null ).isEmpty() )
  {
    options.append( QString( " -O %1" ).arg( KProcess::quote( p.config()->readEntry( "Socket Options", QString::null ) ) ) );
  }

  return options;
}


/****************************************************************************
   Returns the global options for nmblookup
****************************************************************************/

const QString Smb4KGlobal::nmblookupOptions()
{
  QString options;

  p.config()->setGroup( "Samba" );

  if ( !p.config()->readEntry( "NetBIOS Name", QString::null ).isEmpty() )
  {
    options.append( QString( " -n %1" ).arg( KProcess::quote( p.config()->readEntry( "NetBIOS Name", QString::null ) ) ) );
  }

  if ( !p.config()->readEntry( "NetBIOS Scope", QString::null ).isEmpty() )
  {
    options.append( QString( " -i %1" ).arg( KProcess::quote( p.config()->readEntry( "NetBIOS Scope", QString::null ) ) ) );
  }

  if ( !p.config()->readEntry( "Socket Options", QString::null ).isEmpty() )
  {
    options.append( QString( " -O %1" ).arg( KProcess::quote( p.config()->readEntry( "Socket Options", QString::null ) ) ) );
  }

  if ( !p.config()->readEntry( "Domain", QString::null ).isEmpty() )
  {
    options.append( QString( " -W %1" ).arg( KProcess::quote( p.config()->readEntry( "Domain", QString::null ) ) ) );
  }

  if ( !p.config()->readEntry( "NMB Broadcast", QString::null ).isEmpty() )
  {
    options.append( QString( " -B %1" ).arg( KProcess::quote( p.config()->readEntry( "NMB Broadcast", QString::null ) ) ) );
  }

  if ( p.config()->readBoolEntry( "NMB Port 137", false ) )
  {
    options.append( " -r" );
  }

  return options;
}


/****************************************************************************
   Reads the [global section of the smb.conf file
****************************************************************************/

const QMap<QString,QString> Smb4KGlobal::globalSMBOptions()
{
  QStringList paths;
  paths << "/etc";
  paths << "/etc/samba";
  paths << "/usr/local/etc";
  paths << "/usr/local/etc/samba";

  QFile f( "smb.conf" );

  QStringList contents;

  // Locate the file and read its contents:
  for ( QStringList::Iterator it = paths.begin(); it != paths.end(); ++it )
  {
    QDir::setCurrent( *it );

    if ( f.exists() )
    {
      if ( f.open( IO_ReadOnly ) )
      {
        QTextStream ts( &f );

        contents = QStringList::split( '\n', ts.read(), false );
      }

      f.close();

      break;
    }
    else
    {
      continue;
    }
  }

  // Process the file contents.
  for ( QStringList::Iterator it = contents.erase( contents.begin(), ++(contents.find( "[global]" )) ); it != contents.end(); ++it )
  {
    if ( (*it).stripWhiteSpace().startsWith( "#" ) || (*it).stripWhiteSpace().startsWith( ";" ) )
    {
      *it = QString::null;
    }
    else if ( (*it).stripWhiteSpace().startsWith( "include" ) )
    {
      // Put the contents of the included at this position.
      QString file = (*it).section( "=", 1, 1 ).stripWhiteSpace();
      *it = QString::null;
      f.setName( file );

      QStringList include;

      if ( f.exists() )
      {
        if ( f.open( IO_ReadOnly ) )
        {
          QTextStream ts( &f );

          include = QStringList::split( '\n', ts.read(), false );
        }

        f.close();
      }

      for ( QStringList::Iterator i = include.begin(); i != include.end(); ++i )
      {
        if ( !(*i).stripWhiteSpace().isEmpty() )
        {
          contents.insert( it, *i );

          continue;
        }
        else
        {
          continue;
        }
      }

      continue;
    }
    else if ( (*it).startsWith( "[" ) )
    {
      contents.erase( it, contents.end() );

      break;
    }
    else
    {
      continue;
    }
  }

  contents.remove( QString::null );

  QMap<QString,QString> map;

  // Write all options into the map:
  for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
  {
    QString key = (*it).section( "=", 0, 0 ).stripWhiteSpace().lower();
    map[key] = QString( (*it).section( "=", 1, 1 ).stripWhiteSpace().upper() );
  }

  // Post-processing. Some values should be entered with their defaults, if they are
  // not already present.
  if ( !map.contains( "netbios name" ) )
  {
    size_t hostnamelen = 255;
    char *hostname = new char[hostnamelen];

    if ( gethostname( hostname, hostnamelen ) == 0 )
    {
      map["netbios name"] = ( QString( "%1" ).arg( hostname ) ).upper();
    }

    delete [] hostname;
  }

  return map;
}



/****************************************************************************
   Returns the WINS server the system is using
****************************************************************************/

const QString Smb4KGlobal::winsServer()
{
  QString wins = QString::null;
  QMap<QString, QString> options = globalSMBOptions();

  if ( options.find( "wins server" ) != options.end() )
  {
    wins = options["wins server"];
  }
  else if ( options.find( "wins support" ) != options.end() &&
            (QString::compare( options["wins support"].lower(), "yes" ) == 0 ||
             QString::compare( options["wins support"].lower(), "true" ) == 0) )
  {
    wins = "127.0.0.1";
  }

  return wins;
}


/****************************************************************************
   Returns the kernel version
****************************************************************************/

const QString Smb4KGlobal::kernelVersion()
{
  struct utsname system_info;

  uname( &system_info );

  return QString( "%1" ).arg( system_info.release ).section( "-", 0, 0 );
}


/****************************************************************************
   Returns the system name
****************************************************************************/

const QString Smb4KGlobal::systemName()
{
  struct utsname system_info;

  uname( &system_info );

  return QString( "%1" ).arg( system_info.sysname );
}


/****************************************************************************
   Returns the UMASK
****************************************************************************/

int Smb4KGlobal::getUMASK()
{
  return umask( 0 );
}


/****************************************************************************
   Returns the global options of the net command
****************************************************************************/

const QString Smb4KGlobal::netOptions( const QString &command )
{
  QMap<QString,QString> smb_map = globalSMBOptions();

  QString options = QString::null;

  p.config()->setGroup( "Samba" );

  if ( QString::compare( command.lower(), "share" ) == 0 )
  {
    if ( QString::compare( p.config()->readEntry( "Net Protocol", "auto" ), "auto" ) != 0 )
    {
      // The 'share' command only works with the RAP or RPC protocol.
      // Because the user can also choose the ADS protocol, we have to
      // permit only RAP and RPC protocol here:
      QString protocol = QString::compare( p.config()->readEntry( "Net Protocol", "auto" ).lower(), "rap" ) == 0 ? "rap" : "rpc";
      options.append( " "+protocol );
    }

    options.append( " share" );
    options.append( " -l" );
  }
  else if ( QString::compare( command.lower(), "share_no_protocol" ) == 0 )
  {
    options.append( " share" );
    options.append( " -l" );
  }
  else if ( QString::compare( command.lower(), "server domain" ) == 0 )
  {
    options.append( " rap "+command );
  }
  else if ( command.lower().startsWith( "lookup host" ) )
  {
    options.append( " "+command );
  }
  else if ( command.lower().startsWith( "lookup master" ) )
  {
    options.append( " "+command );
  }
  else if ( QString::compare( command.lower(), "domain" ) == 0 )
  {
    options.append( " rap "+command );
  }
  else
  {
    KMessageBox::error( 0, i18n( "The net command '%1' is not supported by Smb4K." ).arg( command ) );

    return options;
  }

  options.append( p.config()->readBoolEntry( "Net Machine Account", false ) ? " -P" : "" );
  options.append( QString( " -p %1" ).arg( p.config()->readNumEntry( "Port", 139 ) ) );
  options.append( QString( " -W %1" ).arg( KProcess::quote( p.config()->readEntry( "Domain", smb_map["workgroup"] ) ) ) );
  options.append( QString( " -n %1" ).arg( KProcess::quote( p.config()->readEntry( "NetBIOS Name", smb_map["netbios name"] ) ) ) );

  return options;
}


/***************************************************************************
   Returns a pointer to the global KConfig object.
***************************************************************************/

KConfig *Smb4KGlobal::config()
{
  return p.config();
}


/***************************************************************************
   Returns a pointer to the global timer object
***************************************************************************/

QTimer *Smb4KGlobal::timer()
{
  return p.timer();
}


/***************************************************************************
   Returns the timer interval
***************************************************************************/

const int Smb4KGlobal::timerInterval()
{
  return TIME;
}


/***************************************************************************
   Will return a specified user for a homes share
***************************************************************************/

const QString Smb4KGlobal::specifyUser( const QString &host )
{
  return p.specifyUser( host );
}


/***************************************************************************
  Shows an error message according to the passed error code
***************************************************************************/

void Smb4KGlobal::showCoreError( int code, const QString &message )
{
  switch( code )
  {
    case ERROR_GETTING_BROWSELIST:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to get the browse list." ), message );
      break;
    case ERROR_GETTING_MEMBERS:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to get the list of workgroup members." ), message );
      break;
    case ERROR_GETTING_SHARES:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to get the list of shares." ), message );
      break;
    case ERROR_GETTING_PREVIEW:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to get the preview." ), message );
      break;
    case ERROR_UNMOUNTING_NOT_ALLOWED:
      KMessageBox::error( 0, i18n( "You are not allowed to unmount this share." ) );
      break;
    case ERROR_MOUNTING_SHARE:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to mount the share." ), message );
      break;
    case ERROR_UNMOUNTING_SHARE:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to unmount the share." ), message );
      break;
    case ERROR_UNMOUNTING_ALL:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to unmount all shares." ), message );
      break;
    case ERROR_FILE_NOT_FOUND:
        KMessageBox::error( 0, i18n( "The file %1 could not be found." ).arg( message ) );
      break;
    case ERROR_READING_FILE:
      KMessageBox::error( 0, i18n( "The file %1 could not be read." ).arg( message ) );
      break;
    case ERROR_UNKNOWN:
      KMessageBox::detailedError( 0, i18n( "An error occurred." ), message );
      break;
    case ERROR_GETTING_HOSTNAME:
      KMessageBox::error( 0, i18n( "The hostname could not be determined." ) );
      break;
    case ERROR_MIMETYPE_NOT_SUPPORTED:
      KMessageBox::error( 0, i18n( "The mimetype \"%1\" is not supported. Please convert the file to PostScript or PDF." ).arg( message ) );
      break;
    case ERROR_LOCKED:
      {
        QString file = message.section( ":", 1, 1 );
        QString user = message.section( ":", 0, 0 );
        KMessageBox::error( 0, i18n( "The file %1 is currently edited by the user %2. Please try again later." ).arg( file, user ) );
        break;
      }
    case ERROR_MKDIR_FAILED:
      KMessageBox::error( 0, i18n( "The directory %1 could not be created." ).arg( message ) );
      break;
    case ERROR_MISSING_PROGRAMS:
      KMessageBox::error( 0, i18n( "Either your PATH environment variable is not set correctly or there are the following programs missing on your system:\n%1\nPlease correct this and start Smb4K again." ).arg( message ) );
      break;
    case ERROR_SUID_PROGRAM_MISSING:
      KMessageBox::information( 0, i18n( "You previously chose to use %1, but now it is missing on your system. Smb4K will disable this feature." ).arg( message ) );
      break;
    case ERROR_WRITING_FILE:
      KMessageBox::error( 0, i18n( "The file %1 could not be written successfully." ).arg( message ) );
      break;
    case ERROR_GETTING_USAGE:
      KMessageBox::detailedError( 0, i18n( "The disk usage could not be determined." ), message );
      break;
    case ERROR_MOUNTPOINT_EMPTY:
      KMessageBox::error( 0, i18n( "This share could not be unmounted because the mount point string was empty." ) );
      break;
    case ERROR_WRONG_KERNEL:
      KMessageBox::error( 0, i18n( "Your kernel (%1) does not support this operation." ).arg( message ) );
      break;
    case ERROR_FEATURE_NOT_ENABLED:
      KMessageBox::error( 0, i18n( "This feature has not been enabled." ) );
      break;
    case ERROR_WRONG_BOOKMARK_TYPE:
      KMessageBox::error( 0, i18n( "Printers cannot be bookmarked." ) );
      break;
    case ERROR_IP_CANNOT_BE_USED:
      KMessageBox::sorry( 0, i18n( "The search method you chose (smbclient) can't handle IP addresses correctly. Please choose \"Use nmblookup\" in the configuration dialog and try again." ) );
      break;
    case ERROR_COMMAND_NOT_FOUND:
      KMessageBox::error( 0, i18n( "The command %1 could not be found." ).arg( message ) );
      break;
    case ERROR_OPENING_WALLET:
      KMessageBox::error( 0, i18n( "The wallet \"%1\" could not be opened." ).arg( message ) );
      break;
    case ERROR_PRINTING:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to print." ), message );
      break;
    case ERROR_CREATING_TEMP_DIR:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to create a temporary directory." ), message );
      break;
    case ERROR_CREATING_TEMP_FILE:
      KMessageBox::detailedError( 0, i18n( "An error occurred while trying to create a temporary file." ), message );
      break;
    case ERROR_DIRECTORY_NOT_FOUND:
      KMessageBox::error( 0, i18n( "The directory %1 could not be found." ).arg( message ) );
      break;
    case ERROR_LOCK_FILE_IS_SYMLINK:
      KMessageBox::error( 0, i18n( "The lock file (%1) is a symlink. Since this might indicate that someone trys to exploit your system, please inform your system administrator. No action has been performed." ).arg( message ) );
      break;
    case ERROR_SHARE_WENT_OFFLINE:
//       KMessageBox::error( 0, i18n( "The share %1 went offline." ).arg( message ) );
//       KPassivePopup::message( kapp->caption(), i18n( "The share %1 went offline." ).arg( message ), kapp->icon(), KApplication::desktop(), 0, 0 );
      break;
    case ERROR_GETTING_PERMISSIONS:
      KMessageBox::detailedError( 0, i18n( "An error occurred while determining file permissions." ), message );
      break;
    case ERROR_SYNCHRONIZING:
      KMessageBox::detailedError( 0, i18n( "An error occurred during synchronization." ), message );
      break;
    case ERROR_OPENING_FILE:
      KMessageBox::error( 0, i18n( "The file %1 could not be opened." ).arg( message ) );
      break;
    case ERROR_IMPORTING_SHARES:
    {
      if ( message.stripWhiteSpace().isEmpty() )
      {
        KMessageBox::error( 0, i18n( "The list of mounted SMBFS and CIFS shares could not be imported.\nUnfortunately, there was no error message." ) );
      }
      else
      {
        KMessageBox::detailedError( 0, i18n( "The list of mounted SMBFS and CIFS shares could not be imported.\nRead the error message under 'Details' to find out more." ), message );
      }

      break;
    }
    default:
      break;
  }
}

#include "smb4kglobal.moc"
