/***************************************************************************
    smb4koptionsdlg.cpp  -  description
                             -------------------
    begin                : Sam Feb 22 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qlayout.h>
#include <qtimer.h>
#include <qregexp.h>
#include <qvaluelist.h>
#include <qmap.h>

#include <qscrollview.h>

// KDE includes
#include <klocale.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <kapplication.h>
#include <kstandarddirs.h>

// system includes
#include <sys/types.h>
#include <sys/stat.h>

// application specific includes
#include "smb4koptionsdlg.h"
#include "../core/smb4kauthinfo.h"
#include "../core/smb4kcore.h"
#include "../core/smb4kglobal.h"

using namespace Smb4KGlobal;


// This is to reverse the settings for
// super user privileges.
QString super_prog_prev = QString::null;
bool use_force_prev = false;
bool use_suid_prev = false;


/**
 * This class handles the configuration dialog. It initializes the pages and saves
 * and reads the options.
 */

Smb4KOptionsDlg::Smb4KOptionsDlg( QWidget *parent, const char *name, bool modal )
: KDialogBase( IconList, i18n( "Configuration" ), Ok|Apply|Cancel, Ok, parent, name, modal, true )
{
  setWFlags( Qt::WDestructiveClose );

  setupAppearanceOptions();
  setupNetworkOptions();
  setupShareOptions();
  setupAuthOptions();
  setupSambaOptions();
  setupRsyncOptions();
  setupSuperUserOptions();

  config()->reparseConfiguration();

  readOptions();

  // Connections this the core
  connect( smb4k_core->fileIO(),  SIGNAL( finished_suid_writing() ),
           this,                  SLOT( slotFinishedSUIDWriting() ) );
  connect( smb4k_core->fileIO(),  SIGNAL( error_suid_writing() ),
           this,                  SLOT( slotSUIDWritingFailed() ) );

  connect( m_superPage->m_remove, SIGNAL( clicked() ),
           this,                  SLOT( slotRemoveSuperUserEntries() ) );
}


Smb4KOptionsDlg::~Smb4KOptionsDlg()
{
}


/****************************************************************************
   Set up the network options page.
****************************************************************************/

void Smb4KOptionsDlg::setupNetworkOptions()
{
  QFrame *frame = addPage( i18n( "Network" ), i18n( "Network" ), DesktopIcon( "network" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_networkPage = new Smb4KNetworkOptions( container->viewport() );
  container->addChild( m_networkPage, 0, 0 );

  frameLayout->addWidget( container, 0, 0, 0 );
}


/****************************************************************************
   Set up the share options page.
****************************************************************************/

void Smb4KOptionsDlg::setupShareOptions()
{
  QFrame *frame = addPage( i18n( "Shares" ), i18n( "Shares" ), DesktopIcon( "hdd_mount" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_sharePage = new Smb4KShareOptions( container->viewport() );
  container->addChild( m_sharePage, 0, 0 );

  frameLayout->addWidget( container, 0, 0, 0 );
}


/****************************************************************************
   Set up the authentication management page.
****************************************************************************/

void Smb4KOptionsDlg::setupAuthOptions()
{
  QFrame *frame = addPage( i18n( "Authentication" ), i18n( "Authentication" ), DesktopIcon( "identity" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_authPage = new Smb4KAuthOptions( container->viewport() );
  container->addChild( m_authPage, 0, 0 );

  frameLayout->addWidget( container, 0, 0, 0 );
}


/****************************************************************************
   Set up the apprearance options page.
****************************************************************************/

void Smb4KOptionsDlg::setupAppearanceOptions()
{
  QFrame *frame = addPage( i18n( "Appearance" ), i18n( "Appearance" ), DesktopIcon( "colorize" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_appearancePage = new Smb4KAppearanceOptions( container->viewport() );
  container->addChild( m_appearancePage, 0, 0 );

  frameLayout->addWidget( container, 0, 0, 0 );
}


/****************************************************************************
   Set up the advanced Samba options page.
****************************************************************************/

void Smb4KOptionsDlg::setupSambaOptions()
{
  QFrame *frame = addPage( i18n( "Samba" ), i18n( "Samba Options" ), DesktopIcon( "samba" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_sambaPage = new Smb4KSambaOptions( container->viewport() );
  container->addChild( m_sambaPage, 0, 0 );

  frameLayout->addWidget( container, 0, 0 );
}


/****************************************************************************
   Set up the super user options page.
****************************************************************************/

void Smb4KOptionsDlg::setupSuperUserOptions()
{
  QFrame *frame = addPage( i18n( "Super User" ), i18n( "Super User Privileges" ), DesktopIcon( "penguin" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_superPage = new Smb4KSuperUserOptions( container->viewport() );
  container->addChild( m_superPage, 0, 0 );

  frameLayout->addWidget( container, 0, 0 );
}


/****************************************************************************
   Set up the synchronize options page
****************************************************************************/

void Smb4KOptionsDlg::setupRsyncOptions()
{
  QFrame *frame = addPage( i18n( "Synchronization" ), i18n( "Rsync Options" ), DesktopIcon( "bottom" ) );

  QGridLayout *frameLayout = new QGridLayout( frame );
  frameLayout->setSpacing( 0 );
  frameLayout->setMargin( 0 );

  QScrollView *container = new QScrollView( frame );
  container->setResizePolicy( QScrollView::AutoOneFit );
  container->setFrameShadow( QScrollView::Plain );
  container->setFrameShape( QScrollView::MenuBarPanel );
  container->setLineWidth( 0 );
  container->setMidLineWidth( 0 );
  m_syncPage = new Smb4KRsyncOptions( container->viewport() );
  container->addChild( m_syncPage, 0, 0 );

  frameLayout->addWidget( container, 0, 0 );
}


/****************************************************************************
   Saves the options.
****************************************************************************/


bool Smb4KOptionsDlg::saveOptions()
{
  //
  // General options
  //
  config()->setGroup( "Dialogs" );
  config()->writeEntry( "Configuration Dialog Geometry", size() );

  //
  // Mount options
  //
  config()->setGroup( "Mount Options" );

  // Check, if the default path ends with a slash.
  if ( m_sharePage->m_mount_prefix->url().endsWith( "/" ) )
  {
    config()->writeEntry( "Default Path", m_sharePage->m_mount_prefix->url() );
  }
  else
  {
    config()->writeEntry( "Default Path", m_sharePage->m_mount_prefix->url().append( "/" ) );
  }

  config()->writeEntry( "Unmount All", m_sharePage->m_unmountAll->isChecked() );
  config()->writeEntry( "Mount Recent", m_sharePage->m_mountRecent->isChecked() );
  config()->writeEntry( "Unmount Foreign", m_sharePage->m_allowUnmountForeign->isChecked() );
  config()->writeEntry( "Force Lowercase", m_sharePage->m_lowercase->isChecked() );
  config()->writeEntry( "Check Interval", m_sharePage->m_check_interval->value() );

  //
  // Browse options
  //
  config()->setGroup( "Browse Options" );

  if ( m_networkPage->m_useNMB->isChecked() )
  {
    config()->writeEntry( "Browse List", "lookup" );
  }
  else if ( m_networkPage->m_useMaster->isChecked() )
  {
    config()->writeEntry( "Browse List", "master" );
  }
  else if ( m_networkPage->m_useHost->isChecked() )
  {
    config()->writeEntry( "Browse List", "host" );
  }

  config()->writeEntry( "Query Host", m_networkPage->m_hostName->text().stripWhiteSpace().upper() );

  if ( m_networkPage->m_nmbSearch->isChecked() )
  {
    config()->writeEntry( "Network Search", "nmblookup" );
  }
  else
  {
    config()->writeEntry( "Network Search", "smbclient" );
  }


  //
  // Appearance
  //
  config()->setGroup( "Appearance" );
  config()->writeEntry( "Show Hidden", m_appearancePage->m_showHidden->isChecked() );
  config()->writeEntry( "Show IPC", m_appearancePage->m_showIPC->isChecked() );
  config()->writeEntry( "Show ADMIN", m_appearancePage->m_showAdmin->isChecked() );
  config()->writeEntry( "Show Printer", m_appearancePage->m_showPrinter->isChecked() );
  config()->writeEntry( "Show IP", m_appearancePage->m_showIP->isChecked() );
  config()->writeEntry( "Show Type", m_appearancePage->m_showType->isChecked() );
  config()->writeEntry( "Show Comment", m_appearancePage->m_showComment->isChecked() );

  if ( m_appearancePage->m_shareview->isChecked() )
  {
    config()->writeEntry( "Show Shares", "list" );
  }
  else
  {
    config()->writeEntry( "Show Shares", "icons" );
  }

  config()->writeEntry( "Show Mount Point", m_appearancePage->m_description->isChecked() );
  config()->writeEntry( "Show All Shares", m_appearancePage->m_external->isChecked() );
  config()->writeEntry( "Show Share Tooltip", m_appearancePage->m_showShareTooltip->isChecked() );
  config()->writeEntry( "Show Network Tooltip", m_appearancePage->m_showNetworkItemTooltip->isChecked() );

  //
  // Authentication
  //
  config()->setGroup( "Authentication" );
  config()->writeEntry( "Use Wallet", m_authPage->m_use_wallet->isChecked() );
  config()->writeEntry( "Remember Passwords", m_authPage->m_remember_passwords->isChecked() );
  config()->writeEntry( "Default Authentication", m_authPage->m_defaultAuth->isChecked() );

  if ( m_authPage->m_defaultAuth->isChecked() )
  {
    Smb4KAuthInfo *auth = new Smb4KAuthInfo( QString::null, QString::null, QString::null, m_authPage->m_defaultLogin->text().stripWhiteSpace(), m_authPage->m_defaultPasswd->text().stripWhiteSpace() );
    smb4k_core->passwordHandler()->writeDefaultAuth( auth );
  }

  //
  // Samba options
  //
  config()->setGroup( "Samba" );

  // General
  if ( !m_sambaPage->m_netbios->text().stripWhiteSpace().isEmpty() &&
       QString::compare( *(globalSMBOptions().find( "netbios name" )), m_sambaPage->m_netbios->text().stripWhiteSpace().upper() ) != 0 )
  {
    config()->writeEntry( "NetBIOS Name", m_sambaPage->m_netbios->text().stripWhiteSpace().upper() );
  }
  else
  {
    config()->deleteEntry( "NetBIOS Name" );
  }

  if ( m_sambaPage->m_domain->isEnabled() )
  {
    if ( !m_sambaPage->m_domain->text().stripWhiteSpace().isEmpty() &&
         QString::compare( m_sambaPage->m_domain->text().stripWhiteSpace().upper(), *(globalSMBOptions().find( "workgroup" )) ) != 0 )
    {
      config()->writeEntry( "Domain", m_sambaPage->m_domain->text().stripWhiteSpace().upper() );
    }
    else
    {
      config()->deleteEntry( "Domain" );
    }
  }

  if ( !m_sambaPage->m_sockopt->text().stripWhiteSpace().isEmpty() &&
       QString::compare( *(globalSMBOptions().find( "socket options" )), m_sambaPage->m_sockopt->text().stripWhiteSpace().upper() ) != 0 )
  {
    config()->writeEntry( "Socket Options", m_sambaPage->m_sockopt->text().stripWhiteSpace().upper() );
  }
  else
  {
    config()->deleteEntry( "Socket Options" );
  }

  if ( !m_sambaPage->m_scope->text().stripWhiteSpace().isEmpty() &&
       QString::compare( *(globalSMBOptions().find( "netbios scope")), m_sambaPage->m_scope->text().stripWhiteSpace().upper() ) != 0 )
  {
    config()->writeEntry( "Scope", m_sambaPage->m_scope->text().stripWhiteSpace().upper() );
  }
  else
  {
    config()->deleteEntry( "NetBIOS Scope" );
  }

  config()->writeEntry( "Port", m_sambaPage->m_port->value() );

  if ( m_sambaPage->m_krb->isEnabled() )
  {
    config()->writeEntry( "Use Kerberos", m_sambaPage->m_krb->isChecked() );
  }
  else
  {
    config()->deleteEntry( "Use Kerberos" );
  }

  // mount
  if ( QString::compare( m_sambaPage->m_filesystem->currentText(), "SMBFS" ) == 0 )
  {
    config()->writeEntry( "Mount Filesystem", "smbfs" );
  }
  else
  {
    config()->writeEntry( "Mount Filesystem", "cifs" );
  }

  if ( QString::compare( m_sambaPage->m_uid->currentText(), i18n( "default" ) ) != 0 )
  {
    config()->writeEntry( "Mount UID", m_sambaPage->m_uid->currentText() );
  }
  else
  {
    config()->deleteEntry( "Mount UID" );
  }

  if ( QString::compare( m_sambaPage->m_gid->currentText(), i18n( "default" ) ) != 0 )
  {
    config()->writeEntry( "Mount GID", m_sambaPage->m_gid->currentText() );
  }
  else
  {
    config()->deleteEntry( "Mount GID" );
  }

  if ( QString::compare( m_sambaPage->m_charset->currentText(), i18n( "default" ) ) != 0 &&
       m_sambaPage->m_charset->isEnabled() )
  {
    config()->writeEntry( "Mount Charset", m_sambaPage->m_charset->currentText() );
  }
  else
  {
    config()->deleteEntry( "Mount Charset" );
  }

  if ( QString::compare( m_sambaPage->m_codepage->currentText(), i18n( "default" ) ) != 0 &&
       m_sambaPage->m_codepage->isEnabled() )
  {
    config()->writeEntry( "Mount Codepage", m_sambaPage->m_codepage->currentText() );
  }
  else
  {
    config()->deleteEntry( "Mount Codepage" );
  }

  QString fmask = m_sambaPage->m_fmask->text().stripWhiteSpace();
  QString dmask = m_sambaPage->m_dmask->text().stripWhiteSpace();

  // We need to make sure that the masks have 4 digits:
  if ( fmask.length() < 4 )
  {
    while ( fmask.length() < 4 )
    {
      fmask.prepend( "0" );
    }
  }

  if ( dmask.length() < 4 )
  {
    while ( dmask.length() < 4 )
    {
      dmask.prepend( "0" );
    }
  }

  config()->writeEntry( "Mount FMASK", fmask );
  config()->writeEntry( "Mount DMASK", dmask );

  if ( QString::compare( m_sambaPage->m_read_mode->currentText(), i18n( "read-write" ) ) == 0 )
  {
    config()->writeEntry( "Mount ReadWrite", true );
  }
  else
  {
    config()->writeEntry( "Mount ReadWrite", false );
  }

  if ( m_sambaPage->m_ttl->value() != 1000 && m_sambaPage->m_ttl->isEnabled() )
  {
    config()->writeEntry( "Mount Cache", m_sambaPage->m_ttl->value() );
  }
  else
  {
    config()->deleteEntry( "Mount Cache" );
  }

  if ( m_sambaPage->m_unicode->isEnabled() )
  {
    config()->writeEntry( "Mount Unicode", m_sambaPage->m_unicode->isChecked() );
  }
  else
  {
    config()->deleteEntry( "Mount Unicode" );
  }

  if ( m_sambaPage->m_lfs->isEnabled() )
  {
    config()->writeEntry( "Mount LFS", m_sambaPage->m_lfs->isChecked() );
  }
  else
  {
    config()->deleteEntry( "Mount LFS" );
  }

  config()->writeEntry( "Mount RSize", m_sambaPage->m_rsize->isChecked() );
  config()->writeEntry( "Mount WSize", m_sambaPage->m_wsize->isChecked() );

  // net
  if ( m_sambaPage->m_net_auto->isChecked() )
  {
    config()->writeEntry( "Net Protocol", "auto" );
  }
  else if ( m_sambaPage->m_net_ads->isChecked() )
  {
    config()->writeEntry( "Net Protocol", "ads" );
  }
  else if ( m_sambaPage->m_net_rpc->isChecked() )
  {
    config()->writeEntry( "Net Protocol", "rpc" );
  }
  else if ( m_sambaPage->m_net_rap->isChecked() )
  {
    config()->writeEntry( "Net Protocol", "rap" );
  }

  if ( m_sambaPage->m_net_machine_account->isChecked() )
  {
    config()->writeEntry( "Net Machine Account", true );
  }
  else
  {
    config()->writeEntry( "Net Machine Account", false );
  }

  // smbclient
  if ( !m_sambaPage->m_resolve->text().stripWhiteSpace().isEmpty() &&
       QString::compare( *(globalSMBOptions().find( "name resolve order" )), m_sambaPage->m_resolve->text().stripWhiteSpace().upper() ) != 0 )
  {
    config()->writeEntry( "Client Resolve Order", m_sambaPage->m_resolve->text().stripWhiteSpace().upper() );
  }
  else
  {
    config()->deleteEntry( "Client Resolve Order" );
  }

  if ( m_sambaPage->m_buffer->value() == 65520 )
  {
    config()->deleteEntry( "Client Buffer Size" );
  }
  else
  {
    config()->writeEntry( "Client Buffer Size", m_sambaPage->m_buffer->value() );
  }

  if ( !m_sambaPage->m_broadcast->text().stripWhiteSpace().isEmpty() &&
       QString::compare( *(globalSMBOptions().find( "interfaces" )), m_sambaPage->m_broadcast->text().stripWhiteSpace().upper() ) != 0 )
  {
    config()->writeEntry( "NMB Broadcast", m_sambaPage->m_broadcast->text().stripWhiteSpace().upper() );
  }
  else
  {
    config()->deleteEntry( "NMB Broadcast" );
  }

  config()->writeEntry( "NMB Port 137", m_sambaPage->m_udp->isChecked() );


  //
  // The super user tab
  //
  config()->setGroup( "Super User Privileges" );

  if ( m_superPage->isEnabled() )
  {
    if ( m_superPage->m_super->isChecked() )
    {
      config()->writeEntry( "SUID Program", "super" );
    }
    else if ( m_superPage->m_sudo->isChecked() )
    {
      config()->writeEntry( "SUID Program", "sudo" );
    }
    config()->writeEntry( "Force Unmount", m_superPage->m_unmount_dead->isChecked() );
    config()->writeEntry( "Run SUID", m_superPage->m_use_suid->isChecked() );
  }
  else
  {
    config()->deleteEntry( "Force Unmount" );
    config()->deleteEntry( "Run SUID" );
    config()->deleteEntry( "SUID Program" );
  }

  //
  // Synchronization options
  //

  config()->setGroup( "Rsync" );

  if ( m_syncPage->m_rsync_prefix->url().endsWith( "/" ) )
  {
    config()->writeEntry( "Prefix", m_syncPage->m_rsync_prefix->url() );
  }
  else
  {
    config()->writeEntry( "Prefix", m_syncPage->m_rsync_prefix->url().append( "/" ) );
  }

  config()->writeEntry( "archive", m_syncPage->m_archive->isChecked() );
  config()->writeEntry( "recursive", m_syncPage->m_recursive->isChecked() );
  config()->writeEntry( "links", m_syncPage->m_links->isChecked() );
  config()->writeEntry( "perms", m_syncPage->m_perms->isChecked() );
  config()->writeEntry( "times", m_syncPage->m_times->isChecked() );
  config()->writeEntry( "group", m_syncPage->m_group->isChecked() );
  config()->writeEntry( "owner", m_syncPage->m_owner->isChecked() );
  config()->writeEntry( "devices", m_syncPage->m_devices->isChecked() );
  config()->writeEntry( "relative", m_syncPage->m_relative->isChecked() );
  config()->writeEntry( "omit dir times", m_syncPage->m_omit_dir_times->isChecked() );
  config()->writeEntry( "no implied dirs", m_syncPage->m_no_implied_dirs->isChecked() );
  config()->writeEntry( "update", m_syncPage->m_update->isChecked() );
  config()->writeEntry( "inplace", m_syncPage->m_inplace->isChecked() );
  config()->writeEntry( "dirs", m_syncPage->m_dirs->isChecked() );
  config()->writeEntry( "copy links", m_syncPage->m_copy_links->isChecked() );
  config()->writeEntry( "copy unsafe links", m_syncPage->m_copy_unsafe_links->isChecked() );
  config()->writeEntry( "safe links", m_syncPage->m_safe_links->isChecked() );
  config()->writeEntry( "hard links", m_syncPage->m_hard_links->isChecked() );
  config()->writeEntry( "keep dirlinks", m_syncPage->m_keep_dirlinks->isChecked() );
  config()->writeEntry( "delete", m_syncPage->m_delete->isChecked() );
  config()->writeEntry( "remove sent files", m_syncPage->m_remove_sent_files->isChecked() );
  config()->writeEntry( "delete before", m_syncPage->m_delete_before->isChecked() );
  config()->writeEntry( "delete during", m_syncPage->m_delete_during->isChecked() );
  config()->writeEntry( "delete after", m_syncPage->m_delete_after->isChecked() );
  config()->writeEntry( "delete excluded", m_syncPage->m_delete_excluded->isChecked() );
  config()->writeEntry( "ignore errors", m_syncPage->m_ignore_errors->isChecked() );
  config()->writeEntry( "force", m_syncPage->m_force->isChecked() );
  config()->writeEntry( "whole file", m_syncPage->m_whole_file->isChecked() );
  config()->writeEntry( "sparse", m_syncPage->m_sparse->isChecked() );
  config()->writeEntry( "one file system", m_syncPage->m_one_file_system->isChecked() );
  config()->writeEntry( "existing", m_syncPage->m_existing->isChecked() );
  config()->writeEntry( "delay updates", m_syncPage->m_delay_updates->isChecked() );
  config()->writeEntry( "compress", m_syncPage->m_compress->isChecked() );
  config()->writeEntry( "backup", m_syncPage->m_backup->isChecked() );
  config()->writeEntry( "Use backup dir", m_syncPage->m_backup_dir->isChecked() );
  config()->writeEntry( "backup dir", m_syncPage->m_backup_dir_requester->url() );
  config()->writeEntry( "Use suffix", m_syncPage->m_suffix->isChecked() );
  config()->writeEntry( "suffix", m_syncPage->m_suffix_line->text() );
  config()->writeEntry( "Use max delete", m_syncPage->m_max_delete->isChecked() );
  config()->writeEntry( "max delete", m_syncPage->m_max_delete_input->value() );
  config()->writeEntry( "checksum", m_syncPage->m_checksum->isChecked() );
  config()->writeEntry( "Use block size", m_syncPage->m_block_size->isChecked() );
  config()->writeEntry( "block size", m_syncPage->m_block_size_input->value() );
  config()->writeEntry( "Use checksum seed", m_syncPage->m_checksum_seed->isChecked() );
  config()->writeEntry( "checksum seed", m_syncPage->m_checksum_seed_input->value() );
  config()->writeEntry( "filter", m_syncPage->m_filter_input->text() );
  config()->writeEntry( "Use min size", m_syncPage->m_min_size->isChecked() );
  config()->writeEntry( "min size", m_syncPage->m_min_size_input->value() );
  config()->writeEntry( "Use max size", m_syncPage->m_max_size->isChecked() );
  config()->writeEntry( "max size", m_syncPage->m_max_size_input->value() );
  config()->writeEntry( "partial", m_syncPage->m_partial->isChecked() );
  config()->writeEntry( "Use partial dir", m_syncPage->m_partial_dir->isChecked() );
  config()->writeEntry( "partial dir", m_syncPage->m_partial_dir_requester->url() );
  config()->writeEntry( "cvs exclude", m_syncPage->m_cvs_exclude->isChecked() );
  config()->writeEntry( "F-Filter", m_syncPage->m_f_filter->isChecked() );
  config()->writeEntry( "FF-Filter", m_syncPage->m_ff_filter->isChecked() );
  config()->writeEntry( "Use exclude", m_syncPage->m_exclude->isChecked() );
  config()->writeEntry( "exclude", m_syncPage->m_exclude_input->text() );
  config()->writeEntry( "Use exclude from", m_syncPage->m_exclude_from->isChecked() );
  config()->writeEntry( "exclude from", m_syncPage->m_exclude_from_input->url() );
  config()->writeEntry( "Use include", m_syncPage->m_include->isChecked() );
  config()->writeEntry( "include", m_syncPage->m_include_input->text() );
  config()->writeEntry( "Use include from", m_syncPage->m_include_from->isChecked() );
  config()->writeEntry( "include from", m_syncPage->m_include_from_input->url() );

  config()->sync();
  return true;
}


/****************************************************************************
   Reads the options.
****************************************************************************/

void Smb4KOptionsDlg::readOptions()
{
  int mask = getUMASK();
  QString system = systemName();
  QString kernel = kernelVersion();

  int k_major = kernel.section( ".", 0, 0 ).toInt();
  int k_minor = kernel.section( ".", 1, 1 ).toInt();
  int k_rev = kernel.section( ".", 2, 2 ).toInt();

  config()->setGroup( "Programs" );
  QString super = config()->readPathEntry( "super" );
  QString sudo = config()->readPathEntry( "sudo" );
  QString mount_cifs = config()->readPathEntry( "mount.cifs" );
  QString net = config()->readPathEntry( "net" );
  QString rsync = config()->readPathEntry( "rsync" );

  //
  // Mount options
  //
  config()->setGroup( "Mount Options" );
  m_sharePage->m_mount_prefix->setURL( config()->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) ) );
  m_sharePage->m_unmountAll->setChecked( config()->readBoolEntry( "Unmount All", false ) );
  m_sharePage->m_mountRecent->setChecked( config()->readBoolEntry( "Mount Recent", false ) );
  m_sharePage->m_allowUnmountForeign->setChecked( config()->readBoolEntry( "Unmount Foreign", false ) );
  m_sharePage->m_lowercase->setChecked( config()->readBoolEntry( "Force Lowercase", false ) );
  m_sharePage->m_check_interval->setValue( config()->readNumEntry( "Check Interval", 2500 ) );

  //
  // Browse options
  //
  config()->setGroup( "Browse Options" );
  // Toggle the browse list buttons.
  QString browselist = config()->readEntry( "Browse List", "lookup" );
  QString host = config()->readEntry( "Query Host", QString::null );

  if ( QString::compare( browselist.stripWhiteSpace(), "lookup" ) == 0 )
  {
    m_networkPage->m_useNMB->setChecked( true );
  }
  if ( QString::compare( browselist.stripWhiteSpace(), "master" ) == 0 )
  {
    m_networkPage->m_useMaster->setChecked( true );
  }
  else if ( QString::compare( browselist.stripWhiteSpace(), "host" ) == 0 )
  {
    m_networkPage->m_useHost->setChecked( true );
  }

  if ( !host.isEmpty() )
  {
    m_networkPage->m_hostName->setText( host );
  }

  if ( QString::compare( config()->readEntry( "Network Search", "nmblookup" ), "nmblookup" ) == 0 )
  {
    m_networkPage->m_nmbSearch->setChecked( true );
  }
  else
  {
    m_networkPage->m_clientSearch->setChecked( true );
  }


  //
  // Appearance
  //
  config()->setGroup( "Appearance" );
  m_appearancePage->m_showHidden->setChecked( config()->readBoolEntry( "Show Hidden", true ) );
  m_appearancePage->m_showIPC->setChecked( config()->readBoolEntry( "Show IPC", false ) );
  m_appearancePage->m_showAdmin->setChecked( config()->readBoolEntry( "Show ADMIN", false ) );
  m_appearancePage->m_showPrinter->setChecked( config()->readBoolEntry( "Show Printer", true ) );
  m_appearancePage->m_showIP->setChecked( config()->readBoolEntry( "Show IP", true ) );
  m_appearancePage->m_showType->setChecked( config()->readBoolEntry( "Show Type", true ) );
  m_appearancePage->m_showComment->setChecked( config()->readBoolEntry( "Show Comment", true ) );

  if ( QString::compare( config()->readEntry( "Show Shares", "icons" ), "icons" ) == 0 )
  {
    m_appearancePage->m_shareview->setChecked( false );
  }
  else if ( QString::compare( config()->readEntry( "Show Shares", "icons" ), "list" ) == 0 )
  {
    m_appearancePage->m_shareview->setChecked( true );
  }

  if ( !config()->readBoolEntry( "Show Mount Point", false ) )
  {
    m_appearancePage->m_description->setChecked( false );
  }
  else
  {
    m_appearancePage->m_description->setChecked( true );
  }

  m_appearancePage->m_external->setChecked( config()->readBoolEntry( "Show All Shares", false ) );
  m_appearancePage->m_showShareTooltip->setChecked( config()->readBoolEntry( "Show Share Tooltip", true ) );
  m_appearancePage->m_showNetworkItemTooltip->setChecked( config()->readBoolEntry( "Show Network Tooltip", true ) );


  //
  // General options
  //
  config()->setGroup( "Dialogs" );
  setInitialSize( config()->readSizeEntry( "Configuration Dialog Geometry", new QSize( 800, 600 ) ), false );


  //
  // Authentication
  //
  config()->setGroup( "Authentication" );

  if ( config()->readBoolEntry( "Use Wallet", true ) )
  {
    Smb4KAuthInfo *defaultAuth = smb4k_core->passwordHandler()->readDefaultAuth();

    if ( defaultAuth )
    {
      m_authPage->m_defaultLogin->setText( defaultAuth->user() );
      m_authPage->m_defaultPasswd->setText( defaultAuth->password() );
    }
  }

  bool uses_wallet = config()->readBoolEntry( "Use Wallet", true );
  m_authPage->m_login_box->setEnabled( uses_wallet );
  m_authPage->m_use_wallet->setChecked( uses_wallet );
  m_authPage->m_remember_passwords->setChecked( config()->readBoolEntry( "Remember Passwords", true ) );

  if ( config()->readBoolEntry( "Default Authentication", false ) )
  {
    m_authPage->m_defaultAuth->setChecked( true );
    m_authPage->m_authWidget->setEnabled( true );
  }
  else
  {
    m_authPage->m_defaultAuth->setChecked( false );
    m_authPage->m_authWidget->setEnabled( false );
  }


  //
  // Samba options
  //
  config()->setGroup( "Samba" );
  m_sambaPage->m_netbios->setText( config()->readEntry( "NetBIOS Name", *(globalSMBOptions().find( "netbios name" )) ) );
  m_sambaPage->m_domain->setText( config()->readEntry( "Domain", *(globalSMBOptions().find( "workgroup" )) ) );
  m_sambaPage->m_sockopt->setText( config()->readEntry( "Socket Options", *(globalSMBOptions().find( "socket options" )) ) );
  m_sambaPage->m_scope->setText( config()->readEntry( "NetBIOS Scope", *(globalSMBOptions().find( "netbios scope" )) ) );
  m_sambaPage->m_port->setValue( config()->readNumEntry( "Port", 139 ) );

  m_sambaPage->m_krb->setChecked( config()->readBoolEntry( "Use Kerberos", false ) );
  m_sambaPage->m_unicode->setChecked( config()->readBoolEntry( "Mount Unicode", false ) );
  m_sambaPage->m_lfs->setChecked( config()->readBoolEntry( "Mount LFS", false ) );

  QString filesystem = config()->readEntry( "Mount Filesystem", "smbfs" );

  if ( QString::compare( filesystem, "smbfs" ) == 0 || mount_cifs.isEmpty() )
  {
    m_sambaPage->m_filesystem->setCurrentText( "SMBFS" );
    m_sambaPage->slotSambaFileSystem( "SMBFS" );
  }
  else
  {
    m_sambaPage->m_filesystem->setCurrentText( "CIFS" );
    m_sambaPage->slotSambaFileSystem( "CIFS" );
  }

  // TODO: Check this with a system that has no mount.cifs installed!
  if ( mount_cifs.isEmpty() )
  {
    m_sambaPage->m_filesystem->removeItem( 1 );
  }

  QValueList<Smb4KUser *> list = smb4k_core->fileIO()->getUsers();
  QStringList uid, gid;

  for ( QValueList<Smb4KUser *>::ConstIterator it = list.begin(); it != list.end(); ++it )
  {
    // Put the UIDs in the respective combo box.
    if ( uid.find( QString( "%1" ).arg( (*it)->uid() ) ) == uid.end() )
    {
      uid.append( QString( "%1" ).arg( (*it)->uid() ) );
    }

    // Put the GIDs in the respective combo box.
    if ( gid.find( QString( "%1" ).arg( (*it)->gid() ) ) == gid.end() )
    {
      gid.append( QString( "%1" ).arg( (*it)->gid() ) );
    }
  }

  uid.sort();
  gid.sort();

  m_sambaPage->m_uid->insertStringList( uid, -1 );
  m_sambaPage->m_uid->setCurrentText( config()->readEntry( "Mount UID", i18n( "default" ) ) );

  m_sambaPage->m_gid->insertStringList( gid, -1 );
  m_sambaPage->m_gid->setCurrentText( config()->readEntry( "Mount GID", i18n( "default" ) ) );

#ifdef __linux__
  // Under Linux the codepage and charset are only supported for
  // Kernels >= 2.4.0.
  if ( k_major >= 2 && k_minor >= 4 && k_rev >= 0 )
  {
    m_sambaPage->m_codepage->setCurrentText( config()->readEntry( "Mount Codepage", i18n( "default" ) ) );
    m_sambaPage->m_charset->setCurrentText( config()->readEntry( "Mount Charset", i18n( "default" ) ) );
  }
  else
  {
    ((QWidget *)m_sambaPage->child( "SambaCharsetCodepage", "QGroupBox", true ))->setEnabled( false );
  }
#else
  // Other operating systems than Linux also support the codepage
  // and charset options. Up to now, now restrictions like under
  // Linux are known. Include them here if some emerge:
  m_sambaPage->m_codepage->setCurrentText( config()->readEntry( "Mount Codepage", i18n( "default" ) ) );
  m_sambaPage->m_charset->setCurrentText( config()->readEntry( "Mount Charset", i18n( "default" ) ) );
#endif

  QString fmask = config()->readEntry( "Mount FMASK", QString( "%1" ).arg( 0777
- mask, 0, 8 ) );
  QString dmask = config()->readEntry( "Mount DMASK", QString( "%1" ).arg( 0777 - mask, 0, 8 ) );

  // We need to make sure that the masks have 4 digits:
  if ( fmask.length() < 4 )
  {
    while ( fmask.length() < 4 )
    {
      fmask.prepend( "0" );
    }
  }

  if ( dmask.length() < 4 )
  {
    while ( dmask.length() < 4 )
    {
      dmask.prepend( "0" );
    }
  }

  m_sambaPage->m_fmask->setText( fmask );
  m_sambaPage->m_dmask->setText( dmask );

  if ( config()->readBoolEntry( "Mount ReadWrite", true ) )
  {
    m_sambaPage->m_read_mode->setCurrentText( i18n( "read-write" ) );
  }
  else
  {
    m_sambaPage->m_read_mode->setCurrentText( i18n( "read-only" ) );
  }

  if ( k_major >= 2 && k_minor >= 4 && k_rev >= 2 )
  {
    m_sambaPage->m_ttl->setValue( config()->readNumEntry( "Mount Cache", 1000 ) );
  }
  else
  {
    m_sambaPage->m_ttl->setEnabled( false );
  }

  m_sambaPage->m_rsize->setChecked( config()->readBoolEntry( "Mount RSize", false ) );
  m_sambaPage->m_wsize->setChecked( config()->readBoolEntry( "Mount WSize", false ) );

  m_sambaPage->m_buffer->setValue( config()->readNumEntry( "Client Buffer Size", 65520 ) );
  m_sambaPage->m_resolve->setText( config()->readEntry( "Client Resolve Order", *(globalSMBOptions().find( "name resolve order" )) ) );

  m_sambaPage->m_broadcast->setText( config()->readEntry( "NMB Broadcast", *(globalSMBOptions().find( "interfaces" )) ) );
  m_sambaPage->m_udp->setChecked( config()->readBoolEntry( "NMB Port 137", false ) );

  QString protocol = config()->readEntry( "Net Protocol", "auto" );

  if ( net.isEmpty() )
  {
    ((QWidget *)m_sambaPage->child( "SambaNetTab", "QWidget" ))->setEnabled( false );
  }

  if ( QString::compare( protocol, "auto" ) == 0 )
  {
    m_sambaPage->m_net_auto->setChecked( true );
  }
  else if ( QString::compare( protocol, "ads" ) == 0 )
  {
    m_sambaPage->m_net_ads->setChecked( true );
  }
  else if ( QString::compare( protocol, "rpc" ) == 0 )
  {
    m_sambaPage->m_net_rpc->setChecked( true );
  }
  else if ( QString::compare( protocol, "rap" ) == 0 )
  {
    m_sambaPage->m_net_rap->setChecked( true );
  }

  m_sambaPage->m_net_machine_account->setChecked( config()->readBoolEntry( "Net Machine Account", false ) );


  //
  // Super user options
  //
  config()->setGroup( "Super User Privileges" );

  QString super_prog = super_prog_prev = config()->readEntry( "SUID Program", "super" );
  bool use_force = use_force_prev = config()->readBoolEntry( "Force Unmount", false );
  bool use_suid = use_suid_prev = config()->readBoolEntry( "Run SUID", false );

  if ( super.isEmpty() && sudo.isEmpty() )
  {
    m_superPage->setEnabled( false );
  }
  else if ( !super.isEmpty() && sudo.isEmpty() )
  {
    m_superPage->m_super->setChecked( true );
    m_superPage->m_sudo->setEnabled( false );
  }
  else if ( super.isEmpty() && !sudo.isEmpty() )
  {
    m_superPage->m_sudo->setChecked( true );
    m_superPage->m_super->setEnabled( false );
  }
  else
  {
    if ( QString::compare( super_prog, "super" ) == 0 )
    {
      m_superPage->m_super->setChecked( true );
    }
    else if ( QString::compare( super_prog, "sudo" ) == 0 )
    {
      m_superPage->m_sudo->setChecked( true );
    }
  }

  if ( QString::compare( system, "Linux" ) == 0 && k_major >= 2 && ( ( k_minor == 4 && k_rev >= 11 ) || k_minor > 4 ) )
  {
    m_superPage->m_unmount_dead->setChecked( use_force );
  }
  else
  {
    m_superPage->m_unmount_dead->setChecked( false );
    m_superPage->m_unmount_dead->setEnabled( false );
  }

  m_superPage->m_use_suid->setChecked( use_suid );

  //
  // Synchronization options
  //
  m_syncPage->setEnabled( rsync.isEmpty() ? false : true );

  config()->setGroup( "Rsync" );
  m_syncPage->m_rsync_prefix->setURL( config()->readPathEntry( "Prefix", QDir::homeDirPath()+"/smb4k_sync/" ) );
  m_syncPage->m_archive->setChecked( config()->readBoolEntry( "archive", true ) );
  m_syncPage->m_recursive->setChecked( config()->readBoolEntry( "recursive", true ) );
  m_syncPage->m_links->setChecked( config()->readBoolEntry( "links", true ) );
  m_syncPage->m_perms->setChecked( config()->readBoolEntry( "perms", true ) );
  m_syncPage->m_times->setChecked( config()->readBoolEntry( "times", true ) );
  m_syncPage->m_group->setChecked( config()->readBoolEntry( "group", true ) );
  m_syncPage->m_owner->setChecked( config()->readBoolEntry( "owner", true ) );
  m_syncPage->m_devices->setChecked( config()->readBoolEntry( "devices", true ) );
  m_syncPage->m_relative->setChecked( config()->readBoolEntry( "relative", false ) );
  m_syncPage->m_omit_dir_times->setChecked( config()->readBoolEntry( "omit dir times", false ) );
  m_syncPage->m_no_implied_dirs->setChecked( config()->readBoolEntry( "no implied dirs", false ) );
  m_syncPage->m_update->setChecked( config()->readBoolEntry( "update", true ) );
  m_syncPage->m_inplace->setChecked( config()->readBoolEntry( "inplace", false ) );
  m_syncPage->m_dirs->setChecked( config()->readBoolEntry( "dirs", false ) );
  m_syncPage->m_copy_links->setChecked( config()->readBoolEntry( "copy links", false ) );
  m_syncPage->m_copy_unsafe_links->setChecked( config()->readBoolEntry( "copy unsafe links", false ) );
  m_syncPage->m_safe_links->setChecked( config()->readBoolEntry( "safe links", false ) );
  m_syncPage->m_hard_links->setChecked( config()->readBoolEntry( "hard links", false ) );
  m_syncPage->m_keep_dirlinks->setChecked( config()->readBoolEntry( "keep dirlinks", false ) );
  m_syncPage->m_delete->setChecked( config()->readBoolEntry( "delete", false ) );
  m_syncPage->m_remove_sent_files->setChecked( config()->readBoolEntry( "remove sent files", false ) );
  m_syncPage->m_delete_before->setChecked( config()->readBoolEntry( "delete before", false ) );
  m_syncPage->m_delete_during->setChecked( config()->readBoolEntry( "delete during", false ) );
  m_syncPage->m_delete_after->setChecked( config()->readBoolEntry( "delete after", false ) );
  m_syncPage->m_delete_excluded->setChecked( config()->readBoolEntry( "delete excluded", false ) );
  m_syncPage->m_ignore_errors->setChecked( config()->readBoolEntry( "ignore errors", false ) );
  m_syncPage->m_force->setChecked( config()->readBoolEntry( "force", false ) );
  m_syncPage->m_whole_file->setChecked( config()->readBoolEntry( "whole file", false ) );
  m_syncPage->m_sparse->setChecked( config()->readBoolEntry( "sparse", false ) );
  m_syncPage->m_one_file_system->setChecked( config()->readBoolEntry( "one file system", false ) );
  m_syncPage->m_existing->setChecked( config()->readBoolEntry( "existing", false ) );
  m_syncPage->m_delay_updates->setChecked( config()->readBoolEntry( "delay updates", false ) );
  m_syncPage->m_compress->setChecked( config()->readBoolEntry( "compress", false ) );
  m_syncPage->m_backup->setChecked( config()->readBoolEntry( "backup", false ) );
  m_syncPage->m_backup_dir->setChecked( config()->readBoolEntry( "Use backup dir", false ) );
  m_syncPage->m_backup_dir_requester->setURL( config()->readPathEntry( "backup dir", QDir::homeDirPath() ) );
  m_syncPage->m_suffix->setChecked( config()->readBoolEntry( "Use suffix", false ) );
  m_syncPage->m_suffix_line->setText( config()->readEntry( "suffix", "~" ) );
  m_syncPage->m_max_delete->setChecked( config()->readBoolEntry( "Use max delete", false ) );
  m_syncPage->m_max_delete_input->setValue( config()->readNumEntry( "max delete", 0 ) );
  m_syncPage->m_checksum->setChecked( config()->readBoolEntry( "checksum", false ) );
  m_syncPage->m_block_size->setChecked( config()->readBoolEntry( "Use block size", false ) );
  m_syncPage->m_block_size_input->setValue( config()->readNumEntry( "block size", 0 ) );
  m_syncPage->m_checksum_seed->setChecked( config()->readBoolEntry( "Use checksum seed", false ) );
  m_syncPage->m_checksum_seed_input->setValue( config()->readNumEntry( "checksum seed", 0 ) );
  m_syncPage->m_filter_input->setText( config()->readEntry( "filter", QString::null ) );
  m_syncPage->m_min_size->setChecked( config()->readBoolEntry( "Use min size", false ) );
  m_syncPage->m_min_size_input->setValue( config()->readNumEntry( "min size", 0 ) );
  m_syncPage->m_max_size->setChecked( config()->readBoolEntry( "Use max size", false ) );
  m_syncPage->m_max_size_input->setValue( config()->readNumEntry( "max size", 0 ) );
  m_syncPage->m_partial->setChecked( config()->readBoolEntry( "partial", false ) );
  m_syncPage->m_partial_dir->setChecked( config()->readBoolEntry( "Use partial dir", false ) );
  m_syncPage->m_partial_dir_requester->setURL( config()->readPathEntry( "partial dir", QDir::homeDirPath() ) );
  m_syncPage->m_cvs_exclude->setChecked( config()->readBoolEntry( "cvs exclude", false ) );
  m_syncPage->m_f_filter->setChecked( config()->readBoolEntry( "F-Filter", false ) );
  m_syncPage->m_ff_filter->setChecked( config()->readBoolEntry( "FF-Filter", false ) );
  m_syncPage->m_exclude->setChecked( config()->readBoolEntry( "Use exclude", false ) );
  m_syncPage->m_exclude_input->setText( config()->readEntry( "exclude", QString::null ) );
  m_syncPage->m_exclude_from->setChecked( config()->readBoolEntry( "Use exclude from", false ) );
  m_syncPage->m_exclude_from_input->setURL( config()->readPathEntry( "exclude from", QDir::homeDirPath()+"/exclude.txt" ) );
  m_syncPage->m_include->setChecked( config()->readBoolEntry( "Use include", false ) );
  m_syncPage->m_include_input->setText( config()->readEntry( "include", QString::null ) );
  m_syncPage->m_include_from->setChecked( config()->readBoolEntry( "Use include from", false ) );
  m_syncPage->m_include_from_input->setURL( config()->readPathEntry( "include from", QDir::homeDirPath()+"/include.txt" ) );
}


/****************************************************************************
   Checks, whether everything has been done
****************************************************************************/


bool Smb4KOptionsDlg::check()
{
  bool ok = true;

  //
  // Network page
  //
  if ( m_networkPage->m_useHost->isChecked() && m_networkPage->m_hostName->text().stripWhiteSpace().isEmpty() )
  {
    ok = false;

    if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the host you want to query to retrieve the browse list.\nDo you want to enter it now?" ) ) == KMessageBox::Yes )
    {
      this->showPage( Network );
      m_networkPage->m_hostName->setFocus();
    }
  }

  //
  // Shares page
  //
  if ( m_sharePage->m_mount_prefix->url().stripWhiteSpace().isEmpty() )
  {
    ok = false;

    if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the path where Smb4K will mount the shares to.\nDo you want to enter it now?" ) ) == KMessageBox::Yes )
    {
      this->showPage( Shares );
      m_sharePage->m_mount_prefix->setFocus();
    }
  }

  //
  // Authentication page
  //
  if ( m_authPage->m_defaultAuth->isChecked() )
  {
    if ( m_authPage->m_defaultLogin->text().stripWhiteSpace().isEmpty() && m_authPage->m_defaultPasswd->text().stripWhiteSpace().isEmpty() )
    {
      ok = false;

      if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the default login and password.\nDo you want to enter them now?" ) ) == KMessageBox::Yes )
      {
        this->showPage( Authentication );
        m_authPage->m_defaultLogin->setFocus();
      }
    }
    else if ( m_authPage->m_defaultLogin->text().stripWhiteSpace().isEmpty() && !m_authPage->m_defaultPasswd->text().stripWhiteSpace().isEmpty() )
    {
      ok = false;

      if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the default login.\nDo you want to enter it now?" ) ) == KMessageBox::Yes )
      {
        this->showPage( Authentication );
        m_authPage->m_defaultLogin->setFocus();
      }
    }
    else if ( !m_authPage->m_defaultLogin->text().stripWhiteSpace().isEmpty() && m_authPage->m_defaultPasswd->text().stripWhiteSpace().isEmpty() )
    {
      ok = false;

      if ( KMessageBox::warningYesNo( this, i18n( "You forgot to enter the default password.\nDo you want to enter it now?" ) ) == KMessageBox::Yes )
      {
        this->showPage( Authentication );
        m_authPage->m_defaultPasswd->setFocus();
      }
    }
  }

  return ok;
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KOptionsDlg::slotOk()
{
  bool force = m_superPage->m_unmount_dead->isChecked();
  bool full_use = m_superPage->m_use_suid->isChecked();

  if ( check() )
  {
    enableButtonOK( false );
    enableButtonApply( false );
    m_superPage->m_remove->setEnabled( false );

    if ( force || full_use )
    {
      m_button = "OK";

      QString prog;

      if ( m_superPage->m_super->isChecked() )
      {
        prog = "super";
      }
      else if ( m_superPage->m_sudo->isChecked() )
      {
        prog = "sudo";
      }

      if ( smb4k_core->fileIO()->writeSuperUserEntries( prog, force, full_use ) )
      {
        if ( saveOptions() )
        {
          emit savedOptions();
        }
      }
    }
    else
    {
      if ( saveOptions() )
      {
        emit savedOptions();

        enableButtonOK( true );
        enableButtonApply( true );
        m_superPage->m_remove->setEnabled( true );

        accept();
      }
    }
  }
}


void Smb4KOptionsDlg::slotApply()
{
  bool force = m_superPage->m_unmount_dead->isChecked();
  bool full_use = m_superPage->m_use_suid->isChecked();

  if ( check() )
  {
    enableButtonOK( false );
    enableButtonApply( false );
    m_superPage->m_remove->setEnabled( false );

    if ( force || full_use )
    {
      m_button = "Apply";

      QString prog;

      if ( m_superPage->m_super->isChecked() )
      {
        prog = "super";
      }
      else if ( m_superPage->m_sudo->isChecked() )
      {
        prog = "sudo";
      }

      if ( smb4k_core->fileIO()->writeSuperUserEntries( prog, force, full_use ) )
      {
        if ( saveOptions() )
        {
          emit savedOptions();
        }
      }
    }
    else
    {
      if ( saveOptions() )
      {
        emit savedOptions();

        enableButtonOK( true );
        enableButtonApply( true );
        m_superPage->m_remove->setEnabled( true );
      }
    }
  }
}


void Smb4KOptionsDlg::slotFinishedSUIDWriting()
{
  enableButtonOK( true );
  enableButtonApply( true );
  m_superPage->m_remove->setEnabled( true );

  if ( QString::compare( m_button, "OK" ) == 0 )
  {
    accept();
  }
  else if ( QString::compare( m_button, "Apply" ) == 0 )
  {
    super_prog_prev = m_superPage->m_super->isChecked() ? "super" : "sudo";
    use_force_prev = m_superPage->m_unmount_dead->isChecked();
    use_suid_prev = m_superPage->m_use_suid->isChecked();
  }
}


void Smb4KOptionsDlg::slotSUIDWritingFailed()
{
  config()->setGroup( "Super User Privileges" );

  if ( QString::compare( super_prog_prev, "super" ) == 0 )
  {
    m_superPage->m_super->setChecked( true );
  }
  else if( QString::compare( super_prog_prev, "sudo" ) == 0 )
  {
    m_superPage->m_sudo->setChecked( true );
  }

  config()->writeEntry( "SUID Program", super_prog_prev );

  m_superPage->m_unmount_dead->setChecked( use_force_prev );
  config()->writeEntry( "Force Unmount", use_force_prev );

  m_superPage->m_use_suid->setChecked( use_suid_prev );
  config()->writeEntry( "Run SUID", use_suid_prev );

  config()->sync();
}


void Smb4KOptionsDlg::slotRemoveSuperUserEntries()
{
  QString prog;

  if ( m_superPage->m_super->isChecked() )
  {
    prog = "super";
  }
  else if ( m_superPage->m_sudo->isChecked() )
  {
    prog = "sudo";
  }

  enableButtonOK( false );
  enableButtonApply( false );
  m_superPage->m_remove->setEnabled( false );

  if ( smb4k_core->fileIO()->removeSuperUserEntries( prog ) )
  {
    m_superPage->m_unmount_dead->setChecked( false );
    m_superPage->m_use_suid->setChecked( false );
  }
}

#include "smb4koptionsdlg.moc"
