/***************************************************************************
    smb4k_umount  -  This is the unmount utility of Smb4K.
                             -------------------
    begin                : Sa Sep 25 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <fstream>
#include <cstring>
#include <cerrno>
#include <unistd.h>

#if !defined(__FreeBSD__) && !defined(__solaris__) && !defined(USE_SOLARIS)
#include <sys/statfs.h>
#elif defined(__solaris__) || defined(USE_SOLARIS)
#include <sys/types.h>
#include <sys/statvfs.h>
#elif defined(__FreeBSD__)
#include <sys/param.h>
#include <sys/mount.h>
#elif defined(__irix__)
#include <sys/types.h>
#endif
using namespace std;

#define SMB4K_UMOUNT_VERSION 0.9

//
// Returns the information about the program
//

void info()
{
  cout << "This is smb4k_umount (version " << SMB4K_UMOUNT_VERSION << "), the unmount utility of Smb4K" << endl;
  cout << "Copyright (C) 2004-2006, Alexander Reinholdt" << endl << endl;
  cout << "Usage:" << endl;
  cout << " smb4k_umount {startopt} {filesystem} {mountpoint}" << endl;
  cout << " smb4k_umount --help" << endl << endl;
  cout << "Arguments:" << endl;
  cout << "  {startopt}" << endl;
  cout << "    --no-suid\tinvokes smbumount or umount.cifs" << endl;
  cout << "    --suid\tinvokes umount" << endl;
  cout << endl;
  cout << "  {filesystem}" << endl;
  cout << "    --cifs\tThe share should be unmounted as CIFS share (i.e. umount.cifs" << endl;
  cout << "\t\tor umount -t cifs is used)" << endl;
  cout << "    --smbfs\tThe share should be unmounted as SMBFS share (i.e. smbumount" << endl;
  cout << "\t\tor umount -t smbfs is used)" << endl;
  cout << endl;
  cout << "  {mountpoint}\tThe mountpoint of the share." << endl;
  cout << endl;
  cout << "  --help\tDisplays this help screen and exits." << endl;
  cout << endl;
}


//
// Finds the umount program
//

const char *findprog( const char *name )
{
  const char *paths[] = { "/bin/", "/sbin/", "/usr/bin/", "/usr/sbin/", "/usr/local/bin/", "/usr/local/sbin/" };
  string file = "";

  for ( uint i = 0; i < sizeof( paths ) / sizeof( char * ); i++ )
  {
    string path( paths[i] );
    path.append( name );

    if ( access( path.c_str(), X_OK ) == 0 )
    {
      file.assign( path );
      break;
    }
  }

  if ( !strcmp( file.c_str(), "" ) )
  {
    cerr << "smb4k_umount: Could not find " << name << "." << endl;
    exit( EXIT_FAILURE );
  }

  return file.c_str();
}


//
// Replaces special characters with their ASCII codes. Some
// characters have been omitted, because they either are non-
// critical or are necessary for mounting (like $).
//

const char *replace_special_characters( const char *str )
{
  string s( str );

  for ( uint i = 0; i < s.length(); i++ )
  {
    switch ( s[i] )
    {
      case '\040':
        s.replace( i, 1, "\040" );
        break;
      case '\041':
        s.replace( i, 1, "\041" );
        break;
      case '\042':
        s.replace( i, 1, "\042" );
        break;
      case '\043':
        s.replace( i, 1, "\043" );
        break;
      case '\045':
        s.replace( i, 1, "\045" );
        break;
      case '\046':
        s.replace( i, 1, "\046" );
        break;
      case '\047':
        s.replace( i, 1, "\047" );
        break;
      case '\050':
        s.replace( i, 1, "\050" );
        break;
      case '\051':
        s.replace( i, 1, "\051" );
        break;
      case '\052':
        s.replace( i, 1, "\052" );
        break;
      case '\054':
        s.replace( i, 1, "\054" );
        break;
      case '\074':
        s.replace( i, 1, "\074" );
        break;
      case '\076':
        s.replace( i, 1, "\076" );
        break;
      case '\130':
        s.replace( i, 1, "\130" );
        break;
      case '\131':
        s.replace( i, 1, "\131" );
        break;
      case '\132':
        s.replace( i, 1, "\132" );
        break;
      case '\133':
        s.replace( i, 1, "\133" );
        break;
      case '\134':
        s.replace( i, 1, "\134" );
        break;
      case '\135':
        s.replace( i, 1, "\135" );
        break;
      case '\136':
        s.replace( i, 1, "\136" );
        break;
      case '\137':
        s.replace( i, 1, "\137" );
        break;
      case '\140':
        s.replace( i, 1, "\140" );
        break;
      case '\141':
        s.replace( i, 1, "\141" );
        break;
      case '\142':
        s.replace( i, 1, "\142" );
        break;
      case '\143':
        s.replace( i, 1, "\143" );
        break;
      case '\144':
        s.replace( i, 1, "\144" );
        break;
      case '\145':
        s.replace( i, 1, "\145" );
        break;
      case '\146':
        s.replace( i, 1, "\146" );
        break;
      case '\147':
        s.replace( i, 1, "\147" );
        break;
      case '\150':
        s.replace( i, 1, "\150" );
        break;
      case '\151':
        s.replace( i, 1, "\151" );
        break;
      case '\153':
        s.replace( i, 1, "\153" );
        break;
      case '\154':
        s.replace( i, 1, "\154" );
        break;
      case '\156':
        s.replace( i, 1, "\156" );
        break;
      case '\157':
        s.replace( i, 1, "\157" );
        break;
      case '\160':
        s.replace( i, 1, "\160" );
        break;
      case '\161':
        s.replace( i, 1, "\161" );
        break;
      case '\162':
        s.replace( i, 1, "\162" );
        break;
      case '\163':
        s.replace( i, 1, "\163" );
        break;
      case '\164':
        s.replace( i, 1, "\164" );
        break;
      case '\165':
        s.replace( i, 1, "\165" );
        break;
      case '\173':
        s.replace( i, 1, "\173" );
        break;
      case '\174':
        s.replace( i, 1, "\174" );
        break;
      case '\175':
        s.replace( i, 1, "\175" );
        break;
      case '\176':
        s.replace( i, 1, "\176" );
        break;
      default:
        break;
    }
  }

  return s.c_str();
}


//
// Checks wether the user is allowed to unmount the
// specified filesystem
//
bool check( const char *path, char *fs_buf )
{
#if !defined(__solaris__) && !defined(USE_SOLARIS)
  struct statfs filesystem;
#else
  struct statvfs filesystem;
#endif

#if !defined(__solaris__) && !defined(USE_SOLARIS) && !defined(__irix__)
  if ( statfs( path, &filesystem ) == -1 )
#elif defined(__irix__)
  if ( statfs( path, &filesystem, sizeof( filesystem ), 0 ) == -1 )
#else
  if ( statvfs( path, &filesystem ) == -1 )
#endif
  {
    int err_code = errno;

    if ( err_code != EIO && err_code != EACCES )
    {
      cerr << "smb4k_umount: The filesystem check failed: " << strerror( err_code ) << endl;
      exit( EXIT_FAILURE );
    }
    else
    {
      fs_buf[0] = '\0';

      return true; // Bypass the check below, because it would yield ok == FALSE
                   // and we want to be able to unmount broken shares as well.
    }
  }

  bool ok = false;

#if !defined(__FreeBSD__) && !defined(__solaris__) && !defined(USE_SOLARIS) && !defined(__irix__)
  // First entry is for CIFS, the second for SMBFS.
  if ( (uint)filesystem.f_type == 0xFF534D42 || (uint)filesystem.f_type == 0x517B )
  {
    strcpy( fs_buf, (uint)filesystem.f_type == 0xFF534D42 ? "cifs" : "smbfs" );
#elif defined(__FreeBSD__)
  if ( !strcmp( filesystem.f_fstypename, "smbfs" ) )
  {
    strcpy( fs_buf, "smbfs" );
#elif defined(__solaris__) || defined(USE_SOLARIS)
  if ( (uint)filesystem.f_basetype == 0xFF534D42 || (uint)filesystem.f_basetype == 0x517B )
  {
    strcpy( fs_buf, (uint)filesystem.f_basetype == 0xFF534D42 ? "cifs" : "smbfs" );
#elif defined(__irix__)
  if ( (uint)filesystem.f_fstyp == 0xFF534D42 || (uint)filesystem.f_fstyp == 0x517B )
  {
    strcpy( fs_buf, (uint)filesystem.f_fstyp == 0xFF534D42 ? "cifs" : "smbfs" );
#endif
    ok = true;
  }

  return ok;
}



//
// The main function
//

int main( int argc, char *argv[], char *envp[] )
{
  if ( argc < 2 )
  {
    info();
    exit( EXIT_FAILURE );
  }

  int k = 0;
  bool suid_arg_exists = false;
  bool use_suid = false;
  char *args[5];
  char fs_arg[10];

  for ( int i = 1; i < argc; i++ )
  {
    if ( !strcmp( argv[i], "--help" ) )
    {
      info();
      exit( EXIT_SUCCESS );
    }
    else if ( !strcmp( argv[i], "--no-suid" ) )
    {
      suid_arg_exists = true;
      use_suid = false;
      // Leave first entry in args[] free:
      k++;

      continue;
    }
    else if ( !strcmp( argv[i], "--suid" ) )
    {
      suid_arg_exists = true;
      use_suid = true;
      // Leave first entry in args[] free:
      k++;

      continue;
    }
    else if ( !strcmp( argv[i], "--cifs" ) || !strcmp( argv[i], "--smbfs" ) )
    {
      (void) strcpy( fs_arg, argv[i] );

      continue;
    }
    else
    {
      // Only do something, if the argument is a mount point:

      // FIXME: Honor relative paths etc. as well.

      if ( argv[i][0] == '/' || ((argv[i][0] == '\047' /* ' */|| argv[i][0] == '\042' /* " */) && argv[i][1] == '/') )
      {
        char *path = new char[strlen( argv[i] )+1];

        if ( !path )
        {
          cerr << "smb4k_umount: Out of memory" << endl;
          exit( EXIT_FAILURE );
        }

        args[k++] = strcpy( path, replace_special_characters( argv[i] ) );

        char fs[6];

        if ( check( path, fs ) )
        {
          char *t = new char[255];

          if ( !t )
          {
            cerr << "smb4k_umount: Out of memory" << endl;
            exit( EXIT_FAILURE );
          }

          if ( suid_arg_exists )
          {
            if ( !use_suid )
            {
              if ( !strcmp( fs, "smbfs" ) )
              {
                args[0] = strcpy( t, findprog( "smbumount" ) );
              }
              else if ( !strcmp( fs, "cifs" ) )
              {
                args[0] = strcpy( t, findprog( "umount.cifs" ) );
              }
              else
              {
                // If the check failed, we need to use the user supplied
                // file system info.
                if ( !strcmp( fs_arg, "--smbfs" ) )
                {
                  args[0] = strcpy( t, findprog( "smbumount" ) );
                }
                else if ( !strcmp( fs_arg, "--cifs" ) )
                {
                  args[0] = strcpy( t, findprog( "umount.cifs" ) );
                }
                else
                {
                  delete [] t;

                  cerr << "smb4k_umount: Could not determine umount binary" << endl;

                  exit( EXIT_FAILURE );
                }
              }
            }
            else
            {
              if ( !strcmp( fs, "smbfs" ) || !strcmp( fs, "cifs" ) )
              {
                args[0] = strcpy( t, findprog( "umount" ) );
              }
              else
              {
                if ( !strcmp( fs_arg, "--smbfs" ) || !strcmp( fs_arg, "--cifs" ) )
                {
                  args[0] = strcpy( t, findprog( "umount" ) );
                }
                else
                {
                  delete [] t;

                  cerr << "smb4k_umount: Could not determine umount binary" << endl;

                  exit( EXIT_FAILURE );
                }
              }
            }
          }
          else
          {
            cerr << "smb4k_mount: You have neither supplied the '--suid' nor '--no-suid' argument." << endl;
            exit( EXIT_FAILURE );
          }
        }
        else
        {
          cerr << "smb4k_umount: You are not allowed to unmount this filesystem type." << endl;
          exit( EXIT_FAILURE );
        }

        continue;
      }
      else
      {
        char *t = new char[strlen( argv[i] )+1];

        if ( !t )
        {
          cerr << "smb4k_umount: Out of memory" << endl;
          exit( EXIT_FAILURE );
        }

        args[k++] = strcpy( t, argv[i] );

        continue;
      }
    }
  }

  args[k] = NULL;

  if ( execve( args[0], args, envp ) == -1 )
  {
    int err_code = errno;

    cerr << "smb4k_umount: " << strerror( err_code ) << endl;
    exit( EXIT_FAILURE );
  }

  return EXIT_SUCCESS;
}
