#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stddef.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <dirent.h>
#include <sys/statvfs.h>
#include <time.h>
#include <iconv.h>
#include <pthread.h>
#include <libsmbclient.h>
#include "common.h"
#include "buffer.h"
#include "array.h"
#include "charset.h"
#include "smbctx.h"
#include "smbitem.h"
#include "reconfigure.h"
#include "function.h"

#define SS_FILE		0x01
#define SS_DIR		0x02
#define SS_ANY		(SS_FILE | SS_DIR)

typedef struct{
    BUFFER	*url;
    SmbCtx	*smb;
    SMBCFILE	*fd;
    off_t	offset;
    int		flags;
} SmbState;

BUFFERMANAGER   OpenFiles	= {"OpenFiles", 100, 0, sizeof(SmbState),
				    STATIC_LIST_INITIALIZER(OpenFiles.list),
				    PTHREAD_MUTEX_INITIALIZER
				  };
int		fs_free_space_blocks	= 0;	
int		fs_block_size		= 32768;
int		fs_quiet_flag		= 1;
int		show_dollar_shares	= 0;
int		kde_workaround		= 3;
int		unsafe_truncate		= 0;
int		fast_shutdown		= 1;
int		update_delta		= 5;
time_t		last_event_time		= (time_t) 0;
pthread_mutex_t	m_function		= PTHREAD_MUTEX_INITIALIZER;

pthread_t	t_update;

int SetFsFreeSpaceBlocks(int blocks){
    if (blocks < 0) return 0;
    DPRINT(7, "blocks=%d\n", blocks);
    pthread_mutex_lock(&m_function);
    fs_free_space_blocks = blocks;
    pthread_mutex_unlock(&m_function);
    return 1;
}

int GetFsFreeSpaceBlocks(void){
    int blocks;

    pthread_mutex_lock(&m_function);
    blocks = fs_free_space_blocks;
    pthread_mutex_unlock(&m_function);
    DPRINT(7, "blocks=%d\n", blocks);
    return blocks;
}

int SetFsBlockSize(int size){
    if (size < 4096) return 0;
    DPRINT(7, "size=%d\n", size);
    size -= (size % 4096);
    pthread_mutex_lock(&m_function);
    fs_block_size = size;
    pthread_mutex_unlock(&m_function);
    return 1;
}

int GetFsBlockSize(void){
    int size;

    pthread_mutex_lock(&m_function);
    size = fs_block_size;
    pthread_mutex_unlock(&m_function);
    DPRINT(7, "size=%d\n", size);
    return size;
}

int SetFsQuietFlag(int flag){
    DPRINT(7, "flag=%d\n", flag);
    pthread_mutex_lock(&m_function);
    fs_quiet_flag = flag;
    pthread_mutex_unlock(&m_function);
    return 1;
}

int GetFsQuietFlag(void){
    int flag;

    pthread_mutex_lock(&m_function);
    flag = fs_quiet_flag;
    pthread_mutex_unlock(&m_function);
    DPRINT(7, "flag=%d\n", flag);
    return flag;
}

int SetShowDollarShares(int flag){
    DPRINT(7, "flag=%d\n", flag);
    pthread_mutex_lock(&m_function);
    show_dollar_shares = flag;
    pthread_mutex_unlock(&m_function);
    return 1;
}

int GetShowDollarShares(void){
    int flag;

    pthread_mutex_lock(&m_function);
    flag = show_dollar_shares;
    pthread_mutex_unlock(&m_function);
    DPRINT(7, "flag=%d\n", flag);
    return flag;
}

int SetKDEworkaround(int count){
    if (count < -1) return 0;
    DPRINT(7, "count=%d\n", count);
    pthread_mutex_lock(&m_function);
    kde_workaround = count;
    pthread_mutex_unlock(&m_function);
    return 1;
}

int GetKDEworkaround(void){
    int count;

    pthread_mutex_lock(&m_function);
    count = kde_workaround;
    pthread_mutex_unlock(&m_function);
    DPRINT(7, "count=%d\n", count);
    return count;
}

int SetUnsafeTruncate(int flag){
    DPRINT(7, "flag=%d\n", flag);
    pthread_mutex_lock(&m_function);
    unsafe_truncate = flag;
    pthread_mutex_unlock(&m_function);
    return 1;
}

int GetUnsafeTruncate(void){
    int flag;

    pthread_mutex_lock(&m_function);
    flag = unsafe_truncate;
    pthread_mutex_unlock(&m_function);
    DPRINT(7, "flag=%d\n", flag);
    return flag;
}

int SetFastShutdown(int flag){
    DPRINT(7, "flag=%d\n", flag);
    pthread_mutex_lock(&m_function);
    fast_shutdown = flag;
    pthread_mutex_unlock(&m_function);
    return 1;
}

int GetFastShutdown(void){
    int flag;

    pthread_mutex_lock(&m_function);
    flag = fast_shutdown;
    pthread_mutex_unlock(&m_function);
    DPRINT(7, "flag=%d\n", flag);
    return flag;
}

void SetLastEventTime(time_t time){
    pthread_mutex_lock(&m_function);
    last_event_time = time;
    pthread_mutex_unlock(&m_function);
};

int SetUpdateTimeDelta(int delta){
    struct itimerval	value;

    if (delta < 1) return 0;
    DPRINT(7, "delta=%d\n", delta);
    pthread_mutex_lock(&m_function);
    if (delta != update_delta){
	memset(&value, 0, sizeof(struct itimerval));
	setitimer(ITIMER_REAL, &value, NULL);

	value.it_value.tv_sec = (time(NULL) + delta - last_event_time);
	if (value.it_value.tv_sec <= 0){
	    value.it_value.tv_sec = 0;
	    value.it_value.tv_usec = 1000;
	}
	value.it_interval.tv_sec = delta;
	setitimer(ITIMER_REAL, &value, NULL);
	update_delta = delta;
    }
    pthread_mutex_unlock(&m_function);
    return 1;
}

int GetUpdateTimeDelta(void){
    int	delta;

    pthread_mutex_lock(&m_function);
    delta = update_delta;
    pthread_mutex_unlock(&m_function);
    DPRINT(7, "delta=%d\n", delta);
    return delta;
}

void set_error_state(SmbCtx *smb, int error){
    switch (error){
	case 0:
	case EACCES:
	case EBADF:
	case EBUSY:
	case EEXIST:
	case EISDIR:
	case ENODEV:
	case ENOENT:
	case ENOTDIR:
	case ENOTEMPTY:
	case ENOTSUP:
	case EPERM:
	case EXDEV:
	    return;
	case EINVAL:
	case ENOMEM:
	default:
	    SetSmbCtxFlags(smb, SMBCTX_IS_BAD);
	    return;
    }
}

int KDEworkaround(const char *path){
    const char	*kde_dir_name = ".directory";
    size_t	kde_dir_len = strlen(kde_dir_name);
    int		i, n = GetKDEworkaround();

    path = filename_begin(path);
    for(i = 0; (i < n) || (n == -1); i++){
	if (*path == '\0') break;
	if ((filename_len(path) == kde_dir_len) &&
	    (strncmp(path, kde_dir_name, kde_dir_len) == 0)) return 1;
	path = next_filename(path);
    }
    return 0;
}

BUFFER* GetSmbStateBuffer(struct fuse_file_info *fi){
    return *((BUFFER**)(void*)(&fi->fh));
}

void StoreSmbStateBuffer(struct fuse_file_info *fi, BUFFER *buf){
    *((BUFFER**)(void*)(&fi->fh)) = buf;
}

SmbState* GetSmbState(struct fuse_file_info *fi){
    if (GetSmbStateBuffer(fi) == NULL) return NULL;
    return (SmbState*)(GetSmbStateBuffer(fi))->data;
}

int convert_filename(const char *path, BUFFER **result, file_type *type){
    if ((*result = GetBuffer(&ConvertBuffer)) == NULL){
	*type = PATH_UNKNOWN;
	return ENOMEM;
    }
    *type = GetPathType(path, *result);
    return 0;
}

int convert_as(const char *path, BUFFER **result, file_type res_type){
    file_type	type;
    int		error;

    if ((error = convert_filename(path, result, &type)) != 0) return error;
    if (type != res_type){
	ReleaseBuffer(*result);
	*result = NULL;
	return (type == PATH_UNKNOWN) ? EILSEQ : ENOTSUP;
    }
    return 0;
}

int get_smb_urlctx(const char *path, BUFFER **smburl, SmbCtx **smbctx){
    int		error;

    if ((error = convert_as(path, smburl, SMBFILE)) != 0){
	*smbctx = NULL;
	return error;
    }
    if ((*smbctx = GetSmbCtx((*smburl)->data, SMBCTX_REGULAR)) == NULL){
	ReleaseBuffer(*smburl);
	*smburl = NULL;
	return ENOMEM;
    }
    return 0;
}

#ifdef HAVE_FUSE_25
static int samba_fgetattr(const char *path, struct stat *stbuf,
			struct fuse_file_info *fi){

    int		error;
    int		block_size;
    SmbState	*state;
    SmbCtx	*smb;

    (void) path;

    DPRINT(5, "(%s)\n", path);
    
    if ((state = GetSmbState(fi)) == NULL){
	/* path to / or /smbgroup */
	block_size = GetFsBlockSize();
	memset(stbuf, 0, sizeof(struct stat));
	stbuf->st_mode = 0040777;	    /* protection */
	stbuf->st_nlink = 2;	    /* number of hard links */
	stbuf->st_uid = 0;		    /* user ID of owner */
	stbuf->st_gid = 0;		    /* group ID of owner */
	stbuf->st_size = 0;		    /* total size, in bytes */
	stbuf->st_blksize = block_size; /* blocksize for filesystem I/O */
	return 0;
    }
    if (state->url == NULL) return -EBADF;
    if (state->smb == NULL) return -EBADF;
    if (state->fd  == NULL) return -EBADF;

    error = 0;
    smb = state->smb;
    LockSmbCtx(smb);
    if (smb->ctx->fstat(smb->ctx, state->fd, stbuf) != 0)
	set_error_state(smb, error = errno);
    UnlockSmbCtx(smb);
    return -error;
}
#endif /* HAVE_FUSE_25 */

static int samba_getattr(const char *path, struct stat *stbuf){
    int		error = 0;
    file_type	type;
    int		block_size;
    BUFFER	*buf;
    SmbCtx	*smb;
    
    DPRINT(5, "(%s)\n", path);
    
    if (KDEworkaround(path) != 0) return -ENOENT;
    if ((error = convert_filename(path, &buf, &type)) != 0) return -error;

    block_size = GetFsBlockSize();
    switch(type){
	case PATH_UNKNOWN:
	    ReleaseBuffer(buf);
	    return -EINVAL;

	case SMBNETFS_LINK:
	    memset(stbuf, 0, sizeof(struct stat));
	    stbuf->st_mode = 0120777;	    /* protection */
	    stbuf->st_nlink = 1;	    /* number of hard links */
	    stbuf->st_uid = 0;		    /* user ID of owner */
	    stbuf->st_gid = 0;		    /* group ID of owner */
	    stbuf->st_size = strlen(buf->data);	/* total size, in bytes */
	    stbuf->st_blksize = block_size; /* blocksize for filesystem I/O */
	    ReleaseBuffer(buf);
	    return 0;

	case SMBNETFS_DIR:
	case SMBNAME:
	case SMBSHARE:
	    memset(stbuf, 0, sizeof(struct stat));
	    stbuf->st_mode = 0040777;	    /* protection */
	    stbuf->st_nlink = 2;	    /* number of hard links */
	    stbuf->st_uid = 0;		    /* user ID of owner */
	    stbuf->st_gid = 0;		    /* group ID of owner */
	    stbuf->st_size = 0;		    /* total size, in bytes */
	    stbuf->st_blksize = block_size; /* blocksize for filesystem I/O */
	    ReleaseBuffer(buf);
	    return 0;

	default:
	    if ((smb = GetSmbCtx(buf->data, SMBCTX_REGULAR)) == NULL){
		ReleaseBuffer(buf);
		return -ENOMEM;
	    }

	    LockSmbCtx(smb);
	    if (smb->ctx->stat(smb->ctx, buf->data, stbuf) != 0)
		set_error_state(smb, error = errno);
	    else stbuf->st_blksize = block_size; /* blocksize for filesystem I/O */
	    UnlockSmbCtx(smb);

 	    ReleaseSmbCtx(smb);
	    ReleaseBuffer(buf);
	    return -error;
    }
}

static int samba_readlink(const char *path, char *buf, size_t size){
    BUFFER	*link;
    int		error = 0;

    DPRINT(5, "(%s, %d)\n", path, size);
    *buf = '\0';
    if ((error = convert_as(path, &link, SMBNETFS_LINK)) != 0) return -error;
    strncpy(buf, link->data, size);
    ReleaseBuffer(link);
    return 0;
}

static int samba_mknod(const char *path, mode_t mode, dev_t rdev){
    int		error;
    BUFFER	*url;
    SmbCtx	*smb;
    SMBCFILE	*fd;
    
    (void) rdev;

    DPRINT(5, "(%s, %o)\n", path, mode);
    if ((mode & S_IFMT) != S_IFREG) return -EPERM;
    if ((error = get_smb_urlctx(path, &url, &smb)) != 0) return -error;

    LockSmbCtx(smb);
    if ((fd = smb->ctx->creat(smb->ctx, url->data, mode)) == NULL)
	set_error_state(smb, error = errno);
    else
#ifdef HAVE_LIBSMBCLIENT_3_0_20    
	smb->ctx->close_fn(smb->ctx, fd);
#else
	smb->ctx->close(smb->ctx, fd);
#endif
    UnlockSmbCtx(smb);

    ReleaseSmbCtx(smb);
    ReleaseBuffer(url);
    return -error;
}

static int samba_mkdir(const char *path, mode_t mode){
    int		error;
    BUFFER	*url;
    SmbCtx	*smb;

    DPRINT(5, "(%s, %o)\n", path, mode);
    if ((error = get_smb_urlctx(path, &url, &smb)) != 0) return -error;

    LockSmbCtx(smb);
    if (smb->ctx->mkdir(smb->ctx, url->data, mode) != 0)
	set_error_state(smb, error = errno);
    UnlockSmbCtx(smb);

    ReleaseSmbCtx(smb);
    ReleaseBuffer(url);
    return -error;
}

static int samba_unlink(const char *path){
    int		error;
    BUFFER	*url;
    SmbCtx	*smb;
    
    DPRINT(5, "(%s)\n", path);
    if ((error = get_smb_urlctx(path, &url, &smb)) != 0) return -error;

    LockSmbCtx(smb);
    if (smb->ctx->unlink(smb->ctx, url->data) != 0)
	set_error_state(smb, error = errno);
    UnlockSmbCtx(smb);

    ReleaseSmbCtx(smb);
    ReleaseBuffer(url);
    return -error;
}

static int samba_rmdir(const char *path){
    int		error;
    BUFFER	*url;
    SmbCtx	*smb;
    
    DPRINT(5, "(%s)\n", path);
    if ((error = get_smb_urlctx(path, &url, &smb)) != 0) return -error;

    LockSmbCtx(smb);
    if (smb->ctx->rmdir(smb->ctx, url->data) != 0)
	set_error_state(smb, error = errno);
    UnlockSmbCtx(smb);

    ReleaseSmbCtx(smb);
    ReleaseBuffer(url);
    return -error;
}

static int samba_symlink(const char *from, const char *to){
    (void) from;
    (void) to;

    DPRINT(5, "(%s, %s)\n", from, to);
    return -EPERM;
}

static int samba_rename(const char *from, const char *to){
    int		error;
    BUFFER      *url_from, *url_to;
    SmbCtx	*smb;

    DPRINT(5, "(%s, %s)\n", from, to);
    if ((error = convert_as(from, &url_from, SMBFILE)) != 0) return -error;
    if ((error = get_smb_urlctx(to, &url_to, &smb)) != 0){
	ReleaseBuffer(url_from);
	return -error;
    }

    LockSmbCtx(smb);
    if (smb->ctx->rename(smb->ctx, url_from->data,
	smb->ctx, url_to->data) != 0) set_error_state(smb, error = errno);
    UnlockSmbCtx(smb);

    ReleaseSmbCtx(smb);
    ReleaseBuffer(url_to);
    ReleaseBuffer(url_from);
    return -error;
}

static int samba_link(const char *from, const char *to){
    (void) from;
    (void) to;
    
    DPRINT(5, "(%s, %s)\n", from, to);
    return -EPERM;
}

static int samba_chmod(const char *path, mode_t mode){
    int		error;
    BUFFER	*url;
    SmbCtx	*smb;

    DPRINT(5, "(%s, %o)\n", path, mode);
    if ((error = get_smb_urlctx(path, &url, &smb)) != 0) return -error;

    LockSmbCtx(smb);
    if (smb->ctx->chmod(smb->ctx, url->data, mode) != 0)
	set_error_state(smb, error = errno);
    UnlockSmbCtx(smb);

    ReleaseSmbCtx(smb);
    ReleaseBuffer(url);
    return -error;
}

static int samba_chown(const char *path, uid_t uid, gid_t gid){
    (void) path;
    (void) uid;
    (void) gid;

    DPRINT(5, "(%s, uid=%d, gid=%d)\n", path, uid, gid);
    if (GetFsQuietFlag()) return 0;
    else return -EPERM;
}

#ifdef HAVE_FUSE_25
static int samba_ftruncate(const char *path, off_t size,
			struct fuse_file_info *fi){

    static char data = '\0';
    int		error;
    SmbState    *state;
    SmbCtx	*smb;
    struct stat	st;

    DPRINT(5, "(%s, %lld)\n", path, (long long) size);
    
    if (size < 0) return -EINVAL;
    if ((state = GetSmbState(fi)) == NULL) return -EBADF;
    if ((state->flags & SS_ANY) != SS_FILE) return -EBADF;
    if (state->url == NULL) return -EBADF;
    if (state->smb == NULL) return -EBADF;
    if (state->fd  == NULL) return -EBADF;
    
    error = 0;
    smb = state->smb;
    LockSmbCtx(smb);
    if (smb->ctx->fstat(smb->ctx, state->fd, &st) != 0){
	set_error_state(smb, error = errno);
	goto end;
    }
    if (size < st.st_size){
	if (size > 0){
	    if (!GetUnsafeTruncate()){
		// samba do not support truncate operation
		// so return error for now
		DPRINT(3, "(%s, size=%lld, filesize=%lld): "
		    "operation is not supported\n", 
		    path, (long long) size, (long long)st.st_size);
		error = ENOTSUP;
		goto end;
	    }
	    DPRINT(3, "(%s, size=%lld, filesize=%lld): "
		"trying unsafe operation...\n",
		path, (long long) size, (long long)st.st_size);
	}

#ifdef HAVE_LIBSMBCLIENT_3_0_20
	smb->ctx->close_fn(smb->ctx, state->fd);
#else
	smb->ctx->close(smb->ctx, state->fd);
#endif

	if ((state->fd = smb->ctx->creat(smb->ctx, state->url->data, 0777)) == NULL){
	    state->offset = (off_t) -1;
	    set_error_state(smb, error);
	    goto end;
	}
    }

    if (size > 0){
	if (smb->ctx->lseek(smb->ctx, state->fd, size - 1, SEEK_SET) == (off_t) -1){
	    state->offset = (off_t) -1;
	    set_error_state(smb, error = errno);
	    goto end;
	}
	if (smb->ctx->write(smb->ctx, state->fd, &data, 1) == (ssize_t) -1){
	    state->offset = (off_t) -1;
	    set_error_state(smb, error = errno);
	    goto end;
	}
    }

    state->offset = smb->ctx->lseek(smb->ctx, state->fd, 0, SEEK_CUR);
    if (state->offset == (off_t) -1){
	state->offset = (off_t) -1;
	set_error_state(smb, error = errno);
	goto end;
    }

  end:
    UnlockSmbCtx(smb);
    return -error;
}
#endif /* HAVE_FUSE_25 */

static int samba_truncate(const char *path, off_t size){
    static char data = '\0';
    int		error;
    SmbCtx	*smb;
    SMBCFILE	*fd;
    BUFFER	*url;
    struct stat	st;

    DPRINT(5, "(%s, %lld)\n", path, (long long) size);
    if (size < 0) return -EINVAL;
    if ((error = get_smb_urlctx(path, &url, &smb)) != 0) return -error;

    LockSmbCtx(smb);
    if (smb->ctx->stat(smb->ctx, url->data, &st) != 0){
	set_error_state(smb, error = errno);
	goto end;
    }
    if (size == st.st_size) goto end;
    if (size > st.st_size){
	fd = smb->ctx->open(smb->ctx, url->data, O_WRONLY, 0777);
    	if (fd == NULL){
	    set_error_state(smb, error = errno);
	    goto end;
	}
    }else{
	if (size > 0){
	    if (!GetUnsafeTruncate()){
		// samba do not support truncate operation
		// so return error for now
		DPRINT(3, "(%s, size=%lld, filesize=%lld): "
		    "operation is not supported\n", 
		    path, (long long) size, (long long)st.st_size);
		error = ENOTSUP;
		goto end;
	    }	    
	    DPRINT(3, "(%s, size=%lld, filesize=%lld): "
		"trying unsafe operation...\n", 
		path, (long long) size, (long long)st.st_size);
	}    

	if ((fd = smb->ctx->creat(smb->ctx, url->data, 0777)) == NULL){
	    set_error_state(smb, error = errno);
	    goto end;
	}
    }
    if (size > 0){
	if (smb->ctx->lseek(smb->ctx, fd, size - 1, SEEK_SET) == (off_t) -1){
	    set_error_state(smb, error = errno);
	    goto end_with_close;
	}
	if (smb->ctx->write(smb->ctx, fd, &data, 1) == (ssize_t)(-1)){
	    set_error_state(smb, error = errno);
	    goto end_with_close;
	}
    }
    
  end_with_close:
#ifdef HAVE_LIBSMBCLIENT_3_0_20
    smb->ctx->close_fn(smb->ctx, fd);
#else
    smb->ctx->close(smb->ctx, fd);
#endif

  end:
    UnlockSmbCtx(smb);

    ReleaseSmbCtx(smb);
    ReleaseBuffer(url);
    return -error;
}

static int samba_utime(const char *path, struct utimbuf *buffer){
    int			error;
    BUFFER		*url;
    SmbCtx		*smb;
    struct timeval	tbuf[2];
    
    DPRINT(5, "(%s, %u)\n", path, (unsigned int)buffer->modtime);
    if ((error = get_smb_urlctx(path, &url, &smb)) != 0) return -error;

    tbuf[0].tv_sec = buffer->actime;
    tbuf[0].tv_usec = 0;
    tbuf[1].tv_sec = buffer->modtime;
    tbuf[1].tv_usec = 0;

    LockSmbCtx(smb);
    if (smb->ctx->utimes(smb->ctx, url->data, tbuf) != 0)
	set_error_state(smb, error = errno);
    UnlockSmbCtx(smb);

    ReleaseSmbCtx(smb);
    ReleaseBuffer(url);
    return -error;
}

#ifdef HAVE_FUSE_25
static int samba_create(const char *path, mode_t mode, 
			struct fuse_file_info *fi){

    int		error;
    BUFFER	*url;
    SmbCtx	*smb;
    BUFFER	*stbuf;
    SmbState	*state;

    DPRINT(5, "(%s, mode=%0x, flags=%o, fh=%llx)\n", path, mode, 
	fi->flags, (long long)fi->fh);
    if ((stbuf = GetBuffer(&OpenFiles)) == NULL) return -ENOMEM;
    if ((error = get_smb_urlctx(path, &url, &smb)) != 0){
	ReleaseBuffer(stbuf);
	return -error;
    }

    state = (SmbState*) stbuf->data;
    state->url = url;
    state->smb = smb;
    state->offset = (off_t) -1;
    state->flags = SS_FILE;

    LockSmbCtx(smb);
    if ((state->fd = smb->ctx->creat(smb->ctx, url->data, mode)) == NULL)
	set_error_state(smb, error = errno);
    UnlockSmbCtx(smb);

    if (error != 0){
	ReleaseBuffer(stbuf);
	ReleaseSmbCtx(smb);
	ReleaseBuffer(url);
	return -error;
    }
    StoreSmbStateBuffer(fi, stbuf);
    return 0;
}
#endif /* HAVE_FUSE_25 */

static int samba_open(const char *path, struct fuse_file_info *fi){
    int		error;
    BUFFER	*url;
    SmbCtx	*smb;
    BUFFER	*stbuf;
    SmbState	*state;

    DPRINT(5, "(%s, flags=%o, fh=%llx)\n", path, fi->flags, (long long)fi->fh);
    if ((stbuf = GetBuffer(&OpenFiles)) == NULL) return -ENOMEM;
    if ((error = get_smb_urlctx(path, &url, &smb)) != 0){
	ReleaseBuffer(stbuf);
	return -error;
    }

    state = (SmbState*) stbuf->data;
    state->url = url;
    state->smb = smb;
    state->offset = (off_t) -1;
    state->flags = SS_FILE;

    LockSmbCtx(state->smb);
    if ((state->fd = smb->ctx->open(smb->ctx, url->data, fi->flags, 0777)) == NULL)
	set_error_state(smb, error = errno);
    UnlockSmbCtx(smb);

    if (error != 0){
	ReleaseBuffer(stbuf);
	ReleaseSmbCtx(smb);
	ReleaseBuffer(url);
	return -error;
    }
    StoreSmbStateBuffer(fi, stbuf);
    return 0;
}

static int samba_flush(const char *path, struct fuse_file_info *fi){
    DPRINT(5, "(%s, fh=%llx, flags=%o)\n", path, (long long)fi->fh, fi->flags);
    return 0;
}

static int samba_release(const char *path, struct fuse_file_info *fi){
    int		error;
    SmbState	*state;
    SmbCtx	*smb;

    DPRINT(5, "(%s, fh=%llx, flags=%o)\n", path, (long long)fi->fh, fi->flags);

    error = EBADF;
    if ((state = GetSmbState(fi)) == NULL) return -error;
    if ((state->flags & SS_ANY) != SS_FILE) goto release_smbstate;
    if (state->url == NULL) goto release_smbstate;
    if (state->smb == NULL) goto release_url;
    if (state->fd  == NULL) goto release_context;

    error = 0;
    smb = state->smb;
    LockSmbCtx(state->smb);
#ifdef HAVE_LIBSMBCLIENT_3_0_20
    if (smb->ctx->close_fn(smb->ctx, state->fd) != 0)
#else
    if (smb->ctx->close(smb->ctx, state->fd) != 0)
#endif
	set_error_state(smb, error = errno);
    UnlockSmbCtx(state->smb);

  release_context:
    ReleaseSmbCtx(state->smb);
  release_url:
    ReleaseBuffer(state->url);
  release_smbstate:
    ReleaseBuffer(GetSmbStateBuffer(fi));
    return -error;
}

static int samba_reopen(const char *path, struct fuse_file_info *fi){
    int		error;
    SmbCtx	*smb;
    SmbState	*state;

    DPRINT(5, "(%s, fh=%llx, flags=%o)\n", path, (long long)fi->fh, fi->flags);

    if ((state = GetSmbState(fi)) == NULL) return -EBADF;
    if ((state->flags & SS_ANY) != SS_FILE) return -EBADF;
    if (state->url == NULL) return -EBADF;
    if (state->smb == NULL) return -EBADF;
    if (state->fd != NULL){
	smb = state->smb;
	LockSmbCtx(state->smb);
#ifdef HAVE_LIBSMBCLIENT_3_0_20
	if (smb->ctx->close_fn(smb->ctx, state->fd) != 0)
#else
	if (smb->ctx->close(smb->ctx, state->fd) != 0)
#endif
	    set_error_state(state->smb, errno);
	UnlockSmbCtx(state->smb);
    }

    state->fd = NULL;
    state->offset = (off_t) -1;

    /* It maybe required to get a new context to recover from error. */
    /* Try to get a new samba context, keep old context on error     */
    if ((smb = GetSmbCtx(state->url->data, SMBCTX_REGULAR)) == NULL) return -EBADF;
    ReleaseSmbCtx(state->smb);
    state->smb = smb;

    /* try to open file again */
    error = 0;
    LockSmbCtx(state->smb);
    if ((state->fd = state->smb->ctx->open(state->smb->ctx,
	state->url->data, fi->flags, 0777)) == NULL)
	    set_error_state(state->smb, error = errno);
    UnlockSmbCtx(state->smb);

    return -error;
}

static int samba_read(const char *path, char *buf, size_t size, off_t offset,
		    struct fuse_file_info *fi){
    size_t	block_size;
    int		error, result = 0, err_count = 0;
    SmbState	*state;
    SmbCtx	*smb;

    DPRINT(5, "(%s, %d, fh=%llx, offset=%lld, flags=%o)\n", path, size,
	(long long)fi->fh, (long long)offset, fi->flags);

  restart:
    block_size = GetFsBlockSize();
    if ((state = GetSmbState(fi)) == NULL) return -EBADF;
    if ((state->flags & SS_ANY) != SS_FILE) return -EBADF;
    if (state->url == NULL) return -EBADF;
    if (state->smb == NULL) return -EBADF;
    if (state->fd  == NULL) return -EBADF;

    error = 0;
    smb = state->smb;

    LockSmbCtx(smb);
    if (state->offset != offset){
	state->offset = smb->ctx->lseek(smb->ctx, state->fd, offset, SEEK_SET);
	if (state->offset == (off_t) -1) set_error_state(smb, error = errno);
    }
    while((size > 0) && (error == 0)){
	ssize_t		res;
	size_t		count;

	count = (size <= block_size) ? size : block_size;
	res = smb->ctx->read(smb->ctx, state->fd, buf + result, count);
	if (res == (ssize_t) -1){
	    set_error_state(smb, error = errno);
	    break;
	}
	size -= res; result += res;
	offset += res; state->offset += res;
	if (res != (ssize_t) count) break;
    }
    UnlockSmbCtx(smb);

    if ((error != 0) && (err_count++ < 2)){
	if ((error = samba_reopen(path, fi)) == 0) goto restart;
	return -error;
    }
    return result;
}

static int samba_write(const char *path, const char *buf, size_t size,
		    off_t offset, struct fuse_file_info *fi){
    size_t	block_size;
    int		error, result = 0, err_count = 0;
    SmbState	*state;
    SmbCtx	*smb;

    DPRINT(5, "(%s, %d, fh=%llx, offset=%lld, flags=%o)\n", path, size,
	(long long)fi->fh, (long long)offset, fi->flags);

  restart:
    block_size = GetFsBlockSize();
    if ((state = GetSmbState(fi)) == NULL) return -EBADF;
    if ((state->flags & SS_ANY) != SS_FILE) return -EBADF;
    if (state->url == NULL) return -EBADF;
    if (state->smb == NULL) return -EBADF;
    if (state->fd  == NULL) return -EBADF;

    error = 0;
    smb = state->smb;

    LockSmbCtx(smb);
    if (state->offset != offset){
	state->offset = smb->ctx->lseek(smb->ctx, state->fd, offset, SEEK_SET);
	if (state->offset == (off_t) -1) set_error_state(smb, error = errno);
    }
    while((size > 0) && (error == 0)){
	ssize_t		res;
	size_t		count;

	count = (size <= block_size) ? size : block_size;
	res = smb->ctx->write(smb->ctx, state->fd, (char*)buf + result, count);
	if (res == (ssize_t) -1){
	    set_error_state(smb, error = errno);
	    break;
	}
	size -= res; result += res;
	offset += res; state->offset += res;
	if (res != (ssize_t) count) break;
    }
    UnlockSmbCtx(smb);

    if ((error != 0) && (err_count++ < 2)){
	if ((error = samba_reopen(path, fi)) == 0) goto restart;
	return -error;
    }
    return result;
}

#ifdef HAVE_FUSE_25
static int samba_statfs(const char *path, struct statvfs *stbuf){
    int	free_space_blocks;

    DPRINT(5, "(%s)\n", path);

    free_space_blocks = GetFsFreeSpaceBlocks();
    memset(stbuf, 0, sizeof(struct statvfs));
    stbuf->f_bsize = 4096;
    stbuf->f_frsize = 4096;
    if (free_space_blocks > 0){
	stbuf->f_blocks = free_space_blocks;
	stbuf->f_bfree = free_space_blocks;
	stbuf->f_bavail = free_space_blocks;
	stbuf->f_ffree = 32768;
	stbuf->f_favail = 32768;
    }
    stbuf->f_namemax = FILENAME_MAX;
    return 0;
}
#else 
static int samba_statfs(const char *path, struct statfs *stbuf){
    int	free_space_blocks;

    DPRINT(5, "(%s)\n", path);

    free_space_blocks = GetFsFreeSpaceBlocks();
    memset(stbuf, 0, sizeof(struct statfs));
    stbuf->f_bsize = 4096;
    if (free_space_blocks > 0){
	stbuf->f_blocks = free_space_blocks;
	stbuf->f_bfree = free_space_blocks;
	stbuf->f_bavail = free_space_blocks;
	stbuf->f_ffree = 32768;
    }
#ifdef HAVE_STATFS_F_NAMELEN
    stbuf->f_namelen = FILENAME_MAX;
#endif
#ifdef HAVE_STATFS_F_NAMEMAX
    stbuf->f_namemax = FILENAME_MAX;
#endif
    return 0;
}
#endif /* HAVE_FUSE_25 */

static int samba_fsync(const char *path, int isdatasync,
			struct fuse_file_info *fi){
    (void)isdatasync;
    
    DPRINT(5, "(%s, fh=%llx, flags=%o)\n", path, (long long)fi->fh, fi->flags);
    return 0;
}

static int samba_opendir(const char *path, struct fuse_file_info *fi){
    int		error = 0;
    file_type	type;
    BUFFER	*url, *stbuf;
    SmbState	*state;
    SmbCtx	*smb;

    DPRINT(5, "(%s, flags=%o, fh=%llx)\n", path, fi->flags, (long long)fi->fh);
    if ((error = convert_filename(path, &url, &type)) != 0) return -error;
    if ((stbuf = GetBuffer(&OpenFiles)) == NULL){
	ReleaseBuffer(url);
	return -ENOMEM;
    }

    state = (SmbState*) stbuf->data;
    state->url = url;
    state->offset = (off_t) -1;
    state->flags = SS_DIR;

    switch(type){
	case SMBNETFS_DIR:
	    state->smb = NULL;
	    state->fd = NULL;
	    StoreSmbStateBuffer(fi, stbuf);
	    return 0;

	case SMBNAME:
	case SMBSHARE:
	case SMBFILE:
	    if ((smb = GetSmbCtx(url->data, SMBCTX_REGULAR)) == NULL){
		ReleaseBuffer(stbuf);
		ReleaseBuffer(url);
		return -ENOMEM;
	    }
	    state->smb = smb;

	    LockSmbCtx(smb);
	    if ((state->fd = smb->ctx->opendir(smb->ctx, url->data)) == NULL)
		set_error_state(state->smb, error = errno);
	    UnlockSmbCtx(smb);

	    if (error != 0){
		ReleaseSmbCtx(smb);
		ReleaseBuffer(stbuf);
		ReleaseBuffer(url);
		return -error;
	    }

	    StoreSmbStateBuffer(fi, stbuf);
	    return 0;

	default:
	    ReleaseBuffer(stbuf);
	    ReleaseBuffer(url);
	    return -ENOTDIR;
    }
}

static int samba_releasedir(const char *path, struct fuse_file_info *fi){
    int		error;
    SmbState	*state;
    SmbCtx	*smb;

    DPRINT(5, "(%s, fh=%llx, flags=%o)\n", path, (long long)fi->fh, fi->flags);

    error = EBADF;
    if ((state = GetSmbState(fi)) == NULL) return -error;
    if ((state->flags & SS_ANY) != SS_DIR) goto release_smbstate;
    if (state->url == NULL) goto release_smbstate;
    if (state->smb == NULL) goto release_url;
    if (state->fd  == NULL) goto release_context;

    error = 0;
    smb = state->smb;
    LockSmbCtx(smb);
    if (smb->ctx->closedir(smb->ctx, state->fd) != 0)
	set_error_state(smb, error = errno);
    UnlockSmbCtx(smb);

  release_context:
    ReleaseSmbCtx(state->smb);
  release_url:
    ReleaseBuffer(state->url);
  release_smbstate:
    ReleaseBuffer(GetSmbStateBuffer(fi));
    return -error;
}

static int samba_readdir(const char *path, void *buf, fuse_fill_dir_t filler,
			off_t offset, struct fuse_file_info *fi){
    int			i;
    int			error;
    struct smbc_dirent	*dirent;
    SmbState		*state;
    ARRAY		array;
    BUFFER		*buf1, *buf2;
    SmbCtx		*smb;
    smbitem		*item;
    struct stat 	st;
    
    DPRINT(5, "(%s)\n", path);

    (void) offset;

    memset(&st, 0, sizeof(st));
    if ((state = GetSmbState(fi)) == NULL) return -EBADF;
    if ((state->flags & SS_ANY) != SS_DIR) return -EBADF;
    if (state->url == NULL) return -EBADF;

    if (state->smb == NULL){
	getdir(&array, path, USERPATH | AUTOPATH);

	st.st_mode = S_IFDIR;
	if (filler(buf, ".",  &st, 0)) goto error0;
	if (filler(buf, "..", &st, 0)) goto error0;

	for(i = 0; i < array_count(&array); i++){
	    item = array_elem(&array, i);
	    if (item->type == SMBITEM_LINK) st.st_mode = S_IFLNK;
	    else st.st_mode = S_IFDIR;
	    if (filler(buf, item->name.string, &st, 0)) break;
	}

      error0:
        DeleteSmbItemArray(&array);
	return 0;
    }

    if (state->fd == NULL) return -EBADF;
    st.st_mode = S_IFDIR;

    if ((buf1 = GetBuffer(&ConvertBuffer)) == NULL) return 0;

    error = 0;
    smb = state->smb;
    LockSmbCtx(smb);

    if (smb->ctx->lseekdir(smb->ctx, state->fd, (off_t) 0) != 0){
	set_error_state(smb, error = errno);
	goto error1;
    }
    if (filler(buf, ".",  &st, 0)){
	set_error_state(state->smb, error = EINVAL);
	goto error1;
    }
    if (filler(buf, "..", &st, 0)){
	set_error_state(state->smb, error = EINVAL);
 	goto error1;
     }

    while((dirent = smb->ctx->readdir(smb->ctx, state->fd)) != NULL){
	if (strcmp(dirent->name, "") == 0) continue;
	if (strcmp(dirent->name, ".") == 0) continue;
	if (strcmp(dirent->name, "..") == 0) continue;

	switch(dirent->smbc_type){
	    case SMBC_WORKGROUP:
		error = EBADF;
		goto error1;

	    case SMBC_SERVER:
		SetSmbCtxFlags(smb, SMBCTX_NEEDS_CLEANUP | SMBCTX_DO_NOT_UPDATE);
		st.st_mode = S_IFDIR;
		
		mkgroup(path, AUTOPATH);

		if (BufferSize(buf1) < strlen(path) + 4) break;
		if ((buf2 = GetBuffer(&ConvertBuffer)) == NULL) break;

		buf1->data[0] = '/';
		strcpy(buf1->data + 1, path);
		buf1->data[strlen(path) + 1] = '/';
		if (! smb2local(buf1->data + strlen(path) + 2, dirent->name,
		    BufferSize(buf1) - strlen(path) - 2)) goto conv_error;

		strcpy(buf2->data, "../");
		if (! smb2local(buf2->data + 3, dirent->name, 
		    BufferSize(buf2) - 3)) goto conv_error;

		mklink(buf1->data, buf2->data, AUTOPATH);
	      conv_error:
		ReleaseBuffer(buf2);
		break;
	
	    case SMBC_FILE_SHARE:
		if ((dirent->name[strlen(dirent->name) - 1] == '$') &&
		    !GetShowDollarShares()) continue;
		st.st_mode = S_IFDIR;
		break;

	    case SMBC_DIR:
		st.st_mode = S_IFDIR;
		break;

	    case SMBC_FILE:
		st.st_mode = S_IFREG;
		break;

	    case SMBC_LINK:
	    case SMBC_PRINTER_SHARE:
	    case SMBC_COMMS_SHARE:
	    case SMBC_IPC_SHARE:
	    default:
		continue;
	}
	if (! smb2local(buf1->data, dirent->name, BufferSize(buf1))) continue;
	if (filler(buf, buf1->data, &st, 0)) break;
    }
  error1:
    UnlockSmbCtx(state->smb);
    ReleaseBuffer(buf1);
    return -error;
}

void* UpdateThread(void *data){
    int			sig_number;
    sigset_t		signal_set;
    time_t		current_time;
    int			delta;
    struct itimerval	value;

    (void)data;

    /* init system timer */
    delta = GetUpdateTimeDelta();
    memset(&value, 0, sizeof(struct itimerval));
    setitimer(ITIMER_REAL, &value, NULL);
    value.it_value.tv_sec = delta;
    value.it_interval.tv_sec = delta;
    setitimer(ITIMER_REAL, &value, NULL);

    /* set signals to watch */
    sigemptyset(&signal_set);
    sigaddset(&signal_set, SIGHUP);
    sigaddset(&signal_set, SIGALRM);
    
    SetLastEventTime(time(NULL));
    /* ReadConfig();  --- skip this, as we read config file already */
    UpdateSambaTree();
    while(1){
	sigwait(&signal_set, &sig_number);
	current_time = time(NULL);
	SetLastEventTime(current_time);

	if ((sig_number == SIGHUP) || isTimeForRereadConfig(current_time))
	    ReadConfig();
	if (isTimeForRescanSambaTree(current_time))
	    UpdateSambaTree();
	RefreshOldSmbCtxs(current_time - GetSmbCtxUpdateInterval());
    }
    return NULL;
}	

static void* samba_init(void){
    if (pthread_create(&t_update, NULL, UpdateThread, NULL) != 0){
	fprintf(stderr, "Could not create update thread\n");
	exit(1);
    }
    return NULL;
}

static void samba_destroy(void *private_data){
    (void)private_data;
    
    DPRINT(1, "Destroy update thread\n");
    pthread_cancel(t_update);
    if (!GetFastShutdown()) pthread_join(t_update, NULL);
}

struct fuse_operations smb_oper = {
    .getattr	= samba_getattr,
    .readlink	= samba_readlink,
    .mknod	= samba_mknod,
    .mkdir	= samba_mkdir,
    .unlink	= samba_unlink,
    .rmdir	= samba_rmdir,
    .symlink	= samba_symlink,
    .rename	= samba_rename,
    .link	= samba_link,
    .chmod	= samba_chmod,
    .chown	= samba_chown,
    .truncate	= samba_truncate,
    .utime	= samba_utime,
    .open	= samba_open,
    .read	= samba_read,
    .write	= samba_write,
    .statfs	= samba_statfs,
    .flush	= samba_flush,
    .release	= samba_release,
    .fsync	= samba_fsync,
//    .setxattr	= samba_setxattr,
//    .getxattr	= samba_getxattr,
//    .listxattr	= samba_listxattr,
//    .removexattr= samba_removexattr,    
    .opendir	= samba_opendir,
    .readdir	= samba_readdir,
    .releasedir	= samba_releasedir,
//    .fsyncdir	= samba_fsyncdir,
    .init	= samba_init,
    .destroy	= samba_destroy,
#ifdef HAVE_FUSE_25
//    .access	= samba_access,
    .create	= samba_create,
    .ftruncate	= samba_ftruncate,
    .fgetattr	= samba_fgetattr,
#endif /* HAVE_FUSE_25 */
};
