/*
 *
 *
 */

#ifndef _SPAM_EVAL_H
#define _SPAM_EVAL_H

/*
 * This is a simple story how to use smtpguard evaluator.
 * See test/eval-test.c for more details. Currently, this is the
 * complete example.
 *
 * 1) Prepare context dependant callbacks such as
 *       - variable evaluator
 *       - variable assigner
 *       - function evaluator
 *
 *           ctx.variable = spam_context_variable;
 *           ctx.assign = spam_context_assign;
 *           ctx.func_call = spam_context_funccall;
 * 
 *       - lookuper of ruleid ( rule string to integer value )
 *	 - semantics checker (OPTIONAL)
 *
 *	     ctx.lookup_ruleid = lookup_ruleid;
 *           ctx.semantics = spam_context_semantics;
 *
 *    You can use private_ctx in spam_context_t for your private purpose.
 *
 * 2) Parse config file.
 *
 *      spam_def_t *spamdef = spam_parse_def( stdin, &ctx ) )
 *
 *    Static variables are initialized here.
 *
 * 3) Change your context and call spam_ruleset_eval each time
 *
 *    while()....
 *      MYCTX(ctx)->some_variable = "new value";
 *      spam_ruleset_eval( &ctx, spamdef, lookup_ruleid("SOME_RULE") );
 *
 */

/** spam definition */
typedef struct spam_def spam_def_t;

/** rule */
typedef struct spam_rule spam_rule_t;

/** condition */
typedef struct spam_cond spam_cond_t;

/** context */
typedef struct spam_context spam_context_t;

/** result of evaluation */
typedef struct spam_eval spam_eval_t;

/******************************************************************/
/*  runtime callbacks  */
/******************************************************************/

/**
 * callback function to get the context dependant value of
 * the specified variable.
 **/
typedef spam_eval_t spam_context_variable_func ( const char *varname, spam_context_t *ctx );

/**
 * callback function to assign the context dependant value to
 * the specified variable.
 **/
typedef spam_eval_t spam_context_assign_func ( const char *varname, spam_context_t *ctx, spam_eval_t *val );

/**
 * callback function to call the context dependant function.
 **/
typedef spam_eval_t spam_context_funccall_func ( const char *funcame, spam_context_t *ctx, spam_eval_t **arg_list );

/******************************************************************/
/* parse-time callbacks */
/******************************************************************/

/**
 * callback function to get the unique id of the specified rule string.
 **/
typedef int spam_context_lookup_ruleid_func( const char *name );

/**
 * callback function to check whether the parsed syntax is allowed.
 **/
typedef char *spam_context_semantics_func( spam_cond_t *cond );

/** 
 * spam_action_t is a list of action. spam_cond_t list is interpreted
 * as each condition is connected as AND while in spam_action_t list
 * every action must be evaluated.
 * */
typedef struct spam_action spam_action_t;
typedef int spam_action_func (spam_context_t *ctx, void *arg);

typedef enum {
	SPAM_EVAL_BOOL,
	SPAM_EVAL_NUMBER,
	SPAM_EVAL_STRING
} spam_eval_type;

struct spam_eval
{
	spam_eval_type type;
	union {
            int number;
            const char *string;
	} node;
};


typedef enum {
	SPAM_COND_NUMBER,
	SPAM_COND_VARIABLE,
	SPAM_COND_STRING,
	SPAM_COND_AND,
	SPAM_COND_OR,
	SPAM_COND_NOT,
	SPAM_COND_EQ,
	SPAM_COND_APPROX_EQ,
	SPAM_COND_LT,
	SPAM_COND_GT,
	SPAM_COND_LE,
	SPAM_COND_GE,
	SPAM_COND_ASSIGN,
	SPAM_COND_FUNC_CALL
} spam_cond_type;

struct spam_def
{
	char *mail;
	int expire_status;
	int expir_history;

	spam_rule_t *rule_list;

};

typedef enum {
	SPAM_RULE_ALL = 0,
	SPAM_RULE_RESULT = 255
} spam_rule_type;

struct spam_rule
{
	unsigned int type;
	spam_cond_t *cond_tree;
	spam_action_t *action_list;

	spam_rule_t *next_rule;	
};

struct spam_cond
{
	spam_cond_type type;
	union {
            int number;
            const char *name;
	} node;
	spam_cond_t *children;

	spam_cond_t *next_sibling;	
};

struct spam_action
{
	spam_action_func *action;
	void *arg;
	spam_action_t *next_action;
};

struct spam_context
{
	void *private_ctx;

	/* runtime callbacks */
	spam_context_variable_func *variable;
	spam_context_assign_func *assign;
	spam_context_funccall_func *func_call;

	/* semantics callbacks */
	spam_context_lookup_ruleid_func *lookup_ruleid;
	spam_context_semantics_func *semantics;
};

spam_eval_t spam_cond_eval(spam_cond_t *cond, spam_context_t *ctx );
int spam_do_action(spam_action_t *action, spam_context_t *ctx );

/*******************************************************************/
/* Macros */
/*******************************************************************/
/*
static inline spam_eval_t spam_eval_build_bool( int val );
static inline spam_eval_t spam_eval_build_number( int val );
static inline spam_eval_t spam_eval_build_string( const char *buf );
*/
static inline spam_eval_t spam_eval_build_bool( int val )
{
	spam_eval_t eval = { SPAM_EVAL_BOOL, { val } };
	return eval;
}

static inline spam_eval_t spam_eval_build_number( int val )
{
	spam_eval_t eval = { SPAM_EVAL_NUMBER, { val } };
	return eval;
}

static inline spam_eval_t spam_eval_build_string( const char *buf )
{
	spam_eval_t eval = { SPAM_EVAL_STRING, { string: buf } };
	return eval;
}

/*******************************************************************/
/* Global API */
/*******************************************************************/

/**
 * Evaluate the specified context by the specified spam definition
 * for the specified rule type.
 **/ 
int spam_ruleset_eval( spam_context_t *ctx, spam_def_t *def, int type );

#endif

