%{

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include "eval.h"

#include "config.h"

#define SPAM_PARSE_ERR_MAX 10

typedef struct spam_parse_error
{
	const char *reason;
} spam_parse_error_t;

static spam_parse_error_t spam_parse_errlist[SPAM_PARSE_ERR_MAX] = {};

static spam_def_t *spamdef;
static spam_context_t *yyctx;

typedef void spam_parse_error_func(char *s, ...);
typedef struct spam_parse_func
{
	spam_parse_error_func *errfunc;
} spam_parse_func_t;

void yyerror(char *s);

void spam_parse_func_error_fprintf(char *s, ...)
{
	va_list ap;
	va_start(ap, s);
	vfprintf(stderr, s, ap);
	va_end(ap);
}

spam_parse_func_t spam_parse_funcset = { spam_parse_func_error_fprintf };

#define SPAM_PARSE_ERROR(arg...) ((spam_parse_funcset.errfunc)?spam_parse_funcset.errfunc( arg ):0)

static int spam_parse_error_count()
{
	int i;
	for( i = 0; spam_parse_errlist[i].reason != NULL || i == SPAM_PARSE_ERR_MAX; i++);
	return i;
}

static int spam_parse_error_init()
{
	int i, max = spam_parse_error_count();
	for( i = 0; i < max; i++ ) {
		free( (char *)spam_parse_errlist[i].reason );
		spam_parse_errlist[i].reason = NULL;
	}
    return 0;
}

static int spam_parse_error_add(spam_parse_error_t *newerror)
{
	int i = spam_parse_error_count();
	if( i == ( SPAM_PARSE_ERR_MAX ) ) {
		SPAM_PARSE_ERROR("Too many errors");
		return -1;
	} else if( i == ( SPAM_PARSE_ERR_MAX - 1 ) ) {
		if ( (spam_parse_errlist[i].reason = strdup("Too many errors")) == NULL ) {
			SPAM_PARSE_ERROR("spam_parse_error_add: %s\n", strerror(errno));
		}
		return -1;
	} else {
		spam_parse_errlist[i] = *newerror;
		if ( (spam_parse_errlist[i].reason = strdup(newerror->reason)) == NULL ) {
			SPAM_PARSE_ERROR("spam_parse_error_add: %s\n", strerror(errno));
			return -1;
		}
	}

	return i+1;
}

static int spam_parse_perror(char *prefix)
{
	spam_parse_error_t perr;
#ifdef SOLARIS8
	perr.reason = strerror(errno);
#else
	perr.reason = sys_errlist[errno];
#endif
	return spam_parse_error_add( &perr );
}

static void spam_print_cond_t(spam_cond_t *cond)
{
	switch( cond->type ) {
	case SPAM_COND_VARIABLE:
		SPAM_PARSE_ERROR("VAR:", cond->node.name );
	case SPAM_COND_STRING:
		SPAM_PARSE_ERROR("%s", cond->node.name );
		break;

	case SPAM_COND_NUMBER:
		SPAM_PARSE_ERROR("%d", cond->node.number );
		break;

	case SPAM_COND_EQ:
		SPAM_PARSE_ERROR("==" );
		break;

	case SPAM_COND_AND:
		SPAM_PARSE_ERROR("&" );
		break;

	default:
		SPAM_PARSE_ERROR("UNKNOWN");
		break;
	}

	SPAM_PARSE_ERROR(" ");
}

static void spam_print_cond_tree(spam_cond_t *cond)
{
	spam_print_cond_t(cond);
	if ( cond->children != NULL ) {
		SPAM_PARSE_ERROR("( ");
		spam_print_cond_tree(cond->children);
		SPAM_PARSE_ERROR(") ");
	}

	if ( cond->next_sibling != NULL ) {
			spam_print_cond_tree( cond->next_sibling);
	}

}

#if 0 /* unused */
static void spam_print_rule_t(spam_rule_t *rule)
{
	while( rule != NULL ) {
		SPAM_PARSE_ERROR( "rule: ");
		spam_print_cond_tree( rule->cond_tree );
		rule = rule->next_rule;
		SPAM_PARSE_ERROR("\n");
	}

}
#endif

static int spam_action_eval(spam_context_t *ctx, void *arg)
{
	spam_cond_eval( (spam_cond_t *)arg, ctx );
	return 0;
}

static int spam_parse_check_semantics( spam_cond_t *cond )
{
	if( yyctx->semantics != NULL ) {
		char *reason = yyctx->semantics( cond );
		if( reason != NULL ) {
			spam_parse_error_t e = { reason };
			if ( spam_parse_error_add( &e ) < 0 ) {
				free( reason );
				return -1;
			}
			free( reason );
		}
	}
	return 0;
}
extern char *yytext;

%}

%union {
	spam_cond_t cond, *condp;
	spam_rule_t *rulep;
	spam_action_t *actionp;
	int number;
	spam_cond_type type;
	char *string;
}

%token <number> SPAM_LEX_NUMBER
%token SPAM_LEX_STRING SPAM_LEX_VARIABLE '(' ')'
%token <type> SPAM_LEX_EQ SPAM_LEX_NEQ SPAM_LEX_APPROX_EQ SPAM_LEX_APPROX_NEQ SPAM_LEX_LE SPAM_LEX_GE SPAM_LEX_INC '>' '<'

%type <cond> unary expr_specifier negative_expr_specifier func_name
%type <condp> cond cond_list func_call func_arg arg arg_list
%type <rulep> rule rule_list
%type <string> variable string
%type <actionp> action action_list
%type <number> label

%%

spam_def:	rule_list {}
	;

rule_list:
		rule rule_list
	|	rule
	;

rule: 
		label ':' cond_list ':' action_list '\n'
			{
				spam_rule_t *next_rule = spamdef->rule_list;
				spamdef->rule_list = (spam_rule_t *)malloc(sizeof(spam_rule_t));
				if( spamdef->rule_list == NULL ) {
					spam_parse_perror("malloc");
					YYABORT;
				}
				spamdef->rule_list->type = $1;
				spamdef->rule_list->cond_tree = $3;
				spamdef->rule_list->next_rule = next_rule;
				spamdef->rule_list->action_list = $5;
			}
	|	variable '=' unary '\n'
			{
				spam_eval_t eval = spam_cond_eval(&$3, yyctx);
				yyctx->assign( $1, yyctx, &eval );
				free( $1 );
			}
			
	|	'\n'	{}
	;

variable: SPAM_LEX_VARIABLE
			{ 
				$$ = strdup(yytext);
				if( $$ == NULL ) {
					spam_parse_perror("strdup");
					YYABORT;
				}
			}
	;

string: SPAM_LEX_STRING
			{ 
				$$ = strdup(yytext);
				if( $$ == NULL ) {
					spam_parse_perror("strdup");
					YYABORT;
				}
			}
	;

label: 		SPAM_LEX_STRING
			{
				$$ = (int) yytext; /* FIXME */
			}
	|	SPAM_LEX_NUMBER
			{
				$$ = atoi( yytext );
			}
	|
		variable
			{
				if(( $$ = yyctx->lookup_ruleid( $1 ) ) < 0 ) {
					spam_parse_error_t err = { "Unknown Rule" };
					spam_parse_error_add(&err);
				}
				free( $1 );
			}
	;

cond_list:
		cond ',' cond_list
			{
				spam_cond_t cond = {
					SPAM_COND_AND,
					{ 0 },
					NULL, NULL
				};
				$$ = (spam_cond_t *)malloc(sizeof(spam_cond_t));
				if( $$ == NULL ) {
					spam_parse_perror("malloc");
					YYABORT;
				}
				*$$ = cond;
				$$->children = $1;
				$$->children->next_sibling = $3;
			}
	|	cond
	|
		{
				spam_cond_t cond = {
					SPAM_COND_NUMBER,
					{ 1 },
					NULL, NULL
				};
				$$ = (spam_cond_t *)malloc(sizeof(spam_cond_t));
				if( $$ == NULL ) {
					spam_parse_perror("malloc");
					YYABORT;
				}
				*$$ = cond;
		}
	;

cond:
		unary expr_specifier cond
			{
				$$ = (spam_cond_t *)malloc(sizeof(spam_cond_t));
				if( $$ == NULL ) {
					spam_parse_perror("malloc");
					YYABORT;
				}
				*$$ = $2;
				$$->children = (spam_cond_t *)malloc(sizeof(spam_cond_t));
				if( $$->children == NULL ) {
					spam_parse_perror("malloc");
					YYABORT;
				}
				*$$->children = $1;
				$$->children->next_sibling = $3;
			}
	|	unary negative_expr_specifier cond
			{
				spam_cond_t ncond = {
					SPAM_COND_NOT,
					{ 0 },
					NULL, NULL
				};

				spam_cond_t *cond = (spam_cond_t *)malloc(sizeof(spam_cond_t));
				if( cond == NULL ) {
					spam_parse_perror("malloc");
					YYABORT;
				}
				$$ = (spam_cond_t *)malloc(sizeof(spam_cond_t));
				if( $$ == NULL ) {
					spam_parse_perror("malloc");
					YYABORT;
				}
				*$$ = ncond;
				*cond = $2;
				cond->children = (spam_cond_t *)malloc(sizeof(spam_cond_t));
				if( cond->children == NULL ) {
					spam_parse_perror("malloc");
					YYABORT;
				}
				*cond->children = $1;
				cond->children->next_sibling = $3;
				$$->children = cond;
			}
	|	unary
			{
				$$ = (spam_cond_t *)malloc(sizeof(spam_cond_t));
				if( $$ == NULL ) {
					spam_parse_perror("malloc");
					YYABORT;
				}
				*$$ = $1;
			}
	|	func_call
	;

unary:
		string
			{
				spam_cond_t cond = {
					SPAM_COND_STRING,
					{ 0 },
					NULL, NULL
				};
				cond.node.name = $1;
				$$ = cond;
			}
	|	SPAM_LEX_NUMBER
			{
				spam_cond_t cond = {
					SPAM_COND_NUMBER,
					{ 0 },
					NULL, NULL
				};
				cond.node.number = atoi(yytext);
				$$ = cond;
			}
	|	SPAM_LEX_VARIABLE
			{
				spam_cond_t cond = {
					SPAM_COND_VARIABLE,
					{ 0 },
					NULL, NULL
				};
				cond.node.name = strdup(yytext);
				if( cond.node.name == NULL ) {
					spam_parse_perror("strdup");
					YYABORT;
				}
				$$ = cond;
				if( spam_parse_check_semantics( &$$ ) < 0 ) {
					YYABORT;
				}

			}
	;

func_call:
		func_name '(' func_arg ')'
			{
				spam_cond_t *cond =
					(spam_cond_t *)malloc(sizeof(spam_cond_t));
				if( cond == NULL ) {
					spam_parse_perror("malloc");
					YYABORT;
				}
				cond->type = SPAM_COND_FUNC_CALL;
				cond->node.name = $1.node.name;
				$$ = cond;
				$$->next_sibling = $3;
			}
	;

func_name:	string
			{
				spam_cond_t cond = {
					SPAM_COND_STRING,
					{ 0 },
					NULL, NULL
				};
				$$ = cond;
				$$.node.name = $1;
			}
	;

func_arg:
		arg_list
	|
		{
			$$ = NULL;
		}
	;

arg_list:
		arg ',' arg_list
			{
				$1->next_sibling = $3;
				$$ = $1;
			}
	|	arg
	;

arg:
		cond
	;

expr_specifier:
		SPAM_LEX_EQ
			{
				spam_cond_t cond = {
					SPAM_COND_EQ,
					{ 0 },
					NULL, NULL
				};
				$$ = cond;
			}
	|	SPAM_LEX_APPROX_EQ
			{
				spam_cond_t cond = {
					SPAM_COND_APPROX_EQ,
					{ 0 },
					NULL, NULL
				};
				$$ = cond;
			}
	|	SPAM_LEX_LE
			{
				spam_cond_t cond = {
					SPAM_COND_LE,
					{ 0 },
					NULL, NULL
				};
				$$ = cond;
			}
	|	SPAM_LEX_GE
			{
				spam_cond_t cond = {
					SPAM_COND_GE,
					{ 0 },
					NULL, NULL
				};
				$$ = cond;
			}
	|	'>'
			{
				spam_cond_t cond = {
					SPAM_COND_GT,
					{ 0 },
					NULL, NULL
				};
				$$ = cond;
			}
	|	'<'
			{
				spam_cond_t cond = {
					SPAM_COND_LT,
					{ 0 },
					NULL, NULL
				};
				$$ = cond;
			}
	|	'='
			{
				spam_cond_t cond = {
					SPAM_COND_ASSIGN,
					{ 0 },
					NULL, NULL
				};
				$$ = cond;
			}
	;

negative_expr_specifier:
		SPAM_LEX_NEQ
			{
				spam_cond_t cond = {
					SPAM_COND_EQ,
					{ 0 },
					NULL, NULL
				};
				$$ = cond;
			}
	|
		SPAM_LEX_APPROX_NEQ
			{
				spam_cond_t cond = {
					SPAM_COND_APPROX_EQ,
					{ 0 },
					NULL, NULL
				};
				$$ = cond;
			}
	;

action_list:
		action ',' action_list
			{
				$$ = $1;
				$1->next_action = $3;
			}
	|	action
	;

action:
		cond
			{
				$$ = (spam_action_t *)malloc(sizeof(spam_action_t));
				if( $$ == NULL ) {
					spam_parse_perror("malloc");
					YYABORT;
				}
				$$->action = spam_action_eval;
				$$->arg = (void *)$1;
				$$->next_action = NULL;
			}
	;


%%

extern FILE *yyin;

spam_def_t *spam_parse_def(FILE *fin, spam_context_t *ctx)
{
	int i;
	int max;
	yyin = fin;
	yyctx = ctx;
	spamdef = (spam_def_t *)malloc(sizeof(spam_def_t));
	if( spamdef == NULL ) {
		SPAM_PARSE_ERROR("spam_parse_def: %s\n", strerror(errno));
		return NULL;
	}
	spamdef->rule_list = NULL;
	spam_parse_error_init();
	while( !feof(yyin) ) {
		yyparse();
		if( (max = spam_parse_error_count()) > 0 ) {
			for( i = 0; i < max; i++ ) {
				SPAM_PARSE_ERROR("E: %s\n", spam_parse_errlist[i].reason);
			}
			free( spamdef );
			spamdef = NULL;
		}
	}
	spam_parse_error_init();
	return spamdef;
}

static void spam_cond_free( spam_cond_t *cond )
{
	switch( cond->type ) {
	case SPAM_COND_VARIABLE:
	case SPAM_COND_STRING:
		free( (char *)cond->node.name );
		break;
    default:
        break;
	}

	if ( cond->children != NULL ) {
		spam_cond_free( cond->children );
		cond->children = NULL;
	}

	if ( cond->next_sibling != NULL ) {
		spam_cond_free( cond->next_sibling );
		cond->next_sibling = NULL;
	}
	free( cond );

}

static void spam_action_free( spam_action_t *action_list )
{
	if ( action_list->next_action != NULL ) {
		spam_action_free( action_list->next_action );
		action_list->next_action = NULL;
	}
	spam_cond_free( (spam_cond_t *)action_list->arg );
}

static void spam_rule_list_free( spam_rule_t *rule_list )
{
	if ( rule_list->next_rule != NULL ) {
		spam_rule_list_free( rule_list->next_rule );
		rule_list->next_rule = NULL;
	}

	spam_cond_free( rule_list->cond_tree );
	rule_list->cond_tree = NULL;
	spam_action_free( rule_list->action_list );
	rule_list->action_list = NULL;

}

void spam_free_def( spam_def_t *def )
{
	spam_rule_list_free( def->rule_list);
	def->rule_list = NULL;
	free( def );
}

void yyerror(char *s)
{
	spam_parse_error_t err;
	err.reason = s;
	SPAM_PARSE_ERROR("%s\n", s);
	spam_parse_error_add( &err );
}

