/*==================================================================
 * uif_sfundo.c - User interface sound font undo/redo routines
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include <stdio.h>
#include <gtk/gtk.h>
#include "glade_interface.h"
#include "uif_sftree.h"
#include "sfundo.h"
#include "util.h"
#include "i18n.h"

/* # of remembered items in state history (NOT # of undo levels!) */
#define UIDO_STATE_HISTORY_LENGTH 10

static gboolean uido_cb_history_dialog_destroy (GtkWidget *dialog);
static gint uido_update_history_curpos (gboolean forcesel);
static void uido_update_state_history (void);
static gint uido_add_history_item (GNode *node, gint addpos);
static void uido_update_history_dialog (void);
static void uido_cb_branch (GtkWidget *colbtn, gint dir);

static gboolean uido_history_active = FALSE;
static GtkWidget *uido_history_dialog = NULL;
static GtkWidget *uido_history_clist;
static GNode *uido_dispos;	/* current display pos in action history */
static gint uido_curpos;	/* active position in action history */

/* state history */
static GList *uido_state_history = NULL;
static GList *uido_state_curpos = NULL;

void
uido_create_history_dialog (void)
{
  if (util_activate_unique_dialog ("do", 0)) return;

  uido_history_dialog = create_undowin ();
  util_register_unique_dialog (uido_history_dialog, "do", 0);

  gtk_signal_connect (GTK_OBJECT (uido_history_dialog), "destroy",
		      (GtkSignalFunc)uido_cb_history_dialog_destroy, NULL);

  uido_history_active = TRUE;

  uido_history_clist = gtk_object_get_data (GTK_OBJECT (uido_history_dialog),
					   "CLSTdo");
  gtk_clist_set_column_justification (GTK_CLIST (uido_history_clist), 1,
				      GTK_JUSTIFY_CENTER);
  gtk_clist_set_column_justification (GTK_CLIST (uido_history_clist), 2,
				      GTK_JUSTIFY_CENTER);

  gtk_signal_connect (GTK_OBJECT (GTK_CLIST (uido_history_clist)
				  ->column[1].button),
		      "clicked", GTK_SIGNAL_FUNC (uido_cb_branch),
		      GINT_TO_POINTER (0));
  gtk_signal_connect (GTK_OBJECT (GTK_CLIST (uido_history_clist)
				  ->column[2].button),
		      "clicked", GTK_SIGNAL_FUNC (uido_cb_branch),
		      GINT_TO_POINTER (1));

  uido_dispos = sfdo_tree.curpos;
  uido_curpos = -1;

  uido_update_history_dialog ();

  gtk_widget_show (uido_history_dialog);
}

static gboolean
uido_cb_history_dialog_destroy (GtkWidget *dialog)
{
  uido_history_active = FALSE;
  return (FALSE);
}

static gint
uido_update_history_curpos (gboolean forcesel)
{
  gint row;
  GList *sel;

  if (!uido_history_active) return (-1);

  row = gtk_clist_find_row_from_data (GTK_CLIST (uido_history_clist),
				      sfdo_tree.curpos);
  if (uido_curpos != -1)
    gtk_clist_set_text (GTK_CLIST (uido_history_clist), uido_curpos, 0, NULL);

  gtk_clist_set_text (GTK_CLIST (uido_history_clist), row, 0, ">>");

  /* see if old current position is selected in history list */
  if (!forcesel)
    {
      sel = GTK_CLIST (uido_history_clist)->selection;
      if (sel && GPOINTER_TO_INT (sel->data) == uido_curpos) forcesel = TRUE;
    }

  /* keep curpos selected? */
  if (forcesel) gtk_clist_select_row (GTK_CLIST (uido_history_clist),
				      row, 0);
  uido_curpos = row;

  return (row);
}

/* add sfdo_tree.curpos to the state history */
static void
uido_update_state_history (void)
{
  GList *p, *p2;

  /* if not duplicate state history item, then add it to the state history */
  if (!uido_state_curpos || uido_state_curpos->data != sfdo_tree.curpos)
    {
      /* delete all history items "forward" from current position */
      p = uido_state_curpos;
      if (p) p = g_list_previous (p);
      while (p)
	{
	  p2 = p;
	  p = g_list_previous (p);
	  uido_state_history = g_list_remove_link (uido_state_history, p2);
	  g_list_free_1 (p2);
	}

      /* add the new state item (prepend) */
      uido_state_history = g_list_prepend(uido_state_history, sfdo_tree.curpos);
      uido_state_curpos = uido_state_history; /* new current position */
    }

  /* truncate state history if needed */
  p = g_list_nth (uido_state_history, UIDO_STATE_HISTORY_LENGTH);
  while (p)
    {
      p2 = p;
      p = g_list_next (p);
      uido_state_history = g_list_remove_link (uido_state_history, p2);
      g_list_free_1 (p2);
    }

  /* make sure curpos didn't get truncated */
  if (g_list_position (uido_state_history, uido_state_curpos) == -1)
    uido_state_curpos = g_list_last (uido_state_history);
}

static gint
uido_add_history_item (GNode *node, gint addpos)
{
  SFDoEntry *entry;
  gchar *coltxt[4] = { NULL, NULL, NULL, NULL };
  gint prev, next;
  gint row;

  if (!uido_history_active) return (-1);

  if (!node->data)		/* root node? */
    {
      coltxt[3] = _("<ROOT>");
      gtk_clist_prepend (GTK_CLIST (uido_history_clist), coltxt);
      gtk_clist_set_row_data (GTK_CLIST (uido_history_clist), 0, node);
      return (0);
    }

  entry = (SFDoEntry *)(node->data);

  if (node->parent)
    {
      next = g_node_child_position (node->parent, node);
      prev = g_node_n_children (node->parent) - next - 1;
    }

  if (prev) coltxt[1] = g_strdup_printf ("[%2d]", prev);
  if (next) coltxt[2] = g_strdup_printf ("[%2d]", next);
  coltxt[3] = entry->descr;

  row = gtk_clist_insert (GTK_CLIST (uido_history_clist), addpos, coltxt);

  gtk_clist_set_row_data (GTK_CLIST (uido_history_clist), row, node);

  if (coltxt[1]) g_free (coltxt[1]);
  if (coltxt[2]) g_free (coltxt[2]);

  return (row);
}

/* update the undo history dialog */
static void
uido_update_history_dialog (void)
{
  GNode *n;
  gboolean up = TRUE;	/* go up/down tree toggle */

  if (!uido_history_active) return;

  gtk_clist_freeze (GTK_CLIST (uido_history_clist));
  gtk_clist_clear (GTK_CLIST (uido_history_clist));

  /* traverse up, then down, the tree.. The End */
  n = uido_dispos;
  while (n)
    {
      if (up)
	{
	  uido_add_history_item (n, 0); /* prepend item */
	  n = n->parent;

	  if (!n)
	    {
	      up = FALSE;
	      n = uido_dispos;
	      n = n->children;
	    }
	}
      else
	{
	  uido_add_history_item (n, -1); /* append item */
	  n = n->children;
	}
    }

  uido_update_history_curpos (TRUE);

  gtk_clist_thaw (GTK_CLIST (uido_history_clist));
}

/* user interface wrapper for sfdo_undo */
void
uido_undo (void)
{
  SFTREE_FREEZE ();
  sfdo_undo ();
  SFTREE_THAW ();

  uido_update_history_curpos (FALSE);
  uido_update_state_history ();
}

/* user interface wrapper for sfdo_redo (node is child node of curpos or
   NULL for most recent branch) */
void
uido_redo (GNode *node)
{
  SFTREE_FREEZE ();
  sfdo_redo (node);
  SFTREE_THAW ();

  uido_update_history_curpos (FALSE);
  uido_update_state_history ();
}

/* to satisfy glade interface callbacks */
void
uido_redo_simple (void)
{
  uido_redo (NULL);
}

/* user interface jump routine (jump state to selected history item) */
void
uido_jump (void)
{
  GList *sel;
  GNode *n;

  if (!uido_history_active) return;

  sel = GTK_CLIST (uido_history_clist)->selection;
  if (!sel) return;

  n = gtk_clist_get_row_data (GTK_CLIST (uido_history_clist),
			      GPOINTER_TO_INT (sel->data));
  if (!n) return;

  SFTREE_FREEZE ();
  sfdo_jump (n);
  SFTREE_THAW ();

  uido_update_history_curpos (FALSE);
  uido_update_state_history ();
}

/* called after an undo toplevel group is closed (new undo item) */
void
uido_toplevel_group_done (void)
{
  gint row;
  gint i;

  uido_update_state_history ();

  if (!uido_history_active) return;

  /* find parent row of new node */
  row = gtk_clist_find_row_from_data (GTK_CLIST (uido_history_clist),
				      sfdo_tree.curpos->parent);
  if (row < 0) return;		/* not displayed?, return */
  row++;			/* new row index is after parent so ++ */

  gtk_clist_freeze (GTK_CLIST (uido_history_clist));

  /* # of rows to delete (everything after parent row) */
  i = GTK_CLIST(uido_history_clist)->rows - row;

  /* delete all rows after parent row */
  for (; i > 0; i--)
    gtk_clist_remove (GTK_CLIST (uido_history_clist), row);

  uido_add_history_item (sfdo_tree.curpos, -1); /* append new item */
  uido_update_history_curpos (FALSE);
  gtk_clist_thaw (GTK_CLIST (uido_history_clist));
}

static void
uido_cb_branch (GtkWidget *colbtn, gint dir)
{
  GList *sel;
  GNode *n;
  gint row;
  gint i;

  if (!uido_history_active) return;

  sel = GTK_CLIST (uido_history_clist)->selection;
  if (!sel) return;
  row = GPOINTER_TO_INT (sel->data);

  n = gtk_clist_get_row_data (GTK_CLIST (uido_history_clist), row);
  if (!n) return;

  /* advance to the next/previous sibling as requested by 'dir'ection */
  if (dir == 0) n = n->next;	/* older branches */
  else n = n->prev;		/* newer branches */
  if (!n) return;

  uido_dispos = n;

  gtk_clist_freeze (GTK_CLIST (uido_history_clist));

  /* # of rows to delete (old row and all its children) */
  i = GTK_CLIST(uido_history_clist)->rows - row;

  /* delete rows */
  for (; i > 0; i--)
    gtk_clist_remove (GTK_CLIST (uido_history_clist), row);

  /* append new branch */
  while (n)
    {
      uido_add_history_item (n, -1);
      n = n->children;
    }

  uido_update_history_curpos (FALSE);

  gtk_clist_select_row (GTK_CLIST (uido_history_clist), row, 0);
  gtk_clist_thaw (GTK_CLIST (uido_history_clist));
}

void
uido_back (void)
{
  if (!uido_state_history) return;
  if (!uido_state_curpos || !g_list_next (uido_state_curpos)) return;

  uido_state_curpos = g_list_next (uido_state_curpos);

  SFTREE_FREEZE ();
  sfdo_jump ((GNode *)(uido_state_curpos->data));
  SFTREE_THAW ();

  if (uido_update_history_curpos (FALSE) == -1)
    {
      uido_dispos = (GNode *)(uido_state_curpos->data);
      uido_update_history_dialog ();
    }
}

void
uido_forward (void)
{
  if (!uido_state_history) return;
  if (!uido_state_curpos || !g_list_previous (uido_state_curpos)) return;

  uido_state_curpos = g_list_previous (uido_state_curpos);

  SFTREE_FREEZE ();
  sfdo_jump ((GNode *)(uido_state_curpos->data));
  SFTREE_THAW ();

  if (uido_update_history_curpos (FALSE) == -1)
    {
      uido_dispos = (GNode *)(uido_state_curpos->data);
      uido_update_history_dialog ();
    }
}
