/*
** Copyright (C) 1998,1999,2000,2001 Martin Roesch <roesch@clark.net>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* $Id: spo_unified.c,v 1.14.2.1 2002/01/20 18:12:46 roesch Exp $ */

/* spo_unified 
 * 
 * Purpose:
 *
 * This plugin generates the new unified alert and logging formats
 *
 * Arguments:
 *   
 * filename of the alert and log spools
 *
 * Effect:
 *
 * Packet logs are written (quickly) to a unified output file
 *
 * Comments:
 *
 * The future...
 */

/* your output plugin header file goes here */
#include "spo_unified.h"

/* ----------------External variables -------------------- */
extern char *file_name;    /* current rules file filename */
extern int file_line;      /* current line being processed in rules file */
extern OptTreeNode *otn_tmp;
extern int thiszone;

/* ------------------ Data structures --------------------------*/
typedef struct _UnifiedData
{
    char *filename;
    FILE *stream;
    unsigned int limit;
    unsigned int current;
} UnifiedData;

/* -------------------- Local Functions -----------------------*/
UnifiedData *ParseUnifiedArgs(char *, char *);
void UnifiedInitLogFile(UnifiedData *);
void UnifiedInitAlertFile(UnifiedData *);
void UnifiedLogInit(u_char *);
void UnifiedAlertInit(u_char *);
void LogUnified(Packet *, char *, void *, Event *);
void AlertUnified(Packet *, char *, void *, Event *);
void SpoUnifiedCleanExitFunc(int, void *);
void SpoUnifiedRestartFunc(int, void *);
void SpoUnifiedLogRotateFile(UnifiedData *data);
void SpoUnifiedAlertRotateFile(UnifiedData *data);




/*
 * Function: SetupUnified()
 *
 * Purpose: Registers the output plugin keyword and initialization 
 *          function into the output plugin list.  This is the function that
 *          gets called from InitOutputPlugins() in plugbase.c.
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 */
void SetupUnified()
{
    /* link the preprocessor keyword to the init function in 
       the preproc list */
    RegisterOutputPlugin("log_unified", NT_OUTPUT_LOG, UnifiedLogInit);
    RegisterOutputPlugin("alert_unified", NT_OUTPUT_ALERT, UnifiedAlertInit);

    DebugMessage(DEBUG_INIT, "Output plugin: Unified logging/alerting "
                             "is setup...\n");
}


/*
 * Function: UnifiedInit(u_char *)
 *
 * Purpose: Calls the argument parsing function, performs final setup on data
 *          structs, links the preproc function into the function list.
 *
 * Arguments: args => ptr to argument string
 *
 * Returns: void function
 *
 */
void UnifiedLogInit(u_char *args)
{
    UnifiedData *UnifiedInfo;

    DebugMessage(DEBUG_INIT, "Output: Unified Log Initialized\n");

    /* tell command line loggers to go away */
    pv.log_plugin_active = 1;

    /* parse the argument list from the rules file */
    UnifiedInfo = ParseUnifiedArgs(args, "snort-unified.log");

    LogMessage("UnifiedLogFilename = %s\n", UnifiedInfo->filename);

    UnifiedInitLogFile(UnifiedInfo);

    pv.log_bitmap |= LOG_UNIFIED;

    /* Set the preprocessor function into the function list */
    AddFuncToOutputList(LogUnified, NT_OUTPUT_LOG, UnifiedInfo);
    AddFuncToCleanExitList(SpoUnifiedCleanExitFunc, UnifiedInfo);
    AddFuncToRestartList(SpoUnifiedRestartFunc, UnifiedInfo);
}



void UnifiedAlertInit(u_char *args)
{
    UnifiedData *data;

    DebugMessage(DEBUG_INIT, "Output: Unified Alert Initialized\n");

    pv.alert_plugin_active = 1;

    /* parse the argument list from the rules file */
    data = ParseUnifiedArgs(args, "snort-unified.alert");

    UnifiedInitAlertFile(data);


    LogMessage("UnifiedAlertFilename = %s\n", data->filename);
    /* Set the preprocessor function into the function list */
    AddFuncToOutputList(AlertUnified, NT_OUTPUT_ALERT, data);
    AddFuncToCleanExitList(SpoUnifiedCleanExitFunc, data);
    AddFuncToRestartList(SpoUnifiedRestartFunc, data);
}


/*
 * Function: ParseUnifiedArgs(char *)
 *
 * Purpose: Process the preprocessor arguements from the rules file and 
 *          initialize the preprocessor's data struct.  This function doesn't
 *          have to exist if it makes sense to parse the args in the init 
 *          function.
 *
 * Arguments: args => argument list
 *
 * Returns: void function
 *
 */
UnifiedData *ParseUnifiedArgs(char *args, char *default_filename)
{
    UnifiedData *tmp;
    int limit = 0;

    tmp = (UnifiedData *)calloc(sizeof(UnifiedData), sizeof(char));

    if(tmp == NULL)
    {
        FatalError("Unable to allocate Unified Data struct!\n");
    }

    DebugMessage(DEBUG_PLUGIN, "Args: %s\n", args);

    if(args != NULL)
    {
        char **toks;
        int num_toks;
        int i = 0;
        toks = mSplit(args, ",", 31, &num_toks, '\\');
        for(i = 0; i < num_toks; ++i)
        {
            char **stoks;
            int num_stoks;
            char *index = toks[i];
            while(isspace((int)*index))
                ++index;
          

            stoks = mSplit(index, " ", 2, &num_stoks, 0);
            
            if(strcasecmp("filename", stoks[0]) == 0)
            {
                if(num_stoks > 1 && tmp->filename == NULL)
                    tmp->filename = strdup(stoks[1]);
                else
                    LogMessage("Argument Error in %s(%i): %s\n",
                            file_name, file_line, index);
            }
            if(strcasecmp("limit", stoks[0]) == 0)
            {
                if(num_stoks > 1 && limit == 0)
                {
                    limit = atoi(stoks[1]);
                }
                else
                {
                    LogMessage("Argument Error in %s(%i): %s\n",
                            file_name, file_line, index);
                }
            }
            do
                free(stoks[--num_stoks]);
            while(num_stoks);
        }
        do
            free(toks[--num_toks]);
        while(num_toks);
    }

    if(tmp->filename == NULL)
        tmp->filename = strdup(default_filename);
    
    LogMessage("limit == %i\n", limit);

    if(limit <= 0)
    {
        limit = 128;
    }
    if(limit > 512)
    {
        LogMessage("spo_unified => Lowering limit of %iMB to 512MB\n");
        limit = 512;
    }
    
    tmp->limit = limit << 20;

    return tmp;
}


/*
 * Function: UnifiedInitLogFile()
 *
 * Purpose: Initialize the unified log file header
 *
 * Arguments: data => pointer to the plugin's reference data struct 
 *
 * Returns: void function
 */
void UnifiedInitLogFile(UnifiedData *data)
{
    time_t curr_time;      /* place to stick the clock data */
    char logdir[STD_BUF];
    int value;
    UnifiedLogFileHeader hdr;

    bzero(logdir, STD_BUF);
    curr_time = time(NULL);

    if(data == NULL)
    {
        FatalError("Can't get unified plugin context, that's bad\n");
    }

    if(*(data->filename) == '/')
        value = snprintf(logdir, STD_BUF, "%s%s.%lu", 
                chrootdir == NULL ? "" : chrootdir, data->filename, 
                curr_time);
    else
        value = snprintf(logdir, STD_BUF, "%s%s/%s.%lu",
                chrootdir == NULL ? "" : chrootdir, pv.log_dir,  
                data->filename, curr_time);

    if(value == -1)
    {
        FatalError("ERROR: log file logging path and file name are "
                "too long, aborting!\n");
    }

    printf("Opening %s\n", logdir);

    if((data->stream = fopen(logdir, "w")) == NULL)
    {
        FatalError("UnifiedInitLogFile(%s): %s\n", logdir, strerror(errno));
    }

    /* write the log file header */
    hdr.magic = LOG_MAGIC;
    hdr.version_major = SNORT_VERSION_MAJOR;
    hdr.version_minor = SNORT_VERSION_MINOR;
    hdr.timezone = thiszone;
    hdr.snaplen = snaplen;
    hdr.sigfigs = 0;
    hdr.linktype = datalinks[0];

    if(fwrite((char *)&hdr, sizeof(hdr), 1, data->stream) != 1)
    {
        FatalError("UnifiedLogInit(): %s", strerror(errno));
    }

    fflush(data->stream);

    return;
}


/*
 * Function: UnifiedInitAlertFile()
 *
 * Purpose: Initialize the unified log alert file
 *
 * Arguments: data => pointer to the plugin's reference data struct 
 *
 * Returns: void function
 */
void UnifiedInitAlertFile(UnifiedData *data)
{
    time_t curr_time;      /* place to stick the clock data */
    char logdir[STD_BUF];
    int value;
    UnifiedAlertFileHeader hdr;

    bzero(logdir, STD_BUF);
    curr_time = time(NULL);

    if(data->filename[0] == '/')
        value = snprintf(logdir, STD_BUF, "%s%s.%lu", 
                chrootdir == NULL ? "" : chrootdir, data->filename, 
                curr_time);
    else
        value = snprintf(logdir, STD_BUF, "%s%s/%s.%lu",
                chrootdir == NULL ? "" : chrootdir, pv.log_dir, 
                data->filename, curr_time);

    if(value == -1)
    {
        FatalError("ERROR: log file logging path and file name are "
                "too long, aborting!\n");
    }

#ifdef DEBUG
    printf("Opening %s\n", logdir);
#endif

    if((data->stream = fopen(logdir, "w+")) == NULL)
    {
        FatalError("UnifiedInitAlertFile(%s): %s\n", logdir, strerror(errno));
    }

    hdr.magic = ALERT_MAGIC;
    hdr.version_major = 1;
    hdr.version_minor = 81;
    hdr.timezone = thiszone;

    if(fwrite((char *)&hdr, sizeof(hdr), 1, data->stream) != 1)
    {
        FatalError("UnifiedAlertInit(): %s\n", strerror(errno));
    }
        
    fflush(data->stream);

    return;
}


/**  R U N   T I M E   F U N C T I O N S  ************************************/

/*
 * Function: LogUnified(Packet *, char *msg, void *arg)
 *
 * Purpose: Perform the preprocessor's intended function.  This can be
 *          simple (statistics collection) or complex (IP defragmentation)
 *          as you like.  Try not to destroy the performance of the whole
 *          system by trying to do too much....
 *
 * Arguments: p => pointer to the current packet data struct 
 *
 * Returns: void function
 */
void LogUnified(Packet *p, char *msg, void *arg, Event *event)
{
    UnifiedLog logheader;
    UnifiedData *data = (UnifiedData *)arg;

    if(event != NULL)
    {
        logheader.event.sig_generator = event->sig_generator;
        logheader.event.sig_id = event->sig_id;
        logheader.event.sig_rev = event->sig_rev;
        logheader.event.classification = event->classification;
        logheader.event.priority = event->priority;
        logheader.event.event_id = event->event_id;
        logheader.event.event_reference = event->event_reference;
        logheader.event.ref_time.tv_sec = event->ref_time.tv_sec;
        logheader.event.ref_time.tv_usec = event->ref_time.tv_usec;

#ifdef DEBUG
        printf("------------\n");
        printf("gen: %lu\n", logheader.event.sig_generator);
        printf("sid: %lu\n", logheader.event.sig_id);
        printf("rev: %lu\n", logheader.event.sig_rev);
        printf("cls: %lu\n", logheader.event.classification);
        printf("pri: %lu\n", logheader.event.priority);
        printf("eid: %lu\n", logheader.event.event_id);
        printf("erf: %lu\n", logheader.event.event_reference);
        printf("sec: %lu\n", logheader.event.ref_time.tv_sec);
        printf("usc: %lu\n", logheader.event.ref_time.tv_usec);
#endif

    }

    if(p)
    {
        logheader.flags = p->packet_flags;

        /* 
         * this will have to be fixed when we transition to the pa_engine
         * code (p->pkth is libpcap specific)
         */ 
        memcpy(&logheader.pkth, p->pkth, sizeof(SnortPktHeader));
    }
    else
    {
        logheader.flags = 0;
        logheader.pkth.ts.tv_sec = 0;
        logheader.pkth.ts.tv_usec = 0;
        logheader.pkth.caplen = 0;
        logheader.pkth.pktlen = 0;
    }
    
    if((data->current + sizeof(UnifiedLog) + logheader.pkth.caplen) > 
            data->limit)
        SpoUnifiedLogRotateFile(data);
    
    fwrite((char*)&logheader, sizeof(UnifiedLog), 1, data->stream);

    if(p)
    {
        fwrite((char*)p->pkt, p->pkth->caplen, 1, data->stream);
    }
    fflush(data->stream);

    data->current += sizeof(UnifiedLog) + logheader.pkth.caplen;
    
}

void AlertUnified(Packet *p, char *msg, void *arg, Event *event)
{
    UnifiedData *data = (UnifiedData *)arg;
    UnifiedAlert alertdata;

    bzero(&alertdata, sizeof(alertdata));

    if(event != NULL)
    {
        alertdata.event.sig_generator = event->sig_generator;
        alertdata.event.sig_id = event->sig_id;
        alertdata.event.sig_rev = event->sig_rev;
        alertdata.event.classification = event->classification;
        alertdata.event.priority = event->priority;
        alertdata.event.event_id = event->event_id;
        alertdata.event.event_reference = event->event_reference;
        alertdata.event.ref_time.tv_sec = event->ref_time.tv_sec;
        alertdata.event.ref_time.tv_usec = event->ref_time.tv_usec;

    }

    if(p)
    {
        alertdata.ts.tv_sec = p->pkth->ts.tv_sec;
        alertdata.ts.tv_usec = p->pkth->ts.tv_usec;

        if(p->iph != NULL)
        {
            alertdata.sip = p->iph->ip_src.s_addr;
            alertdata.dip = p->iph->ip_dst.s_addr;
            if(p->iph->ip_proto == IPPROTO_ICMP)
            {
                if(p->icmph != NULL)
                {
                    alertdata.sp = p->icmph->type;
                    alertdata.dp = p->icmph->code;
                }
            }
            else
            {
                alertdata.sp = p->sp;
                alertdata.dp = p->dp;
            }
            alertdata.protocol = p->iph->ip_proto;
            alertdata.flags = p->packet_flags;
        }
    }
    
    if((data->current + sizeof(UnifiedAlert)) > data->limit)
        SpoUnifiedLogRotateFile(data);

    fwrite((char *)&alertdata, sizeof(UnifiedAlert), 1, data->stream);
    fflush(data->stream);

    data->current += sizeof(UnifiedAlert);
}


/*
 * Function: SpoUnifiedCleanExitFunc()
 *
 * Purpose: Cleanup at exit time
 *
 * Arguments: signal => signal that caused this event
 *            arg => data ptr to reference this plugin's data
 *
 * Returns: void function
 */
void SpoUnifiedCleanExitFunc(int signal, void *arg)
{
    /* cast the arg pointer to the proper type */
    UnifiedData *data = (UnifiedData *)arg;

    DebugMessage(DEBUG_FLOW, "SpoUnifiedCleanExitFunc\n");

    fclose(data->stream);

    /* free up initialized memory */
    free(data->filename);
    free(data);
}



/*
 * Function: SpoUnifiedRestartFunc()
 *
 * Purpose: For restarts (SIGHUP usually) clean up structs that need it
 *
 * Arguments: signal => signal that caused this event
 *            arg => data ptr to reference this plugin's data
 *
 * Returns: void function
 */
void SpoUnifiedRestartFunc(int signal, void *arg)
{
    UnifiedData *data = (UnifiedData *)arg;

    DebugMessage(DEBUG_FLOW, "SpoUnifiedRestartFunc\n");

    fclose(data->stream);
    free(data->filename);
    free(data);
}


void SpoUnifiedLogRotateFile(UnifiedData *data)
{

    fclose(data->stream);
    data->current = 0;
    UnifiedInitLogFile(data);
}


void SpoUnifiedAlertRotateFile(UnifiedData *data)
{

    fclose(data->stream);
    data->current = 0;
    UnifiedInitAlertFile(data);
}

