/*
 * ------------------------------------------------------------------------
 * SRAM Emulation
 *
 * (C) 2004  Lightmaze Solutions AG
 *   Author: Jochen Karrer
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 * ------------------------------------------------------------------------
 *
 */
#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include <bus.h>
#include <configfile.h>

typedef struct SRam {
	BusDevice bdev;
	uint8_t *host_mem;
	uint32_t  size;
	uint32_t flags;
} SRam;

static void
SRam_Map(void *module_owner,uint32_t base,uint32_t mapsize,uint32_t flags) 
{
	SRam *sram = module_owner;
	flags &= MEM_FLAG_READABLE|MEM_FLAG_WRITABLE;
	Mem_MapRange(base,sram->host_mem,sram->size,mapsize,flags);
}

static void
SRam_UnMap(void *module_owner,uint32_t base,uint32_t mapsize) {
	Mem_UnMapRange(base,mapsize); 
}

static uint32_t 
parse_memsize (char *str)
{
	uint32_t size;
	char c;
	if(sscanf(str,"%d",&size)!=1) {
		return 0;
	}
	if(sscanf(str,"%d%c",&size,&c)==1) {
		return size;
	}
	switch(tolower(c)) {
		case 'm':		
			return size*1024*1024;
		case 'k':		
			return size*1024;
	}
	return 0;
}

/*
 * --------------------
 * DRAM New
 * --------------------
 */
BusDevice *
SRam_New(char *sram_name) {
	char *sizestr;
	uint32_t size=0;
	SRam *sram;
	sizestr=Config_ReadVar(sram_name,"size");
	if(sizestr) {
		size=parse_memsize(sizestr);
		if(size==0) {
			fprintf(stderr,"SRAM bank \"%s\" not present\n",sram_name);
			return NULL;
		}
	} else {
		fprintf(stderr,"SRAM bank \"%s\" not present\n",sram_name);
		return NULL;
	}
	sram = malloc(sizeof(SRam));
	if(!sram) {
		fprintf(stderr,"Out of memory\n");
		exit(556);
	}
	memset(sram,0,sizeof(SRam));
	sram->host_mem=malloc(size);
	if(!sram->host_mem) {
		fprintf(stderr,"Can not allocate memory(%d bytes) for SRAM emulation\n",size);
		exit(546);
	}
	memset(sram->host_mem,0xff,size);
	sram->size=size;
	sram->bdev.first_mapping=NULL;
	sram->bdev.Map=SRam_Map;
	sram->bdev.UnMap=SRam_UnMap;
	sram->bdev.owner=sram;
	sram->bdev.hw_flags=MEM_FLAG_WRITABLE|MEM_FLAG_READABLE;
	fprintf(stderr,"SRAM bank \"%s\" with  size %ukB\n",sram_name,size/1024);
	return &sram->bdev;
}
