/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#include "player.h"
#include "sock.h"
#include "other.h"
#include "util.h"
#include "sds/sds.h"

int sock_unix_test (char *s);

sock_struct *
sock_parse (char *data)
{
  char str[SIZE_BUFFER];
  int port = 0;
  int type;
  sock_struct *sock;

  if (!data)
    return NULL;

  /* Standard UNIX connection */
  if (!strcmp (data, "unix"))
    {
      int len = 0;
      DIR *d;
      struct dirent *dir;

      if (!(d = opendir (get_tmp_dir ())))
	{
	  msg_error ("I can't open %s directory!", get_tmp_dir ());
	  return NULL;
	}

      while ((dir = readdir (d)))
	{

	  if (!strncmp (dir->d_name, "sds-", 4)
	      && !strcmp (dir->d_name + strlen (dir->d_name) - 5, ".sock"))
	    {

	      snprintf (str, SIZE_BUFFER, "%s/%s", get_tmp_dir (),
			dir->d_name);

	      if (!sock_unix_test (str))
		{
		  len++;
		  break;
		}
	    }
	}

      if (!len)
	{
	  msg_error ("No sds found!");
	  return NULL;
	}

      type = SOCKET_UNIX;
    }

  /* Standard TCP connection */
  else if (!strcmp (data, "tcp"))
    {
      type = SOCKET_TCP;
      snprintf (str, SIZE_BUFFER, "127.0.0.1");
      port = SDS_PORT;
    }

  /* Check Unix connection */
  else if (!strncmp (data, "unix://", 7))
    {
      type = SOCKET_UNIX;
      strncpy (str, data + 7, SIZE_BUFFER);
    }

  /* Check Tcp connection */
  else if (!strncmp (data, "tcp://", 6))
    {
      strncpy (str, data + 6, SIZE_BUFFER);

      for (type = 0; type < strlen (str); type++)
	{
	  if (str[type] == ':')
	    {
	      if (type + 1 < strlen (str))
		port = atoi (&str[type + 1]);
	      str[type] = 0;
	    }
	}

      if (!port)
	port = SDS_PORT;

      type = SOCKET_TCP;
    }

  else
    {
      msg_error (_("Error in audiodevice parameter: %s"), data);
      return NULL;
    }

  if (!(sock = (sock_struct *) malloc (sizeof (sock))))
    fatal (_("Error: memory."));

  if (!(sock->server = strdup (str)))
    fatal (_("Error: memory."));
  sock->port = port;
  sock->type = type;

  return sock;
}

int
sock_connect (sock_struct * data)
{
  int fd;

  if (data->type == SOCKET_TCP)
    {
      struct sockaddr_in sock;
      struct hostent *hp;

      if ((fd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
	{
	  msg_error (_("Socket error."));
	  return 0;
	}

      if (!(hp = gethostbyname (data->server)))
	{
	  msg_error (_("Error: unknown host."));
	  return 0;
	}

      memset ((void *) &sock, 0, sizeof (sock));
      sock.sin_family = AF_INET;
      sock.sin_addr.s_addr = ((struct in_addr *) (hp->h_addr))->s_addr;
      sock.sin_port = htons (data->port);

      if (connect (fd, (struct sockaddr *) &sock, sizeof (sock)) < 0)
	{
	  msg_error (_("Connect error."));
	  return 0;
	}
    }

  /* UNIX SOCKET */
  else
    {
      struct sockaddr_un saddr;

      if ((fd = socket (AF_UNIX, SOCK_STREAM, 0)) < 0)
	{
	  msg_error (_("Socket error."));
	  return 0;
	}

      saddr.sun_family = AF_UNIX;
      strncpy (saddr.sun_path, data->server, 108);

      if (connect (fd, (struct sockaddr *) &saddr, sizeof (saddr)) == -1)
	{
	  close (fd);
	  msg_error (_("Connect error."));
	  return 0;
	}
    }

  return fd;
}

void
sock_free (sock_struct * data)
{
  if (!data)
    return;

  if (data->server)
    free (data->server);

  free (data);
}

int
sock_unix_test (char *s)
{
  int fd;
  struct sockaddr_un saddr;

  msg (_("Socket test: %s"), s);

  if ((fd = socket (AF_UNIX, SOCK_STREAM, 0)) < 0)
    {
      msg (_("Error test: Socket error."));
      return 1;
    }

  saddr.sun_family = AF_UNIX;
  strncpy (saddr.sun_path, s, 108);

  if (connect (fd, (struct sockaddr *) &saddr, sizeof (saddr)) == -1)
    {
      close (fd);
      msg (_("Error test: Connect error."));
      return 1;
    }

  write (fd, SDS_TEST_CONNECT "\n", strlen(SDS_TEST_CONNECT)+1);

  close (fd);
  return 0;
}

/* EOF */
