/*  Sonicmail - POP3 email notifier
 *  Copyright (C) 2000  Cameron Edwards
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "sonicmail-mailcheck.h"


void *sonicmail_mailcheck_thread(void *data)
{
   SonicmailVars *vars;
   int secs = 3;
   char servername[256];
   char username[256];
   char passwd[256];
   char tmp[256];
   
   enum SonicmailState new_state;
   int new_num_mails;
   
   
   vars = (SonicmailVars *)data;
   
   vars->mailcheck_thread_pid = getpid();
   SM_DEBUG_INT("Mailcheck thread PID =", getpid());
   
   
   /* Install a signal handler so that sleep() may be aborted below.
      This will be useful for when the mailcheck period is changed
      by the user: */
   signal(SIGUSR2, sig_handler);
   
   while(1) {
      
      /* Get the latest preferences exclusively: */
      pthread_mutex_lock(&(vars->mut));
         secs = vars->mins * 60 + vars->secs;
         strcpy(servername, vars->servername);
         strcpy(username, vars->username);
         strcpy(passwd, vars->pword);
      pthread_mutex_unlock(&(vars->mut));
      
      /* Now check for mail: */
      sonicmail_mailcheck_go(servername, username, passwd, &new_state, &new_num_mails);
      
      /* Now update the sonicmail state: */
      pthread_mutex_lock(&(vars->mut));
         vars->state = new_state;
         vars->prev_num_mails = vars->curr_num_mails;
         vars->curr_num_mails = new_num_mails;
      pthread_mutex_unlock(&(vars->mut));
      
      /* Refresh the pixmap if need be, and the tooltip: */
      gdk_threads_enter();
         
       sonicmail_pixmap_refresh(vars);   
       pthread_mutex_lock(&(vars->mut));
       
        if(vars->state == Sonicmail_Offline) {
           gtk_tooltips_set_tip(GTK_TOOLTIPS (vars->tooltips), vars->applet, "Offline", "");
        }
        else {
           switch(new_num_mails) {
              case 0:  gtk_tooltips_set_tip(GTK_TOOLTIPS (vars->tooltips), vars->applet, "No new email", "");
                       break;
              case 1: gtk_tooltips_set_tip(GTK_TOOLTIPS (vars->tooltips), vars->applet, "1 new email", "");
                       break;
              default: sprintf(tmp, "%d new emails", new_num_mails);
                       gtk_tooltips_set_tip(GTK_TOOLTIPS (vars->tooltips), vars->applet, tmp, "");
           }
        }
       
       pthread_mutex_unlock(&(vars->mut));
    
      gdk_threads_leave();
      
      /* Play sound if enabled: */
      sonicmail_sound_play(vars);
      
      sleep(secs); /* sleep() will abort if SIGUSR1 is raised */
   }
}


/* Signal handler for SIGUSR1: */
void sig_handler(int signum)
{
   /* Do nothing except reinstall the sig handler: */
   signal(SIGUSR2, sig_handler);
}

void sonicmail_mailcheck_go(char *servername, char *username, char *passwd, enum SonicmailState *new_state, int *new_num_mails)
{
   int sock, len;
   char buff[1024];
   
   SM_DEBUG("\nBeginning to check mail");
   
   /* Default to Sonicmail_Offline state unless explicitly set later: */
   *new_state = Sonicmail_Offline;
   
/* The following code follows RFC1225 Post Office Protocol - Version 3
   as defined by the IETF (see http://www.ietf.org) */

/* State: Disconnected */
   
   SM_DEBUG_STRING("Attempting to connect socket to server", servername);
   if( (sock = socket_connect(servername, POP3_PORT)) == -1 ) {
      SM_DEBUG("socket_connect() failed");
      return;
   }
   
   if( (len = sonicmail_mailcheck_recv(sock, buff, 1024) ) == -1) {
      SM_DEBUG("sonicmail_mailcheck_recv() failed on connect");
      return;
   }
   
/* Successful connection to POP3 server now established (AUTHORIZATION state) */
   SM_DEBUG("Connected to server, proceeding to login...");
   
   /* Proceed to login (send username): */
   sprintf(buff, "USER %s\r\n", username);
   SM_DEBUG_STRING("Sending", buff);
   
   if(socket_send(sock, buff, strlen(buff)) == -1) {
      SM_DEBUG("socket_send() failed at USER");
      socket_close(sock);
      return;
   }
   
   if( (len = sonicmail_mailcheck_recv(sock, buff, 1024) ) == -1) {
      SM_DEBUG("sonicmail_mailcheck_recv() failed at USER");
      return;
   }
   
   SM_DEBUG("Username OK proceeding to send PASS");
   
   /* Send Password: */
   sprintf(buff, "PASS %s\r\n", passwd);
   
   if(socket_send(sock, buff, strlen(buff)) == -1) {
      SM_DEBUG("socket_send() failed at PASS");
      socket_close(sock);
      return;
   }

   if( (len = sonicmail_mailcheck_recv(sock, buff, 1024) ) == -1) {
      SM_DEBUG("sonicmail_mailcheck_recv() failed at PASS");
      return;
   }
   
   
/* Successfully logged into POP3 server (TRANSACTION state) */
   SM_DEBUG("Logged in successfully");
   
   /* Now send the STAT command: */
   sprintf(buff, "STAT\r\n");
   SM_DEBUG_STRING("Sending", buff);
   
   if(socket_send(sock, buff, strlen(buff)) == -1) {
      SM_DEBUG("socket_send() failed at STAT");
      socket_close(sock);
      return;
   }
      
   if( (len = sonicmail_mailcheck_recv(sock, buff, 1024) ) == -1) {
      SM_DEBUG("sonicmail_mailcheck_recv() failed at STAT");
      return;
   }
   
   SM_DEBUG("Disconnecting");
   
   /* Ok, we're done now so disconnect from the server: */
   socket_send(sock, "QUIT\r\n", strlen("QUIT\r\n"));
   socket_close(sock);     
   
   /* Process the drop listing: */
   if(sscanf(buff, "+OK %d %*d", new_num_mails) != 1) return;
   
   SM_DEBUG_INT("Number of mails on server:", *new_num_mails);
   
   if(*new_num_mails == 0)
      *new_state = Sonicmail_Empty;
   else 
      *new_state = Sonicmail_Full;
   
   return;
}

int sonicmail_mailcheck_recv(int sock, char *buff, int size)
{ 
   int len;
   
   if( (len = socket_receive(sock, buff, size)) == -1) {
      socket_send(sock, "QUIT\r\n", strlen("QUIT\r\n"));
      socket_close(sock);
      return(-1);
   }
   
   if(strncmp("+OK", buff, 3) != 0) {
      socket_send(sock, "QUIT\r\n", strlen("QUIT\r\n"));
      socket_close(sock);
      return(-1);
   }
   
   return(len);
}
