<?php
/* $Horde: forwards/lib/Driver.php,v 1.3.2.4 2003/01/20 05:19:58 ericr Exp $
 *
 * Copyright 2001-2003 Mike Cochrane <mike@graftonhall.co.nz>
 *
 * See the enclosed file LICENSE for license information (BSD). If you
 * did not receive this file, see http://www.horde.org/bsdl.php.
 *
 * Forwards_Driver:: defines an API for implementing forwarding backends
 * for Forwards.
 *
 * @author  Mike Cochrane <mike@graftonhall.co.nz>
 * @author  Eric Rostetter <eric.rostter@physics.utexas.edu>
 * @version $Revision: 1.3.2.4 $
 * @since   Forwards 2.1
 * @package forwards
 */

class Forwards_Driver {
    /**
     * Attempts to return a concrete Forwards_Driver instance based on $driver.
     *
     * @param string    $driver     The type of concrete Forwards_Driver subclass
     *                              to return.  The is based on the forwards
     *                              driver ($driver).  The code is dynamically
     *                              included.
     *
     * @param array     $params     (optional) A hash containing any additional
     *                              configuration or connection parameters a
     *                              subclass might need.
     *
     * @return mixed    The newly created concrete Forwards_Driver instance, or
     *                  false on an error.
     */
    function &factory($driver, $params = array())
    {
        $driver = strtolower(basename($driver));
        include_once dirname(__FILE__) . '/Driver/' . $driver . '.php';
        $class = 'Forwards_Driver_' . $driver;
        if (class_exists($class)) {
            return new $class($params);
        } else {
            return false;
        }
    }

    /**
     * Attempts to return a reference to a concrete Forwards_Driver instance
     * based on $driver.  It will only create a new instance if no
     * Forwards_Driver instance with the same parameters currently exists.
     *
     * This should be used if multiple storage sources are required.
     *
     * This method must be invoked as: $var = &Forwards_Driver::singleton()
     *
     * @param string    $driver     The type of concrete Forwards_Driver subclass
     *                              to return.  The is based on the forwards
     *                              driver ($driver).  The code is dynamically
     *                              included.
     *
     * @param array     $params     (optional) A hash containing any additional
     *                              configuration or connection parameters a
     *                              subclass might need.
     *
     * @return mixed    The created concrete Forwards_Driver instance, or false
     *                  on error.
     */
    function &singleton($driver, $params = array())
    {
        static $instances;

        if (!isset($instances)) $instances = array();

        $signature = md5(strtolower($driver) . implode('][', $params));
        if (!isset($instances[$signature])) {
            $instances[$signature] = &Forwards_Driver::factory($driver, $params);
        }

        return $instances[$signature];
    }

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string    $user       The username of the user to enable forwarding for.
     *
     * @param string    $realm      The realm of the user to enable forwarding for.
     *
     * @param string    $target     The email address that mail should be forwarded to.
     *
     * @param optional boolean $keeplocal A flag that, when true, causes a copy of
     *                                    forwarded email to be keeps in the local mailbox.
     *
     * @return boolean  Returns true on success, false on error.
     */
    function enableForwarding($user, $realm = 'default', $pass = '', $target, $keeplocal = false)
    {
        return false;
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string    $user       The username of the user to enable forwarding for.
     *
     * @param string    $realm      The realm of the user to enable forwarding for.
     *
     * @return boolean  Returns true on success, false on error.
     */
    function disableForwarding($user, $realm = 'default', $pass = '')
    {
        return false;
    }

    /**
     * Retrieves current state of mail redirection
     *
     * @param string    $user       The username of the user to enable forwarding for.
     *
     * @param string    $realm      The realm of the user to enable forwarding for.
     *
     * @return mixed    Returns true if forwarding is enabled for the user or false if
     *                  forwarding is currently disabled.
     */
    function isEnabledForwarding($user, $realm = 'default', $password)
    {
        return false;
    }

    /**
     * Retrieves true if a local copy of forwarded messages is being kept
     *
     * @param string    $user       The username of the user to check.
     *
     * @param string    $realm      The realm of the user to check.
     *
     * @return boolean    Returns true if retian copy is enabled for the user else false.
     *
     */
    function isKeepLocal($user, $realm = 'default', $password)
    {
        return false;
    }

    /**
     * Retrieves current target of mail redirection
     *
     * @param string    $user       The username of the user to enable forwarding for.
     *
     * @param string    $realm      The realm of the user to enable forwarding for.
     *
     * @return string   Returns a string containing the current forwarding mail address or
     *                  false on error.
     */
    function currentTarget($user, $realm = 'default', $password)
    {
        return false;
    }

    /**
     * Parse an email address list and return it in a known standard form.
     * This will attempt to add the domain (realm) to unqualified addresses
     * if the realm is non-blank and not 'default'.
     *
     * @param   $user       The email address.
     * @param   $realm      The domain/realm to add if none is present.
     * @return  string      The email address(es) on success, false on error.
     */
    function _make_email_address($user, $realm)
    {
        $domain = ($realm != 'default') ? $realm : '';
        $email = '';

        require_once 'Mail/RFC822.php';
        $parsed_email = Mail_RFC822::parseAddressList($user, $domain, FALSE);
        if (is_array($parsed_email) && count($parsed_email) > 0) {
            for ($i=0; $i < count($parsed_email); $i++) {
               $email .= !empty($email) ? ',' : '';
               if (is_object($parsed_email[$i])) {
                 $email .= $parsed_email[$i]->mailbox;
                 $email .= !empty($parsed_email[$i]->host)
                        ? '@' . $parsed_email[$i]->host
                        : '';
              } else {
                 $email .= $parsed_email[$i];
              }
            }
        } else {
            $this->err_str = _("Can't parse your email address");
            $email = false;
        }

        return $email;
    }

}
